/***
 * TEXTURE MAPPED 3D MESH VISUALISATION
 * by Norman Feske
 ***/

#include <GL/gl.h>
#include <mesh.h>
#include <image.h>



/*** LIST OF PUBLIC VARIABLES ***/
char *tcl_symbols = "long vis_points, long vis_edges, long vis_polys, float depth_scale,\
                     float vis_alpha, float vis_beta, float vis_gamma, string vis_texture_fname";
char *title = "Texture Mapping";

long vis_points=1;
long vis_edges=1;
long vis_polys=1;

float vis_alpha=0.0;
float vis_beta=0.0;
float vis_gamma=0.0;

float depth_scale=0.25;
char vis_texture_fname[256];

struct texture {
	long w,h;
	char *pixels;
};

typedef struct texture texture;


struct visdata {
	long w,h;
	long num_points;	/* number of mesh points */
	mesh_point *points;	/* pointer to point array */
	long num_polys;		/* number of polygons */
	mesh_poly *polys;	/* pointer to polygon array */
	texture *tex;
};

typedef struct visdata visdata;


long new_tex_flag=0;


int roundup(int n)
{
  int val = 1;
  while (val < n) val <<= 1;
  return val;
}

texture *init_texture(image *srcimg) {
	long x,y;
	unsigned char *dst,*src;

	texture *result=(texture *)malloc(sizeof(texture));

	if (srcimg->w & (srcimg->w - 1)) result->w = roundup(srcimg->w);
	else result->w = srcimg->w;
 
	if (srcimg->h & (srcimg->h - 1)) result->h = roundup(srcimg->h);
	else result->h = srcimg->h;

	result->pixels=(char *)malloc(result->w*result->h*4);

	if (!result->pixels) {
		printf("vis: malloc failed during texture initialisation!\n");
		return (void *)0;
	}
	
    memset(result->pixels, 0, result->w*result->h*4);
	
	src=srcimg->pixels;
	dst=result->pixels;
    for (y=0;y<srcimg->h;y++) {
		dst=result->pixels + result->w*y*4;
		for (x=0;x<srcimg->w;x++) {
			*(dst++)=*(src++);
			*(dst++)=*(src++);	
			*(dst++)=*(src++);
			dst++;
			src++;
	  	}
   	}
	
	return result;
}



/* needed parameters: num_hor,num_ver */
visdata *Create_Visualisation(mesh *m,long width,long height) {

	visdata *result;
	mesh_point *srcpt, *dstpt;
	mesh_poly *srcpl, *dstpl;
	long i,j;
	image *tmp;
	
	if (!m) return (void *)0;
	result = (visdata *)malloc(sizeof(visdata));
	result->w=width;
	result->h=height;
	result->num_points=m->num_points;
	result->num_polys=m->num_polys;
	result->points=(mesh_point *)malloc(m->num_points*sizeof(mesh_point));
	result->polys=(mesh_poly *)malloc(m->num_polys*sizeof(mesh_poly));
	
	printf("loading texture: %s\n",vis_texture_fname);
	tmp=Load_RGB_Image(vis_texture_fname,IMAGE_RGBA_CHAR);
	if (!tmp) {
		printf("failed (no texture loaded)\n");
	} else {
		result->tex=init_texture(tmp);
		Destroy_Image(tmp);
	}
	
	srcpt=m->points;
	dstpt=result->points;
	for (i=0;i<m->num_points;i++) {
		dstpt->x =srcpt->x;
		dstpt->y =srcpt->y;
		dstpt->z =srcpt->z;
		dstpt->nx=srcpt->nx;
		dstpt->ny=srcpt->ny;
		dstpt->nz=srcpt->nz;
		srcpt++;dstpt++;
	}

	srcpl=m->polys;
	dstpl=result->polys;
	for (i=0;i<m->num_polys;i++) {
		dstpl->num_points = srcpl->num_points;
		dstpl->nx=srcpl->nx;
		dstpl->ny=srcpl->ny;
		dstpl->nz=srcpl->nz;
		for (j=0;j<srcpl->num_points;j++) {
			dstpl->point_idx[j]=srcpl->point_idx[j];
		}
		srcpl++;dstpl++;
	}
	new_tex_flag=1;
	return result;
}



float get_min_x(visdata *v) {
	long i;
	float result=99999.9;
	mesh_point *p=v->points;
	for (i=0;i<v->num_points;i++) {
		if (p->x<result) result=p->x;
		p++;
	}
	return result;
}


float get_max_x(visdata *v) {
	long i;
	float result=-99999.9;
	mesh_point *p=v->points;
	for (i=0;i<v->num_points;i++) {
		if (p->x>result) result=p->x;
		p++;
	}
	return result;
}


float get_min_y(visdata *v) {
	long i;
	float result=99999.9;
	mesh_point *p=v->points;
	for (i=0;i<v->num_points;i++) {
		if (p->y<result) result=p->y;
		p++;
	}
	return result;
}


float get_max_y(visdata *v) {
	long i;
	float result=-99999.9;
	mesh_point *p=v->points;
	for (i=0;i<v->num_points;i++) {
		if (p->y>result) result=p->y;
		p++;
	}
	return result;
}


float get_min_z(visdata *v) {
	long i;
	float result=99999.9;
	mesh_point *p=v->points;
	for (i=0;i<v->num_points;i++) {
		if ((p->z<result) && (p->z>0.0)) result=p->z;
		p++;
	}
	return result;
}


float get_max_z(visdata *v) {
	long i;
	float result=-99999.9;
	mesh_point *p=v->points;
	for (i=0;i<v->num_points;i++) {
		if (p->z>result) result=p->z;
		p++;
	}
	return result;
}




void Draw_Visualisation(visdata *v) {
	long i;
	mesh_point *point;
	mesh_poly *poly;
	mesh_point *p1,*p2,*p3;
	float txscale,tyscale;
	GLfloat xmin,xmax,zmin,zmax,ymin,ymax,w,h,d,xcenter,ycenter,zcenter;

	if (!v) {
		printf("visdata is zero.\n");
		return;
	}
	if (!v->points) {
		printf("visdata_points is zero.\n");
		return;
	}
	if (!v->num_points) {
		printf("visdata->num_points is zero.\n");
		return;
	}
	
	if (!v->polys) {
		printf("visdata->polys is zero.\n");
		return;	
	}

	if (!v->num_polys) {
		printf("visdata->num_polys is zero.\n");
		return;
	}
	
	if (!v->tex) {
		printf("visdata->tex is zero.\n");
		return;	
	}
	

	if (new_tex_flag) {	
		glTexImage2D(GL_TEXTURE_2D, 0, 3, v->tex->w, v->tex->h, 0, GL_RGBA, GL_UNSIGNED_BYTE, v->tex->pixels);
		glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);		   
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,GL_NEAREST);
		glDisable(GL_TEXTURE_GEN_S);
		glDisable(GL_TEXTURE_GEN_T);
		new_tex_flag=0;
	}

	xmin=get_min_x(v);
	xmax=get_max_x(v);
	ymin=get_min_y(v);
	ymax=get_max_y(v);
	zmin=get_min_z(v);
	zmax=get_max_z(v);
	w=xmax-xmin;
	h=ymax-ymin;
	d=zmax-zmin;
	xcenter=(xmin+xmax)/2;
	ycenter=(ymin+ymax)/2;
	zcenter=(zmin+zmax)/2;
	
    glPushMatrix ();
	glTranslatef(v->w/2.0,v->h/2.0,zcenter*depth_scale);
    glRotatef (vis_gamma, 0.0, 0.0, 1.0);
    glRotatef (vis_beta, 0.0, 1.0, 0.0);
    glRotatef (vis_alpha, 1.0, 0.0, 0.0);
	glTranslatef(-v->w/2.0,-v->h/2.0,-zcenter*depth_scale);

	txscale=1.0/(float)v->tex->w;
	tyscale=1.0/(float)v->tex->h;

	poly=v->polys;
	for (i=0;i<v->num_polys;i++) {
	
		p1=v->points + poly->point_idx[0];
		p2=v->points + poly->point_idx[1];
		p3=v->points + poly->point_idx[2];	

		glEnable(GL_TEXTURE_2D);  
		glColor3f(1.0,1.0,1.0);
		
		if (vis_polys) {	
			glBegin(GL_TRIANGLES);
			
//			glColor4f(0.0,0.0,depth_scale*p1->z/256.0,0.0);
			glTexCoord2f(txscale*p1->x,tyscale*p1->y);
//			glTexCoord2f(0.0,0.0);
			glVertex3f(p1->x,p1->y,p1->z*depth_scale);
			
//			glColor4f(0.0,0.0,depth_scale*p2->z/256.0,0.0);
			glTexCoord2f(txscale*p2->x,tyscale*p2->y);
//			glTexCoord2f(0.25,0.0);
			glVertex3f(p2->x,p2->y,p2->z*depth_scale);
			
//			glColor4f(0.0,0.0,depth_scale*p3->z/256.0,0.0);
			glTexCoord2f(txscale*p3->x,tyscale*p3->y);
//			glTexCoord2f(0.25,0.25);
			glVertex3f(p3->x,p3->y,p3->z*depth_scale);		
			glEnd();
		}

		glDisable(GL_TEXTURE_2D);  
		
		if (vis_edges) {
			glBegin(GL_LINES);
			glColor3f(0.8,0.7,0.7);
			glVertex3f(p1->x,p1->y,p1->z*depth_scale);
			glVertex3f(p2->x,p2->y,p2->z*depth_scale);
			glVertex3f(p2->x,p2->y,p2->z*depth_scale);
			glVertex3f(p3->x,p3->y,p3->z*depth_scale);
			glVertex3f(p3->x,p3->y,p3->z*depth_scale);
			glVertex3f(p1->x,p1->y,p1->z*depth_scale);
			glEnd();
		}
		
		poly++;
	}	
	
	if (vis_points) {
		glPointSize(3.0);
		point=v->points;
		for (i=0;i<v->num_points;i++) {
			glBegin(GL_POINTS);
			glColor3f(1.0,1.0,1.0);
			glVertex3f(point->x,point->y,point->z*depth_scale);
			glEnd();
			point++;
		}
	}
	
    glPopMatrix ();

}



void Destroy_Visualisation(visdata *v) {
	if (!v) return;
	if (v->points) free(v->points);
	if (v->polys) free(v->polys);
	if (v->tex) {
		if (v->tex->pixels) free(v->tex->pixels);
		free(v->tex);
	}
	free(v);
}
