/***
 * LEARNING VECTOR QUANTISATION MODULE
 * by Norman Feske
 ***/

#include <math.h>
#include <GL/gl.h>
#include <distrib.h>
#include <mesh.h>


/*** LIST OF PUBLIC VARIABLES ***/
char *tcl_symbols = "long num_hor,long num_ver,\
                     long disp_nodes,long disp_connections,float adaption,long iterations";
char *title = "Learning Vector Quantisation";

long num_hor=20,num_ver=20;
long disp_nodes=1,disp_connections=1,iterations=1000;
float adaption=0.2;


struct node {
	float	x;
	float	y;
	float	weight;
};

typedef struct node node;

struct connection {
	long	from;
	long	to;
};

typedef struct connection connection;

struct network {
	long		num_nodes;
	long		num_connections;
	node		*nodes;
	connection	*connections;
};

typedef struct network network;




/* needed parameters: num_hor,num_ver */
network *Create_Network(long init_w,long init_h) {
	long i,j;
	float hor_step=(float)init_w/(num_hor-0.5);
	float ver_step=(float)init_h/(num_ver-1);
	float h_offset;
	node *n;
	connection *c;
	network	*result=(network *)malloc(sizeof(network));
	result->num_nodes=num_hor*num_ver;
	result->num_connections=num_ver*(num_hor-1) + num_hor*(num_ver-1) + (num_hor-1)*(num_ver-1);
	result->nodes=(node *)malloc(result->num_nodes*sizeof(node));
	result->connections=(connection *)malloc(result->num_connections*sizeof(connection));

	printf("create network with %lu x %lu nodes... \n",num_hor,num_ver);
	
	/* define nodes */	
	n=result->nodes;
	h_offset=0;
	for (j=0;j<num_ver;j++) {
		for (i=0;i<num_hor;i++) {
			n->x=i*hor_step + h_offset;
			n->y=j*ver_step;
			n->weight=0.0;
			n++;
		}
		if (h_offset>0.0) h_offset=0.0;
		else h_offset=hor_step/2.0;
	}
	
	/* define connections */
	c=result->connections;
	/* horizontal connections */
	for (j=0;j<num_ver;j++) {
		for (i=0;i<num_hor-1;i++) {
			c->from=j*num_hor+i;
			c->to=j*num_hor+i+1;
			c++;
		}
	}
	/* vertical connections */
	for (i=0;i<num_hor;i++) {
		for (j=0;j<num_ver-1;j++) {
			c->from=j*num_hor+i;
			c->to=(j+1)*num_hor+i;
			c++;
		}
	}
	/* diagonal connections */
	for (j=0;j<num_ver-1;j++) {
		for (i=0;i<num_hor-1;i++) {
			if (j&1) {
				c->from=j*num_hor+i;
				c->to=(j+1)*num_hor+i+1;
			} else {
				c->from=j*num_hor+(i+1);
				c->to=(j+1)*num_hor+i;
			}
			c++;
		}
	}

	return result;
}


void Destroy_Network(network *net) {
	if (net == (void *)0) return;
	free(net->nodes);
	free(net->connections);
	free(net);
}


void draw_nodes(network *net) {
	long i;
	node *n=net->nodes;

	if (glIsList(1)) {
//		glCallList(1);
//		return;
	}

	glNewList(1,GL_COMPILE_AND_EXECUTE);
	glEnable(GL_POINT_SMOOTH);
	glPointSize(3.0);

//	printf("drawing %lu nodes...\n",net->num_nodes);

	for (i=0;i<net->num_nodes;i++) {
		glBegin(GL_POINTS);
		glColor3f(0.7,0.8,1.0);
		glVertex2fv((GLfloat *)n++);
		glEnd();
	}
	glPointSize(1.0);
	glEndList();
}


void draw_weighted_nodes(network *net) {
	long i;
	node *n=net->nodes;

	if (glIsList(1)) {
//		glCallList(1);
//		return;
	}

	glNewList(1,GL_COMPILE_AND_EXECUTE);
	glPointSize(3.0);
	glEnable(GL_POINT_SMOOTH);

	printf("drawing %lu weighted nodes...\n",net->num_nodes);

	for (i=0;i<net->num_nodes;i++) {
		glBegin(GL_POINTS);
		glColor3f(n->weight,n->weight,n->weight);
		glVertex2fv((GLfloat *)n++);
		glEnd();
	}
	glPointSize(1.0);
	glEndList();
}



void draw_connections(network *net) {
	long i;
	connection *c=net->connections;
	node *n=net->nodes;

	if (glIsList(2)) {
//		glCallList(2);
//		return;
	}

	glNewList(2,GL_COMPILE_AND_EXECUTE);
	glColor3f(0.4,0.5,0.7);
	glBegin(GL_LINES);
	for (i=0;i<net->num_connections;i++) {
		glVertex2fv((GLfloat *)(n+(c->from)));
		glVertex2fv((GLfloat *)(n+(c->to)));
		c++;
	}
	glEnd();
	glEndList();
}


void Draw_Network(network *net) {
	if (net == (void *)0) return;
	if (disp_connections) draw_connections(net);
	if (disp_nodes==1) draw_nodes(net);
	if (disp_nodes==2) draw_weighted_nodes(net);
}



node *get_winner(network *net,dist_point *point) {
	long i;
	node *current=net->nodes;
	node *closest=net->nodes;
	float min_distance=100000000.0;
	float distance;
	float dx,dy;
	
	for (i=0;i<net->num_nodes;i++) {
		dx=current->x - point->x;
		dy=current->y - point->y;
		distance = dx*dx + dy*dy;
		if (distance<min_distance) {
			min_distance=distance;
			closest=current;
		}
		current++;
	}
	return closest;
}



network *Compute_Network(network *net,distribution *dist) {

	dist_point *point;
	node *winner;
	long i;

	for (i=0;i<iterations;i++) {
		
		point = dist->points + (rand()%(dist->num_points));
		winner = get_winner(net,point);
		
		winner->x=winner->x - adaption*(winner->x - point->x);
		winner->y=winner->y - adaption*(winner->y - point->y);
		winner->weight=winner->weight+ 0.1*(1-winner->weight);
		
	}
	return net;
}

float Compute_Error(long iter,network *net,distribution *dist) {
	dist_point *point;
	node *winner;
	float result;
	float dx,dy;
	long i;
	for (i=0;i<iter;i++) {
		point = dist->points + (rand()%(dist->num_points));
		winner = get_winner(net,point);
		dx = winner->x - point->x;
		dy = winner->y - point->y;
		result=result + sqrt(dx*dx + dy*dy);
	}
	return result/((float)iter);
}


mesh *Export_Mesh(network *net) {

	long i;
	long num_points=0;
	node *n;
	mesh *new_mesh;
	mesh_point *mp;	

	/* determine how many points are needed */
	/* the mesh will only contain nodes with a weight > zero */
	n=net->nodes;
	for (i=0;i<net->num_nodes;i++) {
		if (n->weight>0) num_points++;
		n++;
	}
	
	new_mesh = (mesh *)malloc(sizeof(mesh));
	new_mesh->num_polys=0;
	new_mesh->polys=(void *)0;
	new_mesh->num_points=num_points;
	
	new_mesh->points=(mesh_point *)malloc(sizeof(mesh_point)*num_points);
	
	n=net->nodes;	
	mp=new_mesh->points;
	for (i=0;i<net->num_nodes;i++) {
		if (n->weight>0.0) {
			mp->x=n->x;
			mp->y=n->y;
			mp->z=0.0;
			mp++;
		}
		n++;
	}
	
	return new_mesh;
}
