#include <stdio.h>
#include <stdlib.h>
#include "mesh.h"
#include <math.h>
#include <GL/gl.h>


void Destroy_Mesh(mesh *m) {
	if (!m) return;
	if (m->points) free (m->points);
	if (m->polys) free(m->polys);
	free(m);
}


void Calc_Mesh_Normals(mesh *m) {
	mesh_point *p,*p1,*p2,*p3;
	mesh_poly *mp;
	long i,j;
	float ax,ay,az,bx,by,bz,nv;

	if (!m) return;
	if (!m->polys) return;

	/* calculate normal vector for each polygon and accumulate it to each */
	/* point normal vector of the polygon */
	for (i=0;i<m->num_polys;i++) {
		mp=&(m->polys[i]);
		if (mp->num_points>=3) {
		
			/* only three polygon points are used to calc the normal vector */
			p1=&(m->points[ mp->point_idx[0] ]);
			p2=&(m->points[ mp->point_idx[1] ]);
			p3=&(m->points[ mp->point_idx[2] ]);
			
			ax=p2->x - p1->x;
			ay=p2->y - p1->y;
			az=p2->z - p1->z;
			bx=p3->x - p1->x;
			by=p3->y - p1->y;
			bz=p3->z - p1->z;
			
			/* calc normal vector */
			mp->nx=ay*bz - az*by;
			mp->ny=az*bx - ax*bz;
			mp->nz=ax*by - ay*bx;
			
			/* normalize normal vector */
			nv=sqrt(mp->nx*mp->nx + mp->ny*mp->ny + mp->nz*mp->nz);
			if (nv>0.0) {
				mp->nx/=nv;
				mp->ny/=nv;
				mp->nz/=nv;
			}
			
			/* accumulate normal vectors of the poly points */
			for (j=0;j<mp->num_points;j++) {
				p=&(m->points[ mp->point_idx[j] ]);
				p->nx+=mp->nx;
				p->ny+=mp->ny;
				p->nz+=mp->nz;
			}
		}	
	}

	/* normalize point normal vectors */	
	for (i=0;i<m->num_points;i++) {
		p=&(m->points[i]);
		nv=sqrt(p->nx*p->nx + p->ny*p->ny + p->nz*p->nz);
		if (nv>0) {
			p->nx/=nv;
			p->ny/=nv;
			p->nz/=nv;
		}
	}
}


void Draw_Mesh(mesh *m) {
	long i;
	mesh_point *mp;
	mesh_poly *mpoly;
	mesh_point *p1,*p2,*p3;

	if (!m) {
		printf("mesh is zero.\n");
		return;
	}
	if (!m->points) {
		printf("mesh_points is zero.\n");
		return;
	}
	if (!m->num_points) {
		printf("mesh->num_points is zero.\n");
		return;
	}
//	printf("drawing mesh.\n");
	
	glPointSize(2.0);
	mp=m->points;
	for (i=0;i<m->num_points;i++) {
		glBegin(GL_POINTS);
		glColor3f(0.9,0.6,0.0);
		glVertex2f(mp->x,mp->y);
		glEnd();
		mp++;
	}
	
	if (!m->polys) {
		printf("mesh_polys is zero.\n");
		return;	
	}

	if (!m->num_polys) {
		printf("mesh->num_polys is zero.\n");
		return;
	}

	mpoly=m->polys;
	for (i=0;i<m->num_polys;i++) {
	
		/* draw polygon points */
		p1=m->points + mpoly->point_idx[0];
		p2=m->points + mpoly->point_idx[1];
		p3=m->points + mpoly->point_idx[2];
	
	
		glBegin(GL_TRIANGLES);
		glColor4f(0.3,0.1,0.1,0.8);
		glVertex2f(p1->x,p1->y);
		glColor4f(0.0,0.0,0.0,0.8);
		glVertex2f(p2->x,p2->y);
		glColor4f(0.5,0.3,0.3,0.8);
		glVertex2f(p3->x,p3->y);		
		glEnd();

	
		glBegin(GL_LINES);
		glColor3f(0.9,0.8,0.8);
		glVertex2f(p1->x,p1->y);
		glVertex2f(p2->x,p2->y);
//		glEnd();

//		glBegin(GL_LINES);
//		glColor3f(1.0,0.0,1.0);
		glVertex2f(p2->x,p2->y);
		glVertex2f(p3->x,p3->y);
//		glEnd();
		
//		glBegin(GL_LINES);
//		glColor3f(1.0,1.0,1.0);
		glVertex2f(p3->x,p3->y);
		glVertex2f(p1->x,p1->y);
		glEnd();
		
/*		glBegin(GL_POINTS);
		glColor3f(1.0,0.0,1.0);
		glVertex2f(p1->x,p1->y);
		glVertex2f(p2->x,p2->y);
		glVertex2f(p3->x,p3->y);
		glEnd();
	*/	
		mpoly++;
	}


	
}



long find_closest(mesh *m,long p_idx) {
	long i;
	long min_idx=-1;
	float distance;
	float dx,dy,dz;
	float min_distance=10000000.0;
	mesh_point *p = m->points + p_idx;
	mesh_point *lp = m->points;
	
	printf("startpoint is at (%f,%f).\n",p->x,p->y);
	
	for (i=0;i<m->num_points;i++) {
		if (i!=p_idx) {
			dx=p->x - lp->x;
			dy=p->y - lp->y;
			dz=p->z - lp->z;
			distance = dx*dx + dy*dy +dz*dz;
			
//			printf("checking distance = %f\n",distance);
			
			if (distance<min_distance) {
				min_distance=distance;
				min_idx=i;
			}
		}
		lp++;
	}
	return min_idx;
}



/* only x and y are used */
float clockwise(mesh_point *a, mesh_point *b, mesh_point *c)  {
	float x1 = a->x, y1 = a->y;
    float x2 = b->x, y2 = b->y;
    float x3 = c->x, y3 = c->y;

    return (x2*y3-y2*x3) - (x1*y3-y1*x3) + (x1*y2-y1*x2);
}




long triangle_exists(mesh *m,long p1_idx,long p2_idx,long p3_idx) {
	long i,j;
	long result;
	mesh_poly *mp=m->polys;
	
	for (i=0;i<m->num_polys;i++) {
		result=1;	/* assume that the triangle is the same */
		for (j=0;j<mp->num_points;j++) {
			if ((p1_idx!=mp->point_idx[j]) &&
				(p2_idx!=mp->point_idx[j]) && 
				(p3_idx!=mp->point_idx[j])) result=0;
		}
		if (result) return 1;
		mp++;
	}
	return 0;
}



float calc_angle(mesh_point *p1,mesh_point *p2,mesh_point *p3) {
	static float ax,ay,az,bx,by,bz;
	static float q;
	
	/* a = p3 - p1 */
	ax=p3->x - p1->x;
	ay=p3->y - p1->y;
	az=p3->z - p1->z;
	
	/* b = p3 - p2 */
	bx=p3->x - p2->x;
	by=p3->y - p2->y;
	bz=p3->z - p2->z;
	
	/* determine cos(angle) between vectors a and b */
	q = sqrt(ax*ax + ay*ay + az*az)*sqrt(bx*bx + by*by + bz*bz);
	if (q>0.0) return acos((ax*bx + ay*by + az*bz)/q);
	else return 0.0;
}


void build_delaunay(mesh *m,long p1_idx,long p2_idx) {
	mesh_point *p1,*p2,*p3;
	mesh_poly *new_poly,*old_polys;
	long i;
	long p3_idx=p1_idx;	/* crappy init values... */
	float max_angle=0.0;
	float curr_angle;
	
	
//	if (m->num_polys>1000) return;

	/* find third point for the triangle */
	/* the point must be on the right side of the edge */
	/* and fit the maximum angle (p1-p3-p2) criterium */
	p1=m->points + p1_idx;
	p2=m->points + p2_idx;
	p3=m->points;
	for (i=0;i<m->num_points;i++) {
	
		/* searching for a candidate which forms a clockwise triangle */
		if ((i!=p1_idx) && (i!=p2_idx) && (clockwise(p1,p2,p3)>0)) {
//		if ((i!=p1_idx) && (i!=p2_idx)) {
		
//			printf("we are on the right side! \n");
		
			
			/* is the current point better ? */
//	printf("point %lu = (%f,%f,%f)\n",p1_idx,p1->x,p1->y,p1->z);
//	printf("point %lu = (%f,%f,%f)\n",p2_idx,p2->x,p2->y,p2->z);
//	printf("point %lu = (%f,%f,%f)\n",i,p3->x,p3->y,p3->z);
			curr_angle=calc_angle(p1,p2,p3);
//			printf("angle: %f\n",curr_angle);
			if (curr_angle>max_angle) {
				max_angle=curr_angle;
				p3_idx=i;
			}
		}
		p3++;
	}
	p3=m->points + p3_idx;
	
	if (p3_idx==p1_idx) {
		/* no valid point found */
//		printf("no valid point p3 for p1=%lu and p2=%lu found.\n",p1_idx,p2_idx);
		return;
	}
	
	/* yippie, we have a point to build a triangle with */
	if (triangle_exists(m,p1_idx,p2_idx,p3_idx)) {
//		printf("triangle already exists.\n");
		return;
	}
	
	/* ok, the triangle is a new one... add it to the mesh */
	
//	printf("creating new triangle: %lu-%lu-%lu\n",p1_idx,p2_idx,p3_idx);
/*	printf("point %lu = (%f,%f,%f)\n",p1_idx,p1->x,p1->y,p1->z);
	printf("point %lu = (%f,%f,%f)\n",p2_idx,p2->x,p2->y,p2->z);
	printf("point %lu = (%f,%f,%f)\n",p3_idx,p3->x,p3->y,p3->z);
	*/
	m->num_polys++;
//	printf("realloc...\n");
	
	old_polys=m->polys;
	m->polys=(mesh_poly *)realloc(m->polys,m->num_polys*sizeof(mesh_poly));
	/*
	if (m->polys!=old_polys) {
		printf("poly memory block moved.... remove the old one...\n");
		free(old_polys);
	}
	*/
	
//	printf("storing values...\n");
	new_poly=m->polys + m->num_polys - 1;
	new_poly->num_points=3;
	new_poly->point_idx[0]=p1_idx;
	new_poly->point_idx[1]=p2_idx;
	new_poly->point_idx[2]=p3_idx;
//	printf("huh!\n");
	
	/* start calculations for the two new edges */
//	build_delaunay(m,p3_idx,p2_idx);
	build_delaunay(m,p1_idx,p3_idx);
	build_delaunay(m,p3_idx,p2_idx);
}


void Delaunay_Mesh(mesh *m) {
//	mesh_poly *polys = (void *)0;
	long p1_idx;
	long p2_idx;

	if (!m) return;
	if (m->polys) free(m->polys);

//	if (m->num_points) {
//		p1_idx=rand()%(m->num_points);
//	} else {
		p1_idx=0;
//	}

	m->num_polys=0;
	
///	printf("number of mesh_points = %lu\n",m->num_points);
	if (m->num_points<3) return;

	/* an initial edge is chosen */
	/* we take the first point and its nearest neighbour */
	p2_idx=find_closest(m,p1_idx);
	if (p2_idx>-1) {
		build_delaunay(m,p1_idx,p2_idx);
		build_delaunay(m,p2_idx,p1_idx);
		printf("delaunay triangulation consists of %lu triangles.\n",m->num_polys);
	} else {
		printf("no neighbour of startpoint found - strange, eh? \n");
	}
	
	
}

