/******************************************
 *                                        *
 * VNNlab - a stereo image analysing tool *
 *                                        *
 *    written by Norman Feske in 2001     *
 *                                        *
 ******************************************/


#include <stdlib.h>
#include <string.h>
#include <tk.h>
#include <tcl.h>
#include <dlfcn.h>
#include <unistd.h>
#include <signal.h>
#include <pthread.h>

#include <sys/stat.h>
#include <fcntl.h>

#include <math.h>
#include <stdio.h>
#include <GL/gl.h>
#include <GL/glut.h>

#include "handle_gl.h"
#include "distrib.h"
#include "mesh.h"


/* space for storing return values of the new tcl commands */
char TCL_result[256];

/* pointer to the used tcl interpreter */
Tcl_Interp *interp;

/* handles of the used modules */
void *net_so;
void *dist_so;
void *depth_so;
void *vis_so;

/* pointer to current processing data */
void *net;
void *dist;
void *vis;
long depth_initflag;

long visupdate_aniflag=0;
long visupdate_flag=0;

/* file handle for 'quantisation error output' */
FILE *qerr_fh=0;
long qerr_flag=0;

/* size of current used distribution */
long dist_w=200,dist_h=100;

/* semaphores */
pthread_mutex_t mutex_net;
pthread_mutex_t mutex_dist;
pthread_mutex_t mutex_vis;
pthread_mutex_t mutex_depth;

/* some functions to convert data types */
char *ptoa(void *p,char *as) {sprintf(as,"%lu",(long)p);return as;}
char *ltoa(long l,char *as) {sprintf(as,"%lu",l);return as;}
char *ftoa(float f,char *as) {sprintf(as,"%f",f);return as;}
void *atop(char *as) {return (void *)atol(as);}

/* only for debugging */
void *getdlsym(void *so_handle,char *ident) {
	static char *errormsg;
	void *result;
	
	if (!so_handle) return NULL;
	result=dlsym(so_handle,ident);
	errormsg = dlerror();
	if (errormsg != NULL) printf("dlerror [%s]: %s\n",ident,errormsg);
	return result;
}


/************************************************
 *                            
 * VARIABLE SYNCHRONISATION BETWEEN C AND TCL/TK
 *
 ************************************************/


char *fetch_symbol(char *curr_sym,char *dst_type,char *dst_name) {

	/* ignore space at the beginning */
	while (*curr_sym==' ') curr_sym++;

	/* the first component of the symbol is the type */
	/* copy type to dst_type until a space or stringend appears */
	while ((*curr_sym!=' ') && (*curr_sym!=',') && (*curr_sym!=0)) {
		*(dst_type++)=*(curr_sym++);
	}
	
	*(dst_type++)=0; /* terminate type string */
	
	if (*(curr_sym-1) == 0) {
		*dst_name=0;
		return (char *)0;
	}
	
	/* ignore space between type and name */
	while (*curr_sym==' ') curr_sym++;
	
	/* now comes the name */
	while ((*curr_sym!=',') && (*curr_sym!=0)) {
		(*dst_name++)=(*curr_sym++);
	}
	
	*(dst_name++)=0; /* terminate name string */
	
	/* check if the last symbol was zero */
	if (*(curr_sym)==0) return (char *)0;
	
	/* check if the last symbol was a comma */
	if (*(curr_sym)==',') return curr_sym+1;
	
	return (char *)0;
}


/*** UPDATES THE SHARED C VARIABLES OF A SHARED OBJECT ***/
void update_so_vars(void *so) {
	static char sym_type[256];
	static char sym_name[256];
	char **so_vars;
	char *new_value;
	void *sym_adr;
	char *curr_pos;
	
	if (!so) return;
	so_vars = (char **)getdlsym(so,"tcl_symbols");
	if (!so_vars) return;
	curr_pos=*so_vars;

//	printf("symbols: %s\n",curr_pos);
	
	while (curr_pos) {
	
		curr_pos=fetch_symbol(curr_pos,sym_type,sym_name);
//		printf("symbol type: %s, ",sym_type);
//		printf("name: %s, ",sym_name);

		sym_adr=getdlsym(so,sym_name);
		if (sym_adr) {
			new_value=Tcl_GetVar(interp,sym_name,0);
//			printf("new value: %s\n",new_value);
			if (new_value) {
				if (strcmp(sym_type,"long")==0) {
					*((long *)sym_adr)=atol(new_value);
				} else if (strcmp(sym_type,"float")==0) {
					*((float *)sym_adr)=atof(new_value);
				} else if (strcmp(sym_type,"string")==0) {
					strncpy((char *)sym_adr,new_value,256);
				}
			}
		}
	
	}
	
//	printf("so symbols ok.\n");

}




/******************************
 *                            
 * GLUT CALLBACK
 *
 ******************************/


int network_win_id;
int result_win_id;


void glut_redraw_network(void) {

	void (*Draw_Network)      (void *net) = NULL;
	void (*Draw_Distribution) (void *dist) = NULL;
		
	printf("redraw_netwin: start...\n");	
	printf("redraw_netwin: draw dist...\n");	
	
	pthread_mutex_lock(&mutex_dist);
	if (dist) {
		Draw_Distribution = getdlsym(dist_so,"Draw_Distribution");
		(*Draw_Distribution)(dist);	
	} else {
		glClear(GL_COLOR_BUFFER_BIT);
	}
	pthread_mutex_unlock(&mutex_dist);	
	
	printf("redraw_netwin: draw nework...\n");	
	pthread_mutex_lock(&mutex_net);
	if (net) {
		Draw_Network = getdlsym(net_so,"Draw_Network");
		if (Draw_Network && net) (*Draw_Network)(net);
	}
	pthread_mutex_unlock(&mutex_net);
	glutSwapBuffers();	
	printf("redraw_netwin: finished.\n");	
}


void glut_redraw_result(void) {

	void (*Draw_Visualisation) (void *vis) = NULL; 
	
	printf("redraw_reswin: start\n");			

    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	
	/* visualisation draw */
	pthread_mutex_lock(&mutex_vis);
	printf("redraw_reswin: draw visualisation...\n");			
	Draw_Visualisation = getdlsym(vis_so,"Draw_Visualisation");
	if (vis && Draw_Visualisation) (*Draw_Visualisation)(vis);
	pthread_mutex_unlock(&mutex_vis);
	
	glutSwapBuffers();
	printf("redraw_reswin: finished.\n");
}



long update_network_ready;
long update_result_ready;

/* flags for thread synchonisation */
long glut_done=0;
long glut_goon=1;
long glut_halted=0;

void glut_idle(void) {
	static long old_w=0,old_h=0;

//	printf("glut_idle: start\n");

	glutSetWindow(network_win_id);
	if ((old_w!=dist_w) || (old_h!=dist_h)) glutReshapeWindow(dist_w,dist_h);
	if (!update_network_ready) {
		update_network_ready=1;
		glutPostRedisplay();
	}

	glutSetWindow(result_win_id);
	if ((old_w!=dist_w) || (old_h!=dist_h)) glutReshapeWindow(dist_w,dist_h);
	if (!update_result_ready) {
		update_result_ready=1;
		glutPostRedisplay();
	}
	old_w=dist_w;old_h=dist_h;
	
	glut_done=1;
	if (!glut_goon) {
		printf("glutthread: ohh... I will wait.\n");
		glut_halted=1;
	}
	while (!glut_goon) {glut_halted=1;};
	glut_halted=0;
//	printf("glut_idle: finished.\n");
}

long glutsync_cnt=0;
void SyncGlutThread(void) {
	glutsync_cnt++;
	if (glutsync_cnt>1) return;

		if (glut_halted) {
			printf("glutthreadsync: leaving because thread is halted already.\n");
			return;
		} else {
			printf("glutthreadsync: waiting...\n");
		}
		/* force the thread to halt after the current computation */
		glut_goon=0;
		/* wait until the thread finished the current computation */
		glut_done=0;
		while (!glut_done);
		printf("glutthreadsync: glutthread is ready.");
}


void ContinueGlutThread(void) {
		/* tell the learning thread to go on */
		printf("continueglutthread.\n");
		glut_goon=1;
	glutsync_cnt--;
}


/*** FORCE AN UPDATE OF THE GLUT NETWORK OUTPUT WINDOW ***/
void UpdateNetworkWindow(void) {	
	update_network_ready=0;
}

/*** FORCE AN UPDATE OF THE GLUT RESULT OUTPUT WINDOW ***/
void UpdateResultWindow(void) {	
	update_result_ready=0;
}





/***************
 *                            
 * LEARN THREAD
 *
 ***************/


pthread_t learn_th;

/* flag is 1 while the learnthread is active */
long learnflag=0;

void *LearnThread(void *arg) {

	void *         (*Compute_Network)  (void *net,distribution *dist) = NULL;
	distribution * (*Get_Distribution) (void *dist) = NULL;
	float          (*Compute_Error)    (long iter,void *net,distribution *dist) = NULL;

	distribution *d;
		
	if (!dist) return (void *)0;

	do {
		printf("learning: start\n");

		pthread_mutex_lock(&mutex_dist);
		printf("learning: get distribution\n");
		Get_Distribution = getdlsym(dist_so,"Get_Distribution");
		if (Get_Distribution && dist) d=(*Get_Distribution)(dist);
		else d=(void *)0;
		pthread_mutex_unlock(&mutex_dist);
		
		if (d) printf("learning: distribution with %lu points.\n",d->num_points);

		pthread_mutex_lock(&mutex_net);
		printf("learning: update net vars\n");
		update_so_vars(net_so);

		printf("learning: compute network\n");
		Compute_Network = getdlsym(net_so, "Compute_Network");
		if (Compute_Network && net && d) net = (*Compute_Network)(net,d);

		/* error estimation */
		if (qerr_fh && qerr_flag) {
			printf("learning: calc quantisation error\n");
			Compute_Error   = getdlsym(net_so, "Compute_Error");
			fprintf(qerr_fh,"%f\n",(*Compute_Error)(1000,net,d));
			fflush(qerr_fh);
		}
		pthread_mutex_unlock(&mutex_net);

		
		UpdateNetworkWindow();	

		if (visupdate_aniflag) visupdate_flag=1;
		printf("learning: finished.\n");
		
	} while (learnflag);
	return NULL;
}


void StopLearning() {
	if (learnflag) {
		learnflag=0;
		pthread_join(learn_th,NULL);
	}
}





/******************************
 *                            
 * VISUALISATION UPDATE THREAD
 *
 ******************************/

pthread_t vis_update_th;

void *VisUpdateThread(void *arg) {
	
	mesh * (*Export_Mesh)           (void *net) = NULL;
	void   (*Destroy_Visualisation) (void *vis) = NULL;
	void * (*Create_Visualisation)  (void *mesh,long width,long height) = NULL;
	float  (*Get_Disparity)         (long x, long y) = NULL;

	mesh *m;
	float mindisp;	
	mesh_point *mp;
	long i;
	
	while (1) {
	
		if (visupdate_flag) {
			printf("start vis_update\n");
	
			/* visualisation init */
			printf("vis_update: export mesh\n");
			pthread_mutex_lock(&mutex_net);
			Export_Mesh = getdlsym(net_so,"Export_Mesh");
			if (Export_Mesh && net) m=(*Export_Mesh)(net);
			else m=(void *)0;
			pthread_mutex_unlock(&mutex_net);
				
			printf("vis_update: delaunay\n");
			if (Delaunay_Mesh && m) Delaunay_Mesh(m);

			pthread_mutex_lock(&mutex_depth);
			if (depth_so && m) {
				printf("vis_update: update shared depth variables\n");	
				update_so_vars(depth_so);
			
	
				printf("vis_update: eval depth\n");	
				mp=m->points;
				printf("vis_update: check initflag\n");	
				/* evaluate disparities of all points */
				Get_Disparity = getdlsym(depth_so,"Get_Disparity");
				if (Get_Disparity && depth_initflag && mp) {
					printf("vis_update: start loop\n");	
					for (i=0;i<m->num_points;i++) {
						mp->z=Get_Disparity(mp->x,mp->y);
						mp++;
					}
				}
				printf("vis_update: eval depth finished.\n");
			}
			pthread_mutex_unlock(&mutex_depth);

			if (m) {
				printf("vis_update: calc z\n");	
				/* find lowest disparity */				
				mindisp=-999.0;
				mp=m->points;
				if (mp) {
					for (i=0;i<m->num_points;i++) {
						if (mp->z<mindisp) mindisp=mp->z;
						mp++;
					}
				}

				/* calculate depth */
				mp=m->points;
				if (mp) {
					for (i=0;i<m->num_points;i++) {
						if (mp->z<mindisp) mindisp=100.0/(mp->z-mindisp+1.0);
						mp++;
					}
				}
			}

			//// Normal vector calculation ////
				
			pthread_mutex_lock(&mutex_vis);		
			printf("vis_update: destroy old visualisation...\n");
			Destroy_Visualisation = getdlsym(vis_so,"Destroy_Visualisation");
			if (Destroy_Visualisation && vis) Destroy_Visualisation(vis);
			vis=(void *)0;
			
			printf("vis_update: create new visualisation...\n");
			Create_Visualisation  = getdlsym(vis_so,"Create_Visualisation");
			if (Create_Visualisation && m) vis=(*Create_Visualisation)(m,dist_w,dist_h);
			pthread_mutex_unlock(&mutex_vis);
				
			printf("vis_update: destroy mesh...\n");
			if (m) Destroy_Mesh(m);
			m=(void *)0;
	
			/* force glut to redraw result */
			UpdateResultWindow();
			
			/* visualisation update job finished */
			visupdate_flag=0;
	
			printf("Visualisation Update finished.\n");			
		}
	} /* end of thread mainloop */
}






/***********************************
 *                            
 * HANDLING OF 'EVENTS' FROM TCL/TK
 *
 ***********************************/



/*** PREPARE THE CHANGE OF A MODULE ***
 * usage: PrepareModuleChange <module type>
 */
int TCL_PrepareModuleChange(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	void (*Destroy_Network) (void *net);
	void (*Destroy_Distribution) (void *dist);
	void (*Destroy_Visualisation) (void *vis);

	printf("prepare module change: %s\n",argv[1]);
	if (!strcmp(argv[1],"dist")) {

		//StopLearning();
		
		/* destroy current distribution */
		pthread_mutex_lock(&mutex_dist);
		Destroy_Distribution = getdlsym(dist_so,"Destroy_Distribution");
		if (Destroy_Distribution) (*Destroy_Distribution) (dist);
		dist_so = (void *)0;
		dist = (void *)0;
		pthread_mutex_unlock(&mutex_dist);
	}
	if (!strcmp(argv[1],"net")) {

		StopLearning();

		/* destroy current network */
		pthread_mutex_lock(&mutex_net);
		Destroy_Network = getdlsym(net_so,"Destroy_Network");
		if (Destroy_Network) (*Destroy_Network) (net);
		net_so = (void *)0;
		net = (void *)0;
		pthread_mutex_unlock(&mutex_net);
	}
	if (!strcmp(argv[1],"depth")) {
		
		depth_initflag=0;

		/* destroy current visualisation */
		pthread_mutex_lock(&mutex_vis);
		Destroy_Visualisation = getdlsym(vis_so,"Destroy_Visualisation");
		if (Destroy_Visualisation) (*Destroy_Visualisation) (vis);
		vis = (void *)0;
		depth_so = (void *)0;
		pthread_mutex_unlock(&mutex_vis);
	}
	if (!strcmp(argv[1],"vis")) {

		/* destroy current visualisation */
		pthread_mutex_lock(&mutex_vis);
		Destroy_Visualisation = getdlsym(vis_so,"Destroy_Visualisation");
		if (Destroy_Visualisation) (*Destroy_Visualisation) (vis);
		vis_so = (void *)0;
		vis = (void *)0;
		pthread_mutex_unlock(&mutex_vis);
	}
	
	printf("prepare module change of %s finished.\n",argv[1]);
	return TCL_OK;
}



/*** SET THE SHARED OBJECT TO USE ***
 * usage: SetModule <module type> <handle of shared object>
 */
int TCL_SetModule(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	void *so_handle = atop(argv[2]);

	printf("set_module: %s\n",argv[1]);
	
	if (!strcmp(argv[1],"net"))   {
		pthread_mutex_lock(&mutex_net);
		net_so = so_handle;
		pthread_mutex_unlock(&mutex_net);
	}
	if (!strcmp(argv[1],"dist"))  {
		pthread_mutex_lock(&mutex_dist);
		dist_so = so_handle;
		pthread_mutex_unlock(&mutex_dist);
	}
	if (!strcmp(argv[1],"depth")) {
		pthread_mutex_lock(&mutex_depth);
		printf("set_module: setting up new so value%s\n",argv[1]);
		
		depth_so = so_handle;
		depth_initflag=0;
		pthread_mutex_unlock(&mutex_depth);
	}
	if (!strcmp(argv[1],"vis")) {
		pthread_mutex_lock(&mutex_vis);
		vis_so = so_handle;
		pthread_mutex_unlock(&mutex_vis);
	}
	
	printf("set_module: finished\n");
	return TCL_OK;
}


/*** FORCE THE INITIALISATION OF A DISTRIBUTION ***
 * arguments: <command name>
 */
int TCL_InitDistribution(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	void * (*Create_Distribution) (void);
	void (*Destroy_Distribution) (void *dist);
	long (*Get_Distribution_Width) (void *dist);
	long (*Get_Distribution_Height) (void *dist);
	void *newdist;

	if (argc==1) {
	
		printf("dist_init: get adresses of distribution module functions...\n");				
	
		Create_Distribution     = dlsym(dist_so,"Create_Distribution");
		Destroy_Distribution    = dlsym(dist_so,"Destroy_Distribution");
		Get_Distribution_Width  = dlsym(dist_so,"Get_Distribution_Width");
		Get_Distribution_Height = dlsym(dist_so,"Get_Distribution_Height");
				
//		Tcl_Eval(interp,"update_c_syms $dist_so $dist_symbols\n");
		printf("dist_init: update shared distribution variables...\n");	
		
		pthread_mutex_lock(&mutex_dist);		
		update_so_vars(dist_so);		
		
		printf("dist_init: create new distribution...\n");				
		if (Create_Distribution) newdist = (*Create_Distribution)();
		else newdist=(void *)0;
				
		printf("dist_init: destroying old distribution...\n");				
		if (newdist) {
			printf("Destroying old distribution \n");
			if (dist && Destroy_Distribution) (*Destroy_Distribution)(dist);
			dist=newdist;
		}
		pthread_mutex_unlock(&mutex_dist);
		
		printf("adress after creation of net = %lu\n",(long)dist);


		if (dist) {
			printf("dist_init: set new distribution size...\n");				

			SyncGlutThread();
			pthread_mutex_lock(&mutex_dist);
			if (Get_Distribution_Width) dist_w=(*Get_Distribution_Width)(dist);
			if (Get_Distribution_Height) dist_h=(*Get_Distribution_Height)(dist);
			pthread_mutex_unlock(&mutex_dist);
			ContinueGlutThread();

		} else {
			printf("dist_init: something went wrong, damn...\n");				
		}
		
		printf("dist_init: force an update of the glut window...\n");				
		UpdateNetworkWindow();	
	}
	printf("dist_init: finished.\n");				
	return TCL_OK;
}


/*** FORCE THE INITIALISATION OF A NETWORK ***
 * arguments: <command name>
 */
int TCL_InitNetwork(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	void * (*Create_Network) (long init_w,long init_h);
	void (*Destroy_Network) (void *net);
	void *newnet;

	if (argc==1) {
	
		if (!dist) return TCL_OK;
		
		Create_Network  = getdlsym(net_so,"Create_Network");
		Destroy_Network = getdlsym(net_so,"Destroy_Network");
				
		pthread_mutex_lock(&mutex_net);
		if (!learnflag) update_so_vars(net_so);
		newnet = (*Create_Network)(dist_w,dist_h);		
		
		if (net) (*Destroy_Network)(net);
		net = newnet;
		
		pthread_mutex_unlock(&mutex_net);
				
		if (!learnflag) UpdateNetworkWindow();
	
	}
	return TCL_OK;
}


/*** FORCE THE INITIALISATION OF A CORRESPONDENCE ANALYSING MODULE ***
 * arguments: <command name>
 */
int TCL_InitDepthEval(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	void (*Init_DepthEval) (void);
	
	printf("init_depth: start\n");
	
	pthread_mutex_lock(&mutex_depth);
	printf("init_depth: get adresses of module functions...\n");				
	Init_DepthEval = getdlsym(depth_so,"Init_DepthEval");

	printf("init_depth: update shared variables...\n");
	if (depth_so) update_so_vars(depth_so);
				
//	SyncVisThread();
				
	printf("init_depth: init module...\n");
	
	if (Init_DepthEval) {
		(*Init_DepthEval)();
		depth_initflag=1;
	}
	
	pthread_mutex_unlock(&mutex_depth);
	
	printf("init_depth: finished.\n");
	return TCL_OK;
}


/*** FORCE A VISUALISATION UPDATE ***
 * arguments: <command name>
 */
int TCL_UpdateVisualisation(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	while (visupdate_flag);
	pthread_mutex_lock(&mutex_vis);
	if (vis_so) update_so_vars(vis_so);
	pthread_mutex_unlock(&mutex_vis);
	visupdate_flag=1;
	return TCL_OK;
}


/*** BEGIN A LEARN PROCESS ***
 * arguments: <command name>
 */
int TCL_BeginLearning(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {

	if ((argc==1) && (!learnflag) && (net) && (dist)) {

		pthread_mutex_lock(&mutex_net);
		update_so_vars(net_so);
		pthread_mutex_unlock(&mutex_net);
		learnflag=1;
		printf("creating thread for learning...\n");
		pthread_create(&learn_th,NULL,&LearnThread,(void *)interp);		

	}
	return TCL_OK;
}


/*** FINISH A LEARN PROCESS ***
 * arguments: <command name>
 */
int TCL_StopLearning(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	StopLearning();
	return TCL_OK;
}


/*** FORCE A REDRAW OF THE NETWORK ***
 * arguments: <command name>
 */
int TCL_RedrawNetwork(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
//	SyncLearnThread();
	if (net_so) {
		pthread_mutex_lock(&mutex_net);
		update_so_vars(net_so);
		pthread_mutex_unlock(&mutex_net);
	}
//	ContinueLearnThread();		
	UpdateNetworkWindow();
	return TCL_OK;
}


/*** FORCE A REDRAW OF THE DISTRIBUTION ***
 * arguments: <command name>
 */
int TCL_RedrawDistribution(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	if (dist_so) {
		pthread_mutex_lock(&mutex_dist);
		update_so_vars(dist_so);
		pthread_mutex_unlock(&mutex_dist);
	}
	UpdateNetworkWindow();
	return TCL_OK;
}


/*** FORCE A REDRAW OF THE VISUALISATION WINDOW ***
 * arguments: <command name>
 */
int TCL_RedrawVisualisation(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	if (vis_so) {
		pthread_mutex_lock(&mutex_vis);
		update_so_vars(vis_so);
		pthread_mutex_unlock(&mutex_vis);
	}
	UpdateResultWindow();
	return TCL_OK;
}


/* usage: <command name> */
int TCL_StartVisAnimation(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	visupdate_aniflag=1;
	return TCL_OK;
}


/* usage: <command name> */
int TCL_StopVisAnimation(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	visupdate_aniflag=0;
	return TCL_OK;
}




/******************************
 *                            
 * HANDLING OF SHARED OBJECTS
 *
 ******************************/
 


/*** OPENS A SHARED OBJECT FILE ***
 * usage:   dlOpen <filename of shared object>
 * returns: handle of this shared object
 */
int TCL_dlOpen(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	void *so_handle;
	if (argc==2) {
		so_handle=dlopen(argv[1],RTLD_LAZY);
		interp->result=ptoa(so_handle,(char *)&TCL_result);
	}
	return TCL_OK;
}


/*** CLOSES A SHARED OBJECT FILE ***
 * usage: dlClose <handle of shared object>
 */
int TCL_dlClose(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	if (argc==2) dlclose(atop(argv[1]));
	return TCL_OK;
}


/*** RETURNS A STRING WHICH IS DEFINED INSIDE A SHARED OBJECT ***
 * usage:   dlGetString <handle of shared object> <identifier>
 * returns: corresponding identifier or the error message "not defined"
 */
int TCL_dlGetString(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	char **so_string;
	void *so_handle = atop(argv[1]);
	if (argc==3) {
		so_string=(char **)dlsym(so_handle,argv[2]);
		if (dlerror()) {
			interp->result="not defined";
		} else {
			interp->result=*so_string;
		}
	}
	return TCL_OK;
}


/*** SET VALUE OF A SPECIFIED MEMORY LOCATION - INSPIRED BY A BASIC COMMAND :-) ***
 * arguments: dlPoke <pointer> <type> <value>
 */
int TCL_dlPoke(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	if (argc==4) {
		if (!strcmp(argv[2],"long")) {
			*((long *)atop(argv[1]))=atol(argv[3]);
			return TCL_OK;	
		}
		if (!strcmp(argv[2],"float")) {
			*((float *)atop(argv[1]))=atof(argv[3]);
			return TCL_OK;	
		}
		if (!strcmp(argv[2],"string")) {
			strncpy((char *)atop(argv[1]),argv[3],256);
			return TCL_OK;	
		}
	}
	return TCL_OK;
}


/*** GETS VALUE OF A SPECIFIED MEMORY LOCATION ***
 * arguments: dlPeek <pointer> <type>
 * returns:   value of the specified variable
 */
int TCL_dlPeek(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	if (argc==3) {
		if (!strcmp(argv[2],"long")) {
			interp->result=ltoa(*((long *)atop(argv[1])),TCL_result);
			return TCL_OK;
		}
		if (!strcmp(argv[2],"float")) {
			interp->result=ftoa(*((float *)atop(argv[1])),TCL_result);
			return TCL_OK;
		}
		if (!strcmp(argv[2],"string")) {
			strncpy(TCL_result,(char *)atop(argv[1]),256);
/*			interp->result=ftoa(*((float *)atop(argv[1])),TCL_result);*/
			return TCL_OK;
		}
	}
	return TCL_OK;
}


/*** RETURNS ADRESS OF A SYMBOL OF A SHARED OBJECT ***
 * arguments: dlGetAdress <handle of shared object> <symbol>
 * returns:   adress of the specified symbol in shared object
 */
int TCL_dlGetAdress(ClientData clientdata, Tcl_Interp *interp, int argc, char *argv[]) {
	void *sym_adr;
	void *so_handle = atop(argv[1]);
	if (argc==3) {
		sym_adr=dlsym(so_handle,argv[2]);
		if (dlerror()) {
			interp->result="not defined";
		} else {
			interp->result=ptoa(sym_adr,TCL_result);
		}
	}
	return TCL_OK;
}



/*** TCL/TK SHELL INITIALISATION - CALLED BY Tk_Main ***/
int Tcl_AppInit(Tcl_Interp *interpreter) {

	interp=interpreter;

	if (Tcl_Init(interp) == TCL_ERROR) return TCL_ERROR;
	Tk_Init(interp);	

	/* add new vnnlab specific tcl commands to the shell */
//	Tcl_CreateCommand(interp,"SetDistModule",TCL_SetDistModule,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"InitDistribution",TCL_InitDistribution,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);	
	
	Tcl_CreateCommand(interp,"PrepareModuleChange",TCL_PrepareModuleChange,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"SetModule",TCL_SetModule,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);

//	Tcl_CreateCommand(interp,"SetNetModule",TCL_SetNetModule,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"InitNetwork",TCL_InitNetwork,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"BeginLearning",TCL_BeginLearning,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"StopLearning",TCL_StopLearning,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"RedrawNetwork",TCL_RedrawNetwork,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"RedrawDistribution",TCL_RedrawDistribution,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);

//	Tcl_CreateCommand(interp,"SetDepthModule",TCL_SetDepthModule,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"InitDepthEval",TCL_InitDepthEval,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);

//	Tcl_CreateCommand(interp,"SetVisModule",TCL_SetVisModule,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"RedrawVisualisation",TCL_RedrawVisualisation,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"UpdateVisualisation",TCL_UpdateVisualisation,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"StartVisAnimation",TCL_StartVisAnimation,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"StopVisAnimation",TCL_StopVisAnimation,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);

	/* provide tcl commands to access to shared object files */
	Tcl_CreateCommand(interp,"dlOpen",TCL_dlOpen,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"dlClose",TCL_dlClose,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"dlGetString",TCL_dlGetString,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"dlGetAdress",TCL_dlGetAdress,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"dlPeek",TCL_dlPeek,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);
	Tcl_CreateCommand(interp,"dlPoke",TCL_dlPoke,(ClientData)NULL,(Tcl_CmdDeleteProc *)NULL);

	Tcl_SetVar(interp, "tcl_rcFileName", "gui.tcl",TCL_GLOBAL_ONLY);
	return TCL_OK;
}


int main(int argc, char **argv) {
	int log_fh;
	char *prgname="vnnlab";
	char *tclargv[1]={prgname};

	/* redirect stdio to the the file 'vnnlab.log' */
	if (argc==1) {
		log_fh=open("vnnlab.log",O_CREAT+O_WRONLY,S_IREAD+S_IWRITE);
		close(1);
		dup2(log_fh,1);
		close(log_fh);
	}

	/* open file for quantisation error output */
	qerr_fh=fopen("qerror.log","w");

	Init_OpenGL();
//	glutDisplayFunc(glut_redraw);
	glutIdleFunc(glut_idle);
	UpdateNetworkWindow();

	/* init semaphores */
	pthread_mutex_init(&mutex_net,NULL);
	pthread_mutex_init(&mutex_dist,NULL);
	pthread_mutex_init(&mutex_vis,NULL);
	pthread_mutex_init(&mutex_depth,NULL);
	
	printf("creating thread for updating visualisation...\n");
	pthread_create(&learn_th,NULL,&VisUpdateThread,(void *)interp);		

	Tk_Main(1, tclargv, Tcl_AppInit);
	return 0;
}



