/***
 * SOBEL FILTER BASED DISTRIBUTION
 * by Norman Feske
 ***/

#include <math.h>
#include <GL/gl.h>
#include <image.h>
#include <distrib.h>

#define DISP_BG_NONE 0
#define DISP_BG_ORIGINAL 1
#define DISP_BG_SOBEL 2
#define DISP_BG_DIST 3

/*** LIST OF PUBLIC VARIABLES ***/
char *tcl_symbols = "string src_img_fname,float threshold,\
                     long stepsize,long disp_bg";
char *title = "Sobel Based Distribution";

char src_img_fname[256];

float threshold=30.0;
long stepsize=2;
long disp_bg=DISP_BG_DIST;

struct dist_data {
	distribution *dist;
	image *src_img;
	image *filter_img;
	image *threshold_img;
	image *vis_img;
	long vis_state;
};



/*** SOBEL FILTER ***
 * This function generates two output images:
 * dst is the sobel result
 * dst2 is the sobel image with an applied threshold
 */
void sobel_image (image *src_img,long threshold,long step,image *dst_img,image *dst_img2) {
	long x,y;
	long w=src_img->w;
	long h=src_img->h;
	float *src=(float *)(src_img->pixels);
	float *dst=(float *)(dst_img->pixels);
	float *dst2=(float *)(dst_img2->pixels);
	float h_value,v_value;
	
	long pix_e=1*step,pix_w=-1*step,pix_n=-w*step,pix_s=w*step;
	long pix_ne=pix_n+pix_e;
	long pix_nw=pix_n+pix_w;
	long pix_se=pix_s+pix_e;
	long pix_sw=pix_s+pix_w;
	
	src+=w*step+step;
	dst+=w*step+step;
	dst2+=w*step+step;
	for (y=step;y<h-step;y++) {
		for (x=step;x<w-step;x++) {
			
			v_value=  *(src+pix_ne) - *(src+pix_nw) +
					 (*(src+pix_e ) - *(src+pix_w )) * 2 +
					  *(src+pix_se) - *(src+pix_sw);

			h_value=  *(src+pix_sw) - *(src+pix_nw) +
					 (*(src+pix_s ) - *(src+pix_n )) * 2 +
					  *(src+pix_se) - *(src+pix_ne);
			
			*dst = sqrt(v_value*v_value + h_value*h_value)*0.2;
			
			if (*(dst++)>threshold) *(dst2++)=100;
			else *(dst2++)=0;
			src++;
		}
		
		/* skip last pixel of current line and first pixel of next line */
		src+=2*step;
		dst+=2*step;
		dst2+=2*step;
	}
}




distribution *image_to_distribution(image *src_img) {

	float *pix=src_img->pixels;
	distribution *result=(distribution *)malloc(sizeof(distribution));
	long x,y;
	dist_point *points=(dist_point *)malloc(sizeof(dist_point)*src_img->w*src_img->h);
	
	result->num_points=0;
	result->points=points;

	for (y=0;y<src_img->h;y++) {
		for (x=0;x<src_img->w;x++) {
			if (*(pix++) > 50) {
				points->x=x;points->y=y;
				points++;
				result->num_points++;
			}
		}
	}

	
	printf("number of distribution points: %lu\n",result->num_points);
	return result;
}


struct dist_data *Create_Distribution(void) {

	struct dist_data *new_dist = (struct dist_data *)malloc(sizeof(struct dist_data));
	long w,h;
	
	printf("create_distribution: source image filename: %s\n",src_img_fname);
	
	/* load source image */
	printf("create_distribution: load file\n");
	new_dist->src_img = Load_RGB_Image(src_img_fname,IMAGE_GRAY_FLOAT);
	printf("create_distribution: checking zero image\n");
	if (new_dist->src_img == (void *)0) {
		printf("create_distribution: failed - freeing memory and returning...\n");
		free(new_dist);
		return (void *)0;
	}
	w=new_dist->src_img->w;
	h=new_dist->src_img->h;
	printf("sobel_dist: image successfully loaded.\n");
	
	/* create buffer for the filter result */
	printf("sobel_dist: create filter image buffer...\n");
	new_dist->filter_img=Create_Image(w,h,IMAGE_GRAY_FLOAT);
	
	/* create buffer for the thresholded filter result */
	printf("sobel_dist: create threshold image buffer...\n");
	new_dist->threshold_img=Create_Image(w,h,IMAGE_GRAY_FLOAT);

	/* create buffer for displaying */
	printf("sobel_dist: create display image buffer...\n");
	new_dist->vis_img=Create_Image(w,h,IMAGE_RGBA_CHAR);
	
	
	printf("sobel_dist: sobel filtering....");
	sobel_image(new_dist->src_img,threshold,stepsize,new_dist->filter_img,new_dist->threshold_img);
	printf("done.\n");
	
	printf("sobel_dist: creating discrete distribution...");
	new_dist->dist=image_to_distribution(new_dist->threshold_img);
	printf("done.\n");
	
	new_dist->vis_state=DISP_BG_NONE;
	return new_dist;
}



void Destroy_Distribution(struct dist_data *d) {
	if (d) {
		printf("freeing dist");
		if (d->dist) {
			free(d->dist->points);
			free(d->dist);
		}
		printf(", source image");
		Destroy_Image(d->src_img);
		printf(", filter image");
		Destroy_Image(d->filter_img);
		printf(", threshold image");
		Destroy_Image(d->threshold_img);
		printf(", visualisation image");
		Destroy_Image(d->vis_img);
		printf(", dist_data struct");
		free(d);
		printf(".\n");
	}
}


distribution *Get_Distribution(struct dist_data *d) {
	if (d) return d->dist;
	return (void *)0;
}


void Draw_Distribution(struct dist_data *d) {
	if (d == (void *)0) return;
	
	/*printf("displaying distribution bg=%lu \n",disp_bg);*/
	
	switch (disp_bg) {
	case DISP_BG_NONE:
	        glClear(GL_COLOR_BUFFER_BIT);
		break;
	case DISP_BG_ORIGINAL:
		if (d->vis_state!=DISP_BG_ORIGINAL) {
			Image_float2rgba(d->src_img,d->vis_img);
			d->vis_state = DISP_BG_ORIGINAL;
		}
		glRasterPos2i(0,0);
		glDrawPixels(d->vis_img->w,d->vis_img->h,GL_RGBA,GL_UNSIGNED_BYTE,(GLvoid *)d->vis_img->pixels);
		break;
	case DISP_BG_SOBEL:
		if (d->vis_state!=DISP_BG_SOBEL) {
			Image_float2rgba(d->filter_img,d->vis_img);
			d->vis_state = DISP_BG_SOBEL;
		}
		glRasterPos2i(0,0);
		glDrawPixels(d->vis_img->w,d->vis_img->h,GL_RGBA,GL_UNSIGNED_BYTE,(GLvoid *)d->vis_img->pixels);
		break;
	case DISP_BG_DIST:
		if (d->vis_state!=DISP_BG_DIST) {
			Image_float2rgba(d->threshold_img,d->vis_img);
			d->vis_state = DISP_BG_DIST;
		}
		glRasterPos2i(0,0);
		glDrawPixels(d->vis_img->w,d->vis_img->h,GL_RGBA,GL_UNSIGNED_BYTE,(GLvoid *)d->vis_img->pixels);
		break;
	}
	
	/*printf("displaying done.\n");*/
}




long Get_Distribution_Width(struct dist_data *d) {
	if (d) {
		return d->src_img->w;
	}
	return 0;
}



long Get_Distribution_Height(struct dist_data *d) {
	if (d) {
		return d->src_img->h;
	}
	return 0;
}
