/***
 * BLOCKMATCHING DEPTH EVALUATOR
 * by Norman Feske
 ***/

#include <stdio.h>
#include <image.h>
#include <math.h>

void cdft2d(int, int, int, double **, double *, int *, double *);
#include "alloc.c"
#define MAX(x,y) ((x) > (y) ? (x) : (y))


/*** LIST OF PUBLIC VARIABLES ***/
char *tcl_symbols = "string dep_left_img_fname, string dep_right_img_fname,\
                     long disp_min, long disp_max";
char *title = "Fourier Based Similarity";


char dep_left_img_fname[256];
char dep_right_img_fname[256];

long disp_min=0;
long disp_max=400;

image *left_img=(void *)0;
image *right_img=(void *)0;

long blk_width=16;
long blk_height=16;


/* FFT variables */
int *left_ip, left_n1, left_n2, left_n;
double **left_a, *left_t, *left_w, left_err;

int *right_ip, right_n1, right_n2, right_n;
double **right_a, *right_t, *right_w, right_err;



void init_fft_buffers(int n1, int n2) {
    left_a = alloc_2d_double(n1, 2*n2);
    left_t = alloc_1d_double(2 * n1);
    left_n = MAX(n1, n2);
    left_ip = alloc_1d_int(2 + (int) sqrt(left_n + 0.5));
    left_n = MAX(n1 * 5 / 4, n2 * 5 / 2) + n2 / 2;
    left_w = alloc_1d_double(left_n);

    right_a = alloc_2d_double(n1, 2*n2);
    right_t = alloc_1d_double(2 * n1);
    right_n = MAX(n1, n2);
    right_ip = alloc_1d_int(2 + (int) sqrt(right_n + 0.5));
    right_n = MAX(n1 * 5 / 4, n2 * 5 / 2) + n2 / 2;
    right_w = alloc_1d_double(right_n);

}


void free_fft_buffers() {
	if (left_w)  free_1d_double(left_w);
    if (left_ip) free_1d_int(left_ip);
    if (left_t)  free_1d_double(left_t);
    if (left_a)  free_2d_double(left_a);
	left_w=left_t=(void *)0;
	left_a=(void *)0;
	left_ip=(void *)0;

    if (right_w)  free_1d_double(right_w);
    if (right_ip) free_1d_int(right_ip);
    if (right_t)  free_1d_double(right_t);
    if (right_a)  free_2d_double(right_a);
	right_w=right_t=(void *)0;
	right_a=(void *)0;
	right_ip=(void *)0;
}


void Init_DepthEval() {
	if (left_img) Destroy_Image(left_img);
	if (right_img) Destroy_Image(right_img);
	left_img =Load_RGB_Image(dep_left_img_fname, IMAGE_RGBA_CHAR);
	right_img=Load_RGB_Image(dep_right_img_fname,IMAGE_RGBA_CHAR);
	init_fft_buffers(blk_width,blk_height);
}



void Deinit_DepthEval() {
	if (left_img) Destroy_Image(left_img);
	if (right_img) Destroy_Image(right_img);
	left_img=(void *)0;
	right_img=(void *)0;
	free_fft_buffers();
}



/*** RETURNS A VALUE OF SIMILARITY BETWEEN TWO PIXEL BLOCKS ***
 * arguments: left, right are pointers to rgba pixelblocks inside a picture
 */
float red_sim,green_sim,blue_sim;
float Get_Similarity(long *left, long *right, long w) {

	static long i,j;
	unsigned char *l,*r;
	static float result;
	float  left_power,right_power;
	float  abserr;
	long   xoffset=blk_width/2;
	long   yoffset=blk_height/2;
	
	/* copy images blocks to fft buffers */
	for (j=0;j<blk_width;j++) {
		for (i=0;i<blk_height;i++) {
			l=(unsigned char *)(left + w*(j-xoffset) + (i-yoffset));
			left_a[i][j*2]    = (double)(*l) + (double)(*(l+1)) + (double)(*(l+2));
			left_a[i][j*2+1]  = 0.0;
			
			r=(unsigned char *)(right + w*(j-xoffset) + (i-yoffset));			
			right_a[i][j*2]   = (double)(*r) + (double)(*(r+1)) + (double)(*(r+2));
			right_a[i][j*2+1] = 0.0;
		}
	}
	
    /* 2d fft */
    cdft2d(left_n1, left_n2*2, 1, left_a, left_t, left_ip, left_w);
    cdft2d(right_n1, right_n2*2, 1, right_a, right_t, right_ip, right_w);
	
	/* compute the similarity of both fourier images */
	abserr=0.0;
	for (j=0;j<blk_width;j++) {
		for (i=0;i<blk_height;i++) {
			left_power=sqrt(left_a[i][j*2]*left_a[i][j*2] + left_a[i][j*2+1]*left_a[i][j*2+1]);
			right_power=sqrt(right_a[i][j*2]*right_a[i][j*2] + right_a[i][j*2+1]*right_a[i][j*2+1]);
			abserr=abserr + fabs(left_power - right_power);
		}
	}
	
	result=abserr;
	
	return result;
}



float blk_sim[1000];


float Get_Disparity(long x,long y) {
	static long *l,*r;
	long img_w;
	long img_h;
	static long xbeg;
	static long xend;
	float lowest_sim=100000.0;
	float lowest_grad=100000.0;
	long lowest_x=0;
	long lowest_xg=0;
	static long i;
	long bm_sizex=blk_width/2;
	long bm_sizey=blk_height/2;
		
	if (!left_img) return -1;
	if (!right_img) return -1;

	img_w=right_img->w;
	img_h=right_img->h;

	if ((y<=bm_sizey) || (y>=img_h-bm_sizey)) return -1;
	if ((x<=bm_sizex) || (x>=img_w-bm_sizex)) return -1;

	xbeg=x+disp_min;
	xend=x+disp_max;
	
	if (xbeg<0) xbeg=0;
	if (xend>img_w) xend=img_w;

	l=(long *)left_img->pixels + (long)img_w*y;
	r=(long *)right_img->pixels + (long)img_w*y + x;

	/* calculate similarities for the pixels at the line y */
	for (i=0;i<img_w;i++) {
		blk_sim[i]=Get_Similarity(l++,r,img_w);		
	}
	
	/* find best value */
	for (i=xbeg;i<xend;i++) {
		if (blk_sim[i]<lowest_sim) {
			lowest_sim=blk_sim[i];
			lowest_x=i;
		}			
	}
	
	return (float)(lowest_x - x);
}

