//
//	Galil ISA bus device	driver for Linux
//
// These symbols must be defined before	including any header files */
//
#ifndef	MODULE
	#define	MODULE
#endif	
#ifndef	__KERNEL__
	#define	__KERNEL__ 
#endif

#include <linux/pci.h>
#include <linux/module.h>
#include <linux/version.h>

char kernal_version[] =	 UTS_RELEASE;
 
#include <linux/config.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/types.h>
#include <linux/ioctl.h>
#include <linux/ioport.h>
#include <linux/proc_fs.h>
#include <linux/interrupt.h>
#include <asm/segment.h>  /* memcpy and such */

#include "galilpci.h"

// Global variables
static int galilpci_major = DEVICE_MAJOR;
static int galilpci_model0 = DEFAULT_MODEL;
static int galilpci_addr0 = DEFAULT_ADDRESS;
static int galilpci_irq0 = DEFAULT_INTERRUPT;

static int context = -1;

/*
#ifdef MODULE
MODULE_PARM(galilpci_major,"i");
MODULE_PARM(galilpci_model0,"i");
MODULE_PARM(galilpci_addr0,"i");
MODULE_PARM(galilpci_irq0,"i");
MODULE_PARM(galilpci_model1,"i");
MODULE_PARM(galilpci_addr1,"i");
MODULE_PARM(galilpci_irq1,"i");
MODULE_PARM(galilpci_model2,"i");
MODULE_PARM(galilpci_addr2,"i");
MODULE_PARM(galilpci_irq2,"i");
MODULE_PARM(galilpci_model3,"i");
MODULE_PARM(galilpci_addr3,"i");
MODULE_PARM(galilpci_irq3,"i");
#endif
*/

//#define pci_resource_start(dev,bar)((dev)->resource[(bar)]&PCI_BASE_ADDRESS_MEM_MASK)

//int DeviceCount = 0;
GALILDEVICE	GalilDevice[MAX_DEVICES];

/* Define symbol table */
/*
static struct symbol_table galilpci_syms =
{
	#include <linxu/symtab_begin.h>
		X(),
	#include <linux/symtab_end.h>
};

register_symtab(&galilpci_syms);
*/

// 
//	Purpose	:	inline code for turning off interrupts on the 1802/1417 board 
//
//	Returns	:	nothing
//
static inline void disableInterrupts1802(
GALILDEVICE* pDevice )
{
	unsigned short wTemp = (unsigned short)inb( pDevice->Address2 + 0x4c );
 	if ( wTemp & 0x40 )
   	outb( (wTemp ^ 0x40), pDevice->Address2 + 0x4c );
}

// 
//	Purpose	:	inline code for turning off interrupts on the everyother board 
//
//	Returns	:	nothing
//
static inline void disableInterrupts(
GALILDEVICE* pDevice )
{
	unsigned char uchTemp = inb( pDevice->Address+4);
  if (uchTemp & 0x40)	// if it is ours
		outb( (uchTemp ^ 0x40), pDevice->Address+4 );
}

// 
//	Purpose	:	inline code for turning off interrupts on the 1804 board 
//
//	Returns	:	nothing
//
static inline void enableInterrupts1802(
GALILDEVICE* pDevice )
{
	unsigned char uchTemp;
	
	if (pDevice->Model == MODEL_1802)
	{
		// Must re-init interrupts whenever we clear
		outb( 0x02, pDevice->Address+1 );	// Tell DMC to start interrupts
		outb( 0x04, pDevice->Address+1 );
		outb( 0x06, pDevice->Address+1 );	// Clear any old interrupts
		inb( pDevice->Address+1 );
	}

	// Start interrupts on the pci hardware for both 1802 and 1417
	uchTemp = inl( pDevice->Address2+0x4c );
	outl( (uchTemp | 0x40), pDevice->Address2+0x4c );
}

// 
//	Purpose	:	inline code for turning off interrupts on the everyother board 
//
//	Returns	:	nothing
//
static inline void enableInterrupts(
GALILDEVICE* pDevice )
{
	unsigned char uchTemp;
	
 	inb( pDevice->Address+8 );             		// Clear any old interrupts
	uchTemp = inb( pDevice->Address+4 );   		// Start interrupts on the pci hardware
 	outb( (uchTemp | 0x40), pDevice->Address+4 );
 	inb( pDevice->Address+8 );             		// Clear any old interrupts
}

// 
//	Purpose	:	inline code for clearing FIFO on non 1802 boards
//
//	Returns	:	nothing
//
static inline void clearFIFO(
GALILDEVICE* pDevice )
{
	outb( (unsigned char)6, pDevice->Address+8 );
}

// 
//	Purpose	:	inline code for clearing FIFO on 1802 boards
//
//	Returns	:	nothing
//
static inline void clearFIFO1802(
GALILDEVICE* pDevice )
{
	
	disableInterrupts1802(pDevice);
	
	if (pDevice->Model == MODEL_1802)
	{
		inb( pDevice->Address );
		outb( (unsigned char)0x01, pDevice->Address );
		outb( (unsigned char)0x80, pDevice->Address);
		outb( (unsigned char)0x01, pDevice->Address);
		outb( (unsigned char)0x80, pDevice->Address);
		inb( pDevice->Address );
		outb( (unsigned char)0x05, pDevice->Address);	// Set up the FIFO for 128 bytes
		outb( (unsigned char)0xF0, pDevice->Address);	// Bidirectionally
	}
	
	enableInterrupts1802( pDevice );
}

//
//	Purpose	:	Handles	setting	the interrupt
//
//	Returns	:	nothing
//
static void inline EnableInterrupts(
GALILDEVICE* pDevice)
{
	switch( pDevice->IOType )
	{
		case DMC1600IOStyle:	enableInterrupts( pDevice );				break;
		case DMC1000IOStyle:	clearFIFO1802( pDevice );
													enableInterrupts1802( pDevice );		break;
}	}

// 
//	Purpose	:	
//
//	Returns	:	
//
static void inline DisableInterrupts(
GALILDEVICE* pDevice)
{
	switch( pDevice->IOType )
	{
		case DMC1600IOStyle:	disableInterrupts( pDevice );			break;
		default:							disableInterrupts1802( pDevice );	break;
}	}


//
//	Purpose	:	Data management: read and write
//
//	Returns	:	BytesRead
//
ssize_t	galilpci_read(
struct file*	filp, 
char*					buf, 
size_t				count, 
loff_t*				ppos	)
{
	int						MaxBytes = min(count, (count_t)BUFFER_SIZE);
	size_t				BytesRead = 0;
	unsigned char	usTestValue, usStatus;
	GALILDEVICE*	pDevice = (GALILDEVICE*)filp->private_data;

	printk( "GALILPCI: galilpci_read - Entering function.\n" );

	if (pDevice->IOType == DMC1600IOStyle)
	{
		usStatus = 4;
 		usTestValue	= 4;
 	}
 	else
 	{
 		if (pDevice->Model == MODEL_1802)
 		{
 			usTestValue	= 32;
 		}
 		else
 		{
 			// it must be a 1417
 			usTestValue = 0x01;
 		}
 		usStatus = 1;
 	}

	if (  ( inb( pDevice->Address+usStatus )  & usTestValue) == 0)
	{
		printk( "GALILPCI: galilpci_read - 1417 ready to read.\n" );
	}
	else
	{
		printk( "GALILPCI: galilpci_read - 1417 not ready to read.\n");
	}
	// Read the data info a temporary buffer
	while(	( (((unsigned char)inb( pDevice->Address+usStatus )) &	usTestValue) == 0) && 
				(BytesRead < MaxBytes))
	{
		//pDevice->Buffer[ BytesRead++ ] =	(char)ReadData(pDevice);
		pDevice->Buffer[ BytesRead++ ] =	(char)inb( pDevice->Address );
 	}

	// Transfer data to user buffer
	copy_to_user( buf, pDevice->Buffer, BytesRead );

	*ppos =	0;
	
	return BytesRead;
}

//
//	Purpose	:	
//
//	Returns	:	
//
static inline unsigned	long GetTime()
{
	// Get current time in milliseconds
	return (jiffies	% HZ) *	(1000L / HZ);
}

//
//	Purpose	:	determines if there is data to be read
//
//	Returns	:	TRUE - there is else FALSE
//
static inline int ReadyToWrite(GALILDEVICE* pDevice)
{
	int						test = TRUE;
	unsigned char	TestValue;	
 	unsigned long	CurrentTime;
	unsigned long	Timeout	= GetTime() + DEFAULT_TIMEOUT;		
   
	while( test )
	{
		switch( pDevice->IOType )
		{
			case DMC1600IOStyle:
				TestValue =	inb( pDevice->Address+4 );
	 			if ((TestValue & 0x02) == 0)
					return TRUE;
				break;
				
 			case DMC1000IOStyle:
				TestValue =	inb( pDevice->Address+1 );
					
				if (pDevice->Model == MODEL_1802)
				{
					if ( (TestValue & 0x10) == 0 )
						return TRUE;
				}
				else if (pDevice->Model == MODEL_1417)
				{
					printk(	"GALILPCI: ReadyToWrite - this is a 1417.  TestValue=%d.\n", TestValue );
					if (  (TestValue & 0x02)  ) // 1417 half full bit 0=true, 1=false
					{
						printk(	"GALILPCI: ReadyToWrite - not FIFO half full.\n" );
						return TRUE;
					}
				}
		}
		
		CurrentTime	= GetTime();
      
    if (CurrentTime > Timeout)
    {
			printk(	"GALILPCI not ready to write, timedout\n" );
			printk( "GALILPCI: ReadyToWrite - TestValue=%d.\n", TestValue );	 
			return FALSE;
  }	}
	
	return FALSE;
}

//
//	Purpose	:	determines if there is data to be read
//
//	Returns	:	TRUE - there is else FALSE
//
static inline int ReadyToRead(GALILDEVICE* pDevice)
{
	int						test = TRUE;
	unsigned char	TestValue;	
  unsigned long	CurrentTime;
	unsigned long	Timeout	= GetTime() + DEFAULT_TIMEOUT;		
	
	while( test )
	{
		if (pDevice->IOType != DMC1000IOStyle)
 		{
			TestValue =	inb( pDevice->Address+4 );
 			if ((TestValue & (unsigned char)4) == 0)
				return TRUE;
 		}
 		else
 		{
			TestValue =	inb( pDevice->Address+1 );
			if (pDevice->Model == MODEL_1802)
			{
				if ((TestValue & (unsigned char)32 ) == 0)
					return TRUE;
			}
			else
			{
				// this is a 1417
				if ((TestValue & (unsigned char)1) == 0)
					return TRUE;
			}
		}
		
		CurrentTime	= GetTime();
      
    if (CurrentTime > Timeout)
    {
			printk(	"GALILPCI not ready to be read, timedout\n" );	 
			return FALSE;
  }	}
	
	return FALSE;
}

//
//	Purpose	:	Handle writting to the board
//
//	Returns	:	the number of bytes written
//
ssize_t	galilpci_write(
struct file*	filp, 
const char*		buf, 
size_t				count, 
loff_t*				ppos)
{
  int						BurstCount = 0;
  count_t				BytesWritten = 0;
  GALILDEVICE*	pDevice = (GALILDEVICE*)filp->private_data;

	printk( "GALILPCI: galilpci_write - Entering function.\n" );

	// Transfer data from user buffer
	copy_from_user(pDevice->Buffer, buf, count);
	printk( "GALILPCI: galilpci_write - Buffer=%s, Count=%d.\n", buf, count );
	
  // Recheck the ready	to receive flag
	if (ReadyToWrite(pDevice))
	{
		for( BytesWritten=0; BytesWritten<count; BytesWritten++)
		{
			outb( pDevice->Buffer[ BytesWritten ], pDevice->Address );
			//printk( KERN_CRIT "GALILPCI: galilpci_write - outp.\n");
			
			// Recheck the ready to	receive	flag 
			if (++BurstCount > 128)
			{
				if (!ReadyToWrite(pDevice))
					break;	       
				BurstCount = 0;	
			 }	
		}
	}
	else
	{
		printk( "GALILPCI: galilpci_write - Leaving  function with error -20.\n" );
		return -20;
	}

	*ppos =	0;
	
	printk( "GALILPCI: galilpci_write - Leaving  function with success.\n" );
	return BytesWritten;
}

//
//	Purpose	:	
//
//	Returns	:	
//
static inline int ClearFIFO(
GALILDEVICE*	pDevice)
{
	if ( pDevice->IOType == DMC1600IOStyle)
 	{
		clearFIFO( pDevice );
	}
  else	
  {
  		// its a 1417 or 1842
 		disableInterrupts1802( pDevice );
		clearFIFO1802( pDevice );
		enableInterrupts1802( pDevice );
	}
	return 0;
}

// 
//	Purpose	:	release	driver interrupts and memory
//
//	Returns	:	FALSE always
//
int galilpci_release (struct inode *inode, struct file *filp)
{
	GALILDEVICE* pDevice = (GALILDEVICE*)filp->private_data;
	
	ClearFIFO( pDevice );
	
	DisableInterrupts( pDevice	);
	
	MOD_DEC_USE_COUNT;

	return 0;
}

//
//	Purpose	:	Handle device opening
//
//	Returns	:	
//
int galilpci_open(
struct inode*	inode, 
struct file*	filp)
{
	//int	type = TYPE(inode->i_rdev);
  int	num = NUM(inode->i_rdev);
   
	// type	0, check the device number
	if (num	>= MAX_DEVICES)
		return -ENODEV;

	if ( GalilDevice[num].IOType == DMC1600IOStyle)
		printk( "GALILPCI open model=%i, serial#=%i, address=%08x, context=%i\n", 
						GalilDevice[ num ].Model, 
						(int)GalilDevice[ num ].SerialNumber,
						(int)GalilDevice[ num ].Address,
						(int)GalilDevice[ num ].context );	 
	else
		printk( "GALILPCI open model=%i, serial#=%i, address=%08x, address2=%08x, context=%i\n", 
						GalilDevice[ num ].Model, 
						(int)GalilDevice[ num ].SerialNumber,
						(int)GalilDevice[ num ].Address, 
						(int)GalilDevice[ num ].Address2,	 
						(int)GalilDevice[ num ].context );	 
	
	// Use filp->private_data to point to the device data
	filp->private_data = &GalilDevice[ num ];
  	cli();
	// Initialize the FIFO
	ClearFIFO( &GalilDevice[ num ] );
		
	// Enable interrupts
	EnableInterrupts( &GalilDevice[	num ] );
	sti();
	
	// Increment the module	count
	MOD_INC_USE_COUNT;

	return 0; /* success */
}

//
//	Purpose	:	
//
//	Returns	:	
//
static inline int GetIntStatus(
GALILDEVICE*	pDevice,
int*					pusShort)
{
  int      					i;
  long     				rc = 0L;
  unsigned short  usIntValue = 0;
  unsigned short	usDataAddress=0;
  unsigned short	usStatusAddress=0;

  // Do not allow recursive interrupts
  usDataAddress = pDevice->Address;
  switch( pDevice->IOType )
  {
  	case DMC1600IOStyle:	
  		usStatusAddress = usDataAddress + 4;
  	case DMC1000IOStyle:	
  		usStatusAddress = usDataAddress + 1;
  }

	// get the first interrupt and pop it off the stack
  if ( pDevice->InterruptCount > 0 )
  {
  	usIntValue = pDevice->InterruptStatus[ 0 ];
  	   					
  	for (i = 0; i < pDevice->InterruptCount; i++)
  	{
  		pDevice->InterruptStatus[ i ] = pDevice->InterruptStatus[ i+1 ];
  		pDevice->InterruptStatus[ i+1 ] = 0;
  	}
  	pDevice->InterruptCount--;
	}

  // 0 is not a meaningful INTERRUPT status; 0xa8 seems to be a spurious INTERRUPT
  if ((usIntValue > 0x00) && (usIntValue <= 0xff) && (usIntValue != 0xa8))
  {
  	*pusShort = usIntValue;
	}
	
  if (	pDevice->InterruptType == DMC1000InterruptStyle ||
	     	pDevice->InterruptType == DMC1700InterruptStyle)
  {
		outb( 0x02, usStatusAddress );    // tell dmc to start interrupts
		outb( 0x04, usStatusAddress );
		outb(	0x06, usStatusAddress );    // request any pending INTERRUPT
		inb( usStatusAddress );           // and ignore the result
	}

	return rc;
}

//
//	Purpose	:	determines if there is data to be read
//
//	Returns	:	TRUE - there is else FALSE
//
static inline int IsCharAvailable(GALILDEVICE* pDevice)
{
	int 						TestValue;
	
	if (pDevice->IOType != DMC1000IOStyle)
	{
		TestValue =	inb( pDevice->Address+4 );
		if ((TestValue & (unsigned char)4) == 0)
			return TRUE;
	}
	else
	{
		TestValue =	inb( pDevice->Address+1 );
		if (pDevice->Model == MODEL_1417)
		{
			if ((TestValue & 0x01) == 0)
				return TRUE;
		}
		else
		{
			if ((TestValue & (unsigned char)32) == 0)
				return TRUE;
		}
	}
	
	return FALSE;
}

//
//	Purpose	:	determines if we can write the freeze bit to access datarecord
//
//	Returns	:	TRUE 	- success bit written
//						FALSE	- failure bit not written
//
static inline int ReadyToWriteFreezeBit(GALILDEVICE* pDevice)
{
	if ( pDevice->IOType == DMC1600IOStyle )
	{
		if ( !( 0x80 & inb( pDevice->Address+4 )) )
		{
    	outb( 0x10, pDevice->Address+4 );
    	return TRUE;
  }	}
	else
	if ( pDevice->IOType == DMC1000IOStyle )
	{
		if (pDevice->Model == MODEL_1417) 
			return FALSE;	// 1417 doesn't have secondary FIFO
		if ( !(0x04 & inb( pDevice->Address+3 )) )
		{
    	outb( 0x02, pDevice->Address+3 );
    	return TRUE;
  }	}
	
	return FALSE;
}

//
//	Purpose	:	determines if we can read 
//
//	Returns	:	TRUE 	- success
//						FALSE	- failure
//
static inline int ReadyToReadFIFO(GALILDEVICE* pDevice)
{
	if ( pDevice->IOType == DMC1600IOStyle )
	{
		if ( ( 0x08 & inb( pDevice->Address+4 )) )
		{
    			outb( 0x00, pDevice->Address+4 );
    			return FALSE;
  		}
  		else	return TRUE;
  	}
	else
	if ( pDevice->IOType == DMC1000IOStyle )
	{
		if ( (0x01 & inb( pDevice->Address+3 )) )
		{
    			outb( 0x00, pDevice->Address+3 );
    			return FALSE;
  		}
  		else	return TRUE;
	}
	// 1417 has no FIFO	
	return FALSE;
}

//
//	Purpose	:	read the actual data records from the fifo stack
//
//	Returns	:	TRUE always
//
static inline int ReadFIFO(GALILDEVICE* pDevice)
{
	int idx=-1;
	int	amt=sizeof(pDevice->Buffer);
	
	for( idx=0; idx<amt; idx++ )
	  pDevice->Buffer[ idx ] = inb_p( pDevice->Address+12 );
	
	outb( 0x00, pDevice->Address+4 );	// Clear freeze bit 1 in dmc_status2 by writing any number to dmc_mailbox2
	
	return TRUE;
}

//
//	Purpose	:	Retrieve FIFO data record
//
//	Returns	:	nothing
//
static inline int GetFIFODR( 
GALILDEVICE* 	pDevice,
char* 				pData )
{
	if ( ReadyToWriteFreezeBit( pDevice ) )
	{
		printk(	"GALILPCI: GetFIFODR - ReadyToWriteFreezeBit returned true.\n" ); 
		if ( ReadyToReadFIFO( pDevice ) )
		{
			printk(	"GALILPCI: GetFIFODR - ReadyToReadFIFO returned true.\n" ); 
			ReadFIFO( pDevice );
			copy_to_user( pData, pDevice->Buffer, sizeof(pDevice->Buffer) );
			return FALSE;
		}	
		else
			printk(	"GALILPCI unable to acquire read access to FIFO data record\n" );  
	}
	else
		printk(	"GALILPCI unable to acquire write access to FIFO data record\n" );  
	
	return -1;
}

//
//	Purpose	:	Retrieve FIFO data record
//
//	Returns	:	nothing
//
static inline int ValidateSN( 
GALILDEVICE* 	pDevice,
int* 					pData )
{
	return (pDevice->SerialNumber == *pData ? TRUE : FALSE);
}

//
//	Purpose	:	Handle all ioctrl 
//
//	Returns	:	0			on success
//						EINVAL	on failure
//
int galilpci_ioctl(
struct inode*	inode, 
struct file*	filp,
unsigned int	cmd, 
unsigned long	arg	)
{
 	GALILDEVICE*	pDevice = (GALILDEVICE*)filp->private_data;

	switch(cmd) 
	{
	  case GALIL_PCI_CLEARFIFO:			return ClearFIFO( pDevice );
	  case GALIL_PCI_CHARAVAIL:			return IsCharAvailable( pDevice );
	  case GALIL_PCI_INTSTATUS:			return GetIntStatus( pDevice, (int*)arg );
	  case GALIL_PCI_GETFIFODR:			return GetFIFODR( pDevice, (char*)arg );
	  case GALIL_PCI_VALIDATESN:		return ValidateSN( pDevice, (int*)arg );
	}

  return -EINVAL;
}

//
//	Struct that defines all	operations
//
struct file_operations galilpci_fops = {
  
  read:    galilpci_read,
  write:   galilpci_write,
  ioctl:   galilpci_ioctl,
  open:    galilpci_open,
  release: galilpci_release,
  /*
  NULL,
    galilpci_read,
    galilpci_write,
    NULL,	   // galilpci_readdir
    NULL,	   // galilpci_select
    galilpci_ioctl,
    NULL,	   //galilpci_mmap
    galilpci_open,
    NULL,
    galilpci_release,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
  */

};

// 
//	Purpose	:	Interrupt handler
//
//	Returns	:	Nothing
//
void HandleInterrupt( 
int							irq,
void*						dev_id,
struct pt_regs*	regs	)
{
	unsigned char	uchTemp=0;
	unsigned long	ulTemp=0;
	
	PGALILDEVICE pDevice = dev_id;
	
	if ( pDevice->context > -1 && pDevice->context < MAX_DEVICES )
	{
		if ( pDevice->IOType == DMC1600IOStyle)
		{
			uchTemp	= (unsigned char)inb( pDevice->Address+4 );
			
			if ( uchTemp & 0x20 )
			{
				outb( (uchTemp	| 0x40), pDevice->Address+4 );
        uchTemp = inb( pDevice->Address+8 );
			
  			// Only queue interrupt if it is non-zero
				if (uchTemp && pDevice->InterruptCount<MAX_INTERRUPTS )
				{
					pDevice->InterruptStatus[ pDevice->InterruptCount++ ] =	uchTemp;
		}	}	}
		else
		if ( pDevice->IOType == DMC1000IOStyle)
		{
			ulTemp = (unsigned long)inb( pDevice->Address2+0x4C );
		
			// Determine if	the interrupt is ours -	we use uchTemp to preserve the other register values
			if ((ulTemp & (unsigned long)0x47) != 0x47 )
				return;
		
			// If this is a 1417, exit.
			if (pDevice->Model == MODEL_1417)
			{
				outl( 0x05, pDevice->Address2+0x4D );
				return;
			}
			// Read	the status register to determine if a master reset occurred
			uchTemp	= inb( pDevice->Address+1 );
			if (uchTemp & 0x80)
			{
				// Reset interrupt hardware
				// Disable interrupts on the PCI hardware
				disableInterrupts1802( pDevice );
					
				outb( 0x05, pDevice->Address+1 );		// Set up the FIFO for 128 bytes
				outb( 0xF0, pDevice->Address+1 );		// Bidirectionally
																				
				enableInterrupts1802( pDevice );		// Enable interrupts on	the PCI	hardware																					
			}
		  else
		  {
				// Clear the interrupt on the	controller by reading the status
				outb( 6, pDevice->Address+1 );
			 	uchTemp = inb( pDevice->Address+1 );
		
		    // Only queue interrupt if it is non-zero
				if (uchTemp && pDevice->InterruptCount<MAX_INTERRUPTS )
				{
					pDevice->InterruptStatus[ pDevice->InterruptCount++ ] =	uchTemp;
	}	}	}	}
	else
		printk( "GALILPCI: invalid context for device %i\n", pDevice->context );
}

//
// Purpose	:	Ascii to long version written for linux
//
// Returns	: the converted value
//
static inline unsigned long gatol(char *str)
{
	unsigned long	val;
	int						base, c;
	char*					sp;

	val = 0;
	sp = str;
	if ((*sp == '0') && (*(sp+1) == 'x')) 
	{
		base = 16;
		sp += 2;
	} 
	else 
	if (*sp == '0') 
	{
		base = 8;
		sp++;
	} 
	else 
	{
		base = 10;
	}

	for (; (*sp != 0); sp++) 
	{
		c = (*sp > '9') ? (TOLOWER(*sp) - 'a' + 10) : (*sp - '0');
		if ((c < 0) || (c >= base)) 
		{
			val = 0;
			break;
		}
		val = (val * base) + c;
	}
	
	return val;
}

//
// Purpose	:
//
// Returns	:
//
static inline void GetSerialNumber(
GALILDEVICE*	pDevice)
{
	u32  	ulSerialNumber = 0;
	int  	i, idx;
	char 	szBuffer[32];
  u16   uchTemp=0;

	if (pDevice->Model == MODEL_1417) 
	{
		pDevice->SerialNumber = 0;	// 1417 has no serial number
		return;
	}
	
	memset( szBuffer, 0, sizeof(szBuffer) );
	
	if ( pDevice->IOType == DMC1000IOStyle)
	{
		inb( pDevice->Address+1 );
		outb( 0x01, pDevice->Address+1 );
		outb( 0x80, pDevice->Address+1 );
		outb( 0x01, pDevice->Address+1 );
		outb( 0x80, pDevice->Address+1 );
		inb( pDevice->Address+1  );
		outb( 0x05, pDevice->Address+1 );	// Set up the FIFO for 128 bytes
		outb( 0xF0, pDevice->Address+1 );	// Bidirectionally

		// Must re-init interrupts whenever we clear
		outb( 0x02, pDevice->Address+1 );	// Tell DMC to start interrupts
		outb( 0x04, pDevice->Address+1 );
		outb( 0x06, pDevice->Address+1 );	// Clear any old interrupts
		inb( pDevice->Address+1  );
	}
	else
	{
		outb( 6, pDevice->Address+8 );
	}

	ReadyToWrite( pDevice );
  outb( 'M', pDevice->Address );
  outb( 'G', pDevice->Address );
  outb( '_', pDevice->Address );
  outb( 'B', pDevice->Address );
  outb( 'N', pDevice->Address );
  outb( '\r', pDevice->Address );

  // Read the data info a temporary buffer
  for( idx=i=0; (i <sizeof(szBuffer)-1); i++ )
  {
  	if (!ReadyToRead( pDevice ))
  		continue;
  		
  	uchTemp = inb( pDevice->Address );

    if (uchTemp == ' ')	     	continue;
    if (uchTemp == 0x0D)    	break;
    if (uchTemp == '.')	    	break;
    switch( uchTemp )
    {
    	case '0':	case '1':	case '2':	case '3':	case '4':	
    	case '5':	case '6':	case '7':	case '8':	case '9':
    	  szBuffer[ idx++ ] = uchTemp;
	}	}
	
  szBuffer[i] = 0;	// teminate it
  ulSerialNumber = gatol(szBuffer);
	
	pDevice->SerialNumber = ulSerialNumber;
}
	
//
//	Purpose	:	Module initialization
//						Looks up vendor and model from PCI linklist
//
//	Returns	:	0 on success
//						result from registering	the device
//
static inline int FindPciDevice()
{
	int		 					r2=0,r=0, rcode=0, irqr=0;
	struct pci_dev*	pdev = NULL;
	u16							uModel;
					
	if ( pci_present() )
	{
		while( (pdev = pci_find_device(	VENDOR_ID, DEVICE_ID, pdev )) )
		{
			rcode =	TRUE;																			// at least one found
			
			GalilDevice[ ++context ].context = context;
			GalilDevice[ context ].Model = galilpci_model0;
			GalilDevice[ context ].Address = galilpci_addr0;
			GalilDevice[ context ].Interrupt = galilpci_irq0;
			GalilDevice[ context ].IOType = DMC1600IOStyle;
			GalilDevice[ context ].InterruptCount = 0;

		 	GalilDevice[ context ].Interrupt = pdev->irq;

			GalilDevice[context].Address = pci_resource_start(pdev, 2);
			GalilDevice[context].Address2 = pci_resource_start(pdev,1);
			//	GalilDevice[ context ].Address	= (pdev->base_address[ 2 ] & PCI_BASE_ADDRESS_IO_MASK );
			//	GalilDevice[ context ].Address2	= (pdev->base_address[ 1 ] & PCI_BASE_ADDRESS_IO_MASK );
	  	GalilDevice[ context ].Model = pdev->device;
			GalilDevice[ context ].InterruptCount = 0;

			//	get the model number
      pci_read_config_word( pdev, PCI_SUBSYSTEM_ID, &uModel );
      switch( uModel )
      {
      	case SUB_DEVICE_1600:
	    		GalilDevice[ context ].Model = MODEL_1600;
					GalilDevice[ context ].IOType = DMC1600IOStyle;
					GalilDevice[ context ].InterruptType = DMC1600InterruptStyle;
          
			  	if ( -1<check_region( GalilDevice[ context ].Address, MEM_SIZE ) )
			  	{
						request_region( GalilDevice[ context ].Address, MEM_SIZE, "GalilPCI" );
		  		}
		  		else
						printk( "GALILPCI: cannot aquire memory address=%08x", (int)GalilDevice[ context ].Address ); 
					break;
					
		case SUB_DEVICE_1800:
	    		GalilDevice[ context ].Model  = MODEL_1800;
					GalilDevice[ context ].IOType = DMC1600IOStyle;
					GalilDevice[ context ].InterruptType = DMC1600InterruptStyle;
          
			  	if ( -1<check_region( GalilDevice[ context ].Address, MEM_SIZE ) )
			  	{
						request_region( GalilDevice[ context ].Address, MEM_SIZE, "GalilPCI" );
		  		}
		  		else 
						printk( "GALILPCI: region already reserved, address=%08x", (int)GalilDevice[ context ].Address ); 
					break;
					
		case SUB_DEVICE_1842:
	    		GalilDevice[ context ].Model  = MODEL_1802;
					GalilDevice[ context ].IOType = DMC1000IOStyle;
					GalilDevice[ context ].InterruptType = DMC1700InterruptStyle;
          
			  	r = check_region( GalilDevice[ context ].Address, MEM1_SIZE-1 );
				  r2 = check_region( GalilDevice[ context ].Address2, MEM2_SIZE-1 );
				  
				 if ( r >= 0 && r2 >= 0 )
				 {
						request_region( GalilDevice[ context ].Address, MEM1_SIZE, "GalilPCI" );
						request_region( GalilDevice[ context ].Address2, MEM2_SIZE, "GalilPCI" );
		  		 }
		  		 else
		  		 {
		  			printk( "GALILPCI: region already reserved drvier terminating\n" );
		  			return 0;
		  		 }  
		  		 break;
		 case SUB_DEVICE_1417:
		 	GalilDevice[ context ].Model  = MODEL_1417;
					GalilDevice[ context ].IOType = DMC1000IOStyle;
					GalilDevice[ context ].InterruptType = DMC1400InterruptStyle;
          
			  		r = check_region( GalilDevice[ context ].Address, MEM1_SIZE-1 );
				  	r2 = check_region( GalilDevice[ context ].Address2, MEM2_SIZE-1 );
				  
				 if ( r >= 0 && r2 >= 0 )
				 {
						request_region( GalilDevice[ context ].Address, MEM1_SIZE, "GalilPCI" );
						request_region( GalilDevice[ context ].Address2, MEM2_SIZE, "GalilPCI" );
		  		 }
		  		 else
		  		 {
		  			printk( "GALILPCI: region already reserved drvier terminating\n" );
		  			return 0;
		  		 }  
		  		 break;
		 }
    	
    	// disable the hardware so it doesn't fire during initialization
			irqr = request_irq( GalilDevice[ context ].Interrupt, &HandleInterrupt, SA_SHIRQ, "GalilPCI", &GalilDevice[ context ] );
			
			switch(	irqr )
		  {
				default:	printk(	"GALILPCI: cannot enable IRQ	%i \n",	GalilDevice[ context ].Interrupt );	 	return 0;
				case 0:		break;
			}
			
			GetSerialNumber( &GalilDevice[ context ] );
	
			printk( "GALILPCI: vendor \"%08x\", Model %s, SN \"%ld\", irq=%i, context %i\n", 
							pdev->vendor, 
							GalilDevice[ context ].Model==MODEL_1802 
								? "1802" 
								: GalilDevice[ context ].Model==MODEL_1800
									? "1800"
									: GalilDevice[ context ].Model==MODEL_1417
									? "1417"
									: "1600",
							GalilDevice[ context ].SerialNumber,
							pdev->irq, 
							context ); 
	  }

		if ( !rcode )
			printk( "GALILPCI:	can not	find PCI device\n" );
	}
	else
		printk( "GALILPCI:	PCI bus	not installed on this machine.\n" );
		
	return rcode;
}

//
//	Purpose	:	Module initialization
//
//	Returns	:	0 on success
//						result from registering	the device
//
int init_module(void)
{
	int result=0;

	#ifndef	GALIL_DEBUG
  register_symtab(NULL); 		// otherwise, leave global symbols visible
	#endif

  #ifndef SET_MODULE_OWNER
     #define SET_MODULE_OWNER(structure)
#endif


  SET_MODULE_OWNER(&galilpci_fops);
	result = register_chrdev( 0, DRIVER_NAME, &galilpci_fops );

	if ( 0 < result	)
	{
		if ( FindPciDevice() )
		{
			galilpci_major = result; 	// dynamic
	}	}
	else
	{
		printk( "GALILPCI: can not	get major %d, return=%i\n", galilpci_major, result );
		return result;
	}
	
	return 0;
}

//
//	Purpose	:	Handle module cleanup
//
//	Returns	:	nothing
//
void cleanup_module(void)
{
	int i;
	
  cli();																							// free the irq
	for( i=0; i<=context; i++ )
	{
		free_irq( GalilDevice[ i ].Interrupt, &GalilDevice[ i ] );
 		switch( GalilDevice[ i ].IOType )									// Release the I/O port	resource
 	 	{
 	 		default:
 				release_region(	GalilDevice[ i ].Address, MEM_SIZE );
				break;
			
 	 		case DMC1000IOStyle:
				release_region(	GalilDevice[ i ].Address, MEM1_SIZE	);
				release_region(	GalilDevice[ i ].Address2, MEM2_SIZE	);
	}	}
	sti();
 
	unregister_chrdev( galilpci_major, DRIVER_NAME ); 	// unregisture the device first
	
  printk( "GALILPCI: unregistured GalilPCI driver\n" );
}

