/* $Id: if_lxfreeze.c,v 1.5 2007/01/25 15:13:06 cbass Exp $ */
/*****************************************************************************/
/**
 *  \file if_lxfreeze.c
 *  \brief lxfreeze idl, interface implementation 
 * 
 *  \date 12/22/06 15:24:14 CET
 *  \author Sebastian Sumpf <Sebastian.Sumpf@inf.tu-dresden.de>
 */ 
/******************************************************************************/
#include <l4/env/env.h>
#include <l4/generic_ts/generic_ts.h>

#ifdef USE_TASKLIB
#include <l4/task/task_server.h>
#endif

#include "lxfreeze-server.h"
#include "__types.h"
#include "__lxcalls.h"
//#include "__ldso_mmap.h"

/* from package ldso (see Makefile private includes) */
#include "emul_linux.h"

static l4env_infopage_t *lx_infopage;
static l4_addr_t lx_eip, lx_esp;

static int lx_image_saved = 0;

/* add ldso mmap dataspaces to us/pager */
static int __handle_mmap_infopage(l4dm_dataspace_t ds, l4_size_t size)
{
	int i, error;
	l4_addr_t map_addr;
	lx_page_fault_t fault;
	struct mmap_infopage_t *  info;

	if((error =  lx_ds_get_map_addr(lx_task_get_frozen(), ds.id, &map_addr)) < 0)
		return error;

	info = (struct mmap_infopage_t *)map_addr;
	
	for(i = 0; i < info->section_num; i++)
	{
		LOGd(DBG_SUSP, "Got mmap_info at: %08lx size: %08x ds_id %u", 
		     info->section[i].addr, info->section[i].size, info->section[i].ds.id);

		fault.virt_addr = info->section[i].addr;
		fault.virt_size = info->section[i].size;
		fault.ds_id     = info->section[i].ds.id;
		fault.flags     = 0;
		
		/* dataspace will be copied at transfer_owned, since l4linux own them as 
		 * "anon memory" dataspaces */
		if((error = lx_pager_add_ds(fault)) < 0)
			return error;
	}	
	return 0;
}
/******************************************************************************
 *** SUSPEND functions
 *****************************************************************************/

/* copy l4env infopage */	
int
if_l4dm_lxfreeze_find_env_infopage_component(CORBA_Object _dice_corba_obj,
                                             const void * page,
                                             l4_size_t size,
                                             CORBA_Server_Environment *_dice_corba_env)
{
	l4_addr_t map_addr;
	lx_page_fault_t fault;
	unsigned long ds_id;
	int error, i, not_found;
		
	not_found = lx_ds_search_equal(lx_task_get_frozen(), (void*)page, size, &ds_id);
	
	if(not_found)
		enter_kdebug("Couldn't find dataspace of L4Env infopage");
	
	LOGd(DBG_SUSP, "Found L4Env infopage at ds_id %lu", ds_id);

	
	fault.virt_addr  = L4ENV_INFOPAGE_AT;
	fault.virt_size  = size;
	fault.ds_id      = ds_id;
	fault.flags      = 0;

	if((error = lx_ds_get_map_addr(lx_task_get_frozen(), ds_id, &map_addr)) < 0)
		return error;
	
	lx_infopage = (l4env_infopage_t *)map_addr;
	
	/* pager handles pagefaults for this dataspace, also */
	if((error = lx_pager_add_ds(fault)) < 0)
		return error;

	for(i = 0; i < lx_infopage->section_num; i++)
	{
		if(l4dm_is_invalid_ds(lx_infopage->section[i].ds))
			continue;

		/* look for mmap-infopage of pkg/ldso (patched version) */
		if(lx_infopage->section[i].info.id == MMAP_SAVE_EXEC_ID)
		{
			LOGd(DBG_SUSP, "Found MMAP info-page at 0x%08x", MMAP_INFOPAGE_AT);
			error = __handle_mmap_infopage(lx_infopage->section[i].ds, 
			                               lx_infopage->section[i].size);
			continue;
		}
		
		fault.virt_addr = lx_infopage->section[i].addr; 
		fault.virt_size = lx_infopage->section[i].size;
		fault.ds_id     = lx_infopage->section[i].ds.id;
		fault.flags     = 0;

		if((error = lx_pager_add_ds(fault)) < 0)
			return error;
	}
	return 0;
}

/* copy inital stack of linux which is paged by loader (l4rm stack) */
int
if_l4dm_lxfreeze_find_stack_component(CORBA_Object _dice_corba_obj,
                                      const void * page,
                                      l4_size_t size,
                                      CORBA_Server_Environment *_dice_corba_env)
{
  int error, not_found;
	lx_page_fault_t  fault;
	unsigned long ds_id;
	
	not_found = lx_ds_search_equal(lx_task_get_frozen(), (void*)page, size, 
	                               &ds_id);
	
	if(not_found)
		enter_kdebug("Couldn't find intial stack dataspace");
	
	LOGd(DBG_SUSP, "Found initial stack at ds_id %lu", ds_id);
	
	fault.virt_addr  = lx_infopage->stack_low;
	fault.virt_size  = size;
	fault.ds_id      = ds_id;
	fault.flags      = 0;

	/* pager handles pagefaults for this dataspace, also */
	if((error = lx_pager_add_ds(fault)) < 0)
		return error;
	
	return 0;
}

/* get id of pager */
int
if_l4dm_lxfreeze_get_start_pager_component (CORBA_Object _dice_corba_obj, l4_threadid_t * pager,
		                                            CORBA_Server_Environment *_dice_corba_env)
{
	*pager = lx_pager_get_id();
	return 0;
}

/* handle pagefaults in addition to dataspace faults for ds */
int
if_l4dm_lxfreeze_handle_pagefault_component(CORBA_Object _dice_corba_obj,
                                            l4_uint32_t ds_id, l4_addr_t addr, l4_size_t size, 
                                            l4_uint32_t flags,
                                            CORBA_Server_Environment *_dice_corba_env)
{
	lx_page_fault_t fault;

	fault.ds_id     = ds_id;
	fault.virt_addr = addr;
	fault.virt_size = size;
	fault.flags     = flags;
	
	return lx_pager_add_ds(fault);
}


void
if_l4dm_lxfreeze_finish_component(CORBA_Object _dice_corba_obj,
                                  l4_addr_t start_eip,
                                  l4_addr_t start_esp,
                                  unsigned long drain_ds_id,
                                  CORBA_Server_Environment *_dice_corba_env)
{
	LOGd(DBG_SUSP, "Received finish request at EIP: 0x%08x, ESP: 0x%08x\n",
                 (unsigned int)start_eip, (unsigned int)start_esp);
	
	/* release LX_RELEASE dataspaces */
	lx_pager_chk_release();

	lx_eip = start_eip;
	lx_esp = start_esp;
	lx_image_saved = 1;

#if PROFILE
lx_timer();
#endif

#ifdef USE_TASKLIB
  lx_infopage->parent_id = l4task_get_server();
#endif
}

/*******************************************************************************
 *** WAKE UP CALL
 ******************************************************************************/
void
if_l4dm_lxfreeze_wake_up_component (CORBA_Object _dice_corba_obj,
                                    l4_taskid_t *task_id,
                                    l4_addr_t start_eip,
                                    l4_addr_t start_esp,
                                    CORBA_Server_Environment *_dice_corba_env)
{
	l4_taskid_t clone_id;
	int error;
	lx_task_t * task;
	l4_threadid_t pager = lx_pager_get_id();	
	if(!lx_image_saved)
		return;
	
	if((error = l4ts_allocate_task(0, &clone_id)))
		return;
	
	LOGd(DBG_SUSP, "New linux id is "l4util_idfmt, l4util_idstr(clone_id));
	
	error = lx_task_add(clone_id);
	if(error < 0) {
		l4ts_free_task(&clone_id);
		return;
	}
	
  lx_task_get_pos(error, &task);
	LOGd(DBG_VERB, "Got new task pos %d, task_id "l4util_idfmt, 
	     error, l4util_idstr(task->task_id));
	
	/* new stats for task */
	lx_stats_get_free(task);

	/* set default memserver to dm_phys */
	lx_infopage->memserv_id = l4dm_memphys_find_dmphys();

	/* set global */
	lx_frozen = 1;
	
#if PROFILE
lx_timer();
#endif
	
	l4ts_create_task(
	  &clone_id,
	  lx_eip,
	  lx_esp,
	  0xff,
	  &pager,
	  L4THREAD_DEFAULT_PRIO,
	  "l4lx_clone",
		0
	);

	*task_id = clone_id;
}

void
if_l4dm_lxfreeze_loaded_component (CORBA_Object _dice_corba_obj,
                                   const l4_threadid_t *signal_id,
                                   CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t *task = lx_task_get_frozen();
	task->task_id = *_dice_corba_obj;
	task->l4x_signal_id = *signal_id;
}

void
if_l4dm_lxfreeze_signal_id_component(CORBA_Object _dice_corba_obj,
                                     const l4_threadid_t *signal_id,
                                     CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t *task;

	if(lx_task_get(*_dice_corba_obj, &task) < 0)
		return;
	task->l4x_signal_id = *signal_id;
}


/*******************************************************************************
 *** required DSM interface implementations 
 ******************************************************************************/
long
if_l4dm_generic_fault_component(CORBA_Object _dice_corba_obj,
                                 unsigned long ds_id,
                                 unsigned long offset,
                                 l4_snd_fpage_t *page,
                                 CORBA_Server_Environment *_dice_corba_env)
{
	l4_addr_t addr;
	int rw = offset & 2;
	lx_ds_t * lxds;
	lx_page_t * lxpage;
	int error;

	if(DBG_FAULT_RO || rw)
		LOGd(DBG_FAULT, "["l4util_idfmt"] Got fault call ds_id: %lu offset: 0x%08lx (%s)\n",
			l4util_idstr(*_dice_corba_obj), ds_id, offset, (rw)?"rw":"ro");
	
	offset &= L4_PAGEMASK;
	page->snd_base = 0;
	page->fpage = l4_fpage(0,0,0,0);

	if(lx_frozen)
	{
		error = lx_page_fault((l4_taskid_t)*_dice_corba_obj, ds_id, offset, rw, &addr);
		if(error) return error;
	}
	else 
	{
		error = lx_ds_get(lx_task_get_frozen(), ds_id, &lxds);
		if(error < 0) return error;
		
		error = lx_page_get(lxds, offset, &lxpage);
		if(error) return error; 
		
		addr = lxpage->addr;
	}

	if(rw)
		page->fpage = l4_fpage(addr, L4_LOG2_PAGESIZE, L4_FPAGE_RW, L4_FPAGE_MAP);
	else	
		page->fpage = l4_fpage(addr, L4_LOG2_PAGESIZE, L4_FPAGE_RO, L4_FPAGE_MAP);
	
	if(DBG_FAULT_RO || rw)	
		LOGd(DBG_FAULT, "Mapping addr: %08lx\n", addr);
	return 0;
	
}


long
if_l4dm_generic_map_component(CORBA_Object _dice_corba_obj,
                              unsigned long ds_id,
                              unsigned long offset,
                              unsigned long size,
                              unsigned long rcv_size2,
                              unsigned long rcv_offs,
                              unsigned long flags,
                              l4_snd_fpage_t *page,
                              CORBA_Server_Environment *_dice_corba_env)
{ 
	l4_addr_t addr;
	int rw = flags & L4DM_RW;
	int error;
	lx_ds_t * lxds;
	lx_page_t * lxpage;
	
	LOGd(DBG_MAP, "["l4util_idfmt"] Got map call ds_id: %lu offset: 0x%08lx (%s)",
	     l4util_idstr(*_dice_corba_obj), ds_id, offset, (rw)?"rw":"ro");

	/* set dummy page */
	page->snd_base = 0;
	page->fpage = l4_fpage(0, 0, 0, 0);
	offset &= L4_PAGEMASK;
	
	if(lx_frozen)
	{
		error = lx_page_fault((l4_taskid_t)*_dice_corba_obj, ds_id, offset, rw, &addr);
		if(error) return error;
	}
	else 
	{
		error = lx_ds_get(lx_task_get_frozen(), ds_id, &lxds);
		if(error < 0) return error;
		
		error = lx_page_get(lxds, offset, &lxpage);
		if(error) return error; 
		
		addr = lxpage->addr;
	}
	/* doesn't work another way, since we will receive normal sized
	 * page faults everywhere */
//	l4_touch_rw((void*)addr, L4_PAGESIZE);
	if(rw) 
	{
		page->fpage = l4_fpage(addr, L4_LOG2_PAGESIZE, L4_FPAGE_RW, L4_FPAGE_MAP);
	}
	else
	{
		page->fpage = l4_fpage(addr, L4_LOG2_PAGESIZE, L4_FPAGE_RO, L4_FPAGE_MAP);
	}
	LOGd(DBG_MAP, "Mapping addr: %08lx\n", addr);
	return 0;
}


/*******************************************************************************
 * OPEN/CLOSE + needed mem/generic interface calls
 ******************************************************************************/
long
if_l4dm_mem_open_component (CORBA_Object _dice_corba_obj,
                            unsigned long size,
                            unsigned long align,
                            unsigned long flags,
                            const char* name,
                            l4dm_dataspace_t *ds,
                            CORBA_Server_Environment *_dice_corba_env)
{
	int err;
	l4dm_dataspace_t ds_local;
	dsmlib_ds_desc_t * ds_desc;
	void * map_addr;
	LOGd(DBG_OPEN, "Open request for %s", name);
	size = (size + L4_PAGESIZE - 1) & L4_PAGEMASK;

	if((err = l4dm_mem_open(L4DM_DEFAULT_DSM,
	                        size, align, flags, name, &ds_local)))
		return err;
	
	if((ds_desc = dsmlib_create_dataspace()) == NULL) 
	{
		l4dm_close(&ds_local);
		return -L4_ENOMEM;
	}

	if((err = lx_ds_add_ds(ds_local, ds_desc, size, &map_addr)))
	{		
			dsmlib_release_dataspace(ds_desc);
			l4dm_close(&ds_local);
			return err;
	}
	
	dsmlib_set_owner(ds_desc, (l4_threadid_t)*_dice_corba_obj);
	dsmlib_set_name(ds_desc, name);
	
	ds->id = dsmlib_get_id(ds_desc);
	ds->manager = l4_myself();
	return 0;
}


long
if_l4dm_mem_resize_component (CORBA_Object _dice_corba_obj,
                              unsigned long ds_id,
                              unsigned long new_size,
                              CORBA_Server_Environment *_dice_corba_env)
{
	lx_ds_t * lxds;
	int err;
	l4_offs_t offs;
	l4_addr_t map_addr;
	
	new_size = (new_size + L4_PAGESIZE - 1) & L4_PAGEMASK;
	lx_ds_get(lx_task_get_frozen(), ds_id, &lxds);
	
	LOGd(DBG_RESIZE, "Resize called for %lu old %08x, new %08lx",
		ds_id, lxds->size, new_size);

	if(new_size == lxds->size)
	{
		LOGd(DBG_RESIZE, "Same size -> continue");
		return 0;
	}

	if((err = l4dm_mem_resize(&(lxds->ds), new_size)))
		return err;
	
	if(new_size > lxds->size)
	{
		if((err = l4rm_attach(&(lxds->ds), new_size - lxds->size, lxds->size,
		                      L4RM_MAP|L4DM_RW|L4RM_LOG2_ALIGNED, 
		                      (void*)&map_addr)))
			return err;
		
		for(offs = lxds->size; offs < new_size; offs += L4_PAGESIZE)
		{
			LOGd(DBG_RESIZE, "Adding offs %08lx addr: %08lx", 
			     offs, map_addr + offs - lxds->size);
			if((err = lx_page_add(lxds, offs, map_addr + offs - lxds->size, 0)))
				return err;
		}
	}
	else 
	{
		for(offs = new_size; offs < lxds->size; offs += L4_PAGESIZE)
			lx_page_del(lxds, offs);
	}
	lxds->size = new_size;
	return 0;
}


long
if_l4dm_generic_share_component (CORBA_Object _dice_corba_obj,
                                 unsigned long ds_id,
                                 const l4_threadid_t *client,
                                 unsigned long flags,
                                 CORBA_Server_Environment *_dice_corba_env)
{
	lx_ds_t * lxds;
	int err;
	if((err = lx_ds_get(lx_task_get_frozen(), ds_id, &lxds)) < 0)
		return err;

	return l4dm_share(&(lxds->ds), *client, flags);
}


long
if_l4dm_generic_transfer_component (CORBA_Object _dice_corba_obj,
                                    unsigned long ds_id,
                                    const l4_threadid_t *new_owner,
                                    CORBA_Server_Environment *_dice_corba_env)
{
	lx_ds_t * lxds;
	int err;
	if((err = lx_ds_get(lx_task_get_frozen(), ds_id, &lxds)) < 0)
		return err;
	dsmlib_set_owner(lxds->ds_desc, *new_owner);
	return 0;
}


long
if_l4dm_generic_close_component (CORBA_Object _dice_corba_obj,
                                 unsigned long ds_id,
                                 CORBA_Server_Environment *_dice_corba_env)
{
	/* don't let the loader close dataspces after Linux suspend */
	if(l4_task_equal(*_dice_corba_obj, loader_id) && lx_image_saved)
		return 0;
	
	return lxif_close_ds(ds_id);
}

long lxif_close_ds(unsigned long ds_id)
{

	lx_ds_t * lxds;
	int err;

	if((err = lx_ds_get(lx_task_get_frozen(), ds_id, &lxds)) < 0)
		return err;
	
	LOGd(DBG_CLOSE, "Closing datsaspace %s (%u)",
	     dsmlib_get_name(lxds->ds_desc), dsmlib_get_id(lxds->ds_desc));

/*
	if((err = l4dm_close(&(lxds->ds))))
		return err;*/

	lx_ds_del(lx_task_get_frozen(), ds_id);
	return 0;
}


long
if_l4dm_mem_phys_addr_component (CORBA_Object _dice_corba_obj,
                                 unsigned long ds_id,
                                 unsigned long offset,
                                 l4_size_t size,
                                 unsigned long *paddr,
                                 l4_size_t *psize,
                                 CORBA_Server_Environment *_dice_corba_env)
{
	lx_ds_t * lxds;
	int err;
	
	if((err = lx_ds_get(lx_task_get_frozen(), ds_id, &lxds)) < 0)
		return err;

return l4dm_mem_ds_phys_addr(&(lxds->ds), offset, size, paddr, psize);
}


long
if_l4dm_mem_size_component (CORBA_Object _dice_corba_obj,
                            unsigned long ds_id,
                            l4_size_t *size,
                            CORBA_Server_Environment *_dice_corba_env)
{
	lx_ds_t * lxds;
	int err;
	
	if((err = lx_ds_get(lx_task_get_frozen(), ds_id, &lxds)) < 0)
		return err;
	
	*size = lxds->size;
	return 0;
}

/*******************************************************************************
 *** EXIT EVENT HANDLING
 ******************************************************************************/
void
if_l4dm_lxfreeze_hard_exit_component(CORBA_Object _dice_corba_obj,
                                     const l4_taskid_t *task_id,
                                     CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t * task;
	l4_umword_t w0 = 0, w1 = 0;
	l4_msgdope_t result;

	if(lx_task_get(*task_id, &task) < 0 || l4_is_invalid_id(task->l4x_signal_id))
		return;

	l4_ipc_send(task->l4x_signal_id, L4_IPC_SHORT_MSG, w0, w1, L4_IPC_NEVER, 
	            &result);
}

void
if_l4dm_lxfreeze_linux_exit_component(CORBA_Object _dice_corba_obj,
                                      const l4_taskid_t *src_id,
                                      CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t * task;
	int error;
	/* one of our tasks ? */
	if((error = lx_task_get(*src_id, &task)) == -L4_ENOTFOUND)
	{
		/* try not yet frozen */
		task = lx_task_get_frozen();
		error = 0;
		if(!l4_task_equal(task->task_id, *src_id) || lx_image_saved)
			return;
	}
	else
		l4ts_free_task(src_id);

	LOGd(DBG_CLEAN, "Cleaning task "l4util_idfmt"\n", l4util_idstr(*src_id));
	l4semaphore_down(&lx_clean_sem);
	lx_task_del(task, error);
	lx_cow_compactify();
	l4semaphore_up(&lx_clean_sem);
}

/*******************************************************************************
 *** STATISTICS
 ******************************************************************************/

void
if_l4dm_lxfreeze_stats_get_component (CORBA_Object _dice_corba_obj,
                                      void  **stats,
                                      l4_size_t *size,
                                      CORBA_Server_Environment *_dice_corba_env)
{
	l4_size_t mem_size, mem_free;
	
	l4dm_memphys_poolsize(L4DM_MEMPHYS_DEFAULT, &mem_size, &mem_free);
	lx_stats.mem_free = mem_free;
	lx_stats.mem_total = mem_size;

	*stats = &lx_stats;
	*size = sizeof(lx_stats_t);
}
