/* $Id:$ */
/*****************************************************************************/
/**
 *  \file hash.c
 *  \brief hashing thread 
 * 
 *  \date 01/21/07 12:56:49 CET
 *  \author Sebastian Sumpf <Sebastian.Sumpf@inf.tu-dresden.de>
 */ 
/******************************************************************************/
#include <l4/log/l4log.h>
#include <l4/sys/syscalls.h>

#include "__types.h"
#include "__lxcalls.h"
#include "__hash.h"


#define PAGE_AGAINST_CHANGED  1
#define PAGE_HASH_COL				  2 
#define PAGE_EQUAL     				3

#define HASHSIZE(n) ((l4_uint32_t)1<<(n))
#define HASHMASK(n) (HASHSIZE(n) - 1)
l4semaphore_t lx_orig_sem = L4SEMAPHORE_INIT(1);
typedef struct queue_t {
	lx_hash_t * hash;
	struct queue_t * next;
} queue_t;

hash_level_t hash_levels[3] = {
	(hash_level_t){3008,1088, 0xFFFFFFFFULL,    0, 32},
	(hash_level_t){1024,  64, 0xFFFFULL << 32, 32, 16},
	(hash_level_t){64  ,   0, 0xFFFFULL << 48, 48, 16}
};

static l4slab_cache_t queue_cache;
static l4slab_cache_t hash_cache;
static queue_t * waiting_queue = NULL;
static queue_t * last = NULL;
static avl_tree hash_avl = AVL_TREE_INITIALIZER;
static int hash_cur_level = -1;


/* zero page dummys */
static lx_page_t zero;
static lx_ds_t zero_ds = LX_DS_INITIALIZER;

/*******************************************************************************
 *** hash level functions 
 ******************************************************************************/
static l4_uint64_t 
hash_func(l4_addr_t addr, l4_uint64_t old_key)
{
	l4_uint32_t key;
	l4_uint64_t ret;
	hash_level_t level = hash_levels[hash_cur_level];

	/* using 4 BYTE chunk-length */
	l4_size_t real_length = level.length / 4;	
	
	key = hashword(((void*)addr + level.offset), real_length, 0);
	if(level.bits < 32)
		key = key & HASHMASK(level.bits);
	
	LOGd(DBG_HASH_FUNC, "32-key %08x, level %d shift %u\n", key, hash_cur_level, level.shift);
	ret = ((l4_uint64_t)key << level.shift) | old_key;
	return ret;
}

static int 
is_hashed(l4_uint64_t key)
{
	int ret;
	l4_uint64_t test;
	test = key & hash_levels[hash_cur_level].mask;
	if(test)
		ret = 1;
	else
		ret = 0;
	return ret;
}	

static int avl_cmp(void * a, void * b)
{
	l4_uint64_t key_a = ((lx_hash_t *)a)->key;
	l4_uint64_t key_b = ((lx_hash_t *)b)->key;
	l4_uint64_t mask = 0;
	l4_int64_t cmp;
	int ret, i;

	for(i = LX_HASH_LEVELS - 1; i >= hash_cur_level; i--)
		mask |= hash_levels[i].mask;

	key_a &= mask;
	key_b &= mask;
	
	cmp = (l4_int64_t)(key_a - key_b);
	
	if(cmp < 0)
		ret = -1;
	else if(cmp > 0)
		ret =  1;
	else
		ret = 0;
	
	LOGd(DBG_HASH_CMP, "level %d, mask %016llx\n\tkey_a %016llx\n\t" \
			"key_b %016llx\n\tdiff %d",
			hash_cur_level, mask, key_a, key_b, ret);
	
	return ret;
}

/*******************************************************************************
 * waiting queue handling
 ******************************************************************************/

static void queue_add(lx_hash_t * hash)
{
	/* last inserted hash element */
	static queue_t * last_elem = NULL;
	static l4semaphore_t sync_sem = L4SEMAPHORE_INIT(1);
	queue_t * new_elem;
	
	if(IS_ZERO_PAGE(hash->page->flags) &&
			GET_PAGE_RETRY(hash->page->flags) >= LX_RO_MAP_RETRY)
	{
		LOGd(DBG_HOT, "Skipped hot zero page at %08lx", hash->page->addr);
		return;
	}
	
	queue_t * new = (queue_t *)lx_slab_alloc(&queue_cache);
	if(new == NULL)
		return;
	
	new->hash = hash;
	new->next = NULL;
	l4semaphore_down(&sync_sem);
	/* always insert previous element, because of lacking inter-task
	 * synchronization. For example, a page can be unmapped by the hashing thread
	 * before it has been read/written to by L4Linux, leading to a double page
	 * fault */
	if(last_elem == NULL)
	{
		last_elem = new;
		l4semaphore_up(&sync_sem);
		return;
	}
	else {
		new_elem = last_elem;
		last_elem = new;
	}
	
	if(waiting_queue == NULL)
		waiting_queue = last = new_elem;
	else
	{
		last->next = new_elem;
		last = new_elem;
	}
	l4semaphore_up(&sync_sem);
}

static lx_hash_t * queue_get_head(void)
{
	queue_t * head = NULL;
	lx_hash_t * hash = NULL;
	int invalid_flag = 0;
	
	do {
		head = waiting_queue;
		if(head == NULL)
			return NULL;

		if(head->next != NULL)
			waiting_queue = head->next;
		else
			waiting_queue = last = NULL;
		
		hash = head->hash;

		/* got invalidated page object, free it */
		if(IS_INVALID(hash->page->flags))
		{
			LOGd(DBG_CLEAN, "Releasing invalid page %08lx\n", hash->page->addr);
			lx_page_release(hash->page);
			invalid_flag = 1;
			hash = NULL;
		}
		else 
		{
			invalid_flag = 0;
		}
		lx_slab_free(&queue_cache, (void*)head);
	} while(invalid_flag);

	return hash;
}

static void unmap(lx_hash_t * hash, int flags)
{
	l4_fpage_unmap(
			l4_fpage(lx_page_get_map_addr(hash->page), 
				L4_LOG2_PAGESIZE, 0, 0),
//			L4_FP_FLUSH_PAGE | L4_FP_OTHER_SPACES);
			//L4_FP_REMAP_PAGE | L4_FP_OTHER_SPACES);
			flags | L4_FP_OTHER_SPACES);
}

static void hash_unmap(lx_hash_t * hash)
{
	unmap(hash, L4_FP_FLUSH_PAGE);
}

static void hash_remap_ro(lx_hash_t * hash)
{
	SET_MAPPED_RO(hash->page->flags);
	unmap(hash, L4_FP_REMAP_PAGE);
}


static void hash_release(lx_hash_t * hash)
{
	avl_remove(&hash_avl, (avl*)hash);
	lx_slab_free(&hash_cache, (void*)hash);
}


static int hash_sanity_check(lx_hash_t * hash)
{
	
	if(IS_MAPPED_RO(hash->page->flags))
		return 0;

	/* check retry count */
	if(GET_PAGE_RETRY(hash->page->flags) == LX_RO_MAP_RETRY)
	{
		LOGd(DBG_HOT, "Hot page %08lx giving up (%d retries)\n",
				hash->page->addr, GET_PAGE_RETRY(hash->page->flags));
		hash_release(hash);	/* no use trying it again, free */
		lx_stats.hash_hot_cnt++;
	}
	else {	
		/* if page has been written to, requeue */
		avl_remove(&hash_avl, (avl*)hash);
		hash->key = 0;
		LOGd(DBG_HOT, "Re-try count %d\n", GET_PAGE_RETRY(hash->page->flags));
		INC_PAGE_RETRY(hash->page->flags);
		queue_add(hash);
	}
	return -1;
}
		
static int full_compare(lx_hash_t * compare, lx_hash_t * against)
{
	lx_page_t * page_comp, * page_against;
	int result;
	
	page_comp = compare->page;
	page_against = against->page;
	
	if(compare == against)
		enter_kdebug("Trying to compare same hash");
	if(page_comp == page_against)
		enter_kdebug("Trying to compare same page");
	
	/* synchronize */
	LOGd(DBG_SEM, "HASH: Downing against %p\n", against->ds);
	l4semaphore_down(&(against->ds->sem));
	/* down same dataspace only once */
	if(against->ds != compare->ds)
		l4semaphore_down(&(compare->ds->sem));
	l4semaphore_down(&(lx_orig_sem));

	if(IS_INVALID(page_comp->flags) || IS_INVALID(page_against->flags))
	{
		l4semaphore_up(&(against->ds->sem));
		if(against->ds != compare->ds)
			l4semaphore_up(&(compare->ds->sem));
		l4semaphore_up(&lx_orig_sem);
		return 	PAGE_AGAINST_CHANGED;
	}
	
	/* if page was changed, during hash compare */
	if(hash_sanity_check(against))
	{
		l4semaphore_up(&(against->ds->sem));
		if(against->ds != compare->ds)
			l4semaphore_up(&(compare->ds->sem));
		l4semaphore_up(&lx_orig_sem);
		return 	PAGE_AGAINST_CHANGED;
	}
	
	/* unmap, for save comparism */
	hash_unmap(compare);
	
	/* compare */
	result = memcmp((void *)lx_page_get_map_addr(page_against),
			(void *)lx_page_get_map_addr(page_comp), L4_PAGESIZE);
	
	if(result == 0) {
		
		if(page_against->addr  == zero.addr)
		{
			LOGd(DBG_HASH_FULL, "ZERO page hit for %08lx\n", lx_page_get_map_addr(page_comp));
			
			/* clear retry, page maybe release (kernel option) */
			if(!IS_ZERO_PAGE(page_comp->flags))
			{
				if(GET_PAGE_RETRY(page_comp->flags) >= LX_RO_MAP_RETRY)
					lx_stats.hash_hot_cnt--;

				CLEAR_PAGE_RETRY(page_comp->flags);
			}
			else 
				INC_PAGE_RETRY(page_comp->flags);
			
			SET_ZERO_PAGE(page_comp->flags);
		
		}
		
		else {
			LOGd(DBG_HASH_FULL, "\nFound matching page %08lx, referenced page %08lx\n",
				page_comp->addr, page_against->addr);
			UNSET_ZERO_PAGE(page_comp->flags);
		}
			
		result = PAGE_EQUAL;
		SET_PAGE_REF(page_comp->flags);
		INC_PAGE_REF_CNT(page_against->flags);
		
		l4util_inc32(&(lx_stats.page_ref));
		l4util_dec32(&(lx_stats.page_copied));
		lx_cow_set_free(page_comp->addr);
		page_comp->addr = (l4_addr_t)page_against;
	}
	else 
	{
		UNSET_ZERO_PAGE(page_comp->flags);
		LOGd(DBG_HASH_FULL, "Hash collision, pages don't match\n");
		result = PAGE_HASH_COL;
	}
	LOGd(DBG_SEM, "HASH against up\n");
	l4semaphore_up(&(against->ds->sem));
	l4semaphore_up(&lx_orig_sem);
	if(against->ds != compare->ds)
	{
		LOGd(DBG_SEM, "HASH compare up\n");
		l4semaphore_up(&(compare->ds->sem));
	}
	return result;
}

static void do_hash(lx_hash_t * compare)
{
	int level;
	lx_hash_t * against = NULL;
	avl_tree tmp_tree;
	int full_cmp;
	int again_flag = 0;	
	
	for(level=LX_HASH_LEVELS - 1; level >= 0; level--)
	{
		hash_cur_level = level;
		
		compare->key = 
			hash_func(lx_page_get_map_addr(compare->page), compare->key);
		LOGd(DBG_HASH_FUNC, "Hashed level: %d, key: %16llx\n", 
				level, compare->key);

		if(GET_PAGE_REF_CNT(compare->page->flags) > 1)
		{
			LOG_printf("Mysterical REF PAGE in hash\n");
			hash_remap_ro(compare);
			avl_insert(&hash_avl, (avl *)compare);
			return;
		}
		
INVALID:		
		/* start at root */
		if(against == NULL)
			against = (lx_hash_t *)avl_get_first(&hash_avl, (avl *)compare);
		
		/* start at last found */
		else
		{
			tmp_tree.root = (struct avl *)against;
			tmp_tree.compar = hash_avl.compar;
	 		against = (lx_hash_t *)avl_get_first(&tmp_tree, (avl *)compare);
		}
		
		/* we hashed to last level, search whole tree again */
		if(against == NULL && level == 0 && !again_flag)
		{
	  	against = (lx_hash_t *)avl_get_first(&hash_avl, (avl *)compare);
			again_flag = 1;
		}

		
		/* nothing found insert */
		if(against == NULL)
		{
			/* set collision for prev level */
			if(level < LX_HASH_LEVELS - 1) {
				l4util_inc32(&(lx_stats.hash_collisions[level + 1]));
			}
			
			LOGd(DBG_HASH, "Adding page to hash addr: %08lx, key: %016llx\n", 
					compare->page->addr, compare->key);
			hash_remap_ro(compare);
			avl_insert(&hash_avl, (avl *)compare);
			break;
		}
	  
		LOGd(DBG_HASH, "Found match level: %d \n\tkey compare: %16llx " \
				"\n\tkey against: %16llx\n", level, compare->key, against->key);
		
		/* set hit for prev hash level */
		if(level < LX_HASH_LEVELS - 1){
			l4util_inc32(&(lx_stats.hash_matches[level + 1]));
		}
		/* got invalidated page, remove and start-over */
		if(IS_INVALID(against->page->flags))
		{
			LOGd(DBG_CLEAN, "Releasing invalid page %08lx\n", against->page->addr);
			lx_page_release(against->page);
			hash_release(against);
			against = NULL;
			if(level < LX_HASH_LEVELS - 1)
			{
				l4util_dec32(&(lx_stats.hash_matches[level + 1]));
			}
			goto INVALID;
		}
		
		/* sanity checks */
		if(hash_sanity_check(against))
		{
			against = NULL;
			if(level < LX_HASH_LEVELS - 1){
				l4util_dec32(&(lx_stats.hash_matches[level + 1]));
			}
			goto INVALID;
		}
			
		
		/* if not at the end, continue hashing */
		if(level > 0)	
		{
			hash_cur_level = level - 1;
			if(!is_hashed(against->key))
			{	
				LOGd(DBG_HASH, "Removing page from hash addr: %08lx, key: %016llx\n", 
					against->page->addr, against->key);
				avl_remove(&hash_avl, (avl*)against);
				against->key =
					hash_func(lx_page_get_map_addr(against->page), against->key);
				avl_insert(&hash_avl, (avl*)against);
			}
			continue;
		}
		
		/* hash matches, full compare */
		full_cmp = full_compare(compare, against);
		
		switch(full_cmp) 
		{
			/* start over from root */
			case PAGE_AGAINST_CHANGED:
				against = NULL;
				if(IS_INVALID(compare->page->flags))
				{
					LOGd(DBG_CLEAN, "Releasing invalid page %08lx\n", compare->page->addr);
					lx_page_release(against->page);
					return;
				}
		
				goto INVALID;
			
			/* start over, root == right side from here */
			case PAGE_HASH_COL:
				against = (lx_hash_t*)against->avl.right;
				l4util_inc32(&(lx_stats.hash_collisions[0]));
				if(against != NULL)
					goto INVALID;
				break;	
			case PAGE_EQUAL:
				/* free for now */
				l4util_inc32(&(lx_stats.hash_matches[0]));
				lx_slab_free(&hash_cache, (void*)compare);
		}
	}
}
		
static void __hash_loop(void * data)
{
	lx_hash_t * compare = NULL;
	l4thread_started(NULL);
	LOGd(DBG_VERB, "Hashing thread is up\n");
	while(1)
	{
		 while((compare = queue_get_head()) == NULL)
		 {
			 LOGd(DBG_HASH, "Waiting queue empty ... sleeping a bit\n");
			 l4thread_sleep(500);
		 }

		 l4semaphore_down(&lx_clean_sem);
		 if(IS_INVALID(compare->page->flags))
		 {
			 LOGd(DBG_CLEAN, "Releasing invalid page %08lx\n", compare->page->addr);
			 lx_page_release(compare->page);
		 	 l4semaphore_up(&lx_clean_sem);
		 	 continue;
		 }
		 do_hash(compare);
		 l4semaphore_up(&lx_clean_sem);
	}
}	

static void hash_setup_zero_page(void)
{
	void * zero_page = l4dm_mem_allocate_named(L4_PAGESIZE, L4RM_MAP, "zero page");
	int i;
	lx_hash_t * hash;
	if(zero_page == NULL)
	{
		LOGd(DBG_ERROR, "Zero page allocation failed\n");
		return;
	}

	memset(zero_page, 0x0, L4_PAGESIZE);
	hash = (lx_hash_t *)lx_slab_alloc(&hash_cache);
	
	zero.offset = 0;
	zero.flags = 0;
	INC_PAGE_REF_CNT(zero.flags);
	SET_MAPPED_RO(zero.flags);
	zero.addr = (l4_addr_t)zero_page;

	hash->avl = AVL_INVALID;
	hash->page = &zero;
	hash->ds   = &zero_ds;
	hash->key  = 0;
	for(i = LX_HASH_LEVELS - 1; i >= 0; i--)
	{
		hash_cur_level = i;
		hash->key = hash_func(zero.addr, hash->key);
	}

	avl_insert(&hash_avl, (avl*)hash);
}
	
/*******************************************************************************
 ** external funcs 
 ******************************************************************************/
int lx_hash_start(void)
{
	int error;
	
	/* no need to start */
	if(!LX_HASH_LEVELS)
		return 0;
	
	if((error = lx_slab_setup(&hash_cache, sizeof(lx_hash_t))))
		return error;
		
	if((error = lx_slab_setup(&queue_cache, sizeof(queue_t))))
		return error;
	
	hash_avl.root = 0;
	hash_avl.compar = avl_cmp;
	
	hash_setup_zero_page();
	
	l4thread_create_long(
			L4THREAD_INVALID_ID,
			__hash_loop, 
			".hasher",
			L4THREAD_INVALID_SP,
			L4THREAD_DEFAULT_SIZE,
			LX_HASH_PRIO,
			NULL, 
			L4THREAD_CREATE_SYNC);
	
	return 0;
}

void lx_hash_register_page(lx_ds_t * ds, lx_page_t * page)
{
	lx_hash_t * hash = lx_slab_alloc(&hash_cache);

	if(hash == NULL)
		return;

	hash->avl = AVL_INVALID;
	hash->ds = ds;
	hash->page = page;
	hash->key = 0;
	queue_add(hash);
}
	
