/* $Id: cow_mem.c,v 1.4 2007/01/25 15:13:06 cbass Exp $ */
/*****************************************************************************/
/**
 *  \file cow_mem.c
 *  \brief cow memory management 
 * 
 *  \date 12/20/06 15:32:23 CET
 *  \author Sebastian Sumpf <Sebastian.Sumpf@inf.tu-dresden.de>
 */ 
/******************************************************************************/

#include <l4/dm_mem/dm_mem.h>
#include <l4/log/l4log.h>
#include <l4/sys/syscalls.h>
#include <l4/lxfreeze/config.h>
#include "__avl.h"
#include "__types.h"
#include "__lxcalls.h"

#define INIT_SIZE L4_SUPERPAGESIZE
#define FREE_LIST_SIZE (LX_COW_AREA_SIZE / (32 * L4_PAGESIZE))

#define GET_I(addr) ((addr - cow_addr) / (32 * L4_PAGESIZE))
#define GET_J(addr) (((addr - cow_addr) % (32 * L4_PAGESIZE)) / L4_PAGESIZE)
#define GET_ADDR(i,j) (cow_addr + i * 32 * L4_PAGESIZE + j * L4_PAGESIZE)

static l4dm_dataspace_t cow_ds = L4DM_INVALID_DATASPACE_INITIALIZER;
static l4_size_t cow_current_size = 0;
static l4_uint32_t free_list[FREE_LIST_SIZE];
static l4_addr_t cow_addr;
static l4slab_cache_t cow_move_cache;
static avl_tree cow_move_tree;
static int save_i = 0;
static l4semaphore_t sync_sem = L4SEMAPHORE_INIT(1);
static int avl_moved_cmp(void * a, void * b)
{
	return ((lx_moved_t *)a)->map_addr - ((lx_moved_t *)b)->map_addr;
}
									 
static int __cow_init(void)
{
	int error;

	error = l4dm_mem_open(L4DM_DEFAULT_DSM, INIT_SIZE, 0, 0, "COW mem",
			&cow_ds);
	if(error) {
		LOGd(DBG_ERROR, "Error: Failed to init cow mem, %s (%d)\n",
				l4env_errstr(error), error);
		return error;
	}

	// attach heap dataspace, already reserve the whole vm area
	error = l4rm_attach(&cow_ds, LX_COW_AREA_SIZE , 0, L4DM_RW, 
			(void*)&cow_addr);
	if(error)
	{
		LOGd(DBG_ERROR, "Error: Failed to attach slab cache, %s (%d)\n",
				l4env_errstr(error), error);
		l4dm_close(&cow_ds);
		return error;
	}

	cow_current_size += INIT_SIZE;
	memset((void*)free_list, 0xFF, FREE_LIST_SIZE * sizeof(l4_uint32_t));
	
	cow_move_tree.compar = avl_moved_cmp;
	cow_move_tree.root   = 0;
	
	return lx_slab_setup(&cow_move_cache, sizeof(lx_moved_t));
}

static void * __cow_get_free(void)
{
	int i;
	int j= 0;
	int error;
	int found = 0;
	l4_addr_t addr = 0;
	lx_moved_t mov;
	mov.map_addr = mov.new_addr = 0;
	
	if(l4dm_is_invalid_ds(cow_ds))
		__cow_init();
	
	for(i = save_i; i < FREE_LIST_SIZE; i++)
	{
		if(free_list[i] == 0) continue;
		
		for(j = 0; j < 32; j++)
		{
			if(free_list[i] & (1 << j))
			{
				addr = GET_ADDR(i,j);
				mov.map_addr = addr;
				
				/* don't allow page, if address is in move-tree */
				if((avl_get_first(&cow_move_tree, (avl *)&mov)) == NULL)
				{			
					free_list[i] ^= (1 << j);
					found = 1;
					break;
				}
			}
		}
		if(found) break;
	}

	if(!found) 
	{
		LOGd(DBG_ERROR, "Error: COW area overflow\n");
		return NULL;
	}
	
	save_i = i;
	
	if(addr >= (cow_addr + cow_current_size))
	{
		error = l4dm_mem_resize(&cow_ds, cow_current_size + L4_SUPERPAGESIZE);
		if(error) {
			LOGd(DBG_ERROR, "Error: COW memory resize failed, %s (%d)\n",
					l4env_errstr(error), error);
			return NULL;
		}
				
		error = l4dm_map_ds(&cow_ds, cow_current_size, cow_addr + cow_current_size,
			L4_SUPERPAGESIZE, L4DM_RW);
		if(error) {
			LOGd(DBG_ERROR, "Error: COW memory resize mapping failed, %s (%d)\n",
					l4env_errstr(error), error);
		  return NULL;
		}
		
		cow_current_size += L4_SUPERPAGESIZE;
		lx_stats.mem_usage += L4_SUPERPAGESIZE;
	} 
	return (void*)addr;
}  

void * lx_cow_get_free(void)
{
	void * ret;
	l4semaphore_down(&sync_sem);
	ret = __cow_get_free();
	l4semaphore_up(&sync_sem);
	return ret;
}

void lx_cow_set_free(l4_addr_t addr)
{	
	int i = GET_I(addr);
	int j = GET_J(addr);
		
	if(addr >= cow_addr && addr < cow_addr + cow_current_size)
	{
		l4semaphore_down(&sync_sem);
		free_list[i] |= (1 << j);
		if(i < save_i)
			save_i = i;
		l4semaphore_up(&sync_sem);
	}
}

void * lx_cow_chk_moved(l4_addr_t addr)
{
	lx_moved_t * mov;
	lx_moved_t ret;

	l4semaphore_down(&sync_sem);
	ret.map_addr = addr;
	ret.new_addr = 0;
	mov = &ret;
		
	if((mov = (lx_moved_t *)avl_get_first(&cow_move_tree, (avl *)mov)) != NULL)
	{
		LOGd(DBG_CLEAN, "Found moved map_addr %08lx new_addr %08lx\n", 
				mov->map_addr, mov->new_addr);
		ret.new_addr = mov->new_addr;
		avl_remove(&cow_move_tree, (avl*)mov);
		lx_slab_free(&cow_move_cache, (void*)mov);
	}	
	l4semaphore_up(&sync_sem);
	return (void *)(ret.new_addr);
}

void lx_cow_compactify(void)
{
	int head_i, head_j, tail_i, tail_j, error;
	l4_addr_t found_targ = 0;
	l4_addr_t found_src  = 0;
	l4_size_t new_size;
	lx_moved_t * move_page;
	lx_moved_t test;
	l4_addr_t i;	
	
	head_i = GET_I(cow_addr);
	head_j = GET_J(cow_addr);
	tail_i = GET_I((cow_addr + cow_current_size));
	tail_j = GET_J((cow_addr + cow_current_size));

	while(head_i != tail_i || head_j != tail_j)
	{

		while(head_i != tail_i || head_j != tail_j)
		{
	
			if((free_list[head_i] & (1 << head_j)))
			{
				found_targ = GET_ADDR(head_i, head_j);
				test.map_addr = found_targ;
				test.new_addr = 0;
				if((avl_get_first(&cow_move_tree, (avl*)&test)) == NULL)
				  break;
			}
			
			head_j = (head_j + 1) % 32;
			head_i += (head_j==0)?1:0;
		}
	
		while((head_i != tail_i || head_j != tail_j) && !found_src)
		{
		
			if(!(free_list[tail_i] & (1 << tail_j)))
			{
				found_src = GET_ADDR(tail_i, tail_j);
				
				break;
			}
			
			tail_j = (tail_j == 0)?31:tail_j-1;
			tail_i -= (tail_j == 31)?1:0;
		}

		/* copy and insert into move tree */
		if(found_src && found_targ)
		{
			
			LOGd(DBG_CLEAN, "Moving %08lx to %08lx", found_src, found_targ);
			
			move_page = (lx_moved_t *)lx_slab_alloc(&cow_move_cache);	
			move_page->map_addr = found_src;
			move_page->new_addr = found_targ;
			
			if((avl_get_first(&cow_move_tree, (avl*)move_page))!=NULL)
				enter_kdebug("Duplicate move page entry");
			
			avl_insert(&cow_move_tree, (struct avl*)move_page);
			/* unmap */
			l4_fpage_unmap(l4_fpage(
						found_src, L4_LOG2_PAGESIZE, 0, 0),
			    L4_FP_FLUSH_PAGE | L4_FP_OTHER_SPACES);
			
			memcpy((void*)found_targ, (void*)found_src, L4_PAGESIZE);
			
			free_list[tail_i] |= (1 << tail_j);
			free_list[head_i] ^= (1 << head_j);
		
			found_src = found_targ = 0;
			 
		}
		
	}
	
	/* if last page is free */
	if((free_list[tail_i] & (1 << tail_j)))
	{
			tail_j = (tail_j == 0)?31:tail_j-1;
			tail_i -= (tail_j == 31)?1:0;
  }
	save_i = tail_i;

	new_size = (GET_ADDR(tail_i, tail_j) + L4_SUPERPAGESIZE - 1 - cow_addr)
		& L4_SUPERPAGEMASK;
	LOGd(DBG_CLEAN, "New size is %d KB (old %d KB)\n", 
			new_size / 1024, cow_current_size / 1024);
	if(new_size < cow_current_size)
	{	
		if(new_size < INIT_SIZE)
		{
			save_i = 0;
			new_size = INIT_SIZE;
		}

		error = l4dm_mem_resize(&cow_ds, new_size);
		if(error) {
			LOGd(DBG_ERROR, "Error: COW memory (shrink) resize failed, %s (%d)\n",
					l4env_errstr(error), error);
			return ;
		}
		/* unmap at own address space */
		for(i = cow_addr + new_size;  i < cow_addr + cow_current_size; 
				i+=L4_SUPERPAGESIZE)
		{
			l4_fpage_unmap(l4_fpage(
						i, L4_LOG2_SUPERPAGESIZE, L4_FPAGE_RW, L4_FPAGE_MAP),
			    L4_FP_FLUSH_PAGE | L4_FP_ALL_SPACES);
			lx_stats.mem_usage -= L4_SUPERPAGESIZE;
		}
		LOGd(DBG_CLEAN, "Resized cow mem old size: %d KB new size %d KB\n",
				cow_current_size / 1024, new_size / 1024);
		cow_current_size = new_size;
	}
}
