/* $Id: __types.h,v 1.4 2007/01/25 15:13:06 cbass Exp $ */
/*****************************************************************************/
/**
 *  \file __types.h
 *  \brief common datatypes 
 * 
 *  \date 12/18/06 18:33:33 CET
 *  \author Sebastian Sumpf <Sebastian.Sumpf@inf.tu-dresden.de>
 */ 
/******************************************************************************/
#include <l4/semaphore/semaphore.h>
#include <l4/sys/types.h>
#include <l4/env/errno.h>
#include <l4/dm_phys/dm_phys.h>
#include <l4/util/util.h>
#include <l4/util/macros.h>
#include <l4/dm_generic/dsmlib.h>
#include <l4/lxfreeze/lxfreeze.h>
#include <l4/lxfreeze/config.h>
#include <l4/lxfreeze/stats.h>
#include "__avl.h"
#include "__debug.h"

/******************************************************************************
 *** page flags manipulation macros
 *****************************************************************************/
#define FLAG_PAGE_RO      0x80000000
#define FLAG_PAGE_REF     0x40000000
#define FLAG_PAGE_INVALID 0x20000000
#define FLAG_PAGE_ZERO    0x10000000
#define FLAG_PAGE_LOCKED  0x08000000
#define FLAG_REF_CNT_MASK 0x00000FFF
#define FLAG_RETRY_MASK   0x0000F000

#define IS_PAGE_REF(flags)          ((flags & FLAG_PAGE_REF)?1:0)
#define SET_PAGE_REF(flags)         (flags |= FLAG_PAGE_REF)
#define UNSET_PAGE_REF(flags)       (flags &= ~FLAG_PAGE_REF)

#define IS_MAPPED_RO(flags)         ((flags & FLAG_PAGE_RO)?1:0)
#define SET_MAPPED_RO(flags)        (flags |= FLAG_PAGE_RO)
#define UNSET_MAPPED_RO(flags)      (flags &= (~FLAG_PAGE_RO))

#define IS_INVALID(flags)           ((flags & FLAG_PAGE_INVALID?1:0))
#define SET_INVALID(flags)          (flags |= FLAG_PAGE_INVALID)

#define IS_ZERO_PAGE(flags)         ((flags & FLAG_PAGE_ZERO)?1:0)
#define SET_ZERO_PAGE(flags)        (flags |= FLAG_PAGE_ZERO)
#define UNSET_ZERO_PAGE(flags)      (flags &= (~FLAG_PAGE_ZERO))

#define IS_LOCKED_PAGE(flags)         ((flags & FLAG_PAGE_LOCKED)?1:0)
#define SET_LOCKED_PAGE(flags)        (flags |= FLAG_PAGE_LOCKED)
#define UNSET_LOCKED_PAGE(flags)      (flags &= (~FLAG_PAGE_LOCKED))

#define GET_PAGE_REF_CNT(flags)     (flags & FLAG_REF_CNT_MASK) 
#define ADD_PAGE_REF_CNT(flags, val)(flags = \
                                    (flags & ~FLAG_REF_CNT_MASK) \
                                    | (GET_PAGE_REF_CNT(flags) + (val)))
#define INC_PAGE_REF_CNT(flags)     ADD_PAGE_REF_CNT(flags, 1)
#define DEC_PAGE_REF_CNT(flags)     ADD_PAGE_REF_CNT(flags, -1)

#define GET_PAGE_RETRY(flags)       ((flags & FLAG_RETRY_MASK) >> 12)
#define INC_PAGE_RETRY(flags)       (flags = \
                                    (flags & ~FLAG_RETRY_MASK) \
                                    | ((GET_PAGE_RETRY(flags) + 1) << 12))
#define CLEAR_PAGE_RETRY(flags)     (flags &= ~FLAG_RETRY_MASK)

/*******************************************************************************
 *** usefule macros
 ******************************************************************************/

#define lx_ds_is_equal(ds1, ds2)    (ds1.id == ds2.id && \
                                    l4_thread_equal(ds1.manager, ds2.manager))
#define LX_INVALID_DS_ID            (l4_uint32_t)-1
#define lx_is_invalid_ds_id(id)     ((id == LX_INVALID_DS_ID))

/*******************************************************************************
 * freezer data types
 ******************************************************************************/

typedef struct lx_page_t {
	struct avl     avl;    ///AVL-tree entry
	unsigned long  offset; ///offset in dataspace
	l4_uint32_t    flags;  ///refer to flag macros above 
	l4_addr_t      addr;   ///map address or reference address
} lx_page_t;

#define LX_DS_INITIALIZER { \
	NULL, \
	L4DM_INVALID_DATASPACE_INITIALIZER, \
	0, \
	AVL_TREE_INITIALIZER, \
	L4SEMAPHORE_INIT(1)}

#define LX_INVALID_DS ((lx_ds_t)LX_DS_INITIALIZER)

typedef struct lx_ds_t {
	dsmlib_ds_desc_t * ds_desc;  ///dataspace of our memory manager
	l4dm_dataspace_t   ds;       ///dataspace of client
	l4_size_t          size;     ///dataspace size
	struct avl_tree    avl_tree; ///avl tree of dataspace of task
	l4semaphore_t      sem;      ///lock AVL-Tree for dataspace/task
} lx_ds_t;

#define LX_TASK_INITIALIZER {task_id:L4_INVALID_ID, l4x_signal_id:L4_INVALID_ID}
#define LX_INVALID_TASK ((lx_task_t)LX_TASK_INITIALIZER)
typedef struct lx_task_t {
	l4_taskid_t   task_id;
	l4_threadid_t l4x_signal_id;
	lx_ds_t       ds[LX_MAX_DS];
} lx_task_t;

typedef struct lx_hash_t {
	struct avl  avl;
	l4_uint64_t key;
	lx_page_t * page;
	lx_ds_t   * ds;
} lx_hash_t;
	
typedef struct lx_moved_t {
	struct avl  avl;
	l4_addr_t   map_addr;
	l4_addr_t   new_addr;
} lx_moved_t;


#define LX_FAULT_INITIALIZER {0, 0, (l4_uint32_t)-1, 0}
#define LX_INVALID_FAULT ((lx_page_fault_t)LX_FAULT_INITIALIZER)
typedef struct lx_page_fault_t {
	l4_addr_t     virt_addr;
	l4_size_t     virt_size;
	unsigned long ds_id;
	l4_uint8_t    flags;
} lx_page_fault_t;

/******************************************************************************
 *** globals
 *****************************************************************************/
extern int            lx_frozen;
extern lx_stats_t     lx_stats;
extern l4_threadid_t  lx_main_thread;
extern l4_threadid_t  loader_id;
extern l4semaphore_t  lx_clean_sem;
extern l4semaphore_t  lx_orig_sem;
