/* $Id:$ */
/*******************************************************************************/
/*
 * \file   main.c
 * \brief  start routine
 *
 * \date   2007-09-27
 * \author Sebastian Sumpf <sumpf@os.inf.tu-dresden.de>
 */
/*******************************************************************************/
/* (c) 2007 Technische Universitaet Dresden
 * This file is part of DROPS, which is distributed under the terms of the
 * GNU General Public License 2. Please see the COPYING file for details.
 */
/* L4 stuff */
#include <l4/sys/types.h>
#include <l4/lxMir/lxfreeze.h>
#include <l4/names/libnames.h>
#include <l4/util/macros.h>
/* Linux native */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>

/* private */
#include "__globals.h"
#include "lxMirback_end-server.h"


l4_threadid_t freezer_id;
void *copy_area;

void *linux_malloc(unsigned long size) 
{
	void * ptr =  malloc((size_t)size);
	if(ptr)
		mlock(ptr, size);
	return ptr;
}


void linux_free(void * ptr)
{
	free(ptr);
}

static void sig(int signo)
{
	char out[255] = "\0";
	switch(signo)
	{
		case SIGSEGV:
			sprintf(out, "%s\n", "Segmentation Fault");
			break;
		case SIGBUS:
			sprintf(out, "%s\n", "Bus Error");
	}
	names_unregister(LX_MIR);
	printf("%s", out); 
	exit(0);
}

static int init(void)
{
	/* page aligned memory */
	int err = posix_memalign(&copy_area, L4_SUPERPAGESIZE, COPY_AREA_SIZE);
	if(err) {
		fprintf(stderr, "Error: Could not reserve copy area\n");
		return err;
	}

	mlock(copy_area, COPY_AREA_SIZE);
 
 signal(SIGINT, sig);
 signal(SIGTERM, sig);
 signal(SIGKILL, sig);
 signal(SIGQUIT, sig);
 signal(SIGSEGV, sig);
 signal(SIGBUS, sig);
 return 0;
}


static int daemonize(void)
{
	pid_t pid;
	
	if(getppid() == 1) 
		return 0;
	
	pid = fork();
	if(pid < 0) {
		fprintf(stderr, "Fork error\n");
		return -1;
	}

	if(pid > 0) {
		_exit(0);
	}

	umask(0);
	if((setsid()) < 0) {
		fprintf(stderr, "Error: Unable to create session\n");
		return -1;
	}
	

	if((chdir("/")) < 0) {
		fprintf(stderr, "Error: Unable to change to /\n");
		return -1;
	}

	freopen("/dev/null", "r", stdin);
	freopen("/dev/null", "w", stdout);
	freopen("/dev/null", "r", stdout);
	
	return 0;
}


int main(int argc, char **argv)
{
	DICE_DECLARE_SERVER_ENV(env);

	if(!names_query_name(LX_FREEZE, &freezer_id)) {
		fprintf(stderr, "Error: Freezer not found\n");
		return 1;
	}
	printf("Freezer id is "l4util_idfmt"\n", l4util_idstr(freezer_id));
	env.malloc = linux_malloc;
	env.free   = linux_free;
	
	if(init()) 
		return 2;
	
	printf("Started Linux mirgration back end ... (backgrounding)\n");
	if(daemonize())
		return 3;

	if(!names_register(LX_MIR)) {
		fprintf(stderr, "Failed to register '%s' at name server\n", LX_MIR);
		return 4;
	}
	
	if_net_lxmir_server_loop(&env);
	
	return 0;
}

