/* $Id:$ */
/*******************************************************************************/
/*
 * \file   slab.c
 * \brief  slab-allocator-memory management
 *
 * \date   Makefile
 * \author Sebastian Sumpf <sumpf@os.inf.tu-dresden.de>
 */
/*******************************************************************************/
/* (c) 2007 Technische Universitaet Dresden
 * This file is part of DROPS, which is distributed under the terms of the
 * GNU General Public License 2. Please see the COPYING file for details.
 */

#include <l4/log/l4log.h>
#include <l4/util/util.h>

#include "__types.h"
#include "__lxcalls.h"
#include "__debug.h"

#define INIT_SIZE L4_PAGESIZE
#define FREE_LIST_SIZE (LX_MAX_SLAB_SIZE / (L4_PAGESIZE * 32))

static l4dm_dataspace_t slab_ds = L4DM_INVALID_DATASPACE_INITIALIZER;
static l4_addr_t slab_addr;
static l4_size_t slab_current_size = 0;
static l4_uint32_t slab_free[FREE_LIST_SIZE];
static l4semaphore_t sync_sem = L4SEMAPHORE_INIT(1);

static void * __find_free(void)
{
	int i = 0;
	int j = 0;
	l4_addr_t addr;
	int found = 0;
	int error;
	
	for(i = 0; i < FREE_LIST_SIZE; i++)
	{
		if(slab_free[i] == 0) continue;
		
		for(j = 0; j < 32; j++)
		{
			if(slab_free[i] & (1 << j)) {
				slab_free[i] ^= (1 << j);
				found = 1;
				break;
			}
		}
		if(found) break;
	}
	
	if(!found) 
	{
		LOGd(DBG_ERROR, "Error: slab cache overflow!\n");
		return NULL;
	}

	addr = slab_addr + i * 32 * L4_PAGESIZE + j * L4_PAGESIZE;

	if(addr >= slab_addr + slab_current_size)
	{
		error = l4dm_mem_resize(&slab_ds, slab_current_size + L4_PAGESIZE);
		if(error) {
			LOGd(DBG_ERROR, "Error: slab cache resize failed, %s (%d)\n",
			     l4env_errstr(error), error);
			return NULL;
	  }
		slab_current_size += L4_PAGESIZE;
	}
	return (void *)addr;
}		

static void __set_free(l4_addr_t addr)
{
	int i = (addr - slab_addr) / (32 * L4_PAGESIZE);
	int j = ((addr - slab_addr) % (32 * L4_PAGESIZE)) / L4_PAGESIZE;
	//int error;
	
	slab_free[i] |= (1 << j);
/*
	if(addr == (slab_addr + slab_current_size - L4_PAGESIZE))
	{
		error = l4dm_mem_resize(&slab_ds, slab_current_size - L4_PAGESIZE);
		if(error) {
			LOGd(DBG_ERROR, "Error: slab cache (shrink) resize failed, %s (%d)\n",
				l4env_errstr(error), error);
			return;
	  }
		slab_current_size -= L4_PAGESIZE;
	}		
*/
}

static int __slab_init(void)
{
	int error;

	error = l4dm_mem_open(L4DM_DEFAULT_DSM, INIT_SIZE, 0, 0, "Freezer-Slab cache",
			&slab_ds);
	if(error) {
		LOGd(DBG_ERROR, "Error: Failed to init slab cache, %s (%d)\n",
				l4env_errstr(error), error);
		return -1;
	}

	 // attach heap dataspace, already reserve the whole vm area 
	 error = l4rm_attach(&slab_ds, LX_MAX_SLAB_SIZE , 0, L4DM_RW, (void*)&slab_addr);
	 if(error)
	 {
			LOGd(DBG_ERROR, "Error: Failed to attach slab cache, %s (%d)\n",
			     l4env_errstr(error), error);
	 		l4dm_close(&slab_ds);
			return -1;
		}
		
	 slab_current_size = INIT_SIZE;
	 memset((void*)slab_free, 0xFF, (FREE_LIST_SIZE * sizeof(l4_uint32_t)));
	 return 0;
}

static void * __grow(l4slab_cache_t * cache, void ** data)
{
	return __find_free();
}

static void __release(l4slab_cache_t * cache, void * page, void * data)
{
		__set_free((l4_addr_t)page);
}

int lx_slab_setup(l4slab_cache_t * cache, l4_size_t obj_size)
{
	int error;

	if(l4dm_is_invalid_ds(slab_ds))
		error = __slab_init();

	error = l4slab_cache_init(cache, obj_size, 0, __grow, __release);
	if(error)
	{
		 LOGd(DBG_ERROR, "Error: Failed to init slab cache, %s (%d)\n",
		      l4env_errstr(error), error);
		return error;
	}
	return 0;
}

void * lx_slab_alloc(l4slab_cache_t * cache)
{
	void * obj;
	l4semaphore_down(&sync_sem);
	obj = l4slab_alloc(cache);
	l4semaphore_up(&sync_sem);
	return obj;
}

void lx_slab_free(l4slab_cache_t * cache, void * obj)
{
	l4semaphore_down(&sync_sem);
	l4slab_free(cache, obj);
	l4semaphore_up(&sync_sem);
}

	
