/* $Id: __types.h,v 1.4 2007/01/25 15:13:06 cbass Exp $ */
/*****************************************************************************/
/**
 *  \file __types.h
 *  \brief common datatypes 
 * 
 *  \date 12/18/06 18:33:33 CET
 *  \author Sebastian Sumpf <Sebastian.Sumpf@inf.tu-dresden.de>
 */ 
/******************************************************************************/
#ifndef __LXMIR_SERVER_TYPES_H__
#define __LXMIR_SERVER_TYPES_H__

#include <l4/semaphore/semaphore.h>
#include <l4/sys/types.h>
#include <l4/env/errno.h>
#include <l4/dm_phys/dm_phys.h>
#include <l4/util/util.h>
#include <l4/util/macros.h>
#include <l4/dm_generic/dsmlib.h>
#include <l4/lxMir/lxfreeze.h>
#include <l4/env/env.h>
#include "__avl.h"
#include "__debug.h"


/*******************************************************************************
 *** usefule macros
 ******************************************************************************/

#define lx_ds_is_equal(ds1, ds2) (ds1.id == ds2.id && \
                       l4_thread_equal(ds1.manager, ds2.manager))
#define LX_INVALID_DS_ID (l4_uint32_t)-1
#define lx_is_invalid_ds_id(id) ((id == LX_INVALID_DS_ID))

#define LX_INVALID_ADDR ~0UL
#define lx_is_invalid_addr(addr) ((addr ==  LX_INVALID_ADDR))


/*******************************************************************************
 * freezer data types
 ******************************************************************************/

/* dataspace initializer */
#define LX_DS_INITIALIZER { \
	NULL, \
	L4DM_INVALID_DATASPACE_INITIALIZER, \
	LX_INVALID_DS_ID, \
	0, \
	0, \
	0, \
	""}
#define LX_INVALID_DS ((lx_ds_t)LX_DS_INITIALIZER)

/* flags */
#define LX_RESIZE  1 /* dataspace has been resized */
#define LX_POSTCPY 2 /* dataspace has COW pages (debugging only) */

/* internal dataspace type */
typedef struct lx_ds_t {
	dsmlib_ds_desc_t * ds_desc;    ///dataspace of out memory manager
	l4dm_dataspace_t   ds;         ///dataspace of client
	unsigned long      virt_ds_id; ///virtual (non-unique) ds id for for task
	l4_size_t          size;       ///dataspace size
	l4_addr_t          map_addr;   ///map address
	l4_uint8_t         flags;
	char               name[L4DM_DS_NAME_MAX_LEN]; 
} lx_ds_t;


/* virtual addr. to dataspace mapping */
#define LX_FAULT_INITIALIZER {0, 0, (l4_uint32_t)-1, 0}
#define LX_INVALID_FAULT ((lx_page_fault_t)LX_FAULT_INITIALIZER)
typedef struct  lx_page_fault_t {
	l4_addr_t     virt_addr;
	l4_size_t     virt_size;
	unsigned long ds_id;
	l4_uint8_t    flags;
} lx_page_fault_t;


/* copy-on-write structure */
#define LX_COW_PAGE_INITIALIZER {(l4_uint32_t)-1, 0, 0, NULL}
#define LX_INVALID_COW_PAGE ((lx_cow_page_t)LX_COW_PAGE_INITIALIZER)
typedef struct lx_cow_page_t {
	unsigned long ds_id;
	unsigned long offs;
	l4_addr_t     map_addr;
	struct lx_cow_page_t *next;
} lx_cow_page_t;


/* internal task type */
enum {
	TASK_INVALID       = 0,
	TASK_LOADING       = 1,
	TASK_SEND_PRECOPY  = 2,
	TASK_SEND_POSTCOPY = 4,
	TASK_RECV_PRECOPY  = 8,
	TASK_LOADED        = 16,
	TASK_MIGRATED      = 32,
	TASK_NO_FAULTS     = 64,
};

#define TASK_FLAG_MASK 0x7
#define LX_TASK_INITIALIZER {task_id:L4_INVALID_ID, \
                             mir_id:L4_INVALID_ID, \
                             back_server_id: L4_INVALID_ID, \
                             l4x_signal_id: L4_INVALID_ID, \
                             state:TASK_INVALID,\
                             lx_infopage:NULL, \
                             start_eip:(l4_addr_t)-1, \
                             start_esp:(l4_addr_t)-1, \
                             cow_page_first:NULL, \
                             cow_page_last:NULL, \
                             cow_page_cur:NULL, \
                             cow_cnt:0}
#define LX_INVALID_TASK ((lx_task_t)LX_TASK_INITIALIZER)

typedef struct lx_task_t {
	l4_taskid_t           task_id; /* l4linux server thread */
	l4_threadid_t         mir_id;  /* worker thread of back end server */
	l4_threadid_t         back_server_id; /* back end migration server */
	l4_threadid_t         l4x_signal_id; /* suspend signal thread */
	lx_ds_t               ds[LX_MAX_DS]; /* managed dataspaces */
	lx_page_fault_t       fault[LX_MAX_DS]; /* managed virt. mem. areas */
	l4_uint32_t           state; /* task state */
	l4env_infopage_t      *lx_infopage; /* L4Env infopage */
	l4_addr_t             start_eip; /* resume info */
	l4_addr_t             start_esp;
	lx_cow_page_t         *cow_page_first; /* cow pages, gathered during pre-copy */
	lx_cow_page_t         *cow_page_last;
	lx_cow_page_t         *cow_page_cur;
	l4_uint32_t           cow_cnt;
} lx_task_t;

typedef struct lx_moved_t {
	struct avl  avl;
	l4_addr_t   map_addr;
	l4_addr_t   new_addr;
} lx_moved_t;



/******************************************************************************
 *** globals
 *****************************************************************************/

extern l4_threadid_t lx_main_thread;
extern l4_threadid_t loader_id;
extern l4semaphore_t lx_clean_sem;

/* FIXME: interim solution to support l4dm_share (currently used by loader) */
extern lx_ds_t * dstolxds[FREEZE_MAX_DS];

/* add a cow page to task */
L4_INLINE void lx_cow_page_add(lx_task_t * task, lx_cow_page_t *cow_page);
L4_INLINE void lx_cow_page_add(lx_task_t * task, lx_cow_page_t *cow_page)
{
	if(task->cow_page_first == NULL) {
		task->cow_page_first =
		task->cow_page_cur   =
		task->cow_page_last  = cow_page;
	}
	else {
		task->cow_page_last->next = (struct lx_cow_page_t *)cow_page;
		task->cow_page_last       = cow_page;
	}
}


/*******************************************************************************
 *** TASK state helpers 
 ******************************************************************************/

static inline void task_set_state(lx_task_t * task, int state) 
{
	task->state |= state;
}

static inline void task_del_state(lx_task_t * task, int state) 
{
	task->state &= ~state;
}

static inline int task_is_ready(lx_task_t * task) 
{
	return ((task->state == TASK_LOADED) || (task->state == TASK_MIGRATED));
}

static inline int task_has_state(lx_task_t * task, int state) 
{
	return (task->state & state);
}

static inline void task_set_ready(lx_task_t * task) 
{
	task->state &= ~TASK_FLAG_MASK;
}

#endif /* __LXMIR_SERVER_TYPES_H__ */
