/* $Id:$ */
/*******************************************************************************/
/*
 * \file   main.c
 * \brief  hybrid migration-client-front end
 *
 * \date   2007-09-27
 * \author Sebastian Sumpf <sumpf@os.inf.tu-dresden.de>
 */
/*******************************************************************************/
/* (c) 2007 Technische Universitaet Dresden
 * This file is part of DROPS, which is distributed under the terms of the
 * GNU General Public License 2. Please see the COPYING file for details.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>

#include <l4/lxMir/config.h>
#include <l4/lxMir/lxMir-client.h>
#include <l4/lxMir/lxMirback_end-client.h>
#include <l4/names/libnames.h>
#include <l4/util/macros.h>
#include <l4/env/errno.h>
static char * path;
static int resume = 0;
static int instance = 0;

static void usage(l4_taskid_t *inst, unsigned int count) 
{
	int i;
	printf("Usage: lxmigrate [OPTION] <FILE>\n\n");
	printf("Available Options\n");
	printf("\t\t-s --suspend=<number>    Suspend L4Linux <number>" \
	       " and write contents to FILE\n");
	printf("\t\t-r --resume              Resume L4Linux using FILE.\n\n");
	printf("Available L4Linux instances:\n");
	
	if(!count) 
		printf("No migrateable L4Linux instance running\n");
	
	for(i = 0; i < count; i++) 
		printf("%d: "l4util_idfmt"\n", i+1, l4util_idstr(inst[i]));
	exit(0);
}

static int parse_args(int argc, char ** argv, l4_taskid_t *inst,
                      unsigned int count)
{
	char c;
	int err = 0;
	int option_index;
	extern int optind;
	static struct option long_options[] = 
	{
		{"resume" , 0, 0, 'r'},
		{"suspend", 0, 0, 's'},
		{"help"   , 0, 0, 'h'},
		{0, 0, 0, 0}
	};
	
	while(1) {
		
		c = getopt_long(argc, argv, "rhs:", long_options, &option_index);
		
		if(c == -1)
			break;

		switch(c) {
			
			case 'r': 
				resume = 1;
				break;
			case 's':
				if(optarg) {
					instance = strtol(optarg, NULL, 0);
					break;
				}
			default: 
				usage(inst, count);
		}		
	}
	
	if(optind < argc) {
		path = argv[optind];
	}
	else 
		usage(inst, count);
	return err;
}

int main(int argc, char **argv)
{
	DICE_DECLARE_ENV(env);
	DICE_DECLARE_SERVER_ENV(env_srv);
	l4_taskid_t instances[LX_MAX_INSTANCES];
	l4_taskid_t task_id = L4_INVALID_ID;
	l4_threadid_t mir_id, freezer_id, dummy_id;
	unsigned int count = 0;
	int err;
	int l4_error;
	
	if(!names_query_name(LX_MIR, &mir_id)) {
		fprintf(stderr, "No mirgation back end running\n");
		return 1;
	}

	if(!names_query_name(LX_FREEZE, &freezer_id)) {
		fprintf(stderr, "Freezer not running\n");
		return 1;
	}
	
	/* get task ids of Linux instances at freezer */
	if_l4dm_lxmir_get_task_ids_call(&freezer_id, instances, &count, &env);

	if(parse_args(argc, argv, instances, count))
		usage(instances, count);

	if(!resume) {
		if(!instance || instance > count){
			fprintf(stderr, "No such instance %d\n", instance);
			return 1;
		}
		task_id = instances[instance - 1];
	}
	
	if((err = if_net_lxmir_migrate_call(&mir_id, &task_id, path, resume, &env))) {
		fprintf(stderr, "Mirgration initialization failed: %s (%d)\n", 
		        strerror(err), err);
		return 1;
	}
	
	printf("Waiting ...\n");
	/* wait while migrating */
	if((err = if_net_lxmir_done_wait(&dummy_id, &l4_error, &env_srv))) {
		fprintf(stderr, "Migration failed, %s error: %s (%d)\n", 
		        (l4_error)?"L4":"Linux",
		        (l4_error)?l4env_errstr(err):strerror(-1*err), err);
		return 1;
	}

	printf("Migration successfull\n");
	exit(0);

}












