/* $Id: liblxfreeze.c,v 1.2 2007/01/12 12:00:59 cbass Exp $ */
/*****************************************************************************/
/**
 *  \file liblxfreeze.c
 *  \brief Linux freezer library 
 * 
 *  \date 11/12/06 19:44:37 CET
 *  \author Sebastian Sumpf <Sebastian.Sumpf@inf.tu-dresden.de>
 */ 
/******************************************************************************/

#include <l4/lxfreeze/lxfreeze.h>
#include <l4/dm_mem/dm_mem.h>
#include <l4/lxfreeze/lxfreeze-client.h>
#include <l4/util/macros.h>
#include <l4/lxfreeze/stats.h>

#include "internal.h"

l4_threadid_t lxfreeze_get_id(void)
{
	if(__find_freezer_id() < 0)
		LOG_printf("Couldn't find lxfreeze server\n");
	 
	return _freezer_id;
}

l4_threadid_t lxfreeze_get_pager_id(void)
{
	if(__find_freezer_pager_id() < 0)
		LOG_printf("Couldn't find lxfreeze pager\n");
	return _freezer_pager_id;
}

int lxfreeze_find_env_infopage(void *page, l4_size_t size)
{
	DICE_DECLARE_ENV(env);
	int ret = 0;
	
	if((ret = __find_freezer_id()))
		return ret;

	if_l4dm_lxfreeze_find_env_infopage_call(&_freezer_id, page, size, &env);

	return ret;
}

int lxfreeze_find_info_stack(void * stack_addr, l4_size_t stack_size)
{
	DICE_DECLARE_ENV(env);
	int ret = 0;
	
	if((ret = __find_freezer_id()))
		return ret;
	
			
	if_l4dm_lxfreeze_find_stack_call(&_freezer_id, stack_addr, stack_size, &env);

	return ret;
}

int lxfreeze_handle_pagefault(unsigned long ds_id, l4_addr_t map_addr, l4_size_t size,
                              l4_uint32_t flags)
{
	
	DICE_DECLARE_ENV(env);
	int ret = 0;
	
	if((ret = __find_freezer_id()))
		return ret;
	
	ret = if_l4dm_lxfreeze_handle_pagefault_call(&_freezer_id, ds_id, map_addr, 
                                               size, flags, &env);

	return ret;
}

void lxfreeze_finish(l4_addr_t resume_eip, l4_addr_t resume_esp, 
                     unsigned long drain_ds_id)
{
	DICE_DECLARE_ENV(env);
	int ret = 0;
	
	if((ret = __find_freezer_id()))
		return;

	if_l4dm_lxfreeze_finish_call(&_freezer_id, resume_eip, resume_esp, drain_ds_id, 
			&env);
}

void lxfreeze_wake_up(l4_taskid_t *clone_id)
{
	DICE_DECLARE_ENV(env);
	int ret = 0;

	if((ret = __find_freezer_id()))
		return;

	if_l4dm_lxfreeze_wake_up_call(&_freezer_id, clone_id, 0, 0, &env);
}

void lxfreeze_kill_clone(l4_taskid_t clone_id)
{
	DICE_DECLARE_ENV(env);
	int ret = 0;

	if((ret = __find_freezer_id()))
		return;

	if_l4dm_lxfreeze_hard_exit_call(&_freezer_id, &clone_id, &env);
}

void lxfreeze_get_stats(lx_stats_t * stats)
{
	DICE_DECLARE_ENV(env);
	l4_size_t size;	
	int ret;
	lx_stats_t lx_stats;
	static void * addr;

	if(!addr)
		addr = l4dm_mem_allocate(sizeof(lx_stats), L4RM_MAP);

	if((ret = __find_freezer_id()))
		return;
	
	if_l4dm_lxfreeze_stats_get_call(&_freezer_id, &addr ,&size,
		&env);
	memcpy((void*)stats, addr, sizeof(lx_stats));

}

void lxfreeze_timer(void)
{
	DICE_DECLARE_ENV(env);
	int ret = 0;

	if((ret = __find_freezer_id()))
		return;

	if_l4dm_lxfreeze_timer_call(&_freezer_id, &env);
}

void lxfreeze_loaded(l4_threadid_t signal_id)
{
	DICE_DECLARE_ENV(env);
	int ret = 0;

	if((ret = __find_freezer_id()))
		return;
	if_l4dm_lxfreeze_loaded_call(&_freezer_id, &signal_id, &env);
}

void lxfreeze_signal_id(l4_threadid_t signal_id) 
{
	DICE_DECLARE_ENV(env);
	int ret = 0;

	if((ret = __find_freezer_id()))
		return;
	if_l4dm_lxfreeze_signal_id_call(&_freezer_id, &signal_id, &env);
}

