/*
 * \brief  Freezer GUI implementation
 * \author Norman Feske
 * \date   2007-01-10
 */

/*** GENERAL INCLUDES ***/
#include <stdio.h>
#include <stdlib.h>

/*** DOPE INCLUDES ***/
#include <l4/thread/thread.h>
#include <l4/semaphore/semaphore.h>
#include <l4/dope/dopelib.h>

/*** LOCAL INCLUDES ***/
#include "freezer_gui.h"

static l4semaphore_t gui_init_lock = L4SEMAPHORE_LOCKED;

/*** LOCAL ***/
static long app_id;    /* DOpE session id */

static void new_instance(dope_event * event, void * args)
{
	l4_taskid_t dummy;
	lxfreeze_wake_up(&dummy);
}

/*** INTERFACE: UPDATE STATISTICS IN DOPE WINDOW ***
 *
 * should be periodically called from the freezer main program
 */
void update_gui(lx_stats_t *stat) {

	int i;
	static int old_num_instances = -1;
//	int mem_payload = stat->mem_usage - stat->mem_overhead;

	/*
	 * (Re-)create new grid for displaying information about
	 * the different instances if their number changed
	 */
	if (stat->task_len != old_num_instances) {

		dope_cmd(app_id, "page_count_grid = new Grid()");
		dope_cmd(app_id, "page_count_grid.columnconfig(1)");

		for (i = 0; i < stat->task_len; i++) {
			dope_cmdf(app_id, "instance_label = new Label(-text \"Instance %d:\")", i);
			dope_cmdf(app_id, "page_count_grid.place(instance_label, -row %d -column 0)", i);
			dope_cmdf(app_id, "page_cnt%d = new Label()", i);
			dope_cmdf(app_id, "page_count_grid.place(page_cnt%d, -row %d -column 2)", i, i);
		}

		/* replace old by new page_count_grid in stat_grid */
		dope_cmd(app_id, "stat_grid.place(page_count_grid, -row 0 -column 1)");

		old_num_instances = stat->task_len;
	}

	/* update page count numbers */
	for (i = 0; i < stat->task_len; i++)
		dope_cmdf(app_id, "page_cnt%d.set(-text \"%d  \(%dKB)\")", i, stat->task[i].page_cnt, 
				(stat->task[i].page_cnt * L4_PAGESIZE)/1024);

	dope_cmdf(app_id, "total_frozen.set(-text \"%d  \(%dMB)\")", stat->page_frozen,
			(stat->page_frozen * L4_PAGESIZE)/(1024*1024));
	dope_cmdf(app_id, "total_copied.set(-text \"%d  \(%dMB)\")", stat->page_copied,
			(stat->page_copied * L4_PAGESIZE)/(1024*1024));
	dope_cmdf(app_id, "total_referenced.set(-text \"%d  \(%dMB)\")", stat->page_ref,
			(stat->page_ref * L4_PAGESIZE)/(1024*1024));
	int z;
	for (i = 0; i < stat->hash_len; i++) {
		z = stat->hash_len - i - 1;
		dope_cmdf(app_id, "hash_coll%d.set(-text %d)",  z, stat->hash_collisions[i]);
		dope_cmdf(app_id, "hash_match%d.set(-text %d)", z, stat->hash_matches[i]);
	}

	dope_cmdf(app_id, "mem_ld.set(-to %d)", stat->mem_total/1024);
	dope_cmdf(app_id, "mem_ld.barconfig(mem_overhead, -value %d)", stat->mem_overhead/1024);
	dope_cmdf(app_id, "mem_ld.barconfig(mem_usage, -value %d)", stat->mem_usage/1024);
	dope_cmdf(app_id, "hash_hot_cnt.set(-text %d)", stat->hash_hot_cnt);

	dope_cmdf(app_id, "mem_overhead.set(-text \"%dKB/%dMB\")", stat->mem_overhead/1024,
			stat->mem_overhead/(1024*1024));
	dope_cmdf(app_id, "mem_usage.set(-text \"%dKB/%dMB\")", stat->mem_usage/1024,
			stat->mem_usage/(1024*1024));
	dope_cmdf(app_id, "mem_free.set(-text \"%dKB/%dMB\")", stat->mem_free/1024,
			stat->mem_free/(1024*1024));
	dope_cmdf(app_id, "mem_total.set(-text \"%dKB/%dMB\")", stat->mem_total/1024,
			stat->mem_total/(1024*1024));
}


/*** DOPE THREAD ***
 *
 * This thread initializes the GUI and blocks for incoming
 * input events.
 */
static void gui_thread(void *arg) {
	int i;
	lx_stats_t * stat = (lx_stats_t *)arg;
	if (dope_init()) return;
	app_id = dope_init_app("Freezer");

	/* create static widgets */
	#include "freezer_gui.dpi"
	
	/* dynamically generate GUI elements for hash statistics */
	for (i = 0; i < stat->hash_len; i++) {
		dope_cmdf(app_id, "l = new Label(-text \"Hashing level %d:\")", i + 1);
		dope_cmdf(app_id, "hash_coll_grid.place(l, -row %d -column 0)", i);
		dope_cmdf(app_id, "hash_coll%d = new Label(-text 0)", i);
		dope_cmdf(app_id, "hash_coll_grid.place(hash_coll%d, -row %d -column 2)", i, i);

		dope_cmdf(app_id, "l = new Label(-text \"Hashing level %d:\")", i + 1);
		dope_cmdf(app_id, "hash_match_grid.place(l, -row %d -column 0)", i);
		dope_cmdf(app_id, "hash_match%d = new Label(-text 0)", i);
		dope_cmdf(app_id, "hash_match_grid.place(hash_match%d, -row %d -column 2)", i, i);
	}

	/* add hash grids to statistics grid */
	dope_cmd(app_id, "stat_grid.place(hash_coll_grid,  -row 20 -column 1)");
	dope_cmd(app_id, "stat_grid.place(hash_match_grid, -row 30 -column 1)");

	dope_bind(app_id, "new_instance", "click", new_instance, NULL);
	/* unblock init_gui() */
	l4semaphore_up(&gui_init_lock);

	/* block for events */
	dope_eventloop(app_id);
}


/*** INTERFACE: INIT DOPE CLIENT ***/
int init_gui(lx_stats_t * stat) {

	/* start freezer gui thread */
	l4thread_create(gui_thread, stat, L4THREAD_CREATE_ASYNC);

	/* wait until the thread finished executing the GUI init code */
	l4semaphore_down(&gui_init_lock);

	return 0;
}
