/* $Id:$ */
/*******************************************************************************/
/*
 * \file   pages.c
 * \brief  task, dataspace, page handling
 *
 * \date   Makefile
 * \author Sebastian Sumpf <sumpf@os.inf.tu-dresden.de>
 */
/*******************************************************************************/
/* (c) 2007 Technische Universitaet Dresden
 * This file is part of DROPS, which is distributed under the terms of the
 * GNU General Public License 2. Please see the COPYING file for details.
 */
#include<l4/log/l4log.h>
#include<l4/l4rm/l4rm.h>
#include<l4/util/atomic.h>
#include<l4/util/bitops.h>
#include<l4/util/memdesc.h>
#include "__types.h"
#include "__lxcalls.h"
#include "__hash.h"

static lx_task_t lx_tasks[LX_MAX_INSTANCES];
static l4slab_cache_t cow_page_cache;
static l4_addr_t lx_page_get_map_addr(lx_ds_t *lxds, l4_offs_t offs);

/*******************************************************************************
 * TASK LEVEL
 ******************************************************************************/

int lx_task_add_pos(l4_taskid_t task_id, int pos)
{
	if(pos < 0)
	{
		pos = 0;
		while(!l4_is_invalid_id(lx_tasks[pos].task_id) && pos < LX_MAX_INSTANCES
				&& !l4_task_equal(task_id, lx_tasks[pos].task_id))
			pos++;
		if(pos >= LX_MAX_INSTANCES)
		{
			LOGd(DBG_ERROR, "Error: Task limit exceeded\n");
			return -L4_ERES;
		}
	}
		
  lx_tasks[pos].task_id = task_id;
	return pos;
}


int lx_task_do_get(l4_taskid_t task_id, int pos, lx_task_t ** task)
{
	static int last = 0;
	
  if(l4_task_equal(task_id, lx_tasks[last].task_id) && pos < 0)
		pos = last;
	else if(pos < 0) 
	{
		pos = 0;
		while(pos < LX_MAX_INSTANCES && 
				!l4_task_equal(task_id, lx_tasks[pos].task_id) &&
				!l4_thread_equal(task_id, lx_tasks[pos].mir_id))
			pos++;
		if(pos >= LX_MAX_INSTANCES)
			return -L4_ENOTFOUND;
	}
	*task = &lx_tasks[pos];
	last = pos;
	return pos;
}


lx_task_t * lx_task_get_loading(void) 
{
	lx_task_t * task = NULL;
	lx_task_get(loader_id, &task);
	return task;
}

int lx_task_create_loading(void)
{
	if(lx_task_get_loading() != NULL)
	{
		LOGd(DBG_ERROR, "Multiple loading tasks");
		return -1;
	}
	lx_task_add(loader_id);
	return 0;
}

void lx_task_del(l4_taskid_t task_id)
{
	lx_task_t * task;
	int pos;
	lx_cow_page_t *cow, *next;

	if((pos = lx_task_get(task_id, &task)) < 0)
		return;

	while(task->ds[0].ds_desc != NULL) 
		lx_ds_del(task, lx_ds_get_id(&(task->ds[0])));
	
	for(cow = task->cow_page_first; cow != task->cow_page_last; 
	    cow = next)
	{
		next = cow->next;
		lx_cow_set_free(cow->map_addr);
		lx_slab_free(&cow_page_cache, (void*)cow);
	}

	if(pos < LX_MAX_INSTANCES - 1)
		memmove((void*)&lx_tasks[pos], (void *)&(lx_tasks[pos + 1]),
		        (LX_MAX_INSTANCES - pos - 1) * sizeof(lx_task_t));

	lx_tasks[LX_MAX_INSTANCES - 1] = LX_INVALID_TASK;
}


void lx_task_get_ids(l4_taskid_t *tasks, l4_uint32_t *count)
{
	int pos = 0;
	
	*count = 0;
	while(pos < LX_MAX_INSTANCES &&
	  !l4_is_invalid_id(lx_tasks[pos].task_id))
		*(tasks + (*count)++) = lx_tasks[pos++].task_id;
}


static void __clear_region_map(l4_uint32_t ds_id, l4_addr_t cur_map_addr) 
{
	l4_addr_t addr, map_addr;
	l4_size_t map_size;
	l4dm_dataspace_t ds;
	l4_offs_t offs;
	l4_threadid_t pager;
	int type;
	l4_addr_t mem_high = l4util_memdesc_vm_high();

	map_addr = map_size;
	addr = 0x1000;
	while(addr < mem_high)
	{
		type = l4rm_lookup_region((void*)addr, &map_addr, &map_size, &ds, &offs, 
		                          &pager);
		
		if(type <= 0) 
		{
			addr += L4_PAGESIZE;
			continue;
		}

		if(type == L4RM_REGION_DATASPACE && ds.id == ds_id 
		   && map_addr != cur_map_addr)
			l4rm_detach((void*)map_addr);

		addr = map_addr + map_size;
	}
}

void lx_task_remap_ro(lx_task_t *task) 
{
	int pos = 0;
	l4_size_t size;
	lx_ds_t * lxds;
	l4_uint32_t unmap_size;
	l4_addr_t unmap_start;

	while(task->ds[pos].ds_desc != NULL && pos < LX_MAX_DS) 
	{
		lxds = &task->ds[pos];
		size = lxds->size;
		unmap_start = lxds->map_addr;
		
		while(size > 0) 
		{
			unmap_size = l4util_bsr(size);
			
			l4_fpage_unmap(l4_fpage(unmap_start, unmap_size, 0, 0),
			               L4_FP_REMAP_PAGE | L4_FP_OTHER_SPACES);

			size        -= (1 << unmap_size);
			unmap_start += (1 << unmap_size);
		}
		
		if(lxds->flags & LX_RESIZE)
			__clear_region_map(lxds->ds.id, lxds->map_addr);
		
		pos++;
	}
}


/*******************************************************************************
 * DATASPACE LEVEL
 ******************************************************************************/

int lx_ds_do_get(lx_task_t *task, int pos, unsigned long ds_id,  lx_ds_t ** ds)
{
	static int last = 0;
	
	if(task->ds[last].ds_desc != NULL 
	     && lx_ds_get_id(&(task->ds[last])) == ds_id && pos < 0)
		pos = last;
	else if(pos < 0)
	{
		pos = 0;
		while(task->ds[pos].ds_desc != NULL
		        && lx_ds_get_id(&(task->ds[pos])) != ds_id 
		        && pos < LX_MAX_DS)
			pos++;
		
		if(pos >= LX_MAX_DS || task->ds[pos].ds_desc == NULL)
			return -L4_ENOTFOUND;
	}
	*ds = &(task->ds[pos]);
	last = pos;
	return pos;
}


static int lx_ds_add(lx_task_t * task, l4dm_dataspace_t ds, 
                     dsmlib_ds_desc_t * ds_desc, l4_size_t size, 
                     l4_addr_t addr)
{
	lx_ds_t new_ds = LX_DS_INITIALIZER;
	int i = 0;
	unsigned long  max_id = 0;
	while(task->ds[i].ds_desc != NULL 
	      && i < LX_MAX_DS)
	{
	  max_id = max_id < task->ds[i].virt_ds_id ? task->ds[i].virt_ds_id : max_id;
		i++;
	}

	if(i >= LX_MAX_DS)
	{
		LOGd(DBG_ERROR, "Error: Dataspaces limit exceeded for task "l4util_idfmt
		     , l4util_idstr(task->task_id));
		return -L4_ERES;
	}

	if(l4dm_dataspace_equal(task->ds[i].ds, ds))
		return i;
	
	new_ds.ds_desc         = ds_desc;
	new_ds.ds              = ds;
	new_ds.size            = size;
	new_ds.map_addr        = addr;
	if(task_has_state(task, TASK_MIGRATED))
		new_ds.virt_ds_id    = max_id + 1;
	else
		new_ds.virt_ds_id      = (unsigned long)dsmlib_get_id(ds_desc);
	
	task->ds[i]            = new_ds;
	
	LOGd(DBG_OPEN, "Adding dataspace at pos %d ds_id %u (mapped at %lu) " \
      "native_id %u (mapped %08lx)", 
	     i, dsmlib_get_id(ds_desc), new_ds.virt_ds_id, ds.id, addr);
	
	return i;
}


void lx_ds_del(lx_task_t * task, unsigned long ds_id)
{
	lx_ds_t * ds;
	int pos;
 
 	if((pos = lx_ds_get(task, ds_id, &ds)) < 0) 
		return;
	
	/* release dataspace */
	dsmlib_release_dataspace(ds->ds_desc);
	l4dm_mem_release((void*)ds->map_addr);
	
	/* delete at pager */
	lx_pager_del_ds(task, ds_id);
	
	/* FIXME: */
	if((task_has_state(task, TASK_LOADED) || task_has_state(task, TASK_LOADING))
	    && dstolxds[lx_ds_get_id(ds) - 1] != NULL)
		dstolxds[lx_ds_get_id(ds) - 1] = NULL;

	if(pos < LX_MAX_DS - 1)
		memmove((void*)&(task->ds[pos]), (void *)&(task->ds[pos + 1]),
		        (LX_MAX_DS - pos - 1) * sizeof(lx_ds_t));
	task->ds[LX_MAX_DS - 1] = LX_INVALID_DS;
}
 

int lx_ds_add_ds(lx_task_t * task, l4dm_dataspace_t ds, 
                 dsmlib_ds_desc_t * ds_desc, l4_size_t size, void * addr)
{
	int error = 0;
	LOGd(DBG_SUSP, "ID: %u DS size %08x\n", ds.id, size);

	error = l4rm_attach(&ds, size, 0, L4DM_RW|L4RM_MAP|L4RM_LOG2_ALIGNED, &addr);
	if(error)
	{
		LOGd(DBG_ERROR, "Failed to attach dataspace: %s (%d)\n",
		     l4env_errstr(error), error);
		return error;
	}
	return lx_ds_add(task, ds, ds_desc, size, (l4_addr_t)addr);
}


/* caution just for continuous memory */
int lx_ds_search_equal(lx_task_t * task, void * data, l4_size_t size, 
                       unsigned long * ds_id)
{
	int pos;
	int ret = 1;

	lx_ds_t * lxds;
	l4_addr_t map_addr;
	size = (size + L4_PAGESIZE - 1) & L4_PAGEMASK;

	for(pos = 0;;pos++)
	{
		if(task->ds[pos].ds_desc == NULL && pos >= LX_MAX_DS)
			break;
		
		lxds = &(task->ds[pos]);
		
		if(lxds->size != size)
			continue;

		map_addr = lxds->map_addr;
		ret = memcmp((void*)map_addr, data, size);
		if(!ret) 
		{
			*ds_id = lx_ds_get_id(lxds);
			break;
		}
	}
	
	return ret;
}


static int __check_virt(lx_task_t * task, unsigned long ds_id, 
                        l4_addr_t * addr, l4_size_t * size)
{
	int i = 0;
	
	while(i < LX_MAX_DS && !lx_is_invalid_ds_id(task->fault[i].ds_id))
	{
		if(task->fault[i].ds_id == ds_id) {
			*addr = task->fault[i].virt_addr;
			*size = task->fault[i].virt_size;
			return 1;
		}
		i++;
	}

	return 0;
}


void lx_ds_get_ids(lx_task_t * task, lx_ds_info_t * ds_info, 
                   l4_uint32_t * count)
{
	lx_ds_t * lxds;
	int pos = 0;
	*count = 0;
	lxds = task->ds;
	while(lxds[pos].ds_desc != NULL && pos < LX_MAX_DS) {
		ds_info[*count].ds_id = lx_ds_get_id(&(lxds[pos]));
		ds_info[*count].size  = lxds[pos].size;
		ds_info[*count].flags = DS_INFO_VALID;
		
		strcpy(ds_info[*count].name, dsmlib_get_name(lxds[pos].ds_desc));
		
		if(__check_virt(task, 
		                lx_ds_get_id(&(lxds[pos])), 
		                &(ds_info[*count].virt_addr), 
		                &(ds_info[*count].virt_size)))
		   ds_info[*count].flags |= DS_INFO_VIRT;
		
		(*count)++; pos++;
	}
}


unsigned long lx_ds_get_id(lx_ds_t * lxds)
{
	return lxds->virt_ds_id;
}


static l4_addr_t lx_page_get_map_addr(lx_ds_t * lxds, l4_offs_t offs)
{
	if(offs >= lxds->size)
		return LX_INVALID_ADDR;
	else
		return lxds->map_addr + offs;
}

/*******************************************************************************
 * PAGEFAULT HANDLING
 ******************************************************************************/


static int __page_copy(void * from, void ** to)
{
	
	/* add page */	
	*to  = lx_cow_get_free();
	if(*to == NULL)
		return -L4_ENOMEM;

	memcpy(*to, (void*)from, L4_PAGESIZE);
	return 0;
}

int
lx_page_fault(l4_threadid_t src, 
              unsigned long ds_id, 
              unsigned long offset,
              unsigned long size,
              int rw,
              l4_addr_t * map_addr)
{
	lx_task_t * task = NULL;
	lx_ds_t * lxds = NULL;
	lx_cow_page_t *cow_page = NULL;
	int error;
	void * new_addr;
	l4_uint32_t rights = (rw) ? (L4DM_WRITE|L4DM_READ) : L4DM_READ;
	
	*map_addr = 0;
	offset &= L4_PAGEMASK;
	
	if((error = lx_task_get(src, &task)) < 0 &&
		  (task = lx_task_get_loading()) == NULL)
	{
		/* check if task is a  client */
		lxds = dstolxds[ds_id -1];
		
		if(lxds != NULL && lx_ds_get_id(lxds) != dsmlib_get_id(lxds->ds_desc))
			enter_kdebug("FIXME: Virtualized access from foreign client");
		
		if(lxds != NULL 
		  && !dsmlib_check_rights(lxds->ds_desc, src, rights)
		  && !dsmlib_is_owner(lxds->ds_desc, src))
			lxds = NULL;
		
	}
	
	if(lxds == NULL && task == NULL)
	{
		/* OOps */
		LOGd(DBG_ERROR,
		     "No such task "l4util_idfmt" or datspace %lu (rights: %u)\n", 
		     l4util_idstr(src), ds_id, rights);
		enter_kdebug("Alien task");
		return error;
	}
	
	if(lxds == NULL)
	{
		error = lx_ds_get(task, ds_id, &lxds);
		if(error == -L4_ENOTFOUND)
			return error;
	}
	
	if(offset + size  > lxds->size)
		return -L4_EINVAL_OFFS;

	if(task != NULL && task_has_state(task, TASK_SEND_PRECOPY) && rw) 
	{
		cow_page           = lx_slab_alloc(&cow_page_cache);
		*cow_page          = LX_INVALID_COW_PAGE;
		cow_page->ds_id    = ds_id;
		cow_page->offs     = offset;
		
		__page_copy((void*)lx_page_get_map_addr(lxds, offset), &new_addr);
		cow_page->map_addr = *map_addr = (l4_addr_t)new_addr;
		
		lx_cow_page_add(task, cow_page);
		task->cow_cnt++;
	}
	else
	  *map_addr = lx_page_get_map_addr(lxds, offset);

	if(lx_is_invalid_addr(*map_addr))
		return -L4_ENOTFOUND;
	
	return 0;
}


/*******************************************************************************
 * INITIALIZATION
 ******************************************************************************/


int lx_task_init(void)
{
	int error;
	int i, j;
	for(i = 0; i < LX_MAX_INSTANCES; i++) {
		lx_tasks[i] = LX_INVALID_TASK;
		for(j = 0; j < LX_MAX_DS; j++) {
			lx_tasks[i].ds[j]                 = LX_INVALID_DS;
			lx_tasks[i].fault[j]              = LX_INVALID_FAULT;
		}
	}
	error = lx_slab_setup(&cow_page_cache, sizeof(lx_cow_page_t));
  return error;
}


#if DS_CHECKSUM
void dbg_lx_ds_hash(lx_task_t * task)
{
	int i = 0;
	l4_uint32_t key;
	lx_ds_t lxds;

	for(i = 0; i < LX_MAX_DS; i++)
	{
		lxds = task->ds[i];
		
		if(lx_is_invalid_ds_id(lxds.virt_ds_id))
			continue;
	  
		key = hashword((void*)(lxds.map_addr), lxds.size / sizeof(l4_uint32_t), 0); 
		
		LOG_printf("%d\tds_id %lu (native %u) size: %08x %s\tkey: %08x\n", 
		           i, lx_ds_get_id(&lxds), dsmlib_get_id(lxds.ds_desc), lxds.size,
	             lxds.flags & LX_POSTCPY? "[P]":"", key);
	}
	LOG_printf("POST COPY cnt: %u\n", task->cow_cnt);
}
#endif

