/*
 * \brief   BMSI command-line client.
 * \date    2007-09-27
 * \author  Carsten Weinhold <weinhold@os.inf.tu-dresden.de>
 * \author  Christelle Braun 
 * \author  Alexander Boettcher <boettcher@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2007 Carsten Weinhold <weinhold@os.inf.tu-dresden.de>
 * Copyright (C) 2007 Christelle Braun <cbraun@os.inf.tu-dresden.de>
 * Technische Universitaet Dresden, Operating Systems Research Group
 *
 * This file is part of the BMSI package, which is distributed under
 * the  terms  of the  GNU General Public Licence 2.  Please see the
 * COPYING file for details.
 */

/* general includes */
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>

/* L4-specific includes */
#include <l4/bmsi/bmsi-oo.h>
#include <l4/bmsi/bmsi.h>

/*
 * **************************************************************************
 */

#define DBG_TEST 1
#define printfD(doit, msg... ) do { if (doit) printf(msg); } while (0)
/*
 * **************************************************************************
 */

static int parse_number(const char *str, long long *number_out) {

    *number_out = strtoll(str, NULL, 0);
    return (errno == 0) ? 0 : -1;
}

/*
 * **************************************************************************
 */

static int init_builder(char *builder_tag, PDBuilder* builder_out,
                        int *version_out) {

    BuilderInitializer builder_params = builder_tag;
    int res;

    /* Get the BMSI version. */
    res = BMSI_getVersion(version_out);
    printfD(DBG_TEST, "BMSI_getVersion(): %s (res=%d) - version is %d\n", 
                      (res == 0 ? "success" : "failed"), res, *version_out);
    if (res != 0)
        return res;

    /* Create a PDBuilder instance ('Builder'). */
    res = BMSI_getBuilder(builder_params, builder_out);
    printfD(DBG_TEST, "BMSI_getBuilder(): %s (res=%d)\n",
                      (res == 0 ? "success" : "failed"), res);

    return res;
}


static int start_new_pd(PDBuilder builder, char *bin_name,
                        char *bin_args, int is_linux, int mem_size,
                        char *pd_tag, int64 *local_id_out) {

    /* Create a PD instance. */
    DomainInitializer pd_params = pd_tag;
    ProtectionDomain  pd;
    Resources         resources;
    L4PDDescriptor    l4_pd_desc;
    int res;
    
    res = PDBuilder_createPD(builder, pd_params, &pd);
    printfD(DBG_TEST, "PDBuilder_createPD(): %s (%d)\n", 
                      (res == 0 ? "success" : "failed"), res);
    if (res != 0)
        return res;

    /* Set Resources for PD1. */
    resources.MemoryMax     = mem_size;
    resources.MemoryCurrent = mem_size;
    resources.Priority      = 1;
    resources.NbrCpu        = 1;

    res = ProtectionDomain_setResources(pd, resources);
    printfD(DBG_TEST, "ProtectionDomain_setResources(): %s (res=%d), "
         "MemMax=%ld, MemCur=%ld, Prio=%d, NbCpus=%d\n",
         (res == 0 ? "success" : "failed"), res,
         resources.MemoryMax, resources.MemoryCurrent,
         resources.Priority, resources.NbrCpu);
    if (res != 0)
        return res;

    /* Build PD1. */
    l4_pd_desc.Name   = bin_name;
    l4_pd_desc.Memory = mem_size;
    l4_pd_desc.L4Lx   = is_linux;
    l4_pd_desc.Params = bin_args;

    res = ProtectionDomain_build(pd, &l4_pd_desc);
    printfD(DBG_TEST, "ProtectionDomain_build(): %s (res=%d)\n", 
                      (res == 0 ? "success" : "failed"), res);
    if (res != 0)
        return res;

    /* Get LocalID of PD2. */
    res = ProtectionDomain_getLocalId(pd, local_id_out);
    printfD(DBG_TEST, "ProtectionDomain_getLocalId: %s (res=%d), local id is %lld\n",
         (res == 0 ? "success" : "failed"), res, *local_id_out);

    return res;
}


static int change_pd_status(PDBuilder builder, int64 local_id,
                            DomainStatus status) {

    PDDescriptor pd;
    int res;

    res = PDBuilder_findPD(builder, local_id, &pd);
    printfD(DBG_TEST, "PDBuilder_findPD(): %s (res=%d)\n",
                      (res == 0 ? "success" : "failed"), res);
    if (res != 0)
        return res;

    switch (status) {
        case DomainStatusRunning:
            res = ProtectionDomain_run(pd);
            printfD(DBG_TEST, "ProtectionDomain_run(): %s (res=%d)\n",
                      (res == 0 ? "success" : "failed"), res);
            break;
        case DomainStatusPaused:
            res = ProtectionDomain_pause(pd);
            printfD(DBG_TEST, "ProtectionDomain_pause(): %s (res=%d)\n",
                      (res == 0 ? "success" : "failed"), res);
        default:
            printf("Unsupported status %d\n", status);
            res = -1;
    }

    return res;
}


static int destroy_pd(PDBuilder builder, int64 local_id) {

    PDDescriptor pd;
    int res;

    res = PDBuilder_findPD(builder, local_id, &pd);
    printfD(DBG_TEST, "PDBuilder_findPD(): %s (res=%d)\n",
                      (res == 0 ? "success" : "failed"), res);
    if (res != 0)
        return res;

    res = ProtectionDomain_destroy(pd);
    printfD(DBG_TEST, "ProtectionDomain_destroy(): %s (res=%d)\n",
                      (res == 0 ? "success" : "failed"), res);

    return res;
}


static int probe_pds(PDBuilder builder, int num_pds_to_probe,
                     int64 local_ids_out[]) {

    PDDescriptor pd;
    int i, res, found = 0;

    for (i = 0; i < num_pds_to_probe; i++) {
        res = PDBuilder_findPD(builder, i, &pd);
//        printfD(DBG_TEST, "PDBuilder_findPD(): %s (res=%d)\n",  
//                      (res == 0 ? "success" : "failed"), res);
        if (res == 0)
        {
            local_ids_out[found] = i;
            found++;
        }
        else if (0 /* res != BMSI_ErrorNotFound */)
            return res;
    }

    /* terminate list */
    local_ids_out[found] = -1;

    return 0;
}

// allow/disallow communication between protection domains
static int comm_rights_pd (PDBuilder builder, int64 pd1, int64 pd2, boolean allow) {

    int res;
    PDDescriptor pdd1;
    PDDescriptor pdd2;

    res = PDBuilder_findPD(builder, pd1, &pdd1);
    printfD(DBG_TEST, "PDBuilder_findPD(): pd1 %s (res=%d)\n",
                      (res == 0 ? "success" : "failed"), res);
    if (res != 0)
        return res;

    res = PDBuilder_findPD(builder, pd2, &pdd2);
    printfD(DBG_TEST, "PDBuilder_findPD(): pd2 %s (res=%d)\n",
                      (res == 0 ? "success" : "failed"), res);
    if (res != 0)
        return res;

    res = ProtectionDomain_allowConnection(pdd1, pdd2, allow);
    printf("ProtectionDomain_allowConnection(): %s (res=%d)\n",
                      (res == 0 ? "success" : "failed"), res);

    return res; 
}

/*
 * **************************************************************************
 */

/* Print information about the PDs currently registered at the BMSI */
static int print_all_pds(PDBuilder builder) {

    int64 local_ids[32];
    int i, res;

    res = probe_pds(builder, 32, local_ids);
    printfD(DBG_TEST, "probe_pds(): %s (res=%d)\n",
                      (res == 0 ? "success" : "failed"), res);
    if (res != 0)
        return res;

    for (i = 0; local_ids[i] >= 0; i++)
        printf("found PD %lld\n", local_ids[i]);

    return 0;
}


static void show_help_and_exit(PDBuilder builder) {

    printf("Invalid or no command specified!\n");
    printf("Usage: bmsic <builder_name> --start|--start-linux <binary_name> <binary_arg_string> <mem_size> <tag>\n");
    printf("   or: bmsic <builder_name> --pause <pd_id>\n");
    printf("   or: bmsic <builder_name> --run <pd_id>\n");
    printf("   or: bmsic <builder_name> --destroy <pd_id>\n");
    printf("   or: bmsic <builder_name> --probe-pds\n");
    printf("   or: bmsic <builder_name> --commu <pd_id1> <pd_id2> <(dis)allow>\n");

    if (builder)
        PDBuilder_free(builder);

    exit(1);
}

/*
 * **************************************************************************
 */

int main(int argc, char** argv) {

    PDBuilder builder = NULL;
    char *cmd;
    int64 local_id, local_id2, mem_size;
    int version, res;

    if (argc < 3)
        show_help_and_exit(builder);

    cmd = argv[1];
    res = init_builder(cmd, &builder, &version);
    printf("init_builder(): %s (res=%d)\n",
           (res == 0 ? "success" : "failed"), res);
    if (res != 0)
        return 1;

    cmd = argv[2];
    printf("BMSI version %d\n", version);

    if (strcmp(cmd, "--start") == 0 || strcmp(cmd, "--start-linux") == 0) {
        if (argc < 7)
            show_help_and_exit(builder);
        if (parse_number(argv[5], &mem_size) != 0)
            show_help_and_exit(builder);
        res = start_new_pd(builder, argv[3], argv[4],
                           strcmp(cmd, "--start-linux") == 0,
                           mem_size, argv[6], &local_id);
        if (res == 0)
            res = change_pd_status(builder, local_id, DomainStatusRunning);
        printf("Start of new PD %s (res=%d), local_id=%lld\n",
                      (res == 0 ? "success" : "failed"), res, local_id);

    } else if (strcmp(cmd, "--run") == 0) {
        if (argc < 4)
            show_help_and_exit(builder);
        if (parse_number(argv[3], &local_id) != 0)
            show_help_and_exit(builder);
        res = change_pd_status(builder, local_id, DomainStatusRunning);

    } else if (strcmp(cmd, "--pause") == 0) {
        if (argc < 4)
            show_help_and_exit(builder);
        if (parse_number(argv[3], &local_id) != 0)
            show_help_and_exit(builder);
        res = change_pd_status(builder, local_id, DomainStatusPaused);

    } else if (strcmp(cmd, "--destroy") == 0) {
        if (argc < 4)
            show_help_and_exit(builder);
        if (parse_number(argv[3], &local_id) != 0)
            show_help_and_exit(builder);
        res = destroy_pd(builder, local_id);

    } else if (strcmp(cmd, "--probe-pds") == 0) {
        res = print_all_pds(builder);
    } else if (strcmp(cmd, "--commu") == 0) {
        if (argc < 6)
            show_help_and_exit(builder);
        
        if (parse_number(argv[3], &local_id) != 0)
            show_help_and_exit(builder);
        if (parse_number(argv[4], &local_id2) != 0)
            show_help_and_exit(builder);
        if (parse_number(argv[5], &mem_size) != 0)
            show_help_and_exit(builder);

        res = comm_rights_pd (builder, local_id, local_id2, mem_size);
    } else {
				printf("Unknown cmd='%s'\n", cmd);
        show_help_and_exit(builder);
    }

    return (res == 0) ? 0 : 1;
}
 
