%{                                            /* -*- C++ -*- */
# include <cstdlib>
# include <errno.h>
# include <string>
# include <limits>
# include "c-parser-driver.hh"
# include "c-parser.tab.hh"
# include "parser/Converter.h"
# include "parser/Preprocessor.h"

    /* Work around an incompatibility in flex (at least versions
       2.5.31 through 2.5.33): it generates code that does
       not conform to C89.  See Debian bug 333231
       <http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=333231>.  */
# undef yywrap
# define yywrap() 1

    /* By default yylex returns int, we use token_type.
       Unfortunately yyterminate by default returns 0, which is
       not of token_type.  */
#define yyterminate() return token::EOF_TOKEN

    /* These are helper variables for the line directives generated by the
     * preprocessor. We store filename and line number across rule boundaries.
     */
static string sNewFilename;
static long nNewLinenumber;
static bool bSetLine = true;
static bool bStdInc = false;
static long nFlags;

%}

%option noyywrap nounput batch debug

/* some rules */
Float_literal1 [0-9]*"."[0-9]+((e|E)[+|-]?[0-9]+)?
Float_literal2 [0-9]+"."((e|E)[+|-]?[0-9]+)?
Float_literal3 [0-9]+((e|E)[+|-]?[0-9]+)
Float          ({Float_literal1}|{Float_literal2}|{Float_literal3})(f|l|F|L)?

Id             [a-zA-Z_][a-zA-Z0-9_]*
Integer_Suffix ([uU][lL]?)|([uU]([lL]{2}))|([lL][uU]?)|([lL]{2}[uU]?)
Integer        [1-9][0-9]*{Integer_Suffix}?
Hexadec        0[xX][a-fA-F0-9]+{Integer_Suffix}?
Octal          0[0-7]*{Integer_Suffix}?
Filename        ("\""|"<")("<")?[a-zA-Z0-9_\./\\:\- ]*(">")?("\""|">")

Char_lit       (L)?\'.\'

Escape         (L)?\'\\[ntvbrfa\\\?\'\"]\'
Oct_char       (L)?\'\\[0-7]{1,3}\'
Hex_char       (L)?\'\\(x|X)[a-fA-F0-9]{1,2}\'
string         (L)?(\"([^\\\n\"]|\\[ntvbrfa\\\?\'\"\n])*\")
C99string      \"([^\"]|\\\")*\"

/* Include support */
%x preproc
%x preproc2

%{
#define YY_USER_ACTION  yylloc->columns (yyleng);
%}
%%
%{
    yylloc->step ();
%}

%{
    typedef yy::c_parser::token token;
%}

"#"         {
    BEGIN(preproc);
    sNewFilename = "";
    bSetLine = true;
    nFlags = 0;
    bStdInc = false;
            }
<preproc,preproc2>[ \t]+ /* eat whitespace */
<preproc>[0-9]+ {
    nNewLinenumber = strtol(yytext, NULL, 10);
            }
<preproc>{Filename} {
    BEGIN(preproc2);
    string s(yytext);
    s = s.substr(1, s.length()-2); // remove "" or <>
    if (!s.empty())
	sNewFilename = s;
    else
	sNewFilename = driver.file;
    if (sNewFilename == "<stdin>")
	sNewFilename = driver.file;
    if (sNewFilename == "<built-in>")
	sNewFilename = driver.file;
    if (sNewFilename == "<command line>")
	sNewFilename = driver.file;
    if (sNewFilename == "<command-line>")
	sNewFilename = driver.file;
            }
<preproc2>[1-4] {
    long n = strtol(yytext, NULL, 10);
    /* flags are optional and have the following meaning:
     * 1 - start of the file specified with sNewFilename
     * 2 - end of the current file, return to file with name sNewFilename
     * 3 - start of a system header file (specified after 1 or 2)
     * 4 - extern "C" start (specified after 1 or 2)
     *
     * Because 3 and 4 only provide additional information that is not
     * interesting for us, we ignore them.
     */
    if (1 == n || 2 == n)
	nFlags = n;
    if (3 == n)
	bStdInc = true;
            }
<preproc,preproc2>("\r")?"\n" {
    BEGIN(INITIAL);
    int line = yylloc->end.line;
    /* always reset the location information */
    yylloc->end.line = nNewLinenumber;
    yylloc->end.column = 1;
    *yylloc->end.filename = sNewFilename;
    yylloc->begin = yylloc->end;

    if (driver.trace_scanning)
	std::cerr << "c-scanner: # " << nNewLinenumber << " \"" << sNewFilename << "\" " << nFlags <<
	    " (old: " << line << ") found\n";
    /* nFlags == 1: start new file with filename */
    if (nFlags == 1)
	driver.enter_file(sNewFilename, nNewLinenumber, bStdInc, line, true, true);
    /* nFlags == 2: end current file, return to file with filename */
    if (nFlags == 2)
	driver.leave_file(sNewFilename);
            }

[ \t]+      yylloc->step ();
\r          /* eat carriage return character */
[\n]+       yylloc->lines (yyleng); yylloc->columns(); yylloc->step ();
\f          yylloc->step ();
"//".*      yylloc->step ();

"/*"        {
    register int c;
    for (;;)
    {
	while ((c = yyinput ()) != '*' && c != EOF && c != '\n')
	    yylloc->step ();    // eat up text of comment
	if (c == '*')
	{
	    while ((c = yyinput ()) == '*')
		yylloc->step ();    // eat up trailing *
	    if (c == '/') break;    // found end
	}
	if (c == '\n')
	    yylloc->step ();
	if (c == EOF)
	{
	    driver.error (*yylloc, "EOF in comment.");
	    yyterminate ();
	    break;
	}
    }
            }


("{"|"<%")      return token::LBRACE;
("}"|"%>")      return token::RBRACE;
("["|"<:")      return token::LBRACKET;
("]"|":>")      return token::RBRACKET;
":"             return token::COLON;
"::"            return token::SCOPE;
","             return token::COMMA;
"("             return token::LPAREN;
")"             return token::RPAREN;
"."             return token::DOT;
".*"            return token::DOTSTAR_OP;
"*"             return token::ASTERISK;
"?"             return token::QUESTION;
"|"             return token::BITOR;
"^"             return token::BITXOR;
"&"             return token::BITAND;
"<"             return token::LT;
">"             return token::GT;
"+"             return token::PLUS;
"-"             return token::MINUS;
"/"             return token::DIV;
"%"             return token::MOD;
"~"             return token::TILDE;
"!"             return token::EXCLAM;
";"             return token::SEMICOLON;
"||"            return token::LOGICALOR;
"&&"            return token::LOGICALAND;
"^^"            return token::LOGICALXOR;
"=="            return token::EQUAL;
"!="            return token::NOTEQUAL;
"<="            return token::LTEQUAL;
">="            return token::GTEQUAL;
"<<"            return token::LSHIFT;
">>"            return token::RSHIFT;
"..."           return token::ELLIPSIS;
"="             return token::IS;
">>="           return token::RS_ASSIGN;
"<<="           return token::LS_ASSIGN;
"+="            return token::ADD_ASSIGN;
"-="            return token::SUB_ASSIGN;
"*="            return token::MUL_ASSIGN;
"/="            return token::DIV_ASSIGN;
"%="            return token::MOD_ASSIGN;
"&="            return token::AND_ASSIGN;
"^="            return token::XOR_ASSIGN;
"|="            return token::OR_ASSIGN;
"++"            return token::INC_OP;
"--"            return token::DEC_OP;
"<?"            return token::MIN;
">?"            return token::MAX;
"<?="           return token::MIN_ASSIGN;
">?="           return token::MAX_ASSIGN;
"->"            return token::PTR_OP;
"->*"           return token::PTRSTAR_OP;

"extern"[[:space:]]+"\""[cC]"\""	return token::EXTERN_LANG_STRING;
"("[[:space:]]*")"			return token::LEFT_RIGHT;
"("[[:space:]]*"*"[[:space:]]*")"	return token::PAREN_STAR_PAREN;


asm             return token::ASM_KEYWORD;
__asm           return token::ASM_KEYWORD;
__asm__         return token::ASM_KEYWORD;
alignof         return token::ALIGNOF;
attribute       return token::ATTRIBUTE;
__attribute     return token::ATTRIBUTE;
__attribute__   return token::ATTRIBUTE;
auto            return token::AUTO;
boolean         {
    yylval->sval = new string(yytext);
	return token::BOOLEAN;
				}
_Bool           {
    yylval->sval = new string(yytext);
	return token::BOOLEAN;
				}
break           return token::BREAK;
bycopy          return token::BYCOPY;
byref           return token::BYREF;
case            return token::CASE;
catch           return token::CATCH;
char            return token::CHAR;
class           return token::CLASS;
_Complex        return token::UCOMPLEX;
const           return token::CONST;
__const         return token::CONST;
__const__       return token::CONST;
const_cast      return token::CONST_CAST;
continue        return token::CONTINUE;
default         return token::DEFAULT;
delete          return token::DELETE;
do              return token::DO;
double          return token::DOUBLE;
dynamic_cast    return token::DYNAMIC_CAST;
else            return token::ELSE;
enum            return token::ENUM;
explicit        return token::EXPLICIT;
export          return token::EXPORT;
extension       /* eat extension keyword */
__extension__   /* eat extension keyword */
extern          return token::EXTERN;
false           return token::FALSE;
float           return token::FLOAT;
for             return token::FOR;
friend          return token::FRIEND;
goto            return token::GOTO;
if              return token::IF;
__imag          return token::IMAGPART;
__imag__        return token::IMAGPART;
_Imaginary      return token::IMAGPART;
in              return token::IN;
inline          return token::INLINE;
__inline        return token::INLINE;
__inline__      return token::INLINE;
inout           return token::INOUT;
int             return token::INT;
label           return token::ID;
__label         return token::ID;
__label__       return token::ID;
long            return token::LONG;
long[[:space:]]+long  return token::LONGLONG;
mutable         return token::MUTABLE;
namespace       return token::NAMESPACE;
new             return token::NEW;
operator        return token::OPERATOR;
out             return token::OUT;
private         return token::PRIVATE;
protected       return token::PROTECTED;
public          return token::PUBLIC;
__real          return token::REALPART;
__real__        return token::REALPART;
register        return token::REGISTER;
reinterpret_cast return token::REINTERPRET_CAST;
restrict        return token::RESTRICT;
__restrict      return token::RESTRICT;
return          return token::RETURN;
short           return token::SHORT;
signed          return token::SIGNED;
__signed__      return token::SIGNED;
sizeof          return token::SIZEOF;
static          return token::STATIC;
static_cast     return token::STATIC_CAST;
struct          return token::STRUCT;
switch          return token::SWITCH;
template        return token::TEMPLATE;
this            return token::THIS;
throw           return token::THROW;
true            return token::TRUE;
try             return token::TRY;
typedef         return token::TYPEDEF;
typeid          return token::TYPEID;
typename        return token::TYPENAME;
typeof          return token::TYPEOF;
__typeof        return token::TYPEOF;
__typeof__      return token::TYPEOF;
union           return token::UNION;
unsigned        return token::UNSIGNED;
using           return token::USING;
virtual         return token::VIRTUAL;
void            return token::VOID;
volatile        return token::VOLATILE;
__volatile__    return token::VOLATILE;
__volatile      return token::VOLATILE;
wchar_t         {
    yylval->sval = new string(yytext);
	return token::WCHAR;
				}
while           return token::WHILE;

__builtin_va_list   {
    yylval->sval = new string(yytext);
    return token::VA_LIST;
           }
__null     {
    yylval->ival = 0;
    return token::LIT_INT;
           }

{Id}       {
	token::yytokentype ret = token::ID;
	yylval->sval = new std::string(yytext);
	// check if this is a defined type
	if (driver.check_token(yytext, CSymbolTable::TYPENAME))
		ret = token::TYPEDEF_ID;
// 	else if (driver.check_token(yytext, CSymbolTable::CLASS))
// 		ret = token::CLASS_ID;
	return ret;
	   }


{Integer}  {
	using namespace dice::parser;
	IntType t;
#if HAVE_ATOLL
	long
#endif
		long n = int_to_long (yytext, t);
	switch (t)
	{
	case INT_ULLONG:
#if HAVE_ATOLL
		yylval->ullval = static_cast<unsigned long long>(n);
		return token::LIT_ULLONG;
		break;
#endif
	case INT_ULONG:
		yylval->ulval = static_cast<unsigned long>(n);
		return token::LIT_ULONG;
		break;
	case INT_LLONG:
#if HAVE_ATOLL
		yylval->llval = n;
		return token::LIT_LLONG;
		break;
#endif
	case INT_LONG:
		yylval->lval = static_cast<long>(n);
		return token::LIT_LONG;
		break;
	case INT_INT:
		yylval->ival = static_cast<int>(n);
		return token::LIT_INT;
		break;
	case INVALID:
		driver.error(*yylloc, "value out of range");
		break;
	}
	   }
{Hexadec}  {
	using namespace dice::parser;
	IntType t;
#if HAVE_ATOLL
	long
#endif
		long n = hex_to_long (yytext, t);
	switch (t)
	{
	case INT_ULLONG:
#if HAVE_ATOLL
		yylval->ullval = static_cast<unsigned long long>(n);
		return token::LIT_ULLONG;
		break;
#endif
	case INT_ULONG:
		yylval->ulval = static_cast<unsigned long>(n);
		return token::LIT_ULONG;
		break;
	case INT_LLONG:
#if HAVE_ATOLL
		yylval->llval = n;
		return token::LIT_LLONG;
		break;
#endif
	case INT_LONG:
		yylval->lval = static_cast<long>(n);
		return token::LIT_LONG;
		break;
	case INT_INT:
		yylval->ival = static_cast<int>(n);
		return token::LIT_INT;
		break;
	case INVALID:
		driver.error(*yylloc, "value is out of range");
		break;
	}
	   }
{Octal}    {
	using namespace dice::parser;
	IntType t;
#if HAVE_ATOLL
	long
#endif
		long n = oct_to_long (yytext, t);
	switch (t)
	{
	case INT_ULLONG:
#if HAVE_ATOLL
		yylval->ullval = static_cast<unsigned long long>(n);
		return token::LIT_ULLONG;
		break;
#endif
	case INT_ULONG:
		yylval->ulval = static_cast<unsigned long>(n);
		return token::LIT_ULONG;
		break;
	case INT_LLONG:
#if HAVE_ATOLL
		yylval->llval = n;
		return token::LIT_LLONG;
		break;
#endif
	case INT_LONG:
		yylval->lval = static_cast<long>(n);
		return token::LIT_LONG;
		break;
	case INT_INT:
		yylval->ival = static_cast<int>(n);
		return token::LIT_INT;
		break;
	case INVALID:
		driver.error(*yylloc, "value is out of range");
		break;
	}
		   }

{Float}    {
	errno = 0;
	double d = strtod (yytext, NULL);
	if (! (std::numeric_limits<double>::min() <= d &&
			d <= std::numeric_limits<double>::max() &&
			errno != ERANGE))
		driver.error (*yylloc, "floating number out of range");
	yylval->dval = d;
	return token::LIT_FLOAT;
	   }
{Char_lit} {
	if (yytext[0] == 'L')
		yylval->cval = yytext[2];
	else
		yylval->cval = yytext[1];
	return token::LIT_CHAR;
	   }
{Escape}   {
	yylval->cval = dice::parser::escape_to_char (yytext);
	return token::LIT_CHAR;
	   }
{Oct_char} {
    yylval->cval = dice::parser::oct_to_char (yytext);
    return token::LIT_CHAR;
	   }
{Hex_char} {
    yylval->cval = dice::parser::hex_to_char (yytext);
    return token::LIT_CHAR;
	   }
{string}   {
	// check if first is 'L'
	if (yytext[0] == 'L')
		yylval->sval = new std::string (&yytext[2]);
	else
		yylval->sval = new std::string (&yytext[1]);
	while (*(yylval->sval->end()-1) == '"')
		yylval->sval->erase(yylval->sval->end()-1);
	return token::LIT_STR;
	   }
{C99string} {
	// allows embedded newlines
	if (yytext[0] == 'L')
		yylval->sval = new std::string (&yytext[2]);
	else
		yylval->sval = new std::string (&yytext[1]);
	while (*(yylval->sval->end()-1) == '"')
		yylval->sval->erase(yylval->sval->end()-1);
	// replace \n with \\n
	string::size_type pos;
	while ((pos = yylval->sval->find("\n")) != string::npos)
	{
		yylval->sval->replace(pos, strlen("\n"), "\\n", strlen("\\n"));
	}
	return token::LIT_STR;
	   }

<<EOF>>    yyterminate();
.          driver.warning (*yylloc, "invalid character");

%%

void
c_parser_driver::scan_begin ()
{
	using dice::parser::CPreprocessor;

	// save currently used input buffer
	previousBuffer = YY_CURRENT_BUFFER;

	yy_flex_debug = trace_scanning;
	CPreprocessor *pre = CPreprocessor::GetPreprocessor();
	if (!(yyin = pre->Preprocess(file, lastPath)))
		error (std::string ("cannot open ") + file);

	// only if we already parsed something, we need to switch to a new input
	// buffer
	if (previousBuffer)
	{
		yy_switch_to_buffer( yy_create_buffer( yyin, YY_BUF_SIZE ) );
		BEGIN(INITIAL);
	}
}

void
c_parser_driver::scan_end ()
{
	if (yyin)
		fclose (yyin);

	// if we stored a previous buffer, restore it
	if (previousBuffer)
	{
		if (YY_CURRENT_BUFFER)
			yy_delete_buffer(YY_CURRENT_BUFFER);
		yy_switch_to_buffer(previousBuffer);
		previousBuffer = 0;
	}
}

