/**
 *  \file    dice/src/Messages.cpp
 *  \brief   contains the implementation of the class CMessages
 *
 *  \date    04/10/2007
 *  \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2007
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "Messages.h"
#include <iostream>
#include <vector>
using std::vector;
#include "fe/FEFile.h"
#include "Compiler.h"

namespace CMessages
{

	/**
	 *  \brief helper function
	 *  \param sMsg the error message to print before exiting
	 *
	 * This method prints an error message and exits the compiler. Any clean up
	 * should be performed  BEFORE this method is called.
	 */
	void Error(const char *sMsg, ...)
	{
		std::cerr << "dice: ";
		va_list args;
		va_start(args, sMsg);
		vfprintf(stderr, sMsg, args);
		va_end(args);

		exit(1);
	}

	/** \brief helper function
	 *  \param pFEObject the front-end object where the error occurred
	 *  \param sMsg the error message
	 *
	 * The given front-end object can be used to determine the file this object
	 * belonged to and the line, where this object has been declared. This is
	 * useful if we do not have a current line number available (any time after
	 * the parsing finished).
	 */
	void GccError(CFEBase * pFEObject, const char *sMsg, ...)
	{
		va_list args;
		va_start(args, sMsg);
		GccErrorVL(pFEObject, sMsg, args);
		va_end(args);
	}

	/** \brief helper function
	 *  \param pFEObject the front-end object where the error occurred
	 *  \param sMsg the error message
	 *  \param vl the argument list to be printed
	 *
	 * The given front-end object can be used to determine the file this object
	 * belonged to and the line, where this object has been declared. This is
	 * useful if we do not have a current line number available (any time after
	 * the parsing finished).
	 */
	void GccErrorVL(CFEBase * pFEObject, const char *sMsg, va_list vl)
	{
		// increment internal error counter
		CCompiler::Error();
		if (pFEObject)
		{
			// iterate through include hierarchy
			CFEFile *pCur = (CFEFile*)pFEObject->GetSpecificParent<CFEFile>(0);
			vector<CFEFile*> *pStack = new vector<CFEFile*>();
			while (pCur)
			{
				pStack->insert(pStack->begin(), pCur);
				// get file of parent because GetFile start with "this"
				pCur = pCur->GetSpecificParent<CFEFile>(1);
			}
			// need at least one file
			if (!pStack->empty())
			{
				// walk down
				if (pStack->size() > 1)
					std::cerr << "In file included ";
				vector<CFEFile*>::iterator iter;
				for (iter = pStack->begin();
					(iter != pStack->end()) && (iter != pStack->end()-1); iter++)
				{
					// we do not use GetFullFileName, because the "normal"
					// filename already includes the whole path it is the filename
					// generated by Gcc
					CFEFile *pFEFile = *iter;
					int nIncludeLine = 1;
					if (iter != pStack->end()-1)
						nIncludeLine = (*(iter+1))->GetIncludedOnLine();
					std::cerr << "from " << pFEFile->GetFileName() << ":" << nIncludeLine;
					if (iter + 2 != pStack->end())
						std::cerr << ",\n                 ";
					else
						std::cerr << ":\n";
				}
				std::cerr << pFEObject->m_sourceLoc;
				std::cerr << ": ";
			}
			// cleanup
			delete pStack;
		}
		vfprintf(stderr, sMsg, vl);
	}
	/**
	 *  \brief helper function
	 *  \param sMsg the warning message to print
	 *
	 * This method prints an error message and returns.
	 */
	void Warning(const char *sMsg, ...)
	{
		va_list args;
		va_start(args, sMsg);
		vfprintf(stderr, sMsg, args);
		va_end(args);
	}

	/** \brief print warning messages
	 *  \param pFEObject the object where the warning occured
	 *  \param sMsg the warning message
	 */
	void GccWarning(CFEBase * pFEObject, const char *sMsg, ...)
	{
		va_list args;
		va_start(args, sMsg);
		GccWarningVL(pFEObject, sMsg, args);
		va_end(args);
	}

	/** \brief print warning messages
	 *  \param pFEObject the object where the warning occured
	 *  \param sMsg the warning message
	 *  \param vl teh variable argument list
	 */
	void GccWarningVL(CFEBase * pFEObject, const char *sMsg, va_list vl)
	{
		// increment warnings counter
		CCompiler::Warning();
		if (pFEObject)
		{
			// iterate through include hierarchy
			CFEFile *pCur = (CFEFile*)pFEObject->GetSpecificParent<CFEFile>(0);
			vector<CFEFile*> *pStack = new vector<CFEFile*>();
			while (pCur)
			{
				pStack->insert(pStack->begin(), pCur);
				// start with parent of current, because GetFile starts with "this"
				pCur = pCur->GetSpecificParent<CFEFile>();
			}
			// need at least one file
			if (!pStack->empty())
			{
				// walk down
				if (pStack->size() > 1)
					std::cerr << "In file included ";
				vector<CFEFile*>::iterator iter;
				for (iter = pStack->begin();
					(iter != pStack->end()) && (iter != pStack->end()-1); iter++)
				{
					std::cerr << "from " << (*iter)->GetFileName() << ":1";
					if (iter+2 != pStack->end())
						std::cerr << ",\n                 ";
					else
						std::cerr << ":\n";
				}
				std::cerr << pFEObject->m_sourceLoc;
				std::cerr << ": warning: ";
			}
			// cleanup
			delete pStack;
		}
		vfprintf(stderr, sMsg, vl);
	}

};
