/* $Id:$ */
/*******************************************************************************/
/*
 * \file   pager.c
 * \brief  L4Linux page-fault handling
 *
 * \date   2007/06/29
 * \author Sebastian Sumpf <sumpf@os.inf.tu-dresden.de>
 */
/*******************************************************************************/
/* (c) 2007 Technische Universitaet Dresden
 * This file is part of DROPS, which is distributed under the terms of the
 * GNU General Public License 2. Please see the COPYING file for details.
 */
#include <l4/l4rm/l4rm.h>
#include <l4/log/l4log.h>
#include <l4/sys/syscalls.h>
#include <l4/rmgr/librmgr.h>
#include <l4/thread/thread.h>
#include <l4/util/macros.h>

#include "__types.h"
#include "__lxcalls.h"

static l4_addr_t bios_map;
static l4_threadid_t pager_id;

static int __find_ds(l4_taskid_t task_id, l4_addr_t fault_addr, 
                     unsigned long * ds_id, unsigned long * offset)
{
  int pos = 0;
	fault_addr &= L4_PAGEMASK;
	lx_task_t * task;
	lx_page_fault_t *pages; 

	if(lx_task_get(task_id, &task) < 0)
		return -L4_ENOTFOUND;

	pages = task->fault;
	
	while(pos < LX_MAX_DS && 
			(fault_addr < pages[pos].virt_addr ||
			 fault_addr >= pages[pos].virt_addr + pages[pos].virt_size)
			&& !lx_is_invalid_ds_id(pages[pos].ds_id))
		pos++;

	if(pos >= LX_MAX_DS || lx_is_invalid_ds_id(pages[pos].ds_id))
		return -L4_ENOTFOUND;

	*ds_id = pages[pos].ds_id;
	*offset =	(fault_addr - pages[pos].virt_addr) & L4_PAGEMASK;
	return 0;
}

static void __signal_doomed(l4_threadid_t src_id)
{
	l4_msgdope_t dummy;
	l4_ipc_send(src_id, L4_IPC_SHORT_MSG, 0, 0, L4_IPC_SEND_TIMEOUT_0, &dummy);

	LOG_printf("Going to sleep ...\n");
	l4_sleep_forever();
}


static void __pager_loop(void * data)
{
	l4_threadid_t src_id;
	l4_msgdope_t result;
	l4_umword_t dw1, dw2, dummy;
	int error;
	l4_addr_t map_addr;
	unsigned long ds_id = 0;
	unsigned long offset = 0;
	pager_id = l4_myself();
	int rw;
	
	while(1) {
		
		error = l4_ipc_wait(&src_id, L4_IPC_SHORT_MSG, &dw1, &dw2, 
				L4_IPC_NEVER, &result);

		while(!error) 
		{
			rw = dw1 & 2;	
			map_addr = 0x0;	
			
			if(DBG_FAULT_RO || rw)
				LOGd(DBG_FAULT, "Received pagefault at: 0x%08lx (%s)\n", dw1, (rw)?"rw":"ro");
				
			/* video / bios forward to rmgr */
			if(dw1 >= 0x0009f000 && dw1 <= 0x000fffff) 
			{
				LOGd(DBG_SUSP, "Bios pagefault\n");
				l4_fpage_unmap(l4_fpage(bios_map, L4_LOG2_PAGESIZE, L4_FPAGE_RW, 
				               L4_FPAGE_MAP), L4_FP_FLUSH_PAGE | L4_FP_ALL_SPACES);
				
				for(;;)
				{
					error = l4_ipc_call(rmgr_pager_id,
					                    L4_IPC_SHORT_MSG, dw1 | 2, 0,
					                    L4_IPC_MAPMSG(bios_map, L4_LOG2_PAGESIZE),
					                    &dummy, &dummy,
					                    L4_IPC_NEVER, &result);

					if(error != L4_IPC_SECANCELED && error != L4_IPC_SEABORTED)
						break;
				}
				if(error || !l4_ipc_fpage_received(result))
				{
					LOGd(DBG_ERROR, "Can't handle pagefault at %08lx eip: %08lx",
							dw1, dw2);
					Panic("Bios");
				}
				map_addr = bios_map;
				
				dw2 = l4_fpage(bios_map, L4_LOG2_PAGESIZE,
				               L4_FPAGE_RW, L4_FPAGE_GRANT).fpage;
				dw1 &= L4_PAGESIZE;
				rw = 1;		
			}	
			else 
			{
			
				if((error = __find_ds(src_id, dw1, &ds_id, &offset))) {
					LOGd(DBG_ERROR, "Unhandeled %s pagefault (not found) at: 0x%08lx ("
					     l4util_idfmt") EIP: 0x%08lx",
					     rw ?"rw":"ro",
					     (dw1 & ~3), l4util_idstr(src_id), dw2);
					__signal_doomed(src_id);
				}
				LOGd(DBG_FAULT, "ds_id %lu offset: %08lx\n", ds_id, offset);
				
				if((error = lx_page_fault(src_id, ds_id, offset, L4_PAGESIZE, rw, 
				                          &map_addr))) 
				{
					LOGd(DBG_ERROR, "Unhandeled pagefault at: 0x%08lx ("l4util_idfmt
					     ") EIP: 0x%08lx ds_id: %lu offset: %08lx\n",
					     (dw1 & ~3), l4util_idstr(src_id), dw2, ds_id, offset);
					__signal_doomed(src_id);
				}
				map_addr &= L4_PAGEMASK;
			
				dw2 = l4_fpage(map_addr, L4_LOG2_PAGESIZE, 
				               rw ? L4_FPAGE_RW : L4_FPAGE_RO, L4_FPAGE_MAP).fpage;
				dw1 &= L4_PAGEMASK;
			}
			
			if(DBG_FAULT_RO || rw)
				LOGd(DBG_FAULT, "Mapping fpage from %08lx to %08lx(%s)\n", 
				     map_addr, dw1, (rw)?"rw":"ro");
			
			error = l4_ipc_reply_and_wait(src_id, L4_IPC_SHORT_FPAGE, dw1, dw2,
			                              &src_id, L4_IPC_SHORT_MSG, &dw1, &dw2, 
			                              L4_IPC_SEND_TIMEOUT_0, &result);

			if(error == L4_IPC_SETIMEOUT && (dw2 & 1)) 
				l4_fpage_unmap((l4_fpage_t)dw2, L4_FP_FLUSH_PAGE | L4_FP_ALL_SPACES);
		}
	}
}

int lx_pager_init(void)
{
	int error;
	l4_uint32_t area_id;
	
	if((error = l4rm_area_reserve(L4_PAGESIZE, 0, &bios_map, &area_id)))
	{
		LOGd(DBG_ERROR, "Error: reserving bios page: %s (%d)", 
		     l4env_errstr(error), error);
		return error;
	}
	
	if((error = l4thread_create_named(__pager_loop, ".pager", NULL, L4THREAD_CREATE_ASYNC)) < 0)
	{
		LOGd(DBG_ERROR, "Error: starting pager thread: %s (%d)", 
		     l4env_errstr(error), error);
		return error;
	}

	return 0;
}

l4_threadid_t lx_pager_get_id(void)
{
	return pager_id;
}

int lx_pager_add_ds(lx_task_t *task, lx_page_fault_t ds) {
	int pos = 0;

	while(pos < LX_MAX_DS && 
	      !lx_is_invalid_ds_id(task->fault[pos].ds_id))
		pos++;

	if(pos >= LX_MAX_DS) {
		LOGd(DBG_ERROR, "Pagefault dataspace limit exceeded\n");
		return -L4_ERES;
	}

	LOGd(DBG_SUSP, "Adding paging region \n\tAddr: %08lx\n\t"
       "Size: %08x\n\tds_id: %lu\n", ds.virt_addr, ds.virt_size, ds.ds_id);
	
	task->fault[pos] = ds;
	return pos;
}

void lx_pager_del_ds(lx_task_t *task, unsigned long ds_id)
{
	int pos = 0;

	while(pos < LX_MAX_DS && task->fault[pos].ds_id != ds_id &&
	      !lx_is_invalid_ds_id(task->fault[pos].ds_id))
		pos++;

	if(pos >= LX_MAX_DS || lx_is_invalid_ds_id(task->fault[pos].ds_id))
		return;
	
	lx_page_fault_t f;
	f = task->fault[pos];

	memmove((void*)&(task->fault[pos]), (void*)&(task->fault[pos+1]), 
	        (LX_MAX_DS - pos - 1) * sizeof(lx_page_fault_t));

	task->fault[LX_MAX_DS - 1] = LX_INVALID_FAULT;
}

void lx_pager_chk_release(lx_task_t *task)
{
	int pos = 0;
	
	while(pos < LX_MAX_DS && 
	      !lx_is_invalid_ds_id(task->fault[pos].ds_id))
	{
		if(task->fault[pos].flags & LX_RELEASE)
		{
			lxif_close_ds(task, task->fault[pos].ds_id);
			lx_pager_del_ds(task, task->fault[pos].ds_id);
		}
		pos++;
	}
}


