/* $Id: __lxcalls.h,v 1.3 2007/01/25 15:13:06 cbass Exp $ */
/*****************************************************************************/
/**
 *  \file __lxcalls.h
 *  \brief global calls 
 * 
 *  \date 12/20/06 12:58:08 CET
 *  \author Sebastian Sumpf <Sebastian.Sumpf@inf.tu-dresden.de>
 */ 
/******************************************************************************/
#ifndef __LXMIR_SERVER_LXCALLS_H__
#define __LXMIR_SERVER_LXCALLS_H__

#include <l4/slab/slab.h>
#include "__types.h"


/******************************************************************************* 
 *** pager.c
 *******************************************************************************/
int lx_pager_init(void);

l4_threadid_t lx_pager_get_id(void);

/* adds a virtual address range and associated dataspace to pager */
int lx_pager_add_ds(lx_task_t * task, lx_page_fault_t ds);

/* delete a dataspace from pager */
void lx_pager_del_ds(lx_task_t * task, unsigned long ds_id);

/* Release all pager dataspaces marked LX_RELEASE */
void lx_pager_chk_release(lx_task_t * task);


/******************************************************************************* 
 *** slab.c
 *******************************************************************************/

/* simple slab wrapper functions */
int lx_slab_setup(l4slab_cache_t * cache, l4_size_t obj_size);

void * lx_slab_alloc(l4slab_cache_t * cache);

void lx_slab_free(l4slab_cache_t * cache, void * obj);


/******************************************************************************* 
 *** cow_memr.c
 *******************************************************************************/
/* get free page from COW memory */
void * lx_cow_get_free(void);

/* release page */
void lx_cow_set_free(l4_addr_t addr);

/* check if a page was moved during compactification */
void * lx_cow_chk_moved(l4_addr_t addr);

/* compacitfy cow mem */
void lx_cow_compactify(void);


/******************************************************************************* 
 *** pages.c
 *******************************************************************************/
int lx_task_init(void);

/* add a lx_task_t */
int lx_task_add_pos(l4_taskid_t task_id, int pos);
L4_INLINE int lx_task_add(l4_taskid_t task_id);	

/* get a task */
int lx_task_do_get(l4_taskid_t task_id, int pos, lx_task_t ** task);
L4_INLINE int lx_task_get_pos(int pos, lx_task_t ** task);
L4_INLINE int lx_task_get(l4_taskid_t task_id, lx_task_t ** task);

/* get loading task, if there is one, NULL otherwise */
lx_task_t * lx_task_get_loading(void);

/* create the loading task */
int lx_task_create_loading(void);

/* retrieve IDs of every task paged by the freezer */
void lx_task_get_ids(l4_taskid_t * tasks, l4_uint32_t * count);

/* delete a task */
void lx_task_del(l4_taskid_t task);

/* remap  memory of a task read-only */
void lx_task_remap_ro(lx_task_t *task);


/* add a dataspace to task */
int lx_ds_add_ds(lx_task_t * task, l4dm_dataspace_t ds, 
                 dsmlib_ds_desc_t * ds_desc, l4_size_t size, void * addr);

/* search a dataspace with contents equal to data */
int lx_ds_search_equal(lx_task_t * task, void * data, l4_size_t size, 
                       unsigned long * ds_id);

/* get a dataspace */
int lx_ds_do_get(lx_task_t * task, int pos, unsigned long ds_id,  lx_ds_t ** ds);
L4_INLINE int lx_ds_get(lx_task_t * task, unsigned long ds_id,  lx_ds_t ** ds);
L4_INLINE int lx_ds_get_pos(lx_task_t * task, int pos,  lx_ds_t ** ds);

/* delete a dataspace */
void lx_ds_del(lx_task_t * task, unsigned long ds_id);

/* get ds_info (lxfreeze.h) of all dataspaces of task */
void lx_ds_get_ids(lx_task_t * task, lx_ds_info_t * ds_info, l4_uint32_t * count);

/* get id of dataspace */
unsigned long lx_ds_get_id(lx_ds_t * lxds);

/* resolve a page fault */
int lx_page_fault(l4_threadid_t src, 
                  unsigned long ds_id, 
                  unsigned long offset,
                  unsigned long size,
                  int rw,
                  l4_addr_t * map_addr);


/******************************************************************************* 
 *** event.c
 *******************************************************************************/
int lx_event_init(void);

/* if_lxfreeze.c */
/******************************************************************************* 
 *** pager.c
 *******************************************************************************/

/* non-IPC versionis of close,exit, and open calls */
long lxif_close_ds(lx_task_t * task, unsigned long ds_id);

void lxif_exit_task(l4_taskid_t);
long lxif_open(l4_taskid_t task_id,
               unsigned long size,
               unsigned long align,
               unsigned long flags,
               const char * name,
               lx_ds_t ** lxds);


/******************************************************************************* 
 *** debugging
 *******************************************************************************/
#if DS_CHECKSUM
void dbg_lx_ds_hash(lx_task_t * task);
#else
static inline void dbg_lx_ds_hash(lx_task_t * task){}
#endif


/*******************************************************************************
 * implementations
 ******************************************************************************/ 
L4_INLINE int lx_task_add(l4_taskid_t task_id)	
{
 return lx_task_add_pos(task_id, -1);
}

L4_INLINE int lx_task_get_pos(int pos, lx_task_t ** task)
{
	return lx_task_do_get(L4_INVALID_ID, pos, task);
}

L4_INLINE int lx_task_get(l4_taskid_t task_id, lx_task_t ** task)
{
	return lx_task_do_get(task_id, -1, task);
}

L4_INLINE int lx_ds_get(lx_task_t * task, unsigned long ds_id,  lx_ds_t ** ds)
{
	return lx_ds_do_get(task, -1, ds_id, ds);
}

L4_INLINE int lx_ds_get_pos(lx_task_t * task, int pos,  lx_ds_t ** ds)
{
	return lx_ds_do_get(task, pos, 0, ds);
}

#endif/* __LXMIR_SERVER_LXCALLS_H__ */
