/*
 * \brief   Test cases for libcrypto
 * \date    2006-07-04
 * \author  Carsten Weinhold <weinhold@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2006 Carsten Weinhold <weinhold@os.inf.tu-dresden.de>
 * Technische Universitaet Dresden, Operating Systems Research Group
 *
 * This file is part of the libcrypto package, which is distributed under
 * the  terms  of the  GNU General Public Licence 2.  Please see the
 * COPYING file for details.
 */

#include <stdlib.h>
#include <string.h>

#include <l4/log/l4log.h>
#include <l4/util/reboot.h>

#include <l4/crypto/oaep.h>
#include <l4/crypto/random.h>
#include <l4/crypto/sha1.h>
#include <l4/crypto/aes.h>
#include <l4/crypto/cbc.h>
#include <l4/crypto/rsaref2/global.h>
#include <l4/crypto/rsaref2/rsaref.h>
#include <l4/crypto/rsaref2/rsa.h>

/*
 * ***************************************************************************
 */

#if 0
#define VERBOSE_TEST 1
#endif

/*
 * ***************************************************************************
 */

#ifdef VERBOSE_TEST
static void print_hex(const char *bytes, int len) {

    int i;

    for (i = 0; i < len; i++) {
        unsigned int n = (unsigned char) bytes[i];
        LOG_printf("%02x ", n);
        if (i % 16 == 15)
            LOG_printf("\n");
    }
    LOG_printf("\n");
}
#endif

/*
 * ***************************************************************************
 */

int crypto_randomize_buf(char *buf, unsigned int buf_len) {

    static unsigned int idx = 0;
    const char seed[2][SHA1_DIGEST_SIZE] =
        { { 0xaa, 0xfd, 0x12, 0xf6, 0x59, 0xca, 0xe6, 0x34,
            0x89, 0xb4, 0x79, 0xe5, 0x07, 0x6d, 0xde, 0xc2,
            0xf0, 0x6c, 0xb5, 0x8f },
          { 0x47, 0xe1, 0xab, 0x71, 0x19, 0xfe, 0xe5, 0x6c,
            0x95, 0xee, 0x5e, 0xaa, 0xd8, 0x6f, 0x40, 0xd0, 
            0xaa, 0x63, 0xbd, 0x33 } };

    if (idx >= 2 || buf_len != SHA1_DIGEST_SIZE) {
        LOG_printf("unexpected call of crypto_randomize_buf()!");
        return -1;
    }
    
    memcpy(buf, &seed[idx][0], buf_len);
    idx++;
    
    return buf_len;
}

/*
 * ***************************************************************************
 */

static int test_mgf1(void) {

    char mask[107];
    const char mask_ref[] =
        { 0x06, 0xe1, 0xde, 0xb2, 0x36, 0x9a, 0xa5, 0xa5,
          0xc7, 0x07, 0xd8, 0x2c, 0x8e, 0x4e, 0x93, 0x24, 
          0x8a, 0xc7, 0x83, 0xde, 0xe0, 0xb2, 0xc0, 0x46,
          0x26, 0xf5, 0xaf, 0xf9, 0x3e, 0xdc, 0xfb, 0x25, 
          0xc9, 0xc2, 0xb3, 0xff, 0x8a, 0xe1, 0x0e, 0x83,
          0x9a, 0x2d, 0xdb, 0x4c, 0xdc, 0xfe, 0x4f, 0xf4, 
          0x77, 0x28, 0xb4, 0xa1, 0xb7, 0xc1, 0x36, 0x2b,
          0xaa, 0xd2, 0x9a, 0xb4, 0x8d, 0x28, 0x69, 0xd5, 
          0x02, 0x41, 0x21, 0x43, 0x58, 0x11, 0x59, 0x1b,
          0xe3, 0x92, 0xf9, 0x82, 0xfb, 0x3e, 0x87, 0xd0, 
          0x95, 0xae, 0xb4, 0x04, 0x48, 0xdb, 0x97, 0x2f,
          0x3a, 0xc1, 0x4e, 0xaf, 0xf4, 0x9c, 0x8c, 0x3b, 
          0x7c, 0xfc, 0x95, 0x1a, 0x51, 0xec, 0xd1, 0xdd,
          0xe6, 0x12, 0x64 };
    const char seed[] =
        { 0xaa, 0xfd, 0x12, 0xf6, 0x59, 0xca, 0xe6, 0x34,
          0x89, 0xb4, 0x79, 0xe5, 0x07, 0x6d, 0xde, 0xc2,
          0xf0, 0x6c, 0xb5, 0x8f };

    memset(mask, 0, 107);

    mgf1(seed, 20, mask, 107);

#ifdef VERBOSE_TEST
    print_hex(mask, 107);
    print_hex(mask_ref, 107);
#endif

    return memcmp(mask_ref, mask, 107);
}


static int test_pad_oaep(void) {

    const char msg[] =
        { 0xd4, 0x36, 0xe9, 0x95, 0x69, 0xfd, 0x32, 0xa7,
          0xc8, 0xa0, 0x5b, 0xbc, 0x90, 0xd3, 0x2c, 0x49 };
    const char padded_msg_ref[] =
        { 0x00, 0xeb, 0x7a, 0x19, 0xac, 0xe9, 0xe3, 0x00,
          0x63, 0x50, 0xe3, 0x29, 0x50, 0x4b, 0x45, 0xe2, 
          0xca, 0x82, 0x31, 0x0b, 0x26, 0xdc, 0xd8, 0x7d,
          0x5c, 0x68, 0xf1, 0xee, 0xa8, 0xf5, 0x52, 0x67, 
          0xc3, 0x1b, 0x2e, 0x8b, 0xb4, 0x25, 0x1f, 0x84,
          0xd7, 0xe0, 0xb2, 0xc0, 0x46, 0x26, 0xf5, 0xaf, 
          0xf9, 0x3e, 0xdc, 0xfb, 0x25, 0xc9, 0xc2, 0xb3,
          0xff, 0x8a, 0xe1, 0x0e, 0x83, 0x9a, 0x2d, 0xdb, 
          0x4c, 0xdc, 0xfe, 0x4f, 0xf4, 0x77, 0x28, 0xb4,
          0xa1, 0xb7, 0xc1, 0x36, 0x2b, 0xaa, 0xd2, 0x9a, 
          0xb4, 0x8d, 0x28, 0x69, 0xd5, 0x02, 0x41, 0x21,
          0x43, 0x58, 0x11, 0x59, 0x1b, 0xe3, 0x92, 0xf9, 
          0x82, 0xfb, 0x3e, 0x87, 0xd0, 0x95, 0xae, 0xb4,
          0x04, 0x48, 0xdb, 0x97, 0x2f, 0x3a, 0xc1, 0x4f, 
          0x7b, 0xc2, 0x75, 0x19, 0x52, 0x81, 0xce, 0x32,
          0xd2, 0xf1, 0xb7, 0x6d, 0x4d, 0x35, 0x3e, 0x2d };
    char padded_msg[128];

    pad_oaep(msg, 16, NULL, 0, 128, padded_msg);

    return memcmp(padded_msg, padded_msg_ref, 128);
}


static int test_rsa1024(void) {

    int ret;
    R_RSA_PUBLIC_KEY pubkey;
    const char modulus[] =
        { 0xbb, 0xf8, 0x2f, 0x09, 0x06, 0x82, 0xce, 0x9c,
          0x23, 0x38, 0xac, 0x2b, 0x9d, 0xa8, 0x71, 0xf7, 
          0x36, 0x8d, 0x07, 0xee, 0xd4, 0x10, 0x43, 0xa4,
          0x40, 0xd6, 0xb6, 0xf0, 0x74, 0x54, 0xf5, 0x1f, 
          0xb8, 0xdf, 0xba, 0xaf, 0x03, 0x5c, 0x02, 0xab,
          0x61, 0xea, 0x48, 0xce, 0xeb, 0x6f, 0xcd, 0x48, 
          0x76, 0xed, 0x52, 0x0d, 0x60, 0xe1, 0xec, 0x46,
          0x19, 0x71, 0x9d, 0x8a, 0x5b, 0x8b, 0x80, 0x7f, 
          0xaf, 0xb8, 0xe0, 0xa3, 0xdf, 0xc7, 0x37, 0x72,
          0x3e, 0xe6, 0xb4, 0xb7, 0xd9, 0x3a, 0x25, 0x84, 
          0xee, 0x6a, 0x64, 0x9d, 0x06, 0x09, 0x53, 0x74,
          0x88, 0x34, 0xb2, 0x45, 0x45, 0x98, 0x39, 0x4e, 
          0xe0, 0xaa, 0xb1, 0x2d, 0x7b, 0x61, 0xa5, 0x1f,
          0x52, 0x7a, 0x9a, 0x41, 0xf6, 0xc1, 0x68, 0x7f, 
          0xe2, 0x53, 0x72, 0x98, 0xca, 0x2a, 0x8f, 0x59,
          0x46, 0xf8, 0xe5, 0xfd, 0x09, 0x1d, 0xbd, 0xcb };
    const char exp[] =
        { 0x0, 0x0, 0x11 };
    const char msg[] =
        { 0x00, 0xeb, 0x7a, 0x19, 0xac, 0xe9, 0xe3, 0x00,
          0x63, 0x50, 0xe3, 0x29, 0x50, 0x4b, 0x45, 0xe2, 
          0xca, 0x82, 0x31, 0x0b, 0x26, 0xdc, 0xd8, 0x7d,
          0x5c, 0x68, 0xf1, 0xee, 0xa8, 0xf5, 0x52, 0x67, 
          0xc3, 0x1b, 0x2e, 0x8b, 0xb4, 0x25, 0x1f, 0x84,
          0xd7, 0xe0, 0xb2, 0xc0, 0x46, 0x26, 0xf5, 0xaf, 
          0xf9, 0x3e, 0xdc, 0xfb, 0x25, 0xc9, 0xc2, 0xb3,
          0xff, 0x8a, 0xe1, 0x0e, 0x83, 0x9a, 0x2d, 0xdb, 
          0x4c, 0xdc, 0xfe, 0x4f, 0xf4, 0x77, 0x28, 0xb4,
          0xa1, 0xb7, 0xc1, 0x36, 0x2b, 0xaa, 0xd2, 0x9a, 
          0xb4, 0x8d, 0x28, 0x69, 0xd5, 0x02, 0x41, 0x21,
          0x43, 0x58, 0x11, 0x59, 0x1b, 0xe3, 0x92, 0xf9, 
          0x82, 0xfb, 0x3e, 0x87, 0xd0, 0x95, 0xae, 0xb4,
          0x04, 0x48, 0xdb, 0x97, 0x2f, 0x3a, 0xc1, 0x4f, 
          0x7b, 0xc2, 0x75, 0x19, 0x52, 0x81, 0xce, 0x32,
          0xd2, 0xf1, 0xb7, 0x6d, 0x4d, 0x35, 0x3e, 0x2d };
    const char enc_msg_ref[] =
      { 0x12, 0x53, 0xe0, 0x4d, 0xc0, 0xa5, 0x39, 0x7b, 0xb4, 0x4a, 0x7a, 0xb8, 0x7e, 0x9b, 0xf2, 0xa0, 
        0x39, 0xa3, 0x3d, 0x1e, 0x99, 0x6f, 0xc8, 0x2a, 0x94, 0xcc, 0xd3, 0x00, 0x74, 0xc9, 0x5d, 0xf7, 
        0x63, 0x72, 0x20, 0x17, 0x06, 0x9e, 0x52, 0x68, 0xda, 0x5d, 0x1c, 0x0b, 0x4f, 0x87, 0x2c, 0xf6, 
        0x53, 0xc1, 0x1d, 0xf8, 0x23, 0x14, 0xa6, 0x79, 0x68, 0xdf, 0xea, 0xe2, 0x8d, 0xef, 0x04, 0xbb, 
        0x6d, 0x84, 0xb1, 0xc3, 0x1d, 0x65, 0x4a, 0x19, 0x70, 0xe5, 0x78, 0x3b, 0xd6, 0xeb, 0x96, 0xa0, 
        0x24, 0xc2, 0xca, 0x2f, 0x4a, 0x90, 0xfe, 0x9f, 0x2e, 0xf5, 0xc9, 0xc1, 0x40, 0xe5, 0xbb, 0x48, 
        0xda, 0x95, 0x36, 0xad, 0x87, 0x00, 0xc8, 0x4f, 0xc9, 0x13, 0x0a, 0xde, 0xa7, 0x4e, 0x55, 0x8d, 
        0x51, 0xa7, 0x4d, 0xdf, 0x85, 0xd8, 0xb5, 0x0d, 0xe9, 0x68, 0x38, 0xd6, 0x06, 0x3e, 0x09, 0x55 };
    char enc_msg[128];
    unsigned int enc_msg_len;

    pubkey.bits = 1024;
    memset(pubkey.modulus, 0, 128 + 128);
    memset(pubkey.exponent, 0, 128 + 128);
    memcpy(pubkey.modulus + 128, modulus, 128);
    memcpy(pubkey.exponent + 128 + 125, exp, 3);

    ret = RSAPublicEncryptRaw((unsigned char *) enc_msg, &enc_msg_len,
                              (unsigned char *) msg, 128, &pubkey);

#if VERBOSE_TEST
    LOG_printf("RSAPublicEncryptRaw(): %d\n", ret);
    print_hex(enc_msg, 128);
    print_hex(enc_msg_ref, 128);
#endif

    return memcmp(enc_msg, enc_msg_ref, sizeof(enc_msg));
}


static int test_rsa2048(void) {

    const char modulus[] =
      { 0xae, 0x45, 0xed, 0x56, 0x01, 0xce, 0xc6, 0xb8, 0xcc, 0x05, 0xf8, 0x03, 0x93, 0x5c, 0x67, 0x4d, 
        0xdb, 0xe0, 0xd7, 0x5c, 0x4c, 0x09, 0xfd, 0x79, 0x51, 0xfc, 0x6b, 0x0c, 0xae, 0xc3, 0x13, 0xa8, 
        0xdf, 0x39, 0x97, 0x0c, 0x51, 0x8b, 0xff, 0xba, 0x5e, 0xd6, 0x8f, 0x3f, 0x0d, 0x7f, 0x22, 0xa4, 
        0x02, 0x9d, 0x41, 0x3f, 0x1a, 0xe0, 0x7e, 0x4e, 0xbe, 0x9e, 0x41, 0x77, 0xce, 0x23, 0xe7, 0xf5, 
        0x40, 0x4b, 0x56, 0x9e, 0x4e, 0xe1, 0xbd, 0xcf, 0x3c, 0x1f, 0xb0, 0x3e, 0xf1, 0x13, 0x80, 0x2d, 
        0x4f, 0x85, 0x5e, 0xb9, 0xb5, 0x13, 0x4b, 0x5a, 0x7c, 0x80, 0x85, 0xad, 0xca, 0xe6, 0xfa, 0x2f, 
        0xa1, 0x41, 0x7e, 0xc3, 0x76, 0x3b, 0xe1, 0x71, 0xb0, 0xc6, 0x2b, 0x76, 0x0e, 0xde, 0x23, 0xc1, 
        0x2a, 0xd9, 0x2b, 0x98, 0x08, 0x84, 0xc6, 0x41, 0xf5, 0xa8, 0xfa, 0xc2, 0x6b, 0xda, 0xd4, 0xa0, 
        0x33, 0x81, 0xa2, 0x2f, 0xe1, 0xb7, 0x54, 0x88, 0x50, 0x94, 0xc8, 0x25, 0x06, 0xd4, 0x01, 0x9a, 
        0x53, 0x5a, 0x28, 0x6a, 0xfe, 0xb2, 0x71, 0xbb, 0x9b, 0xa5, 0x92, 0xde, 0x18, 0xdc, 0xf6, 0x00, 
        0xc2, 0xae, 0xea, 0xe5, 0x6e, 0x02, 0xf7, 0xcf, 0x79, 0xfc, 0x14, 0xcf, 0x3b, 0xdc, 0x7c, 0xd8, 
        0x4f, 0xeb, 0xbb, 0xf9, 0x50, 0xca, 0x90, 0x30, 0x4b, 0x22, 0x19, 0xa7, 0xaa, 0x06, 0x3a, 0xef, 
        0xa2, 0xc3, 0xc1, 0x98, 0x0e, 0x56, 0x0c, 0xd6, 0x4a, 0xfe, 0x77, 0x95, 0x85, 0xb6, 0x10, 0x76, 
        0x57, 0xb9, 0x57, 0x85, 0x7e, 0xfd, 0xe6, 0x01, 0x09, 0x88, 0xab, 0x7d, 0xe4, 0x17, 0xfc, 0x88, 
        0xd8, 0xf3, 0x84, 0xc4, 0xe6, 0xe7, 0x2c, 0x3f, 0x94, 0x3e, 0x0c, 0x31, 0xc0, 0xc4, 0xa5, 0xcc, 
        0x36, 0xf8, 0x79, 0xd8, 0xa3, 0xac, 0x9d, 0x7d, 0x59, 0x86, 0x0e, 0xaa, 0xda, 0x6b, 0x83, 0xbb };
    const char exp[] =
      { 0x01, 0x00, 0x01 };
    const char msg[] =
      { 0x8b, 0xba, 0x6b, 0xf8, 0x2a, 0x6c, 0x0f, 0x86, 0xd5, 0xf1, 0x75, 0x6e, 0x97, 0x95, 0x68, 0x70, 
        0xb0, 0x89, 0x53, 0xb0, 0x6b, 0x4e, 0xb2, 0x05, 0xbc, 0x16, 0x94, 0xee };
    const char enc_msg_ref[] =
       { 0x53, 0xea, 0x5d, 0xc0, 0x8c, 0xd2, 0x60, 0xfb, 0x3b, 0x85, 0x85, 0x67, 0x28, 0x7f, 0xa9, 0x15, 
         0x52, 0xc3, 0x0b, 0x2f, 0xeb, 0xfb, 0xa2, 0x13, 0xf0, 0xae, 0x87, 0x70, 0x2d, 0x06, 0x8d, 0x19, 
         0xba, 0xb0, 0x7f, 0xe5, 0x74, 0x52, 0x3d, 0xfb, 0x42, 0x13, 0x9d, 0x68, 0xc3, 0xc5, 0xaf, 0xee, 
         0xe0, 0xbf, 0xe4, 0xcb, 0x79, 0x69, 0xcb, 0xf3, 0x82, 0xb8, 0x04, 0xd6, 0xe6, 0x13, 0x96, 0x14, 
         0x4e, 0x2d, 0x0e, 0x60, 0x74, 0x1f, 0x89, 0x93, 0xc3, 0x01, 0x4b, 0x58, 0xb9, 0xb1, 0x95, 0x7a, 
         0x8b, 0xab, 0xcd, 0x23, 0xaf, 0x85, 0x4f, 0x4c, 0x35, 0x6f, 0xb1, 0x66, 0x2a, 0xa7, 0x2b, 0xfc, 
         0xc7, 0xe5, 0x86, 0x55, 0x9d, 0xc4, 0x28, 0x0d, 0x16, 0x0c, 0x12, 0x67, 0x85, 0xa7, 0x23, 0xeb, 
         0xee, 0xbe, 0xff, 0x71, 0xf1, 0x15, 0x94, 0x44, 0x0a, 0xae, 0xf8, 0x7d, 0x10, 0x79, 0x3a, 0x87, 
         0x74, 0xa2, 0x39, 0xd4, 0xa0, 0x4c, 0x87, 0xfe, 0x14, 0x67, 0xb9, 0xda, 0xf8, 0x52, 0x08, 0xec, 
         0x6c, 0x72, 0x55, 0x79, 0x4a, 0x96, 0xcc, 0x29, 0x14, 0x2f, 0x9a, 0x8b, 0xd4, 0x18, 0xe3, 0xc1, 
         0xfd, 0x67, 0x34, 0x4b, 0x0c, 0xd0, 0x82, 0x9d, 0xf3, 0xb2, 0xbe, 0xc6, 0x02, 0x53, 0x19, 0x62, 
         0x93, 0xc6, 0xb3, 0x4d, 0x3f, 0x75, 0xd3, 0x2f, 0x21, 0x3d, 0xd4, 0x5c, 0x62, 0x73, 0xd5, 0x05, 
         0xad, 0xf4, 0xcc, 0xed, 0x10, 0x57, 0xcb, 0x75, 0x8f, 0xc2, 0x6a, 0xee, 0xfa, 0x44, 0x12, 0x55, 
         0xed, 0x4e, 0x64, 0xc1, 0x99, 0xee, 0x07, 0x5e, 0x7f, 0x16, 0x64, 0x61, 0x82, 0xfd, 0xb4, 0x64, 
         0x73, 0x9b, 0x68, 0xab, 0x5d, 0xaf, 0xf0, 0xe6, 0x3e, 0x95, 0x52, 0x01, 0x68, 0x24, 0xf0, 0x54, 
         0xbf, 0x4d, 0x3c, 0x8c, 0x90, 0xa9, 0x7b, 0xb6, 0xb6, 0x55, 0x32, 0x84, 0xeb, 0x42, 0x9f, 0xcc };
    char padded_msg[256];
    char enc_msg[256];
    unsigned int enc_msg_len;
    R_RSA_PUBLIC_KEY pubkey;
    int ret;

    pubkey.bits = 2048;
    memset(pubkey.modulus, 0, 256);
    memset(pubkey.exponent, 0, 256);
    memcpy(pubkey.modulus, modulus, 256);
    memcpy(pubkey.exponent + 253, exp, 3);

    ret = pad_oaep(msg, sizeof(msg), NULL, 0, 256, padded_msg);
#if VERBOSE_TEST
    print_hex(padded_msg, 256);
#endif
    
    ret = RSAPublicEncryptRaw((unsigned char *) enc_msg, &enc_msg_len,
                              (unsigned char *) padded_msg, 256, &pubkey);
#if VERBOSE_TEST
    LOG_printf("enc_msg_len=%d\n", enc_msg_len);
    print_hex(enc_msg, 256);
    print_hex(enc_msg_ref, 256);
#endif

    return memcmp(enc_msg, enc_msg_ref, sizeof(enc_msg));
}


static int test_aes128_cbc(void) {

    /* RFC 3602: Test vectors, case #4:
     * Encrypting 64 bytes (4 blocks) using AES-CBC with 128-bit key
     */
    
    const char key[] =
        { 0x56, 0xe4, 0x7a, 0x38, 0xc5, 0x59, 0x89, 0x74,
          0xbc, 0x46, 0x90, 0x3d, 0xba, 0x29, 0x03, 0x49 };
    const char iv[] =
        { 0x8c, 0xe8, 0x2e, 0xef, 0xbe, 0xa0, 0xda, 0x3c,
          0x44, 0x69, 0x9e, 0xd7, 0xdb, 0x51, 0xb7, 0xd9 };
    const char msg_ref[] =
        { 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
          0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
          0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
          0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
          0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
          0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
          0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
          0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf };
    const char msg_enc_ref[] =
        { 0xc3, 0x0e, 0x32, 0xff, 0xed, 0xc0, 0x77, 0x4e,
          0x6a, 0xff, 0x6a, 0xf0, 0x86, 0x9f, 0x71, 0xaa,
          0x0f, 0x3a, 0xf0, 0x7a, 0x9a, 0x31, 0xa9, 0xc6,
          0x84, 0xdb, 0x20, 0x7e, 0xb0, 0xef, 0x8e, 0x4e,
          0x35, 0x90, 0x7a, 0xa6, 0x32, 0xc3, 0xff, 0xdf,
          0x86, 0x8b, 0xb7, 0xb2, 0x9d, 0x3d, 0x46, 0xad,
          0x83, 0xce, 0x9f, 0x9a, 0x10, 0x2e, 0xe9, 0x9d,
          0x49, 0xa5, 0x3e, 0x87, 0xf4, 0xc3, 0xda, 0x55 };
    char msg[64], msg_enc[64];
    int ret;
#if 1
    crypto_aes_ctx_t ctx;
#else
    AES_KEY ctx;
#endif

#if 1
    aes_cipher_set_key(&ctx, key, AES128_KEY_SIZE, 0);
    crypto_cbc_encrypt(aes_cipher_encrypt, &ctx, AES_BLOCK_SIZE,
                       msg_ref, msg_enc, iv, 64);
#else
    char iv_enc[16];
    memcpy(iv_enc, iv, 16);
    AES_set_encrypt_key(key, 128, &ctx);
    AES_cbc_encrypt(msg_ref, msg_enc, 64, &ctx, iv_enc, 1);
#endif
    ret = memcmp(msg_enc, msg_enc_ref, 64);
    if (ret != 0)
        return ret;
   
#if 1
    crypto_cbc_decrypt(aes_cipher_decrypt, &ctx, AES_BLOCK_SIZE,
                       msg_enc, msg, iv, 64);
    return memcmp(msg, msg_ref, 64);
#else
    AES_set_decrypt_key(key, 128, &ctx);
    AES_cbc_encrypt(msg_enc, msg, 64, &ctx, iv, 0);
    return memcmp(msg, msg_ref, 64);
#endif
}

static int test_sha1(void) {
    const char data[] =
        { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
          0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
          0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
          0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
          0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
          0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
          0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
          0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
          0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
          0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
          0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
          0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
          0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
          0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
          0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
          0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
        };

    const char hash_ref[] =
        { 0x3b, 0xd0, 0xa8, 0x8d, 0xd2, 0xa4, 0x0a, 0xa5, 0x90, 0xa8, 
          0x7e, 0x90, 0xc1, 0x8c, 0xa7, 0x04, 0x5b, 0xcd, 0xeb, 0xc7};

    crypto_sha1_ctx_t ctx;
    char hash[SHA1_DIGEST_SIZE];
    unsigned i;

    sha1_digest_setup(&ctx);
    for (i=0; i< sizeof(data); i++)
        sha1_digest_update(&ctx, data, i);
    sha1_digest_final(&ctx, hash);

#ifdef VERBOSE_TEST
    print_hex(hash, SHA1_DIGEST_SIZE);
    print_hex(hash_ref, SHA1_DIGEST_SIZE);
#endif

    return memcmp(hash, hash_ref, SHA1_DIGEST_SIZE);
}

/*
 * ***************************************************************************
 */
int main(int argc, char **argv) {

    LOG_printf("test_mfg1():       %d (0 expected)\n", test_mgf1());
    LOG_printf("test_pad_oaep():   %d (0 expected)\n", test_pad_oaep());
    LOG_printf("test_rsa1024():    %d (0 expected)\n", test_rsa1024());
    LOG_printf("test_rsa2048():    %d (0 expected)\n", test_rsa2048());
    LOG_printf("test_aes128_cbc(): %d (0 expected)\n", test_aes128_cbc());
    LOG_printf("test_sha1():       %d (0 expected)\n", test_sha1());

    /* terminate Fiasco-UX */
    l4util_reboot();
    
    return 0;
}

