%{                                            /* -*- C++ -*- */
# include <cstdlib>
# include <errno.h>
# include <string>
# include <limits>
# include "idl-parser-driver.hh"
# include "idl-parser.tab.hh"
# include "parser/Converter.h"
# include "parser/Preprocessor.h"
# include "Compiler.h"

    /* Work around an incompatibility in flex (at least versions
       2.5.31 through 2.5.33): it generates code that does
       not conform to C89.  See Debian bug 333231
       <http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=333231>.  */
# undef yywrap
# define yywrap() 1

    /* By default yylex returns int, we use token_type.
       Unfortunately yyterminate by default returns 0, which is
       not of token_type.  */
#define yyterminate() return token::EOF_TOKEN

    /* These are helper variables for the line directives generated by the
     * preprocessor. We store filename and line number across rule boundaries.
     */
static string sNewFilename;
static long nNewLinenumber;
static bool bSetLine = true;
static bool bStdInc = false;
static long nFlags;

    /* These are internal state variables for the scanner. They control
     * behaviour and can be set using the #pragma pre-processor directive
     */
static bool bCheckIncludes = true; /**< use #pragma force-include to set to false */

%}

%option noyywrap nounput batch debug

/* some rules */
Float_literal1 [0-9]*"."[0-9]+((e|E)[+|-]?[0-9]+)?
Float_literal2 [0-9]+"."((e|E)[+|-]?[0-9]+)?
Float_literal3 [0-9]+((e|E)[+|-]?[0-9]+)
Float          ({Float_literal1}|{Float_literal2}|{Float_literal3})(f|l|F|L)?

Id             [a-zA-Z_][a-zA-Z0-9_]*
Integer_Suffix ([uU][lL]?)|([uU]([lL]{2}))|([lL][uU]?)|([lL]{2}[uU]?)
Integer        [1-9][0-9]*{Integer_Suffix}?
Hexadec        0[xX][a-fA-F0-9]+{Integer_Suffix}?
Octal          0[0-7]*{Integer_Suffix}?


Char_lit       (L)?\'.\'

Escape         (L)?\'\\[ntvbrfa\\\?\'\"]\'
Oct_char       (L)?\'\\[0-7]{1,3}\'
Hex_char       (L)?\'\\(x|X)[a-fA-F0-9]{1,2}\'
Filename       (\"|"<")("<")?[a-zA-Z0-9_\./\\:\- ]*(">")?(\"|">")
string         (L)?(\"([^\\\n\"]|\\[ntvbrfa\\\?\'\"\n])*\")
Portspec       \"[^\n\\][ \t]*:[ \t]*\[[^\n\\][ \t]*\][ \t]*\"
Uuid       [a-fA-F0-9]{8}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{12}
VersionRep      [1-9][0-9]*([.,][0-9]+)?

/* Include support */
%x preproc
%x preproc2
/* Pragma support */
%x pragma
%x require

%{
#define YY_USER_ACTION  yylloc->columns (yyleng);
%}
%%
%{
    yylloc->step ();
%}

%{
    typedef yy::idl_parser::token token;
%}

"#"         {
    BEGIN(preproc);
    sNewFilename = "";
    bSetLine = true;
    nFlags = 0;
    bStdInc = false;
            }
<preproc,preproc2>[ \t]+ /* eat whitespace */
<preproc>pragma {
    BEGIN(pragma);
	    }
<preproc>[0-9]+ {
    nNewLinenumber = strtol(yytext, NULL, 10);
            }
<preproc>{Filename} {
    BEGIN(preproc2);
    string s(yytext);
    s = s.substr(1, s.length()-2); // remove "" or <>
    if (!s.empty())
	sNewFilename = s;
    else
	sNewFilename = driver.file;
    if (sNewFilename == "<stdin>")
	sNewFilename = driver.file;
    if (sNewFilename == "<built-in>")
	sNewFilename = driver.file;
    if (sNewFilename == "<command line>")
	sNewFilename = driver.file;
            }
<preproc2>[1-4] {
    long n = strtol(yytext, NULL, 10);
    /* flags are optional and have the following meaning:
     * 1 - start of the file specified with sNewFilename
     * 2 - end of the current file, return to file with name sNewFilename
     * 3 - start of a system header file (specified after 1 or 2)
     * 4 - extern "C" start (specified after 1 or 2)
     *
     * Because 3 and 4 only provide additional information that is not
     * interesting for us, we ignore them.
     */
    if (1 == n || 2 == n)
	nFlags = n;
    if (3 == n)
	bStdInc = true;
            }
<preproc,preproc2>("\r")?"\n" {
    BEGIN(INITIAL);
    int line = yylloc->end.line;
    /* always reset the location information */
    yylloc->end.line = nNewLinenumber;
    yylloc->end.column = 1;
    *yylloc->end.filename = sNewFilename;
    yylloc->begin = yylloc->end;

    if (driver.trace_scanning)
	std::cerr << "idl-scanner: # " << nNewLinenumber << " \"" << sNewFilename << "\" " << nFlags <<
	    " (old: " << line << ") found\n";
    /* nFlags == 1: start new file with filename */
    if (nFlags == 1)
	driver.enter_file(sNewFilename, nNewLinenumber, bStdInc, line, bCheckIncludes, true);
    /* nFlags == 2: end current file, return to file with filename */
    if (nFlags == 2)
	driver.leave_file(sNewFilename);
            }

<pragma,require>[ \t]+ /* eat whitespace */
<pragma>force-include {
    bCheckIncludes = false;
	    }
<pragma>require {
    BEGIN(require);
            }
<require>"\"".*"\"" {
    // strip "" from string
    string s(yytext);
    s = s.substr(1, s.length()-2);
    CCompiler::CheckRequire(s.c_str());
            }
<pragma,require>("\r")?"\n" {
    BEGIN(INITIAL);
	    }

[ \t]+		yylloc->step ();
\r		/* eat cariiage return character */
[\n]+		yylloc->lines (yyleng); yylloc->step ();
\f		yylloc->step ();
"//".*		yylloc->step ();

"/*"        {
    register int c;
    for (;;)
    {
	while ((c = yyinput ()) != '*' && c != EOF && c != '\n')
	    yylloc->step ();    // eat up text of comment
	if (c == '*')
	{
	    while ((c = yyinput ()) == '*')
		yylloc->step ();    // eat up trailing *
	    if (c == '/') break;    // found end
	}
	if (c == '\n')
	    yylloc->step ();
	if (c == EOF)
	{
	    driver.error (*yylloc, "EOF in comment.");
	    yyterminate ();
	    break;
	}
    }
            }


("{"|"<%")      return token::LBRACE;
("}"|"%>")      return token::RBRACE;
("["|"<:")      return token::LBRACKET;
("]"|":>")      return token::RBRACKET;
":"             return token::COLON;
"::"            return token::SCOPE;
","             return token::COMMA;
"("             return token::LPAREN;
")"             return token::RPAREN;
"."             return token::DOT;
"\""            return token::QUOT;
"*"             return token::ASTERISK;
"'"             return token::SINGLEQUOT;
"?"             return token::QUESTION;
"|"             return token::BITOR;
"^"             return token::BITXOR;
"&"             return token::BITAND;
"<"             return token::LT;
">"             return token::GT;
"+"             return token::PLUS;
"-"             return token::MINUS;
"/"             return token::DIV;
"%"             return token::MOD;
"~"             return token::TILDE;
"!"             return token::EXCLAM;
";"             return token::SEMICOLON;
"||"            return token::LOGICALOR;
"&&"            return token::LOGICALAND;
"=="            return token::EQUAL;
"!="            return token::NOTEQUAL;
"<="            return token::LTEQUAL;
">="            return token::GTEQUAL;
"<<"            return token::LSHIFT;
">>"            return token::RSHIFT;
".."            return token::DOTDOT;
"="             return token::IS;
">>="           return token::RS_ASSIGN;
"<<="           return token::LS_ASSIGN;
"+="            return token::ADD_ASSIGN;
"-="            return token::SUB_ASSIGN;
"*="            return token::MUL_ASSIGN;
"/="            return token::DIV_ASSIGN;
"%="            return token::MOD_ASSIGN;
"&="            return token::AND_ASSIGN;
"^="            return token::XOR_ASSIGN;
"|="            return token::OR_ASSIGN;
"++"            return token::INC_OP;
"--"            return token::DEC_OP;
"->"            return token::PTR_OP;
"void"[[:space:]]*"*" return token::VOID_PTR;
"char"[[:space:]]*"*" return token::CHAR_PTR;

abstract        return token::ABSTRACT;
any             return token::ANY;
attribute       return token::ATTRIBUTE;
__attribute__	return token::ATTRIBUTE;
boolean         return token::BOOLEAN;
byte            return token::BYTE;
case            return token::CASE;
char            return token::CHAR;
const           return token::CONST;
context         return token::CONTEXT;
control         return token::CONTROL;
custom          return token::CUSTOM;
default         return token::DEFAULT;
double          return token::DOUBLE;
enum            return token::ENUM;
error_status_t  return token::ERROR_STATUS_T;
exception       return token::EXCEPTION;
factory         return token::FACTORY;
false           return token::FALSE;
FALSE           return token::FALSE;
fixed           return token::FIXED;
flexpage        return token::FLEXPAGE;
fpage           return token::FLEXPAGE;
float           return token::FLOAT;
handle_t        return token::HANDLE_T;
hyper           return token::HYPER;
import          return token::IMPORT;
in              return token::IN;
inout           return token::INOUT;
int             return token::INT;
interface       return token::INTERFACE;
ISO_LATIN_1     return token::ISO_LATIN_1;
ISO_MULTI_LINGUAL     return token::ISO_MULTI_LINGUAL;
ISO_UCS         return token::ISO_UCS;
library         return token::LIBRARY;
module          return token::LIBRARY;
long            return token::LONG;
long[[:space:]]+long    return token::LONGLONG;
native          return token::NATIVE;
null            return token::EXPNULL;
octet           return token::OCTET;
out             return token::OUT;
pipe            return token::PIPE;
private         return token::PRIVATE;
public          return token::PUBLIC;
raises          return token::RAISES;
readonly        return token::READONLY;
refstring       return token::REFSTRING;
sequence        return token::SEQUENCE;
short           return token::SHORT;
small           return token::SMALL;
string		return token::STRING;
struct          return token::STRUCT;
supports        return token::SUPPORTS;
switch          return token::SWITCH;
true            return token::TRUE;
TRUE            return token::TRUE;
truncable       return token::TRUNCABLE;
typedef         return token::TYPEDEF;
union           return token::UNION;
unsigned        return token::UNSIGNED;
signed          return token::SIGNED;
sizeof          return token::SIZEOF;
valuetype       return token::VALUETYPE;
void            return token::VOID;
wchar           return token::WCHAR;
wstring         return token::WSTRING;


{Id}       {
    // Id is an attribute and we expect an attribute driver will find out
    token::yytokentype t = driver.find_attribute(yytext);
    if (t != token::INVALID)
	return t; // attribute token
    // check for type in symbol table
    // otherwise its a simple identifier
    yylval->sval = new std::string(yytext);
    if (driver.check_token(yytext, CSymbolTable::TYPENAME))
	return token::TYPENAME;
    return token::ID;
	   }

{Integer}  {
    using namespace dice::parser;
    IntType t;
#if HAVE_ATOLL
    long
#endif
	long n = int_to_long (yytext, t);
    switch (t)
    {
    case INT_ULLONG:
#if HAVE_ATOLL
	yylval->ullval = static_cast<unsigned long long>(n);
	return token::LIT_ULLONG;
	break;
#endif
    case INT_ULONG:
	yylval->ulval = static_cast<unsigned long>(n);
	return token::LIT_ULONG;
	break;
    case INT_LLONG:
#if HAVE_ATOLL
	yylval->llval = n;
	return token::LIT_LLONG;
	break;
#endif
    case INT_LONG:
	yylval->lval = static_cast<long>(n);
	return token::LIT_LONG;
	break;
    case INT_INT:
	yylval->ival = static_cast<int>(n);
	return token::LIT_INT;
	break;
    case INVALID:
	driver.error(*yylloc, "value out of range");
	break;
    }
	   }
{Hexadec}  {
    using namespace dice::parser;
    IntType t;
#if HAVE_ATOLL
    long
#endif
	long n = hex_to_long (yytext, t);
    switch (t)
    {
    case INT_ULLONG:
#if HAVE_ATOLL
	yylval->ullval = static_cast<unsigned long long>(n);
	return token::LIT_ULLONG;
	break;
#endif
    case INT_ULONG:
	yylval->ulval = static_cast<unsigned long>(n);
	return token::LIT_ULONG;
	break;
    case INT_LLONG:
#if HAVE_ATOLL
	yylval->llval = n;
	return token::LIT_LLONG;
	break;
#endif
    case INT_LONG:
	yylval->lval = static_cast<long>(n);
	return token::LIT_LONG;
	break;
    case INT_INT:
	yylval->ival = static_cast<int>(n);
	return token::LIT_INT;
	break;
    case INVALID:
	driver.error(*yylloc, "value is out of range");
	break;
    }
	   }
{Octal}    {
    using namespace dice::parser;
    IntType t;
#if HAVE_ATOLL
    long
#endif
	long n = oct_to_long (yytext, t);
    switch (t)
    {
    case INT_ULLONG:
#if HAVE_ATOLL
	yylval->ullval = static_cast<unsigned long long>(n);
	return token::LIT_ULLONG;
	break;
#endif
    case INT_ULONG:
	yylval->ulval = static_cast<unsigned long>(n);
	return token::LIT_ULONG;
	break;
    case INT_LLONG:
#if HAVE_ATOLL
	yylval->llval = n;
	return token::LIT_LLONG;
	break;
#endif
    case INT_LONG:
	yylval->lval = static_cast<long>(n);
	return token::LIT_LONG;
	break;
    case INT_INT:
	yylval->ival = static_cast<int>(n);
	return token::LIT_INT;
	break;
    case INVALID:
	driver.error(*yylloc, "value is out of range");
	break;
    }
	   }
{VersionRep} {
    yylval->sval = new std::string(yytext);
    return token::VERSION_STR;
	   }
{Float}    {
    errno = 0;
    double d = strtod (yytext, NULL);
    if (! (std::numeric_limits<double>::min() <= d &&
	    d <= std::numeric_limits<double>::max() &&
	    errno != ERANGE))
	driver.error (*yylloc, "floating number out of range");
    yylval->dval = d;
    return token::LIT_FLOAT;
	   }
{Char_lit} {
    if (yytext[0] == 'L')
    {
	yylval->cval = yytext[2];
	return token::LIT_WCHAR;
    }
    else
	yylval->cval = yytext[1];
    return token::LIT_CHAR;
	   }
{Escape}   {
    yylval->cval = dice::parser::escape_to_char (yytext);
    return token::LIT_CHAR;
	   }
{Oct_char} {
    yylval->cval = dice::parser::oct_to_char (yytext);
    return token::LIT_CHAR;
	   }
{Hex_char} {
    yylval->cval = dice::parser::hex_to_char (yytext);
    return token::LIT_CHAR;
	   }
{string}   {
    // check if first is 'L'
    if (yytext[0] == 'L')
    {
	yylval->sval = new std::string (&yytext[2]);
	while (*(yylval->sval->end()-1) == '"')
	    yylval->sval->erase(yylval->sval->end()-1);
	return token::LIT_WSTR;
    } else {
	yylval->sval = new std::string (&yytext[1]);
	while (*(yylval->sval->end()-1) == '"')
	    yylval->sval->erase(yylval->sval->end()-1);
    }
    return token::LIT_STR;
	   }
{Filename} {
    string s = string(yytext);
    s = s.substr(1, s.length() - 2); // remove "" or <>
    yylval->sval = new std::string(s);
    if (yytext[0] == '<')
	return token::STD_FILENAME;
    return token::FILENAME;
	   }
{Portspec} return token::PORTSPEC;
{Uuid}     {
    yylval->sval = new std::string (yytext);
    return token::UUID_STR;
           }

<<EOF>>    yyterminate();
.          driver.warning (*yylloc, "invalid character");

%%

void
idl_parser_driver::scan_begin ()
{
    using dice::parser::CPreprocessor;

    // save currently used input buffer
    previousBuffer = YY_CURRENT_BUFFER;

    yy_flex_debug = trace_scanning;
    CPreprocessor *pre = CPreprocessor::GetPreprocessor();
    if (!(yyin = pre->Preprocess(file, lastPath)))
	error (std::string ("cannot open ") + file);

    // only if we already parsed something, we need to switch to a new input
    // buffer
    if (previousBuffer)
    {
	yy_switch_to_buffer( yy_create_buffer( yyin, YY_BUF_SIZE ) );
	BEGIN(INITIAL);
    }
}

void
idl_parser_driver::scan_end ()
{
    if (yyin)
	fclose (yyin);

    // if we stored a previous buffer, restore it
    if (previousBuffer)
    {
	if (YY_CURRENT_BUFFER)
	    yy_delete_buffer(YY_CURRENT_BUFFER);
	yy_switch_to_buffer(previousBuffer);
	previousBuffer = 0;
    }
}
