/* $Id:$ */

//linux stuff
#include <linux/suspend.h>
#include <linux/console.h>
#include <linux/freezer.h>
#include <linux/cpu.h>
#include <linux/interrupt.h>

//L4 stuff
#include <l4/log/l4log.h>
#include <l4/util/util.h>
#include <l4/util/macros.h>
#include <l4/util/bitops.h>
#include <l4/util/setjmp.h> //test
#include <l4/l4rm/l4rm.h>
#include <l4/dm_mem/dm_mem.h>
#include <l4/sys/types.h>
#include <l4/semaphore/semaphore.h>
#include <l4/env/env.h>
#include <l4/env/errno.h>
#include <l4/generic_ts/generic_ts.h>
#include <l4/sys/utcb.h>
#include <l4/lxfreeze/lxfreeze.h>

//l4linux stuff
#include <asm/uaccess.h>
#include <asm/generic/suspres.h>
#include <asm/generic/kthreads.h>
#include <asm/api-l4env/api.h>
#include <asm/l4lxapi/thread.h>
#include <asm/generic/stack_id.h>
#include <asm/generic/tamed.h>
#include <asm/generic/dispatch.h>
#include <asm/l4lxapi/thread.h>
#include <asm/generic/setup.h>
#include <asm/generic/stack_id.h>
#include <asm/generic/server_opcode.h>

#include "power.h"

/* Debugging */
#define DBG_SUSP  0
#define DBG_ERROR 1


static int block_suspend = 0;
static l4_thread_jmp_buf l4x_jmp_buf;
static l4_thread_jmp_buf main_jmp_buf;
static l4_threadid_t signal_thread;

/* EXTERNS we need, quite a lot */
extern l4env_infopage_t *l4env_infopage;

/* l4rm stuff to owerwrite */
extern l4_threadid_t l4rm_service_id;
extern l4_threadid_t l4rm_task_pager_id;
extern const l4_addr_t l4rm_heap_start_addr;


/* linux main memory */
extern void * l4x_main_memory_start;
extern unsigned long l4env_mainmem_size;

/* linux arch specific */
#ifdef ARCH_x86
extern l4_utcb_t *l4_utcb_l4lx_server[NR_CPUS];
extern struct desc_struct boot_gdt;
#endif

/* threadlib */
extern int l4th_have_stack_area;
extern l4_addr_t l4th_stack_area_start;
extern l4_addr_t l4th_stack_area_end;
extern const l4_addr_t l4thread_stack_area_addr;
extern const l4_addr_t l4thread_tcb_table_addr;
extern l4_threadid_t l4semaphore_thread_l4_id;
/* funcs */
extern void exit(int);

/*******************************************************************************
 * LOCALS
 ******************************************************************************/

static void unprepare_processes(void)
{
		thaw_processes();
		enable_nonboot_cpus();
		pm_restore_console();
}	

static int prepare_processes(void)
{

	pm_prepare_console();

	if(freeze_processes()) {
		unprepare_processes();
		return -EBUSY;
	}
	return 0;
}	

/* update l4_thread_ids on stacks */
static void update_stacks(void)
{
	struct task_struct *p, *prev;

	/* kernel */
	l4x_stack_setup(current_thread_info());
	
	/* swapper */
	l4x_stack_setup(task_thread_info(&init_task));
	
	read_lock(&tasklist_lock);
	
	/* have to reset all stack id's, even of non-freezeable tasks */
	for_each_process(p) {
		prev = p;
		do {
			LOGd(DBG_SUSP, "Setting up stack for %s", task_thread_info(p)->task->comm);
		  	   l4x_stack_setup(task_thread_info(p));
		} while((p = next_thread(p)) != prev);
	} 
  
	read_unlock(&tasklist_lock);
}


#ifdef ARCH_x86
static void linux_arch(struct task_struct *prev)
{

	l4x_fiasco_gdt_entry_offset = fiasco_gdt_get_entry_offset();
	
	asm volatile("movl %%ds,  %%eax \n"
	             "movl %%eax, %%fs  \n"  // fs == ds
		     "xorl %%eax, %%eax \n"
		     "movl %%eax, %%gs  \n"  // clear gs
		     : : : "eax", "memory");
}
#else
static inline void linux_arch(struct task_struct *prev) { }
#endif

static void update_id(l4_threadid_t ref, l4_threadid_t * update)
{
	l4thread_t save;

	save = update->id.lthread;
	*update = ref;
	update->id.lthread = save;
}

static void update_global_ids(void)
{
	
	l4_threadid_t myself = l4_myself();
	
	/* region mapper thread */
	l4rm_service_id = myself;
	
	/* startup thread */
	update_id(myself, &l4x_start_thread_id);

	/* startup pager */
	l4x_start_thread_pager_id = myself;

	/* l4linux server */
	update_id(myself, &linux_server_thread_id);
	
	/* kernel taskno */
	l4x_kernel_taskno = myself.id.task;
	
	/* main memory dataspace */
	l4env_ds_mainmem.manager = lxfreeze_get_id(); 
}


static void __review_region_map(void)
{
	
	l4_addr_t addr;
	int region_type = 0;
	l4_addr_t map_addr = 0;
	l4_size_t map_size = 0;
	l4_offs_t offset;
	l4dm_dataspace_t ds;
	l4_threadid_t	cur_pager;
	l4_threadid_t freezer_pager = lxfreeze_get_pager_id();
	int error;

	for(addr = l4env_infopage->vm_low; 
	    (map_addr + map_size) < l4env_infopage->vm_high;
      addr = map_addr + map_size) {
		
		ds.id = 0;
		addr &= L4_PAGEMASK;	
		
		region_type = l4rm_lookup_region((void *)addr, &map_addr, &map_size, &ds,
		                                 &offset, &cur_pager);

		if(region_type <= 0) {
			LOGd(DBG_SUSP, "Nothing found for %08lx\n", addr);
			map_size = L4_PAGESIZE;
			continue;
		}
		
		switch(region_type) {

			case L4RM_REGION_PAGER:

				if((error = l4rm_area_clear_region(map_addr))) {
					LOGd(DBG_ERROR, "%s\n", l4env_errstr(error));
					break;
				}

				if((error = l4rm_area_setup_region(map_addr, map_size, 
				                                   L4RM_DEFAULT_REGION_AREA,
				                                   L4RM_REGION_PAGER, 0, 
				                                   freezer_pager)))
					LOGd(DBG_ERROR, "Re-setup error %s\n", l4env_errstr(error));
				break;
		}
	}
	
}

static void __rm_startup(void)
{
	l4_addr_t map_addr;
	l4_size_t size;
	l4dm_dataspace_t ds;
	l4_offs_t offs;
	l4_threadid_t pager;
	int i;
	LOG_printf("Good day, i am alive "l4util_idfmt"\n", 
	           l4util_idstr(l4_myself()));
	LOG_flush();

	/* update global thread ids */
	update_global_ids();
	
	/* switch to freezer given memserver */
	l4env_set_default_dsm(l4env_infopage->memserv_id);
	
	/* reconnect to task server */
	l4ts_connect();

	/* reset thread library */
	l4thread_reset_init();
	
	/* make myself known to threadlib */
	l4thread_setup(l4rm_service_id,".rm", l4env_infopage->stack_low, 
	               l4env_infopage->stack_high);	
	
	/* release after exit */
	if((l4rm_lookup((void*)l4thread_tcb_table_addr, &map_addr, &size, &ds, &offs,
	                &pager)) == L4RM_REGION_DATASPACE)
		lxfreeze_handle_pagefault(ds.id, map_addr, size, LX_RELEASE);


	l4semaphore_init();
	LOGd(DBG_SUSP, "Semaphore initialized\n");

	for(i = 0; i < l4env_infopage->section_num; i++) {
		
		if(l4dm_is_invalid_ds(l4env_infopage->section[i].ds))
			continue;
		
		if(l4dm_map_ds(&l4env_infopage->section[i].ds, 0, 
		               l4env_infopage->section[i].addr, 
		               l4env_infopage->section[i].size,
		               L4DM_RO)) 
			LOGd(DBG_ERROR, "Error mapping dataspace  %u", 
			     l4env_infopage->section[i].ds.id);
	}

	/* start main thread */
	l4_thread_longjmp(l4x_start_thread_id, main_jmp_buf, 0);
	l4lx_thread_pager_change(l4x_start_thread_id, l4rm_region_mapper_id());
	
	l4rm_service_loop();
}

static int pf_callback(l4_addr_t addr, l4_addr_t eip, l4_threadid_t src)
{
	int error;
	l4_umword_t dw0, dw1;
	l4_msgdope_t result;

  LOGd(DBG_SUSP, "["l4util_idfmt"] forward to pager "l4util_idfmt" addr: %08lx\n",
        l4util_idstr(src), l4util_idstr(lxfreeze_get_pager_id()), addr);
  for (;;) {
      /* we may get l4_thread_ex_regs'ed ... */
      error = l4_ipc_call(lxfreeze_get_pager_id(),
                          L4_IPC_SHORT_MSG, addr, eip,
                          L4_IPC_MAPMSG(addr, L4_LOG2_PAGESIZE),
                          &dw0, &dw1,
                          L4_IPC_NEVER, &result);
      if (error != L4_IPC_SECANCELED && error != L4_IPC_SEABORTED)
        break;
	}
	
	if(!dw1) {
		l4rm_show_region_list();
		enter_kdebug("Unresolved PF");
	}

  /* done */
  return L4RM_REPLY_EMPTY;
}


/*******************************************************************************
 *** GLOBALS
 ******************************************************************************/

/* swsusp interface call */
int pm_suspend_ds(void)
{
	int error = 0;
	static l4_addr_t lx_high, lx_low;
	static struct task_struct *last_task;
	l4_msgdope_t msg_result;
	
	if(block_suspend)
		return -EBUSY;
	
	if(l4_is_invalid_id(lxfreeze_get_id()))
		goto Error;
	
	l4thread_get_stack(linux_server_thread_id.id.lthread,&lx_low, &lx_high);
	LOGd(DBG_SUSP, "Linux stack %08lx - %08lx\n", lx_low, lx_high);	
	
	error = prepare_processes();
	if(error) {
		LOGd(DBG_ERROR, "prepare processes returned error");
		return error;
	}
	
	suspend_console();
	error = device_suspend(PMSG_FREEZE);
	if(error)
	{
		LOGd(DBG_ERROR, "Some devices failed to suspend");
		goto DevRes;
	}

	disable_nonboot_cpus();

	if((error = arch_prepare_suspend()))
		return error;

	local_irq_disable();

	if((error = device_power_down(PMSG_FREEZE))) {
		LOGd(DBG_ERROR, "Some devices failed to freeze");
		goto IRQ;
	}

	if(l4_thread_setjmp(l4x_jmp_buf))
		goto Resume;
	
	/* send suspend call to startup thread */
	LOG_printf("Good night ...\n");
	LOG_flush();
	local_irq_enable();
	
	last_task = current;
	
	l4_ipc_send(l4x_start_thread_id, L4_IPC_SHORT_MSG,
	            L4X_SERVER_SUSPEND, 0, L4_IPC_NEVER, &msg_result);

	/* done  */
	l4_sleep_forever();

Resume:
	LOG_printf("Linux's been woken up stack at %08lx - %08lx\n", lx_low, lx_high);
	
	update_stacks();
	
	/*register at thread lib */
	l4thread_setup(linux_server_thread_id, "l4linux (cloned)", lx_low, lx_high);
	l4thread_set_prio(linux_server_thread_id.id.lthread, CONFIG_L4_PRIO_SERVER);
	
	/* linux server tcb */
	l4_utcb_set_l4lx(0, l4_utcb_get());
	l4_utcb_inherit_fpu(l4_utcb_get(), 1);
	linux_arch(last_task);
	
	local_irq_disable();
	device_power_up();

IRQ:	
	local_irq_enable();

DevRes:
	device_resume();
	resume_console();
	unprepare_processes();

Error:
	return error;
}

/* suspend function of main thread */
void l4x_linux_suspend_ds(void)
{
	l4_umword_t old_eflags, dummy;
	l4_threadid_t pager = L4_INVALID_ID;
	l4_threadid_t preempter = L4_INVALID_ID;
	static l4_addr_t main_stack_low, main_stack_high;
	l4_umword_t l4rm_esp;
	int i = 0;
	
	LOGd(DBG_SUSP, "Startup thread received suspend request\n");
	
#ifdef CONFIG_L4_TAMED
	for(i = 0; i < NR_CPUS; i++)
		l4x_tamed_shutdown(i);
#endif
	
	l4thread_get_stack_current(&main_stack_low, &main_stack_high);	
	
	LOGd(DBG_SUSP, "info test secs: %d\n", l4env_infopage->section_num);
	
	pm_suspend_ds_free_stack(l4semaphore_thread_l4_id);
	LOGd(DBG_SUSP, "Killed semaphore\n");
	
	/* save l4rm stack pointer */
	pager = lxfreeze_get_pager_id();
	l4_thread_ex_regs_flags(l4rm_region_mapper_id(), (l4_umword_t)~0, 
	                        (l4_umword_t)~0, &preempter, &pager, &old_eflags,
	                        &dummy, &l4rm_esp, L4_THREAD_EX_REGS_NO_CANCEL);

/* resumed */
	if(l4_thread_setjmp(main_jmp_buf))
		goto Resume;

	l4lx_thread_pager_change(l4x_start_thread_id, lxfreeze_get_pager_id());
	
  l4rm_detach((void*)l4thread_tcb_table_addr);
	
	/* send eip, esp */
	lxfreeze_finish((l4_addr_t)__rm_startup, (l4_addr_t)l4rm_esp, ~0);
	exit(0);

Resume:

	LOGd(DBG_SUSP, "Main stack %08lx - %08lx\n", main_stack_low, main_stack_high);
	/* make main thread known to thre threadlib */
	l4thread_setup(l4x_start_thread_id, ".main", main_stack_low, main_stack_high);
	
#ifdef CONFIG_L4_TAMED
	l4x_tamed_init(0);
#endif
	
	LOG_printf("Waking up linux (ID: "l4util_idfmt")\n",
	           l4util_idstr(linux_server_thread_id));

	/* resume linux */
	l4_thread_longjmp(linux_server_thread_id, l4x_jmp_buf, 0);
	l4lx_thread_pager_change(linux_server_thread_id, l4x_start_thread_id);
	
	/* return to server loop */	
}

void pm_suspend_ds_free_stack(l4_threadid_t thread)
{
	l4_addr_t stack_low, stack_high;

	l4thread_get_stack(thread.id.lthread, &stack_low, &stack_high);
	l4dm_mem_release((void *)stack_low); 
}


/*******************************************************************************
 *** sysfs suspend hook
 ******************************************************************************/

static ssize_t l4state_show(struct kset *kset,  char *buf)
{
	return sprintf(buf, "ds\n");
}

static ssize_t l4state_store(struct kset *kset, const char *buf, size_t n)
{
	char * p;
	int len, err;
	p = memchr(buf, '\n', n);
	len = p ? p - buf : n;

	if((err = strncmp(buf, "ds", len)) == 0)
		pm_suspend_ds();
	
	return err ? -EINVAL : n;
};

power_attr(l4state);

static struct attribute * g[] = {
	&l4state_attr.attr,
	NULL,
};

static struct attribute_group attr_group = {
	.attrs = g,
};


/*******************************************************************************
 *** IPC suspend hook
 ******************************************************************************/
static DECLARE_MUTEX_LOCKED(susp_sem);

static int pm_suspend_ds_wrapper(void * data)
{
	daemonize("kl4suspend");
	current->flags |= PF_NOFREEZE;
	while(1) {
		down(&susp_sem);
		pm_suspend_ds();
	}
	return 0;
}

/* waits for freeze IPC */
static void kernel_signal_thread(void *data) 
{
	l4_umword_t w0 = 0, w1 = 0;
	l4_msgdope_t result;
	int ret = 1;
	
	l4x_stack_setup(current_thread_info());
	while(1) {
		ret = l4_ipc_receive(lxfreeze_get_id(), L4_IPC_SHORT_MSG,
		                     &w0, &w1, L4_IPC_NEVER, &result);
	  
	  if(!ret) 
			up(&susp_sem);
	}

}

static l4_threadid_t kernel_signal_thread_start(void) 
{
	return l4lx_thread_create(kernel_signal_thread, NULL, NULL, 0,
	                          CONFIG_L4_PRIO_SERVER - 1, 
	                          "Suspend Signal Thread");
}

static void kernel_signal_suspend_resume(enum l4x_suspend_resume_state state)
{
	switch(state) {
		case L4X_SUSPEND:
			l4lx_thread_shutdown(signal_thread);
			break;
		case L4X_RESUME:
			signal_thread = kernel_signal_thread_start();
			lxfreeze_signal_id(signal_thread);
	}
}


/*******************************************************************************
 *** init function 
 ******************************************************************************/

void pm_suspend_ds_register(void)
{
	l4_addr_t map_addr;
	l4_offs_t offs;
	l4_size_t size;
	l4dm_dataspace_t ds;
	l4_threadid_t pager;
	l4_threadid_t preempter = L4_INVALID_ID;
	l4_umword_t dummy;
	int err;
	LOGd(DBG_SUSP, "Registering power management\n");
	if(!l4_thread_equal(lxfreeze_get_id(), l4env_infopage->memserv_id)) {
		block_suspend = 1;
		return;
	}
	
	/* start kernel signal thread */
	kernel_thread(pm_suspend_ds_wrapper, NULL, CLONE_KERNEL);
	signal_thread = kernel_signal_thread_start();
	l4x_suspend_resume_register(kernel_signal_suspend_resume);
	
	/* create sys/power/l4state */
	err = sysfs_create_group(&power_subsys.kobj, &attr_group);
	
	/* signal loaded to freezer */
	lxfreeze_loaded(signal_thread);
	
	/* send virtual addresses to handle */

	/* send infopage */
	lxfreeze_find_env_infopage((void *)l4env_infopage, L4_PAGESIZE);

/* initial stack */
	lxfreeze_find_info_stack((void*)l4env_infopage->stack_low, 
	                         l4env_infopage->stack_high 
	                         - l4env_infopage->stack_low);
	
	/* l4rm heap */	
	LOGd(DBG_SUSP, "L4RM heap at %08lx", l4rm_heap_start_addr);
	if((l4rm_lookup((void*)l4rm_heap_start_addr, &map_addr, &size, &ds, &offs,
	                &pager)) == L4RM_REGION_DATASPACE)
		lxfreeze_handle_pagefault(ds.id, map_addr, size, 0);
	

/* release after exit */
	if((l4rm_lookup((void*)l4thread_tcb_table_addr, &map_addr, &size, &ds, &offs,
	                &pager)) == L4RM_REGION_DATASPACE)
		lxfreeze_handle_pagefault(ds.id, map_addr, size, LX_RELEASE);

	/* set pager of region mapper to freezer */
	pager = lxfreeze_get_pager_id();
	l4_thread_ex_regs_flags(l4rm_service_id,
	                        (l4_umword_t)~0,
	                        (l4_umword_t)~0,
	                        &preempter, &pager,
	                        &dummy, &dummy, &dummy,
	                        L4_THREAD_EX_REGS_NO_CANCEL);

	l4rm_set_unkown_pagefault_callback(pf_callback);
	
	/* update pager regions in region map */
	__review_region_map();
}
