/* $Id: lxfreeze.h,v 1.2 2007/01/12 12:00:59 cbass Exp $ */
/*****************************************************************************/
/**
 *  \file lxfreeze.h
 *  \brief Linux freezer library 
 * 
 *  \date 11/12/06 19:44:37 CET
 *  \author Sebastian Sumpf <Sebastian.Sumpf@inf.tu-dresden.de>
 */ 
/******************************************************************************/

#include <l4/dm_generic/dm_generic.h>
#include <l4/sys/types.h>

#define LX_FREEZE "lxfreeze"
/// Release address/dataspace after L4Linux shutdown
#define LX_RELEASE 1

/**
 * \brief Get id of lxfreeze server
 *
 * \return thread id on success, otherwise:
 *				 - L4_INVALID_ID
 */
l4_threadid_t 
lxfreeze_get_id(void);

/**
 * \brief Get id of lxfreeze pager thread 
 *
 * \return thread id on success, otherwise:
 *				 - L4_INVALID_ID
 */
l4_threadid_t 
lxfreeze_get_pager_id(void);

/**
 * \brief Copy L4Env-Infopage
 *
 * \param page the pointer to infopage
 * \param size size of infopage
	*
 * \return 0 on success, otherwise:
 * 				- -#L4_ENOTFOUND if lxfreezer server could not be located
 */
int 
lxfreeze_find_env_infopage(void *page, l4_size_t size);

/**
 * \brief Copy L4Env-Infopage stack
 *
 * \param stack_addr pointer to lower stack address
 * \param size size of stack
 *
 * \return 0 on success, otherwise:
 * 				- -#L4_ENOTFOUND if lxfreezer server could not be located
 */
int lxfreeze_find_info_stack(void * stack_addr, l4_size_t stack_size);

/** \brief Asign lxfreeze to handle pagefaults for virtual memory area
 *
 *  \param ds_id ID of dataspace to handle pagefaults for
 *  \param map_addr virtual address (in callers address space)
 *  \param size size of area to handle
 *  \param flags	Flags:
 *  							- #LX_RELEASE release area after exit of L4Linux
 *  							
 * \return 0 on success, otherwise:
 * 				- -#L4_ENOTFOUND if lxfreezer server could not be located
 */
int lxfreeze_handle_pagefault(unsigned long ds_id, l4_addr_t map_addr, 
		l4_size_t size, l4_uint32_t flags);


/**
 * \brief Let lxfreeze peform restart preparations
 *
 * \param resume_eip Address to resume L4Linux
 * \param resume_esp Stack pointer to resume
 */
void lxfreeze_finish(l4_addr_t resume_eip, l4_addr_t resume_esp, 
		unsigned long drain_ds_id);

/**
 * \brief Perform resume, return task id 
 */
void lxfreeze_wake_up(l4_taskid_t *clone_id);

/**
 * \brief Kill a Linux clone (regardless of consequences)
 */
void lxfreeze_kill_clone(l4_taskid_t clone_id);


/* internal */
void lxfreeze_timer(void);
void lxfreeze_loaded(l4_threadid_t signal_id);
void lxfreeze_signal_id(l4_threadid_t signal_id);
