/* $Id:$ */
/*******************************************************************************/
/*
 * \file   thread.c
 * \brief  file I/O, server communication
 *
 * \date   2007-09-27
 * \author Sebastian Sumpf <sumpf@os.inf.tu-dresden.de>
 */
/*******************************************************************************/
/* (c) 2007 Technische Universitaet Dresden
 * This file is part of DROPS, which is distributed under the terms of the
 * GNU General Public License 2. Please see the COPYING file for details.
 */
/* Linux */
#include <pthread.h>
#include <stdio.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <stdlib.h>
#include <unistd.h>

/* Local */
#include "lxMirback_end-server.h"
#include "__globals.h"

/* L4 */
#include <l4/dm_mem/dm_mem.h>
#include <l4/lxMir/lxMir-client.h>
#include <l4/log/l4log.h>
#include <l4/env/errno.h>
#include <l4/util/macros.h>
//#include <l4/util/rdtsc.h>

/* TODO make this thread save using TLS or similiar */
static off_t fd_offset;
static int   l4_error = 0;

/* CAUTION: hybrid apps are not linked against the Log server, dummy implement
 * necessary functions (dm_mem) */
int  inline LOG_printf(const char* fmt, ...){return 0;} 
void inline LOG_logL(const char*file, int line, const char*function,
                     const char*format,...){}


/* find task info for given dataspace id, also return file offset */
static lx_ds_info_t * 
get_ds(lx_task_t * task, unsigned long ds_id, off_t *offs)
{
	int pos = 0;
	while(pos < task->ds_count) {
		if(task->ds[pos].ds_id == ds_id) {
		 *offs = task->ds_offs[pos];
		 return &(task->ds[pos]);
	  }
	  pos++;
	}
	return NULL;
}

static l4_uint64_t get_time(void)
{/*
	static l4_cpu_time_t tsc1 = 0, tsc2 = 0;
	l4_uint64_t tus;
	
	if(!tsc1) {
		tsc1 = l4_rdtsc();
		tus = 0;
	}
	else {
		tsc2 = l4_rdtsc();
		tus = l4_tsc_to_us(tsc2 - tsc1);
		tsc1 = 0;
	}
	return tus;*/
	return 0;
}

/* copy remaining pages to file */
static int post_copy(lx_task_t * task) {
	CORBA_Environment env = dice_default_environment;
	unsigned long ds_id, offs;
	void * page;
	off_t  file_offs;
	int err, write_offs;
	int fd = task->fd;
	int cnt = 0;
	lx_ds_info_t * ds_info;
	l4_size_t size;
	
	env.malloc = linux_malloc;
	env.free   = linux_free;
	
	while(!(err = if_l4dm_lxmir_post_copy_call(&freezer_id, &(task->task_id), 
	                                           &ds_id, &offs, &page, &size, 
	                                           &env))) {
		ds_info = get_ds(task, ds_id, &file_offs);
		if(ds_info == NULL || ds_info->flags == DS_INFO_INVALID) 
			continue;
		
		if((err = lseek(fd, offs + file_offs, SEEK_SET)) == -1) {
			fprintf(stderr, "Seek error %d\n", errno);
			err = -errno;
			goto Error;
		}
		
		write_offs = 0;
		for(;size > 0; size-=err, write_offs += err) 
			if((err = write(fd, page + write_offs, size)) < 0) {
				err = -errno;
				goto Error;
			}
		
		cnt++;
	}
	
	/* write header */
	if(lseek(fd, 0, SEEK_SET) == -1) {
		err = -errno;
		goto Error;
	}

	for(size = sizeof(lx_task_t), write_offs = 0; size > 0; 
	    size -= err, write_offs += err) 
		if((err = write(fd, (void*)task + write_offs, size)) < 0) {
			err = -errno;
			break;
		}

Error:
	err = err < 0 ? err : 0;
	return err;
}

/* map dataspace, mmap file; copy data to--from file */
static int __cpy(int fd, lx_ds_info_t ds, int to_fd)
{
	int err;
	void * ptr, ** ptr_to, ** ptr_from;
	unsigned long mmap_size, size;
	l4dm_dataspace_t l4_ds;
	int mmap_flags, mmap_prot, dm_flags;
	l4_uint64_t time[3];
	l4_ds.id      = ds.ds_id;
	l4_ds.manager = freezer_id;
	size          = 0;
	
	mmap_prot = PROT_READ;
	mmap_flags = 0;  // MAP_POPULATE;
	dm_flags   = L4DM_RW;
	ptr_to    = &copy_area;
	ptr_from  = &ptr;
	
	if(to_fd) {
		mmap_prot |= PROT_WRITE;
		mmap_flags = 0;
		dm_flags   =  L4DM_RO;
		ptr_to    = &ptr;
		ptr_from  = &copy_area;
	}
	
	while(size < ds.size)
	{
		if(ds.size - size < COPY_AREA_SIZE)
			mmap_size = ds.size - size;
		else
			mmap_size = COPY_AREA_SIZE;
		
		get_time();
		ptr = mmap(0, mmap_size, mmap_prot, mmap_flags | MAP_SHARED,
		           fd, fd_offset);
		time[0] = get_time();
		
		if(ptr == MAP_FAILED) {
			fprintf(stderr, "MMap failed for ds %lu, offset: %08lx\n", 
			        ds.ds_id, size);
			return -errno;
		}
		
		get_time();
		err = l4dm_map_ds(&l4_ds, (l4_offs_t)size, (l4_addr_t)copy_area,
		                  (l4_size_t)mmap_size, dm_flags);
		if(err) {
			fprintf(stderr, "Dataspace mapping failed for ds %lu,"
			                " offset: %08lx size: %08x (%08x)at %p",
			        ds.ds_id, size, (l4_size_t)mmap_size, ds.size, copy_area);
			fprintf(stderr, "L4 Error: %s (%d)\n", l4env_errstr(err), err);
			l4_error = 1;
			return err;
		}
		time[1] = get_time();

		get_time();
		memcpy(*ptr_to, *ptr_from, mmap_size);
		time[2] = get_time();
		
		fd_offset += mmap_size;
		size += mmap_size;
		
		if((munmap(ptr, mmap_size)) < 0)
			return -errno;
	}
	
	return 0;
}

static int cpy_ds2file(int fd, lx_ds_info_t ds) 
{
	/* set to beginning of dataspace */
	if(lseek(fd, ds.size - 1, SEEK_CUR) == -1) 
		return -errno;
	
	if(write(fd, "", 1) != 1) 
		return -errno;
	
	return __cpy(fd, ds, 1);
}

static int cpy_file2ds(int fd, lx_ds_info_t ds, off_t offs)
{
	fd_offset = offs;
	return __cpy(fd, ds, 0);
}

/* cleanup and exit, send exit code to front end */ 
static void thread_exit(lx_task_t * task, int exit_code)
{
	CORBA_Server_Environment env_server = dice_default_server_environment;
	env_server.malloc = linux_malloc;
	env_server.free   = linux_free;
	close(task->fd);
	if_net_lxmir_done_send(&(task->caller_id), exit_code, &l4_error, &env_server);
	free((void*)task);
	pthread_exit(NULL);
}

/* store task data in file */
void thread_save(lx_task_t *task) 
{
	CORBA_Environment env = dice_default_environment;
	CORBA_Server_Environment env_srv = dice_default_server_environment;
	int err = 0;
	int i;
	l4_addr_t eip, esp;
	/* reserve task struct header */
	l4_error = 0;
//	l4_tsc_init(L4_TSC_INIT_KERNEL);
	fd_offset = (sizeof(lx_task_t) + L4_PAGESIZE - 1) & L4_PAGEMASK;
	if(lseek(task->fd, fd_offset, SEEK_SET) == -1 || write(task->fd, "", 1) != 1)
		thread_exit(task, err);
	
	if((err = if_l4dm_lxmir_prepare_send_call(&freezer_id, 
	                                          &(task->task_id),
	                                          &(task->back_server_id),
	                                          task->ds, 
	                                          &(task->ds_count), 
	                                          &env)) < 0) 
	{
		l4_error = 1;
		thread_exit(task, err);
	}
	
	for(i = 0; i < task->ds_count; i++) {
		task->ds_offs[i] = fd_offset;
		
		if((err = cpy_ds2file(task->fd, task->ds[i]))) {
			/* signal pre copy error, as long as we still can recover */
			if_l4dm_lxmir_pre_copy_error_call(&freezer_id, &(task->task_id), &env);
			thread_exit(task, err);
		}
	}

	/* signal Linux suspend and wait for exit */
	if_l4dm_lxmir_suspend_send(&freezer_id, &(task->task_id), &env);
	if((err = if_l4dm_lxmir_suspend_done_recv(&freezer_id, &eip, &esp,  &env_srv))) {
		fprintf(stderr ,"Error during Linux suspend\n");
		l4_error = 1;
		thread_exit(task, err);
	}
	task->start_eip = eip;
	task->start_esp = esp;

	/* update invalidated dataspace structures */
	if_l4dm_lxmir_update_ds_info_call(&freezer_id, &(task->task_id), task->ds, 
	                                  &(task->ds_count), &env);

	/* retrieved pages changed during pre-copy */
	if(!(err = post_copy(task)))
		if_l4dm_lxmir_back_end_done_send(&freezer_id, &(task->task_id), &env);
	
	thread_exit(task, err);
}

/* load task data from file */
void thread_restore(lx_task_t * task)
{
	CORBA_Environment env = dice_default_environment;
	int err, i;
	
//	l4_tsc_init(L4_TSC_INIT_KERNEL);
	if((err = if_l4dm_lxmir_prepare_recv_call(&freezer_id,
	                                          &(task->back_server_id),
	                                          task->ds,
	                                          &(task->ds_count),
	                                          task->start_eip,
	                                          task->start_esp,
	                                          &(task->task_id),
	                                          &env))) {
		l4_error = 1;
		thread_exit(task, err);
	}

	if(lseek(task->fd, task->ds_offs[0], SEEK_SET) == -1) 
		thread_exit(task, -errno);
	
	for(i = 0; i < task->ds_count; i++) {
		if(task->ds[i].flags == DS_INFO_INVALID) {
			
			if(i + 1 == task->ds_count)
				continue;

			if(lseek(task->fd, task->ds_offs[i + 1], SEEK_SET) == -1) 
				thread_exit(task, -errno);
			
			continue;
		}
		
		if((err = cpy_file2ds(task->fd, task->ds[i], task->ds_offs[i]))) 
			thread_exit(task, err);
	}

	/* start new Linux instance */
	if_l4dm_lxfreeze_wake_up_call(&freezer_id, &(task->task_id), 0, 0, &env);

	thread_exit(task, 0);
}

