/* $Id: if_lxfreeze.c,v 1.5 2007/01/25 15:13:06 cbass Exp $ */
/*******************************************************************************/
/*
 * \file   if_lxfreeze.c
 * \brief  dm_mem/dm_generic/lxfreeze interfaces implementations
 *
 * \date   2007/09/27
 * \author Sebastian Sumpf <sumpf@os.inf.tu-dresden.de>
 */
/*******************************************************************************/
/* (c) 2007 Technische Universitaet Dresden
 * This file is part of DROPS, which is distributed under the terms of the
 * GNU General Public License 2. Please see the COPYING file for details.
 */

#include <l4/generic_ts/generic_ts.h>
#include <l4/util/bitops.h>

#ifdef USE_TASKLIB
#include <l4/task/task_server.h>
#endif

#include "lxMir-server.h"
#include "__types.h"
#include "__lxcalls.h"
/* from package ldso (see Makefile private includes) */
#include "emul_linux.h"

static lx_task_t * __get_task(l4_taskid_t task_id)
{
	lx_task_t * task;

	if(lx_task_get(task_id, &task) >= 0)
		return task;
	
	if((task = lx_task_get_loading()) == NULL)
	 lx_task_create_loading();
	
	return lx_task_get_loading();
}

/* add ldso mmap dataspaces to us/pager */
static int __handle_mmap_infopage(lx_task_t * task, l4dm_dataspace_t ds, 
                                  l4_size_t size)
{
	int i, error;
	lx_page_fault_t fault;
	lx_ds_t * lxds;
	mmap_infopage_t * info;
	
	if((error = lx_ds_get(task, ds.id, &lxds)) < 0)
		return error;
	
	info = (mmap_infopage_t *)lxds->map_addr;
	
	for(i = 0; i < info->section_num; i++)
	{
		LOGd(DBG_REGISTER, "Got mmap_info at: %08lx size: %08x ds_id %u",
		     info->section[i].addr, info->section[i].size, info->section[i].ds.id);

		fault.virt_addr = info->section[i].addr;
		fault.virt_size = info->section[i].size;
		fault.ds_id     = info->section[i].ds.id;
		fault.flags     = 0;
		
		/* dataspace will be copied at transfer_owned, since l4linux own them as 
		 * "anon memory" dataspaces */
		if((error = lx_pager_add_ds(task, fault)) < 0)
			return error;
	}	
	return 0;
}
/******************************************************************************
 *** SUSPEND functions
 *****************************************************************************/

/* copy l4env infopage */	
int
if_l4dm_lxfreeze_find_env_infopage_component(CORBA_Object _dice_corba_obj,
                                             const void * page,
                                             l4_size_t size,
                                             CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t *task;
	lx_ds_t *lxds;
	lx_page_fault_t fault;
	unsigned long ds_id;
	int error, i, not_found;
	l4env_infopage_t * lx_infopage;

	task = __get_task(*_dice_corba_obj);
	not_found = lx_ds_search_equal(task, (void*)page, size, &ds_id);
	
	if(not_found)
		enter_kdebug("Couldn't find dataspace of L4Env infopage");
	
	LOGd(DBG_SUSP, "Found L4Env infopage at ds_id %lu", ds_id);

	
	fault.virt_addr  = L4ENV_INFOPAGE_AT;
	fault.virt_size  = size;
	fault.ds_id      = ds_id;
	fault.flags      = 0;

	if((error = lx_ds_get(task, ds_id, &lxds)) < 0)
		return error;
	
	lx_infopage = task->lx_infopage = (l4env_infopage_t *)lxds->map_addr;
	
	/* pager handles pagefaults for this dataspace, also */
	if((error = lx_pager_add_ds(task, fault)) < 0)
		return error;

	for(i = 0; i < lx_infopage->section_num; i++)
	{
		if(l4dm_is_invalid_ds(lx_infopage->section[i].ds))
			continue;

		/* look for mmap-infopage of pkg/ldso (patched version) */
		if(lx_infopage->section[i].info.id == MMAP_SAVE_EXEC_ID)
		{
			LOGd(DBG_REGISTER, "Found MMAP info-page at 0x%08lx", 
			     lx_infopage->section[i].addr);
			error = __handle_mmap_infopage(task, lx_infopage->section[i].ds, 
			                               lx_infopage->section[i].size);
			continue;
		}
		
		fault.virt_addr = lx_infopage->section[i].addr; 
		fault.virt_size = lx_infopage->section[i].size;
		fault.ds_id     = lx_infopage->section[i].ds.id;
		fault.flags     = 0;
		if((error = lx_pager_add_ds(task, fault)) < 0)
			return error;
	}
	return 0;
}

/* copy inital stack of linux which is paged by loader (l4rm stack) */
int
if_l4dm_lxfreeze_find_stack_component(CORBA_Object _dice_corba_obj,
                                      const void * page,
                                      l4_size_t size,
                                      CORBA_Server_Environment *_dice_corba_env)
{
  int error, not_found;
  lx_task_t * task;
	lx_page_fault_t fault;
	unsigned long ds_id;
	
	task = __get_task(*_dice_corba_obj);
	not_found = lx_ds_search_equal(task, (void*)page, size, &ds_id);
	
	if(not_found)
		enter_kdebug("Couldn't find intial stack dataspace");
	
	LOGd(DBG_SUSP, "Found initial stack at ds_id %lu", ds_id);
	
	fault.virt_addr  = task->lx_infopage->stack_low;
	fault.virt_size  = size;
	fault.ds_id      = ds_id;
	fault.flags      = 0;

	/* pager handles pagefaults for this dataspace, also */
	if((error = lx_pager_add_ds(task, fault)) < 0)
		return error;
	
	return 0;
}

/* get id of pager */
int
if_l4dm_lxfreeze_get_start_pager_component(CORBA_Object _dice_corba_obj, 
                                           l4_threadid_t * pager,
                                           CORBA_Server_Environment *_dice_corba_env)
{
	
	*pager = lx_pager_get_id();
	return 0;
}

/* handle pagefaults in addition to dataspace faults for ds */
int
if_l4dm_lxfreeze_handle_pagefault_component(CORBA_Object _dice_corba_obj,
                                            l4_uint32_t ds_id, l4_addr_t addr, 
                                            l4_size_t size, l4_uint32_t flags,
                                            CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t * task;
	lx_page_fault_t fault;

	task = __get_task(*_dice_corba_obj);
	fault.ds_id     = ds_id;
	fault.virt_addr = addr;
	fault.virt_size = size;
	fault.flags     = flags;
	
	return lx_pager_add_ds(task, fault);
}


void
if_l4dm_lxfreeze_finish_component(CORBA_Object _dice_corba_obj,
                                  l4_addr_t start_eip,
                                  l4_addr_t start_esp,
                                  unsigned long drain_ds_id,
                                  CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t * task;
	LOGd(DBG_SUSP, "Received finish request at EIP: 0x%08x, ESP: 0x%08x\n",
                 (unsigned int)start_eip, (unsigned int)start_esp);
	
	lx_task_get(*_dice_corba_obj, &task);
	
	/* release LX_RELEASE dataspaces */
	lx_pager_chk_release(task);
	task->start_eip = start_eip;
	task->start_esp = start_esp;
}

/*******************************************************************************
 *** WAKE UP CALL
 ******************************************************************************/
void
if_l4dm_lxfreeze_wake_up_component(CORBA_Object _dice_corba_obj,
                                   l4_taskid_t *task_id,
                                   l4_addr_t start_eip,
                                   l4_addr_t start_esp,
                                   CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t * task;
	l4_threadid_t pager = lx_pager_get_id();

/* TODO: add startup check */
/*	
	if((error = l4ts_allocate_task(&clone_id)))
		return;
	*/

	lx_task_get(*task_id, &task);
	
	/* reset migration information */
	task->state           = TASK_MIGRATED;
	task->mir_id          =
	task->back_server_id  =
	task->l4x_signal_id   = L4_INVALID_ID;

	dbg_lx_ds_hash(task);
	
	/* set default memserver to dm_phys */
	task->lx_infopage->memserv_id = l4dm_memphys_find_dmphys();
#ifdef USE_TASKLIB
  task->lx_infopage->parent_id = l4task_get_server();
#endif

	l4ts_create_task(&(task->task_id),
	                 task->start_eip,
	                 task->start_esp,
	                 0xff,
	                 &pager,
	                 L4THREAD_DEFAULT_PRIO,
	                 "L4Linux",
	                 0
	                );
	
}

void
if_l4dm_lxfreeze_loaded_component(CORBA_Object _dice_corba_obj,
                                  const l4_threadid_t *signal_id,
                                  CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t *task;

	if((task = lx_task_get_loading()) == NULL)
	{
		enter_kdebug("Nothing loading");
	}

	task->task_id = *_dice_corba_obj;
	task->l4x_signal_id = *signal_id;
	task->state = TASK_LOADED;
	LOGd(DBG_VERB, "New L4Linux instance ["l4util_idfmt"]", 
	     l4util_idstr(task->task_id));
}

void
if_l4dm_lxfreeze_signal_id_component(CORBA_Object _dice_corba_obj,
                                     const l4_threadid_t *signal_id,
                                     CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t * task;

	if(lx_task_get(*_dice_corba_obj, &task) < 0)
		return;
	task->l4x_signal_id = *signal_id;
}

/*******************************************************************************
 *** required DSM interface implementations 
 ******************************************************************************/

long
if_l4dm_generic_fault_component(CORBA_Object _dice_corba_obj,
                                 unsigned long ds_id,
                                 unsigned long offset,
                                 l4_snd_fpage_t *page,
                                 CORBA_Server_Environment *_dice_corba_env)
{
	l4_addr_t addr;
	int rw = offset & 2;
	int error;

	if(DBG_FAULT_RO || rw)
		LOGd(DBG_FAULT, 
		"["l4util_idfmt"] Got fault call ds_id: %lu offset: 0x%08lx (%s)",
		l4util_idstr(*_dice_corba_obj), ds_id, offset, (rw)?"rw":"ro");
	
	if(ds_id == 22)
	LOG(
		"["l4util_idfmt"] Got fault call ds_id: %lu offset: 0x%08lx (%s)",
		l4util_idstr(*_dice_corba_obj), ds_id, offset, (rw)?"rw":"ro");

	offset &= L4_PAGEMASK;
	page->snd_base = 0;
	page->fpage = l4_fpage(0,0,0,0);
	
	error = lx_page_fault((l4_taskid_t)*_dice_corba_obj, ds_id, offset, 
	                      L4_PAGESIZE, rw, &addr);
	if(error) 
	{
		LOGd(DBG_ERROR, "["l4util_idfmt"] Pagefault error %s (%d)", 
		     l4util_idstr(*_dice_corba_obj), l4env_errstr(error), error);
		return error;
	}

	if(rw)
		page->fpage = l4_fpage(addr, L4_LOG2_PAGESIZE, L4_FPAGE_RW, L4_FPAGE_MAP);
	else	
		page->fpage = l4_fpage(addr, L4_LOG2_PAGESIZE, L4_FPAGE_RO, L4_FPAGE_MAP);
	
	if(DBG_FAULT_RO || rw)	
		LOGd(DBG_FAULT, "Mapping addr: %08lx", addr);
	
	return 0;
	
}

/* simple implementation, no recv_offset handling respectively fpage.snd_base
 * handling supported, yet.
 */
long
if_l4dm_generic_map_component(CORBA_Object _dice_corba_obj,
                              unsigned long ds_id,
                              unsigned long offset,
                              unsigned long size,
                              unsigned long rcv_size2,
                              unsigned long rcv_offs,
                              unsigned long flags,
                              l4_snd_fpage_t *page,
                              CORBA_Server_Environment *_dice_corba_env)
{ 
	l4_addr_t addr;
	int error, map_size;
	int rw = flags & L4DM_WRITE;

	LOGd(DBG_MAP, "["l4util_idfmt"] Got map call ds_id: %lu offset: 0x%08lx " \
	     "size: 0x%08lx(%s)",
	     l4util_idstr(*_dice_corba_obj), ds_id, offset, size, (rw)?"rw":"ro");
	
/* set dummy page */
	page->snd_base = 0;
	page->fpage = l4_fpage(0, 0, 0, 0);
	offset &= L4_PAGEMASK;
	
	error = lx_page_fault((l4_taskid_t)*_dice_corba_obj, ds_id, offset, size, 
	                      rw, &addr);
	if(error) 
	{
		LOGd(DBG_ERROR, "["l4util_idfmt"] Pagefault error %s (%d)", 
		     l4util_idstr(*_dice_corba_obj), l4env_errstr(error), error);
		return error;
	}
	
	/* max. alignment = min(receive window, requested size, and local page
	 * alignment) */
	map_size = (rcv_size2 < l4util_bsf(size))?rcv_size2:l4util_bsf(size);
	map_size = (l4util_bsf(addr) < map_size)?l4util_bsf(addr):map_size;

	
	if(rw) 
		page->fpage = l4_fpage(addr, map_size, L4_FPAGE_RW, L4_FPAGE_MAP);
	else
		page->fpage = l4_fpage(addr, map_size, L4_FPAGE_RO, L4_FPAGE_MAP);
	
	LOGd(DBG_MAP, "map addr: %08lx recv2: %08lx rcv_offs: %08lx SIZE: %08x\n",
	     addr, rcv_size2, rcv_offs, l4util_bsf(size));

	return 0;
}


/*******************************************************************************
 * OPEN/CLOSE + needed mem/generic interface calls
 ******************************************************************************/

long lxif_open(l4_taskid_t task_id,
               unsigned long size,
               unsigned long align,
               unsigned long flags,
               const char * name,
               lx_ds_t ** lxds)
{
	int err;
	l4dm_dataspace_t ds_local;
	dsmlib_ds_desc_t * ds_desc;
	void * map_addr;
	lx_task_t * task;
	
	task = __get_task(task_id);
	if(task == NULL) 
		return -L4_ENOMEM;
	
	size = (size + L4_PAGESIZE - 1) & L4_PAGEMASK;
	if((err = l4dm_mem_open(L4DM_DEFAULT_DSM,
	                        size, align, flags, name, &ds_local)))
		return err;
	
	if((ds_desc = dsmlib_create_dataspace()) == NULL) 
	{
		l4dm_close(&ds_local);
		return -L4_ENOMEM;
	}

	if((err = lx_ds_add_ds(task, ds_local, ds_desc, size, &map_addr)) < 0)
	{
		dsmlib_release_dataspace(ds_desc);
		l4dm_close(&ds_local);
		return err;
	}
	
	lx_ds_get_pos(task, err, lxds);
	
	dsmlib_set_owner(ds_desc, task_id);
	dsmlib_set_name(ds_desc, name);
	
	LOGd(DBG_OPEN, "["l4util_idfmt"] Open request for %s size: %lu B (%lu)", 
	     l4util_idstr(task_id), name, size, lx_ds_get_id(*lxds));
	return 0;
}

long
if_l4dm_mem_open_component(CORBA_Object _dice_corba_obj,
                           unsigned long size,
                           unsigned long align,
                           unsigned long flags,
                           const char* name,
                           l4dm_dataspace_t *ds,
                           CORBA_Server_Environment *_dice_corba_env)
{
	int err;
	lx_ds_t * lxds;
	
	LOGd(DBG_OPEN, "["l4util_idfmt"] Open component", 
	     l4util_idstr(*_dice_corba_obj));
	
	if((err = lxif_open(*_dice_corba_obj, size, align, flags, name, &lxds)))
		return err;
	
	dstolxds[dsmlib_get_id(lxds->ds_desc) - 1] = lxds;
	ds->id = lx_ds_get_id(lxds);
	ds->manager = l4_myself();
	return 0;
}


long
if_l4dm_mem_resize_component(CORBA_Object _dice_corba_obj,
                             unsigned long ds_id,
                             unsigned long new_size,
                             CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t * task;
	lx_ds_t * lxds;

	int err;
	l4_addr_t map_addr;
	
	task = __get_task(*_dice_corba_obj);
	
	new_size = (new_size + L4_PAGESIZE - 1) & L4_PAGEMASK;
	lx_ds_get(task, ds_id, &lxds);
	
	LOGd(DBG_RESIZE, "["l4util_idfmt"]Resize called for %lu old %08x, new %08lx",
		l4util_idstr(*_dice_corba_obj), ds_id, lxds->size, new_size);

	if(new_size == lxds->size)
	{
		LOGd(DBG_RESIZE, "Same size -> continue");
		return 0;
	}

	if((err = l4dm_mem_resize(&(lxds->ds), new_size)))
		return err;
	
	if(new_size > lxds->size)
	{
		if((err = l4rm_attach(&(lxds->ds), new_size, 0,
		                      L4RM_MAP|L4DM_RW|L4RM_LOG2_ALIGNED, 
		                      (void*)&map_addr)))
			return err;
		
		lxds->map_addr = map_addr;
		lxds->flags   |= LX_RESIZE;
	}
	
	lxds->size = new_size;
	return 0;
}


long
if_l4dm_generic_share_component(CORBA_Object _dice_corba_obj,
                                unsigned long ds_id,
                                const l4_threadid_t *client,
                                unsigned long flags,
                                CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t * task;
	lx_ds_t * lxds;
	l4_uint32_t rights  = flags & L4DM_RIGHTS_MASK;
	int err;
	
	task = __get_task(*_dice_corba_obj);
	
	if((err = lx_ds_get(task, ds_id, &lxds)) < 0)
		return err;
	
	LOGd(DBG_OPEN, "Sharing ds_id %lu with ["l4util_idfmt"] rights %u",
	     ds_id, l4util_idstr(*client), rights);
	
	return dsmlib_add_client(lxds->ds_desc, *client, rights);
}


long
if_l4dm_generic_transfer_component(CORBA_Object _dice_corba_obj,
                                   unsigned long ds_id,
                                   const l4_threadid_t *new_owner,
                                   CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t * task;
	lx_ds_t * lxds;
	int err;
	
	LOGd(DBG_TRANSFER, "Transfering owner ship of ds_id %lu from "l4util_idfmt \
	     " to "l4util_idfmt"\n", 
	     ds_id, l4util_idstr(*_dice_corba_obj), l4util_idstr(*new_owner));
	
	task = __get_task(*_dice_corba_obj);
	
	if((err = lx_ds_get(task, ds_id, &lxds)) < 0)
		return err;
	dsmlib_set_owner(lxds->ds_desc, *new_owner);
	return 0;
}


long
if_l4dm_generic_close_component(CORBA_Object _dice_corba_obj,
                                unsigned long ds_id,
                                CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t * task;
	
	LOGd(DBG_CLOSE, "["l4util_idfmt"] Close request for ds_id %lu", 
	     l4util_idstr(*_dice_corba_obj), ds_id);

	/* don't let the loader close dataspaces after Linux suspend */
	if(lx_task_get(*_dice_corba_obj, &task) < 0 
	   && (task = lx_task_get_loading()) == NULL)
		return 0;

	return lxif_close_ds(task, ds_id);
}


long lxif_close_ds(lx_task_t * task, unsigned long ds_id)
{

	lx_ds_t * lxds;
	int err;

	if((err = lx_ds_get(task, ds_id, &lxds)) < 0)
		return err;
	
	if(task_has_state(task, TASK_LOADED))
		dstolxds[ds_id - 1] = NULL;
	
	LOGd(DBG_CLOSE, "Closing ds %s (%u) physical id %u\n",
	     dsmlib_get_name(lxds->ds_desc), dsmlib_get_id(lxds->ds_desc),
	     lxds->ds.id);
	
	lx_ds_del(task, ds_id);
	return 0;
}


long
if_l4dm_mem_phys_addr_component(CORBA_Object _dice_corba_obj,
                                unsigned long ds_id,
                                unsigned long offset,
                                l4_size_t size,
                                unsigned long *paddr,
                                l4_size_t *psize,
                                CORBA_Server_Environment *_dice_corba_env)
{
	lx_ds_t * lxds;
	lx_task_t * task;
	int err;
	
	task = __get_task(*_dice_corba_obj);
	if((err = lx_ds_get(task, ds_id, &lxds)) < 0)
		return err;

	return l4dm_mem_ds_phys_addr(&(lxds->ds), offset, size, paddr, psize);
}


long
if_l4dm_mem_size_component(CORBA_Object _dice_corba_obj,
                           unsigned long ds_id,
                           l4_size_t *size,
                           CORBA_Server_Environment *_dice_corba_env)
{
	lx_ds_t * lxds;
	
	if((lxds = dstolxds[ds_id - 1]) == NULL)
		return -L4_EINVAL;

	if(!dsmlib_check_rights(lxds->ds_desc, *_dice_corba_obj, L4DM_READ))
		return -L4_EPERM;
	
	*size = lxds->size;
	return 0;
}

/*******************************************************************************
 *** EXIT EVENT HANDLING
 ******************************************************************************/

void lxif_exit_task(l4_taskid_t task_id)
{
	l4semaphore_down(&lx_clean_sem);
	lx_task_del(task_id);
	l4semaphore_up(&lx_clean_sem);
}

void
if_l4dm_lxfreeze_linux_exit_component(CORBA_Object _dice_corba_obj,
                                      const l4_taskid_t *src_id,
                                      CORBA_Server_Environment *_dice_corba_env)
{
	lx_task_t * task;
	int error;
	/* one of our tasks ? */
	if((error = lx_task_get(*src_id, &task)) == -L4_ENOTFOUND)
		return;
	LOG_printf("Got exit event src_id "l4util_idfmt"\n", l4util_idstr(*src_id));
	if(task_has_state(task, TASK_SEND_PRECOPY))
	{
		task_set_state(task, TASK_SEND_POSTCOPY);
		if_l4dm_lxmir_suspend_done_send(&(task->mir_id), 0, &(task->start_eip),
		                                &(task->start_esp), _dice_corba_env);
		return;
	}
	
	if(!task_is_ready(task))
		return;
	
	lxif_exit_task(*src_id);
	

	LOGd(DBG_CLEAN, "Cleaning task "l4util_idfmt"\n", l4util_idstr(*src_id));
	if(task->state == TASK_MIGRATED)
		l4ts_free_task(src_id);
}

/* not supported */
void
if_l4dm_lxfreeze_hard_exit_component (CORBA_Object _dice_corba_obj,
                                      const l4_taskid_t *task_id,
                                      CORBA_Server_Environment *_dice_corba_env)
{}
