/*
 * \brief   Memory subsystem
 * \author  Thomas Friebel <tf13@os.inf.tu-dresden.de>
 * \author  Christian Helmuth <ch12@os.inf.tu-dresden.de>
 * \date    2006-11-03
 *
 * The memory subsystem provides the backing store for DMA-able memory via
 * large malloc and slabs.
 *
 * FIXME check thread-safety and add locks where appropriate
 */

#include <l4/dde/ddekit/memory.h>
#include <l4/dde/ddekit/panic.h>
#include <l4/dde/ddekit/pgtab.h>
#include <l4/dde/ddekit/printf.h>

#include <l4/slab/slab.h>
#include <l4/dm_mem/dm_mem.h>
#include <l4/util/atomic.h>
#include <l4/util/util.h>


/****************
 ** Page cache **
 ****************/

/* FIXME revisit two-linked-lists approach */

/* page cache to minimize allocations from external servers */
struct ddekit_pcache
{
	struct ddekit_pcache *next;
	void *page;
};


/* head of single-linked list containing used page-cache entries */
static struct ddekit_pcache *pcache_used;

/* head of single-linked list containing free page-cache entries */
static struct ddekit_pcache *pcache_free;

/* maximum number of pages to cache. defaults to a minimum of 1 page
 * because having none hits the performance too hard. */
static l4_uint32_t pcache_num_entries = 1;


/**
 * Setup page cache for all slabs
 *
 * \param pages  maximal number of memory pages
 *
 * If the maximal number of unused pages is exceeded, subsequent deallocation
 * will be freed at the memory server. This page cache caches pages from all
 * slabs.
 */
void ddekit_slab_setup_page_cache(unsigned pages)
{
	/* FIXME just allowing to grow at the moment */
	while (pcache_num_entries < pages) {
		struct ddekit_pcache *new;

		/* create new list element */
		new = (struct ddekit_pcache *) ddekit_simple_malloc(sizeof(*new));

		/* insert into list of unused cache entries */
		do {
			new->next = pcache_free;
		} while (!l4util_cmpxchg32((l4_uint32_t*) &pcache_free, (l4_uint32_t) new->next, (l4_uint32_t) new));

		/* increment number of list elements */
		l4util_inc32(&pcache_num_entries);
	}
}


/*******************************
 ** Slab cache implementation **
 *******************************/

/* ddekit slab facilitates l4slabs */
struct ddekit_slab
{
	l4slab_cache_t cache;
};


/**
 * Grow slab cache
 */
static void *_slab_grow(l4slab_cache_t *cache, void **data)
{
	/* the page(s) to be returned */
	void *res = NULL;

	/* free cache entry, will be used afterwards */
	struct ddekit_pcache *head = NULL;

	/* We don't reuse pages for slabs > 1 page, because this makes caching
	 * somewhat harder. */
	if (cache->slab_size <= L4_PAGESIZE) {
		/* try to aquire cached page */
		do {
			head = pcache_used;
			if (!head) break;
		} while (!l4util_cmpxchg32((l4_uint32_t*) &pcache_used, (l4_uint32_t) head, (l4_uint32_t) head->next));
	}

	if (head) {
		/* use cached page */

		res = head->page;

		/* this cache entry is now available */
		do {
			head->next = pcache_free;
		} while (!l4util_cmpxchg32((l4_uint32_t*) &pcache_free, (l4_uint32_t) head->next, (l4_uint32_t) head));
	} else {
		/* allocate new page at memory server */

		int err;
		l4_size_t tmp;
		l4dm_mem_addr_t dm_paddr;
		int num_pages = cache->slab_size / L4_PAGESIZE;

		/* allocate and map new page(s) */
		res = l4dm_mem_allocate_named(num_pages * L4_PAGESIZE, L4DM_PINNED | L4RM_MAP | L4RM_LOG2_ALIGNED,
		                              "ddekit slab");
		if (res == NULL)
			ddekit_debug("__grow: error allocating a new page");

		err = l4dm_mem_phys_addr(res, num_pages, &dm_paddr, 1, &tmp);
		if (err != 1)
			ddekit_debug("__grow: error getting physical address of new page!");

		ddekit_pgtab_set_region(res, dm_paddr.addr, num_pages, PTE_TYPE_UMA);
	}

	/* save pointer to cache in page for ddekit_slab_get_slab() */
	*data = cache;

	return res;
}

/**
 * Shrink slab cache
 */
static void _slab_shrink(l4slab_cache_t *cache, void *page, void *data)
{
	/* cache deallocated page here */
	struct ddekit_pcache *head = NULL;

	/* we do not return slabs to the page cache, if they are larger than one page */
	if (cache->slab_size <= L4_PAGESIZE)
		/* try to aquire free cache entry */
		do {
			head = pcache_free;
			if (!head) break;
		} while (!l4util_cmpxchg32((l4_uint32_t*) &pcache_free, (l4_uint32_t) head, (l4_uint32_t) head->next));

	if (head) {
		/* use free cache entry to cache page */

		/* set the page info */
		head->page = page;

		/* this cache entry contains a free page */
		do {
			head->next = pcache_used;
		} while (! l4util_cmpxchg32((l4_uint32_t*) &pcache_used, (l4_uint32_t) head->next, (l4_uint32_t) head));
	} else {
		/* cache is full */
		
		/* unset pte */
		ddekit_pgtab_clear_region(page, PTE_TYPE_UMA);

		/* free page */
		l4dm_mem_release(page);
	}
}


/**
 * Allocate object in slab
 */
void *ddekit_slab_alloc(struct ddekit_slab * slab)
{
	return l4slab_alloc(&slab->cache);
}


/**
 * Free object in slab
 */
void  ddekit_slab_free(struct ddekit_slab * slab, void *objp)
{
	l4slab_free(&slab->cache, objp);
}


/**
 * Store user pointer in slab cache
 */
void  ddekit_slab_set_data(struct ddekit_slab * slab, void *data)
{
	l4slab_set_data(&slab->cache, data);
}


/**
 * Read user pointer from slab cache
 */
void *ddekit_slab_get_data(struct ddekit_slab * slab)
{
	return l4slab_get_data(&slab->cache);
}


/**
 * Destroy slab cache
 *
 * \param slab  pointer to slab cache structure
 */
void  ddekit_slab_destroy (struct ddekit_slab * slab)
{
	l4slab_destroy(&slab->cache);
	ddekit_simple_free(slab);
}


/**
 * Initialize slab cache
 *
 * \param size  size of cache objects
 *
 * \return pointer to new slab cache or 0 on error
 */
struct ddekit_slab * ddekit_slab_init(unsigned size)
{
	struct ddekit_slab * slab;
	int err;

	slab = (struct ddekit_slab *) ddekit_simple_malloc(sizeof(*slab));
	err = l4slab_cache_init(&slab->cache, size, 0, _slab_grow, _slab_shrink);
	if (err) {
		ddekit_debug("error initializing cache");
		ddekit_simple_free(slab);
		return 0;
	}
	
	return slab;
}


/**********************************
 ** Large block memory allocator **
 **********************************/

/**
 * Free large block of memory
 *
 * This is no useful for allocation < page size.
 */
void ddekit_large_free(void *objp)
{
	/* clear PTEs */
	ddekit_pgtab_clear_region(objp, PTE_TYPE_LARGE);

	/* release */
	l4dm_mem_release(objp);
}


/**
 * Allocate large block of memory
 *
 * This is no useful for allocation < page size.
 */
void *ddekit_large_malloc(int size)
{
	void *res;
	int err;
	l4_size_t tmp;
	int pages;
	l4dm_mem_addr_t dm_paddr;

	size  = l4_round_page(size);
	pages = size >> L4_PAGESHIFT;

	res = l4dm_mem_allocate_named(size,
	                              L4DM_CONTIGUOUS | L4DM_PINNED |
	                              L4RM_MAP | L4RM_LOG2_ALIGNED,
	                              "ddekit mem");
	if (! res)
		return NULL;

	err = l4dm_mem_phys_addr(res, 1, &dm_paddr, 1, &tmp);
	if (err != 1)
		ddekit_debug("ddekit_large_malloc: error getting physical address of new memory!\n");

	ddekit_pgtab_set_region(res, dm_paddr.addr, pages, PTE_TYPE_LARGE);

	return res;
}


/**
 * Allocate large block of memory (special interface)
 *
 * This is no useful for allocation < page size.
 *
 * FIXME implementation missing...
 */
void *ddekit_contig_malloc(unsigned long size,
                           unsigned long low, unsigned long high,
                           unsigned long alignment, unsigned long boundary)
{
#if 0
	void *res;
	int err;
	int pages;
	l4_size_t tmp;
	l4dm_mem_addr_t dm_paddr;

	size  = l4_round_page(size);
	pages = size >> L4_PAGESHIFT;

	res = l4dm_mem_allocate_named(size, L4DM_CONTIGUOUS | L4DM_PINNED | L4RM_MAP | L4RM_LOG2_ALIGNED);

	if (res) {
		/* check if res meets low/high/alignment/boundary
		 * and panic if it is not the case
		 * XXXY
		 */

		/* get physical address */
		err = l4dm_mem_phys_addr(res, 1, &dm_paddr, 1, &tmp);
		if (err != 1)
			ddekit_debug("contigmalloc: error getting physical address of new page!\n");
	
		/* set PTE */
		ddekit_set_ptes(res, dm_paddr.addr, pages, PTE_TYPE_CONTIG);
	}

	return res;
#else
	ddekit_debug("%s: not implemented\n", __func__);
	return 0;
#endif
}
