/*
 *  linux/arch/i386/traps.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *
 *  Pentium III FXSR, SSE support
 *	Gareth Hughes <gareth@valinux.com>, May 2000
 */

/*
 * 'Traps.c' handles hardware traps and faults after we have saved some
 * state in 'asm.s'.
 */
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/timer.h>
#include <linux/mm.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/highmem.h>
#include <linux/kallsyms.h>
#include <linux/ptrace.h>
#include <linux/utsname.h>
#include <linux/kprobes.h>
#include <linux/kexec.h>
#include <linux/unwind.h>

#ifdef CONFIG_EISA
#include <linux/ioport.h>
#include <linux/eisa.h>
#endif

#ifdef CONFIG_MCA
#include <linux/mca.h>
#endif

#include <asm/processor.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/atomic.h>
#include <asm/debugreg.h>
#include <asm/desc.h>
#include <asm/i387.h>
#include <asm/nmi.h>
#include <asm/unwind.h>
#include <asm/smp.h>
#include <asm/arch_hooks.h>
#include <asm/kdebug.h>

#include <asm/api/ids.h>
#include <asm/api/macros.h>

#include <asm/generic/memory.h>
#include <asm/generic/kthreads.h>
#include <asm/generic/task.h>

#include <l4/sys/syscalls.h>
#include <asm/l4lxapi/thread.h>

#include <linux/irq.h>
#include <linux/module.h>

struct desc_struct default_ldt[] = { { 0, 0 }, { 0, 0 }, { 0, 0 },
		{ 0, 0 }, { 0, 0 } };

/* Do we ignore FPU interrupts ? */
char ignore_fpu_irq = 0;

void divide_error(void);
void debug(void);
void nmi(void);
void int3(void);
void overflow(void);
void bounds(void);
void invalid_op(void);
void device_not_available(void);
void double_fault(void);
void coprocessor_segment_overrun(void);
void invalid_TSS(void);
void segment_not_present(void);
void stack_segment(void);
void general_protection(void);
void page_fault(void);
void coprocessor_error(void);
void simd_coprocessor_error(void);
void alignment_check(void);
void spurious_interrupt_bug(void);
#ifdef CONFIG_X86_MCE
void machine_check(void);
#endif

static int kstack_depth_to_print = 24;
#ifdef CONFIG_STACK_UNWIND
static int call_trace = 1;
#else
#define call_trace (-1)
#endif
ATOMIC_NOTIFIER_HEAD(i386die_chain);

int register_die_notifier(struct notifier_block *nb)
{
	vmalloc_sync_all();
	return atomic_notifier_chain_register(&i386die_chain, nb);
}
EXPORT_SYMBOL(register_die_notifier); /* used modular by kdb */

int unregister_die_notifier(struct notifier_block *nb)
{
	return atomic_notifier_chain_unregister(&i386die_chain, nb);
}
EXPORT_SYMBOL(unregister_die_notifier); /* used modular by kdb */

static inline int valid_stack_ptr(struct thread_info *tinfo, void *p)
{
	return	p > (void *)tinfo &&
		p < (void *)tinfo + THREAD_SIZE - 3;
}

/*
 * Print one address/symbol entries per line.
 */
static inline void print_addr_and_symbol(unsigned long addr, char *log_lvl)
{
	printk(" [<%08lx>] ", addr);

	print_symbol("%s\n", addr);
}

static inline unsigned long print_context_stack(struct thread_info *tinfo,
				unsigned long *stack, unsigned long ebp,
				char *log_lvl)
{
	unsigned long addr;

#ifdef	CONFIG_FRAME_POINTER
	while (valid_stack_ptr(tinfo, (void *)(ebp + 4))) {
		addr = *(unsigned long *)(ebp + 4);
		print_addr_and_symbol(addr, log_lvl);
		/*
		 * break out of recursive entries (such as
		 * end_of_stack_stop_unwind_function):
	 	 */
		if (ebp == *(unsigned long *)ebp)
			break;
		ebp = *(unsigned long *)ebp;
	}
#else
	while (valid_stack_ptr(tinfo, stack)) {
		addr = *stack++;
		if (__kernel_text_address(addr))
			print_addr_and_symbol(addr, log_lvl);
	}
#endif
	return ebp;
}

static asmlinkage int
show_trace_unwind(struct unwind_frame_info *info, void *log_lvl)
{
	int n = 0;

	while (unwind(info) == 0 && UNW_PC(info)) {
		n++;
		print_addr_and_symbol(UNW_PC(info), log_lvl);
		if (arch_unw_user_mode(info))
			break;
	}
	return n;
}

static void show_trace_log_lvl(struct task_struct *task, struct pt_regs *regs,
			       unsigned long *stack, char *log_lvl)
{
	unsigned long ebp;

	if (!task)
		task = current;

	if (call_trace >= 0) {
		int unw_ret = 0;
		struct unwind_frame_info info;

		if (regs) {
			if (unwind_init_frame_info(&info, task, regs) == 0)
				unw_ret = show_trace_unwind(&info, log_lvl);
		} else if (task == current)
			unw_ret = unwind_init_running(&info, show_trace_unwind, log_lvl);
		else {
			if (unwind_init_blocked(&info, task) == 0)
				unw_ret = show_trace_unwind(&info, log_lvl);
		}
		if (unw_ret > 0) {
			if (call_trace == 1 && !arch_unw_user_mode(&info)) {
				print_symbol("DWARF2 unwinder stuck at %s\n",
					     UNW_PC(&info));
				if (UNW_SP(&info) >= PAGE_OFFSET) {
					printk("Leftover inexact backtrace:\n");
					stack = (void *)UNW_SP(&info);
				} else
					printk("Full inexact backtrace again:\n");
			} else if (call_trace >= 1)
				return;
			else
				printk("Full inexact backtrace again:\n");
		} else
			printk("Inexact backtrace:\n");
	}

	if (task == current) {
		/* Grab ebp right from our regs */
		asm ("movl %%ebp, %0" : "=r" (ebp) : );
	} else {
		/* ebp is the last but one reg pushed by switch_to */
		ebp = *(unsigned long *) (task->thread.kernel_sp + sizeof(long));
	}

	while (1) {
		struct thread_info *context;
		context = (struct thread_info *)
			((unsigned long)stack & (~(THREAD_SIZE - 1)));
		ebp = print_context_stack(context, stack, ebp, log_lvl);
		stack = (unsigned long*)context->previous_esp;
		if (!stack)
			break;
		printk("%s =======================\n", log_lvl);
	}
}

void show_trace(struct task_struct *task, struct pt_regs *regs, unsigned long * stack)
{
	show_trace_log_lvl(task, regs, stack, "");
}

static void show_stack_log_lvl(struct task_struct *task, struct pt_regs *regs,
			       unsigned long *esp, char *log_lvl)
{
	unsigned long *stack;
	int i;

	if (esp == NULL) {
		if (task)
			esp = (unsigned long*)task->thread.kernel_sp;
		else
			esp = (unsigned long *)&esp;
	}

	stack = esp;
	for(i = 0; i < kstack_depth_to_print; i++) {
		if (kstack_end(stack))
			break;
		if (i && ((i % 8) == 0))
			printk("\n%s       ", log_lvl);
		printk("%08lx ", *stack++);
	}
	printk("\n%sCall Trace:\n", log_lvl);
	show_trace_log_lvl(task, regs, esp, log_lvl);
}

void show_stack(struct task_struct *task, unsigned long *esp)
{
	printk("       ");
	show_stack_log_lvl(task, NULL, esp, "");
}

/*
 * The architecture-independent dump_stack generator
 */
void dump_stack(void)
{
	unsigned long stack;

	show_trace(current, NULL, &stack);
}

EXPORT_SYMBOL(dump_stack);

void show_registers(struct pt_regs *regs)
{
	int i;
	int in_kernel = 1;
	unsigned long esp;
	unsigned short ss;

	esp = (unsigned long) (1+regs);
	ss = 0;

	print_modules();
	printk(KERN_EMERG "CPU:    %d\nEIP:    %04x:[<%08lx>]    %s VLI\n"
			"EFLAGS: %08lx   (%s %.*s) \n",
		smp_processor_id(), 0xffff & regs->xcs, regs->eip,
		print_tainted(), regs->eflags, system_utsname.release,
		(int)strcspn(system_utsname.version, " "),
		system_utsname.version);
	print_symbol(KERN_EMERG "EIP is at %s\n", regs->eip);
	printk(KERN_EMERG "eax: %08lx   ebx: %08lx   ecx: %08lx   edx: %08lx\n",
		regs->eax, regs->ebx, regs->ecx, regs->edx);
	printk(KERN_EMERG "esi: %08lx   edi: %08lx   ebp: %08lx   esp: %08lx\n",
		regs->esi, regs->edi, regs->ebp, esp);
	printk(KERN_EMERG "ds: %04x   es: %04x   ss: %04x\n",
		regs->xds & 0xffff, regs->xes & 0xffff, ss);
	printk(KERN_EMERG "Process %.*s (pid: %d, ti=%p task=%p task.ti=%p)",
		TASK_COMM_LEN, current->comm, current->pid,
		current_thread_info(), current, current->thread_info);
	/*
	 * When in-kernel, we also print out the stack and code at the
	 * time of the fault..
	 */
	if (in_kernel) {
		u8 __user *eip;

		printk("\n" KERN_EMERG "Stack: ");
		show_stack_log_lvl(NULL, regs, (unsigned long *)esp, KERN_EMERG);

		printk(KERN_EMERG "Code: ");

		eip = (u8 __user *)regs->eip - 43;
		for (i = 0; i < 64; i++, eip++) {
			unsigned char c;

#if 0
			if (eip < (u8 __user *)PAGE_OFFSET || __get_user(c, eip)) {
				printk(" Bad EIP value.");
				break;
			}
#else
			c = *(unsigned char *)eip;
#endif
			if (eip == (u8 __user *)regs->eip)
				printk("<%02x> ", c);
			else
				printk("%02x ", c);
		}
	}
	printk("\n");
}	

static void handle_BUG(struct pt_regs *regs)
{
	unsigned long eip = regs->eip;
	unsigned short ud2;

	if (eip < PAGE_OFFSET)
		return;
	if (__get_user(ud2, (unsigned short __user *)eip))
		return;
	if (ud2 != 0x0b0f)
		return;

	printk(KERN_EMERG "------------[ cut here ]------------\n");

#ifdef CONFIG_DEBUG_BUGVERBOSE
	do {
		unsigned short line;
		char *file;
		char c;

		if (__get_user(line, (unsigned short __user *)(eip + 2)))
			break;
		if (__get_user(file, (char * __user *)(eip + 4)) ||
		    (unsigned long)file < PAGE_OFFSET || __get_user(c, file))
			file = "<bad filename>";

		printk(KERN_EMERG "kernel BUG at %s:%d!\n", file, line);
		return;
	} while (0);
#endif
	printk(KERN_EMERG "Kernel BUG at [verbose debug info unavailable]\n");
}

/* This is gone through when something in the kernel
 * has done something bad and is about to be terminated.
*/
void die(const char * str, struct pt_regs * regs, long err)
{
	static struct {
		spinlock_t lock;
		u32 lock_owner;
		int lock_owner_depth;
	} die = {
		.lock =			SPIN_LOCK_UNLOCKED,
		.lock_owner =		-1,
		.lock_owner_depth =	0
	};
	static int die_counter;
	unsigned long flags;

	oops_enter();

	if (die.lock_owner != raw_smp_processor_id()) {
		console_verbose();
		spin_lock_irqsave(&die.lock, flags);
		die.lock_owner = smp_processor_id();
		die.lock_owner_depth = 0;
		bust_spinlocks(1);
	}
	else
		local_save_flags(flags);

	if (++die.lock_owner_depth < 3) {
		int nl = 0;
		unsigned long esp;
		unsigned short ss;

		handle_BUG(regs);
		printk(KERN_EMERG "%s: %04lx [#%d]\n", str, err & 0xffff, ++die_counter);
#ifdef CONFIG_PREEMPT
		printk(KERN_EMERG "PREEMPT ");
		nl = 1;
#endif
#ifdef CONFIG_SMP
		if (!nl)
			printk(KERN_EMERG);
		printk("SMP ");
		nl = 1;
#endif
#ifdef CONFIG_DEBUG_PAGEALLOC
		if (!nl)
			printk(KERN_EMERG);
		printk("DEBUG_PAGEALLOC");
		nl = 1;
#endif
		if (nl)
			printk("\n");
		if (notify_die(DIE_OOPS, str, regs, err,
					current->thread.trap_no, SIGSEGV) !=
				NOTIFY_STOP) {
			show_registers(regs);
			/* Executive summary in case the oops scrolled away */
			esp = (unsigned long) (&regs->esp);
			savesegment(ss, ss);
			if (user_mode(regs)) {
				esp = regs->esp;
				ss = regs->xss & 0xffff;
			}
			printk(KERN_EMERG "EIP: [<%08lx>] ", regs->eip);
			print_symbol("%s", regs->eip);
			printk(" SS:ESP %04x:%08lx\n", ss, esp);
		}
		else
			regs = NULL;
  	} else
		printk(KERN_EMERG "Recursive die() failure, output suppressed\n");

	bust_spinlocks(0);
	die.lock_owner = -1;
	spin_unlock_irqrestore(&die.lock, flags);

	if (!regs)
		return;

	if (kexec_should_crash(current))
		crash_kexec(regs);

	if (in_interrupt())
		panic("Fatal exception in interrupt");

	if (panic_on_oops)
		panic("Fatal exception");

	oops_exit();
	do_exit(SIGSEGV);
}

#ifdef CONFIG_X86_F00F_BUG
void __init trap_init_f00f_bug(void)
{
	/* for CPU code */
}
#endif

static inline void die_if_kernel(const char * str, struct pt_regs * regs, long err)
{
  	die(str, regs, err);
}

void do_kernel_error(const char * str, struct pt_regs * regs, long error_code)
{
/* 	current->thread.error_code = error_code; */
/* 	current->thread.trap_no = trapnr; */
	die(str, regs, error_code);
}

void unexpected_ret_from_exception(const char * str, struct pt_regs * regs, long error_code)
{
  	printk("unexpected_ret_from_exception: %s returned\n", str);
	panic("unexpected_ret_from_exception");
}

#define DO_ERROR(trapnr, signr, die_nr, str, name, tsk)			\
fastcall void __kprobes do_##name(struct pt_regs * regs, long error_code) \
{									\
	if (notify_die(die_nr, str, regs, error_code, trapnr, signr)	\
						== NOTIFY_STOP)		\
		return;							\
	tsk->thread.error_code = error_code;				\
	tsk->thread.trap_no = trapnr;					\
	force_sig(signr, tsk);						\
}

DO_ERROR( 0, SIGFPE,  DIE_TRAP,  "divide error", divide_error, current)
DO_ERROR( 1, SIGTRAP, DIE_DEBUG, "debug", debug, current)
DO_ERROR( 3, SIGTRAP, DIE_INT3,  "int3", int3, current)
DO_ERROR( 4, SIGSEGV, DIE_TRAP,  "overflow", overflow, current)
DO_ERROR( 5, SIGSEGV, DIE_TRAP,  "bounds", bounds, current)
DO_ERROR( 6, SIGILL,  DIE_TRAP,  "invalid opcode", invalid_op, current)
DO_ERROR( 7, SIGSEGV, DIE_TRAP,  "device not available", device_not_available, current)
DO_ERROR( 8, SIGSEGV, DIE_TRAP,  "double fault", double_fault, current)
DO_ERROR( 9, SIGFPE,  DIE_TRAP,  "coprocessor segment overrun", coprocessor_segment_overrun, current)
DO_ERROR(10, SIGSEGV, DIE_TRAP,  "invalid TSS", invalid_TSS, current)
DO_ERROR(11, SIGBUS,  DIE_TRAP,  "segment not present", segment_not_present, current)
DO_ERROR(12, SIGBUS,  DIE_TRAP,  "stack segment", stack_segment, current)
DO_ERROR(13, SIGSEGV, DIE_TRAP,  "general protection", general_protection, current)
DO_ERROR(17, SIGBUS,  DIE_TRAP,  "alignment check", alignment_check, current)
#ifdef CONFIG_X86_MCE
DO_ERROR(18, SIGSEGV, DIE_TRAP,  "machine check", machine_check, current)
#endif
DO_ERROR(19, SIGFPE,  DIE_TRAP,  "simd coprocessor error", simd_coprocessor_error, current)

fastcall void do_nmi(struct pt_regs * regs, long error_code)
{
}

void do_spurious_interrupt_bug(struct pt_regs * regs, long error_code)
{
}

/*
 * Note that we play around with the 'TS' bit in an attempt to get
 * the correct behaviour even in the presence of the asynchronous
 * IRQ13 behaviour
 */
void math_error(void)
{
	struct task_struct * task;

	/*
	 * Save the info for the exception handler and clear the error.
	 */
	task = current;
	task->thread.trap_no = 16;
	task->thread.error_code = 0;
	force_sig(SIGFPE, task);
}

void do_coprocessor_error(struct pt_regs * regs, long error_code)
{
	math_error();
}

/* Called from init/main.c */
void __init trap_init(void)
{
	cpu_init();
}

int l4x_deliver_signal(int exception_nr, int errcode)
{
	/* use exception entry points within kernel
	 * e.g. do_divide_error( struct ptregs * regs, long error);
	 */

	current->thread.regs.orig_eax = -1;

	switch (exception_nr) {
	case 0:
		do_divide_error(&current->thread.regs, errcode);
		break;
	case 1:
		do_debug(&current->thread.regs, errcode);
		break;
	case 2:
		do_nmi(&current->thread.regs, errcode);
		break;
	case 3:
		do_int3(&current->thread.regs, errcode);
		break;
	case 4:
		do_overflow(&current->thread.regs, errcode);
		break;
	case 5:
		do_bounds(&current->thread.regs, errcode);
		break;
	case 6:
		do_invalid_op(&current->thread.regs, errcode);
		break;
	case 7:			/* no */
		do_device_not_available(&current->thread.regs, errcode);
		break;
	case 8:			/* no */
		do_double_fault(&current->thread.regs, errcode);
		break;
	case 9:			/* no */
		do_coprocessor_segment_overrun(&current->thread.regs, errcode);
		break;
	case 10:		/* no */
		do_invalid_TSS(&current->thread.regs, errcode);
		break;
	case 11:
		do_segment_not_present(&current->thread.regs, errcode);
		break;
	case 12:
		do_stack_segment(&current->thread.regs, errcode);
		break;
	case 13:
	case 14:
		do_general_protection(&current->thread.regs, errcode);
		break;
	case 15:
		do_spurious_interrupt_bug(&current->thread.regs, errcode);
		break;
	case 16:
		do_coprocessor_error(&current->thread.regs, errcode);
		break;
	case 17:
		do_alignment_check(&current->thread.regs, errcode);
		break;
#ifdef CONFIG_X86_MCE
	case 18:
		do_machine_check(&current->thread.regs, errcode);
		break;
#endif
	case 19:
		do_simd_coprocessor_error(&current->thread.regs, errcode);
		break;
	default:
		printk("Unknown exception: %d\n", exception_nr);
		enter_kdebug("deliver_signal:unknown exception");
		break;
	}

	if (signal_pending(current)) {
		extern void fastcall do_signal(struct pt_regs *regs);
		do_signal(&current->thread.regs);
		return 1;
	}
	return 0;
}

static int __init kstack_setup(char *s)
{
	kstack_depth_to_print = simple_strtoul(s, NULL, 0);
	return 1;
}
__setup("kstack=", kstack_setup);

#ifdef CONFIG_STACK_UNWIND
static int __init call_trace_setup(char *s)
{
	if (strcmp(s, "old") == 0)
		call_trace = -1;
	else if (strcmp(s, "both") == 0)
		call_trace = 0;
	else if (strcmp(s, "newfallback") == 0)
		call_trace = 1;
	else if (strcmp(s, "new") == 2)
		call_trace = 2;
	return 1;
}
__setup("call_trace=", call_trace_setup);
#endif
