/**
 *    \file    dice/src/be/BEClass.cpp
 *    \brief   contains the implementation of the class CBEClass
 *
 *    \date    Tue Jun 25 2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "BEClass.h"

#include "BEFunction.h"
#include "BEOperationFunction.h"
#include "BEConstant.h"
#include "BETypedef.h"
#include "BEEnumType.h"
#include "BEAttribute.h"
#include "BEContext.h"
#include "BERoot.h"
#include "BEComponent.h"
#include "BEOperationFunction.h"
#include "BEInterfaceFunction.h"
#include "BECallFunction.h"
#include "BEUnmarshalFunction.h"
#include "BEMarshalFunction.h"
#include "BEReplyFunction.h"
#include "BEComponentFunction.h"
#include "BESndFunction.h"
#include "BEWaitFunction.h"
#include "BEWaitAnyFunction.h"
#include "BESrvLoopFunction.h"
#include "BEDispatchFunction.h"
#include "BEHeaderFile.h"
#include "BEImplementationFile.h"
#include "BEClient.h"
#include "BEOpcodeType.h"
#include "BEExpression.h"
#include "BENameSpace.h"
#include "BEDeclarator.h"
#include "BEStructType.h"
#include "BEUnionType.h"
#include "BEUserDefinedType.h"
#include "BEMsgBuffer.h"

#include "fe/FELibrary.h"
#include "fe/FEInterface.h"
#include "fe/FEConstDeclarator.h"
#include "fe/FEOperation.h"
#include "fe/FEUnaryExpression.h"
#include "fe/FEIntAttribute.h"
#include "fe/FEConstructedType.h"
#include "fe/FEUserDefinedType.h"
#include "fe/FETypedDeclarator.h"
#include "fe/FEDeclarator.h"
#include "fe/FEAttributeDeclarator.h"
#include "fe/FESimpleType.h"
#include "fe/FEAttribute.h"
#include "fe/FEFile.h"

#include "Compiler.h"

#include <string>
#include <cassert>


// CFunctionGroup IMPLEMENTATION

CFunctionGroup::CFunctionGroup(CFEOperation *pFEOperation)
: m_Functions(0, (CObject*)0)
{
    m_pFEOperation = pFEOperation;
}

/** \brief destroys a function group object
 *
 * This doesn't really do anything: The string object cleans up itself, and
 * the vector contains references to objects we do not want to delete. What we have
 * to remove are the vector-elements.
 */
CFunctionGroup::~CFunctionGroup()
{ }

/** \brief retrieves the name of the group
 *  \return the name of the group
 */
string CFunctionGroup::GetName()
{
    return m_pFEOperation->GetName();
}

/** \brief access the front-end operation member
 *  \return a reference to the front-end operation member
 */
CFEOperation *CFunctionGroup::GetOperation()
{
    return m_pFEOperation;
}

// CBEClass IMPLEMENTATION

CBEClass::CBEClass()
: m_Attributes(0, this),
  m_Constants(0, this),
  m_TypeDeclarations(0, this),
  m_Typedefs(0, this),
  m_Functions(0, this),
  m_FunctionGroups(0, this),
  m_DerivedClasses(0, (CObject*)0)
{
    m_pMsgBuffer = 0;
    m_pCorbaObject = 0;
    m_pCorbaEnv = 0;
}

/** \brief destructor of target class */
CBEClass::~CBEClass()
{
    if (m_pMsgBuffer)
        delete m_pMsgBuffer;
    if (m_pCorbaObject)
	delete m_pCorbaObject;
    if (m_pCorbaEnv)
	delete m_pCorbaEnv;
}

/** \brief returns the name of the class
 *  \return the name of the class
 */
string CBEClass::GetName()
{
    return m_sName;
}

/** \brief returns the number of functions in this class
 *  \return the number of functions in this class
 */
int CBEClass::GetFunctionCount()
{
    return m_Functions.size();
}

/** \brief returns the number of functions in this class which are written
 *  \return the number of functions in this class which are written
 */
int CBEClass::GetFunctionWriteCount(CBEFile *pFile)
{
    int nCount = 0;
    vector<CBEFunction*>::iterator iter;
    for (iter = m_Functions.begin();
	 iter != m_Functions.end();
	 iter++)
    {
        if (dynamic_cast<CBEHeaderFile*>(pFile) &&
            (*iter)->DoWriteFunction((CBEHeaderFile*)pFile))
            nCount++;
        if (dynamic_cast<CBEImplementationFile*>(pFile) &&
            (*iter)->DoWriteFunction((CBEImplementationFile*)pFile))
            nCount++;
    }

    return nCount;
}

/** \brief adds a new base class from a name
 *  \param sName the name of the class to add
 *
 * search for the class and if found add it
 */
void CBEClass::AddBaseClass(string sName)
{
    CBERoot *pRoot = GetSpecificParent<CBERoot>();
    assert(pRoot);
    // if we cannot find class it is not there, because this should be
    // called way after all classes are created
    CBEClass *pBaseClass = pRoot->FindClass(sName);
    if (!pBaseClass)
    {
	CCompiler::Warning("%s failed because base class \"%s\"" \
	    " cannot be found\n", __func__,
	    sName.c_str());
	return;
    }
    m_vBaseClasses.push_back(pBaseClass);
    // if we add a base class, we add us to that class' derived classes
    pBaseClass->m_DerivedClasses.Add(this);
}

/** \brief retrieves a pointer to the first base Class
 *  \return a pointer to the first base Class
 *
 * If m_vBaseClasses is empty and m_nBaseNameSize is bigger than 0,
 * we have to add the references to the base classes first.
 */
vector<CBEClass*>::iterator CBEClass::GetFirstBaseClass()
{
    return m_vBaseClasses.begin();
}

/** \brief retrieves a pointer to the next base Class
 *  \param iter a pointer to the next base Class
 *  \return a reference to the next base Class
 */
CBEClass* CBEClass::GetNextBaseClass(vector<CBEClass*>::iterator &iter)
{
    if (iter == m_vBaseClasses.end())
        return 0;
    return *iter++;
}

/** \brief creates the members of this class
 *  \param pFEInterface the front-end interface to use as source
 *  \return true if successful
 */
void
CBEClass::CreateBackEnd(CFEInterface * pFEInterface)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s(interf: %s) called\n",
	__func__, pFEInterface->GetName().c_str());

    // call CBEObject's CreateBackEnd method
    CBEObject::CreateBackEnd(pFEInterface);

    // set target file name
    SetTargetFileName(pFEInterface);
    // set own name
    m_sName = pFEInterface->GetName();
    // add attributes
    vector<CFEAttribute*>::iterator iterA;
    for (iterA = pFEInterface->m_Attributes.begin();
	 iterA != pFEInterface->m_Attributes.end();
	 iterA++)
    {
        CreateBackEnd(*iterA);
    }
    // we can resolve this if we only add the base names now, but when the
    // base classes are first used, add the actual references.
    // add references to base Classes
    vector<CFEInterface*>::iterator iterBI;
    for (iterBI = pFEInterface->m_BaseInterfaces.begin();
	 iterBI != pFEInterface->m_BaseInterfaces.end();
	 iterBI++)
    {
	// add base class
	AddBaseClass((*iterBI)->GetName());
    }
    // add constants
    vector<CFEConstDeclarator*>::iterator iterC;
    for (iterC = pFEInterface->m_Constants.begin();
	 iterC != pFEInterface->m_Constants.end();
	 iterC++)
    {
        CreateBackEnd(*iterC);
    }
    // add typedefs
    vector<CFETypedDeclarator*>::iterator iterTD;
    for (iterTD = pFEInterface->m_Typedefs.begin();
	 iterTD != pFEInterface->m_Typedefs.end();
	 iterTD++)
    {
        CreateBackEnd(*iterTD);
    }
    // add tagged decls
    vector<CFEConstructedType*>::iterator iterT;
    for (iterT = pFEInterface->m_TaggedDeclarators.begin();
	 iterT != pFEInterface->m_TaggedDeclarators.end();
	 iterT++)
    {
        CreateBackEnd(*iterT);
    }
    // add types for Class (only for C)
    if (CCompiler::IsBackEndLanguageSet(PROGRAM_BE_C))
	CreateAliasForClass(pFEInterface);
    // first create operation functions
    vector<CFEOperation*>::iterator iterO;
    for (iterO = pFEInterface->m_Operations.begin();
	 iterO != pFEInterface->m_Operations.end();
	 iterO++)
    {
        CreateFunctionsNoClassDependency(*iterO);
    }
    // create class' message buffer
    AddMessageBuffer(pFEInterface);
    // create functions that depend on class' message buffer
    for (iterO = pFEInterface->m_Operations.begin();
	 iterO != pFEInterface->m_Operations.end();
	 iterO++)
    {
        CreateFunctionsClassDependency(*iterO);
    }
    // add attribute interface members
    vector<CFEAttributeDeclarator*>::iterator iterAD;
    for (iterAD = pFEInterface->m_AttributeDeclarators.begin();
	 iterAD != pFEInterface->m_AttributeDeclarators.end();
	 iterAD++)
    {
        CreateBackEnd(*iterAD);
    }
    // add functions for interface
    AddInterfaceFunctions(pFEInterface);
    // add CORBA_Object and CORBA_Environment for server side C++
    // implementation
    if (CCompiler::IsBackEndLanguageSet(PROGRAM_BE_CPP))
    {
	CreateObject();
	CreateEnvironment();
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s finished\n", __func__);
}

/** \brief add the message buffer
 *  \param pFEInterface the front-end interface to use as reference
 *  \return true if successful
 *
 * The class message buffer is used by functions in the interface, e.g.,
 * unmarshal and marshal functions.  Thus is should be declared just before
 * these functions.  On the other hand, the server message buffer might
 * contain other types, which are declared inside the interface and therefore
 * after the message buffer is defined.  For this reason the message buffer
 * should be defined at the end of the interface.
 *
 * To achieve this, we declare a typedef to a union declaration (note: no
 * definition) and define the message buffer after the interface.
 *
 */
void
CBEClass::AddMessageBuffer(CFEInterface *pFEInterface)
{
    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, "\n\n\n");
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::%s(if: %s) called\n", __func__,
	pFEInterface->GetName().c_str());
    
    if (m_pMsgBuffer)
        delete m_pMsgBuffer;
    string exc = string(__func__);
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    m_pMsgBuffer = pCF->GetNewMessageBuffer();
    m_pMsgBuffer->SetParent(this);
    try
    {
	m_pMsgBuffer->CreateBackEnd(pFEInterface);
    }
    catch (CBECreateException *e)
    {
	CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, 
	    "CBEClass::%s creation failed\n", __func__);
        delete m_pMsgBuffer;
        m_pMsgBuffer = 0;
	e->Print();
	delete e;
	
	exc += " failed, because message buffer for interface " +
	    pFEInterface->GetName() + " could not be created";
        throw new CBECreateException(exc);
    }

    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::%s MB created at %p\n", __func__, m_pMsgBuffer);
    // add platform specific members
    if (!m_pMsgBuffer->AddPlatformSpecificMembers(this))
    {
	exc += " failed because platform specific members caused problems.";
	throw new CBECreateException(exc);
    }

    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::%s MB added platform members\n", __func__);
    // function specific initialization
    if (!MsgBufferInitialization())
    {
	exc += " failed, because message buffer could not be initialized.";
	throw new CBECreateException(exc);
    }

    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s MB initialized\n",
	__func__);
    // sort message buffer
    if (!m_pMsgBuffer->Sort(this))
    {
	exc += " failed, because Sort failed.";
	throw new CBECreateException(exc);
    }

    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, "CBEClass:%s MB sorted\n",
	__func__);
    // do post creation stuff
    try
    {
	m_pMsgBuffer->PostCreate(this, pFEInterface);
    }
    catch (CBECreateException *e)
    {
	e->Print();
	delete e;

	exc += " failed, because PostCreate failed.";
	throw new CBECreateException(exc);
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s returns\n",
	__func__);
}

/** \brief make function specific initialization
 *  \return true if successful
 */
bool
CBEClass::MsgBufferInitialization()
{
    // iterate function groups of class, pick a function and use it to
    // initialize struct
    vector<CFunctionGroup*>::iterator iter;
    for (iter = m_FunctionGroups.begin();
	 iter != m_FunctionGroups.end();
	 iter++)
    {
	// iterate the functions of the function group
	vector<CBEFunction*>::iterator iF;
	for (iF = (*iter)->m_Functions.begin();
	     iF != (*iter)->m_Functions.end();
	     iF++)
	{
	    if ((dynamic_cast<CBECallFunction*>(*iF) != 0) ||
		(dynamic_cast<CBESndFunction*>(*iF) != 0) ||
		(dynamic_cast<CBEWaitFunction*>(*iF) != 0))
	    {
		if (!(*iF)->MsgBufferInitialization(m_pMsgBuffer))
		    return false;
	    }
	}
    }
    return true;
}

/** \brief adds the functions for an interface
 *  \param pFEInterface the interface to add the functions for
 *  \return true if successful
 */
void 
CBEClass::AddInterfaceFunctions(CFEInterface* pFEInterface)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "%s called\n", __func__);

    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEInterfaceFunction *pFunction = pCF->GetNewWaitAnyFunction();
    m_Functions.Add(pFunction);
    pFunction->SetComponentSide(true);
    try
    {
	pFunction->CreateBackEnd(pFEInterface);
    }
    catch (CBECreateException *e)
    {
	e->Print();
	delete e;

	m_Functions.Remove(pFunction);
        delete pFunction;
	string exc = string(__func__);
	exc += " failed because wait-any function could not be created";
        throw new CBECreateException(exc);
    }
    pFunction = pCF->GetNewRcvAnyFunction();
    m_Functions.Add(pFunction);
    pFunction->SetComponentSide(true);
    try
    {
	pFunction->CreateBackEnd(pFEInterface);
    }
    catch (CBECreateException *e)
    {
	e->Print();
	delete e;

	m_Functions.Remove(pFunction);
        delete pFunction;
	string exc = string(__func__);
	exc += " failed because recv-any function could not be created";
        throw new CBECreateException(exc);
    }
    pFunction = pCF->GetNewReplyAnyWaitAnyFunction();
    m_Functions.Add(pFunction);
    pFunction->SetComponentSide(true);
    try
    {
	pFunction->CreateBackEnd(pFEInterface);
    }
    catch (CBECreateException *e)
    {
	e->Print();
	delete e;

	m_Functions.Remove(pFunction);
        delete pFunction;
	string exc = string(__func__);
	exc += " failed because reply-wait-any function could not be created";
        throw new CBECreateException(exc);
    }
    if (!(CCompiler::IsOptionSet(PROGRAM_NO_DISPATCHER) &&
          CCompiler::IsOptionSet(PROGRAM_NO_SERVER_LOOP)))
    {
        pFunction = pCF->GetNewDispatchFunction();
        m_Functions.Add(pFunction);
        pFunction->SetComponentSide(true);
	try
	{
	    pFunction->CreateBackEnd(pFEInterface);
	}
	catch (CBECreateException *e)
        {
	    e->Print();
	    delete e;

	    m_Functions.Remove(pFunction);
            delete pFunction;
	    string exc = string(__func__);
	    exc += " failed because dispatch function could not be created";
	    throw new CBECreateException(exc);
        }
    }
    if (!CCompiler::IsOptionSet(PROGRAM_NO_SERVER_LOOP))
    {
        pFunction = pCF->GetNewSrvLoopFunction();
        m_Functions.Add(pFunction);
        pFunction->SetComponentSide(true);
	try
	{
	    pFunction->CreateBackEnd(pFEInterface);
	}
	catch (CBECreateException *e)
        {
	    e->Print();
	    delete e;

	    m_Functions.Remove(pFunction);
            delete pFunction;
	    string exc = string(__func__);
	    exc += " failed because srv-loop function could not be created";
	    throw new CBECreateException(exc);
        }
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s finished\n", __func__);
}

/** \brief creates an alias type for the class
 *  \param pFEInterface the interface to use as reference
 *  \return true if successful
 *
 * In C we have an alias of CORBA_Object type to the name if the interface.
 * In C++ this is not needed, because the class is derived from CORBA_Object
 */
void 
CBEClass::CreateAliasForClass(CFEInterface *pFEInterface)
{
    assert(pFEInterface);
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::%s(interface) called\n", __func__);
    // create the BE type
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBETypedef *pTypedef = pCF->GetNewTypedef();
    m_Typedefs.Add(pTypedef);
    // create CORBA_Object type
    CBEUserDefinedType *pType = static_cast<CBEUserDefinedType*>(
	pCF->GetNewType(TYPE_USER_DEFINED));
    pType->SetParent(pTypedef);
    try
    {
	pType->CreateBackEnd(string("CORBA_Object"));
    }
    catch (CBECreateException *e)
    {
        delete pType;
	e->Print();
	delete e;

	string exc = string(__func__);
	exc += " failed, because alias type could not be created.";
	throw new CBECreateException(exc);
    }
    // finally create typedef
    try
    {
	pTypedef->CreateBackEnd(pType, pFEInterface->GetName(), pFEInterface);
    }
    catch (CBECreateException *e)
    {
	m_Typedefs.Remove(pTypedef);
        delete pTypedef;
        delete pType;
	e->Print();
	delete e;

	string exc = string(__func__);
	exc += " failed, because typedef could not be created.";
	throw new CBECreateException(exc);
    }
    delete pType; // cloned in CBETypedDeclarator::CreateBackEnd

    // set source line and file
    pTypedef->SetSourceLine(pFEInterface->GetSourceLine()-1);
    pTypedef->SetSourceFileName(
	pFEInterface->GetSpecificParent<CFEFile>()->GetFileName());

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::%s(interface) returns\n", __func__);
}

/** \brief internal function to create a constant
 *  \param pFEConstant the respective front-end constant
 *  \return true if successful
 */
void
CBEClass::CreateBackEnd(CFEConstDeclarator *pFEConstant)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s(const) called\n",
	__func__);

    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEConstant *pConstant = pCF->GetNewConstant();
    m_Constants.Add(pConstant);
    pConstant->SetParent(this);
    try
    {
	pConstant->CreateBackEnd(pFEConstant);
    }
    catch (CBECreateException *e)
    {
	m_Constants.Remove(pConstant);
        delete pConstant;
	e->Print();
	delete e;

	string exc = string(__func__);
	exc += " failed because const " + pFEConstant->GetName() +
	    " could not be created";
        throw new CBECreateException(exc);
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::%s(const) returns\n", __func__);
}

/** \brief internal function to create a typedefinition
 *  \param pFETypedef the respective front-end type definition
 *  \return true if successful
 */
void
CBEClass::CreateBackEnd(CFETypedDeclarator *pFETypedef)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::%s(typedef) called\n", __func__);

    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBETypedef *pTypedef = pCF->GetNewTypedef();
    m_Typedefs.Add(pTypedef);
    pTypedef->SetParent(this);
    try
    {
	pTypedef->CreateBackEnd(pFETypedef);
    }
    catch (CBECreateException *e)
    {
	m_Typedefs.Remove(pTypedef);
        delete pTypedef;
	e->Print();
	delete e;

	string exc = string(__func__);
	exc += " failed because typedef could not be created";
        throw new CBECreateException(exc);
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::%s(typedef) returns\n", __func__);
}

/** \brief internal function to create a functions for attribute declarator
 *  \param pFEAttrDecl the respective front-end attribute declarator definition
 *  \return true if successful
 */
void
CBEClass::CreateBackEnd(CFEAttributeDeclarator *pFEAttrDecl)
{
    string exc = string(__func__);
    if (!pFEAttrDecl)
    {
	exc += " failed because attribute declarator is 0";
        throw new CBECreateException(exc);
    }
    
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s(attr-decl) called\n",
	__func__);

    // add function "<param_type_spec> [out]__get_<simple_declarator>();" for
    // each decl if !READONLY:
    //   add function "void [in]__set_<simple_declarator>(<param_type_spec>
    //   'first letter of <decl>');" for each decl
    vector<CFEDeclarator*>::iterator iterD;
    for (iterD = pFEAttrDecl->m_Declarators.begin();
	 iterD != pFEAttrDecl->m_Declarators.end();
	 iterD++)
    {
        // get function
        string sName = string("_get_");
        sName += (*iterD)->GetName();
        CFETypeSpec *pFEType = (CFETypeSpec*)pFEAttrDecl->GetType()->Clone();
        CFEOperation *pFEOperation = new CFEOperation(pFEType, sName, 0);
        pFEType->SetParent(pFEOperation);
        // get parent interface
        CFEInterface *pFEInterface = 
	    pFEAttrDecl->GetSpecificParent<CFEInterface>();
        assert(pFEInterface);
        pFEInterface->m_Operations.Add(pFEOperation);
	try
	{
	    CreateFunctionsNoClassDependency(pFEOperation);
	    CreateFunctionsClassDependency(pFEOperation);
	}
	catch (CBECreateException *e)
        {
            delete pFEOperation;
	    throw;
        }

        // set function
        if (!pFEAttrDecl->m_Attributes.Find(ATTR_READONLY))
        {
            sName = string("_set_");
            sName += (*iterD)->GetName();
            CFEAttribute *pFEAttr = new CFEAttribute(ATTR_IN);
            vector<CFEAttribute*> *pFEAttributes = new vector<CFEAttribute*>();
            pFEAttributes->push_back(pFEAttr);
            pFEType = new CFESimpleType(TYPE_VOID);
            CFETypeSpec *pFEParamType = 
		(CFETypeSpec*)pFEAttrDecl->GetType()->Clone();
            CFEDeclarator *pFEParamDecl = new CFEDeclarator(DECL_IDENTIFIER, 
		(*iterD)->GetName().substr(0,1));
            vector<CFEDeclarator*> *pFEParameters = new vector<CFEDeclarator*>();
            pFEParameters->push_back(pFEParamDecl);
            CFETypedDeclarator *pFEParam = new CFETypedDeclarator(TYPEDECL_PARAM,
                pFEParamType, pFEParameters, pFEAttributes);
            pFEParamType->SetParent(pFEParam);
            pFEParamDecl->SetParent(pFEParam);
            delete pFEAttributes;
            delete pFEParameters;
            // create function
            vector<CFETypedDeclarator*> *pParams = 
		new vector<CFETypedDeclarator*>();
            pParams->push_back(pFEParam);
            pFEOperation = new CFEOperation(pFEType, sName, pParams);
            delete pParams;
            pFEType->SetParent(pFEOperation);
            pFEAttr->SetParent(pFEOperation);
            pFEParam->SetParent(pFEOperation);
            pFEInterface->m_Operations.Add(pFEOperation);

	    try
	    {
		CreateFunctionsNoClassDependency(pFEOperation);
		CreateFunctionsClassDependency(pFEOperation);
	    }
	    catch (CBECreateException *e)
            {
                delete pFEOperation;
                throw;
            }
        }
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "%s called\n", __func__);
}

/** \brief internal function to create the back-end functions
 *  \param pFEOperation the respective front-end function
 *  \return true if successful
 *
 * A function has to be generated depending on its attributes. If it is a call 
 * function, we have to generate different back-end function than for a message
 * passing function.
 * 
 * We depend on the fact, that either the [in] or the [out] attribute are 
 * specified. Never both may appear.
 *
 * In this method we create functions independant of the class' message buffer
 */
void 
CBEClass::CreateFunctionsNoClassDependency(CFEOperation *pFEOperation)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "%s called.\n", __func__);
    
    CFunctionGroup *pGroup = new CFunctionGroup(pFEOperation);
    m_FunctionGroups.Add(pGroup);
    CBEClassFactory *pCF = CCompiler::GetClassFactory();

    string exc = string(__func__);

    if (!(pFEOperation->m_Attributes.Find(ATTR_IN)) &&
        !(pFEOperation->m_Attributes.Find(ATTR_OUT)))
    {
        // the call case:
        // we need the functions call, unmarshal, reply-and-wait, skeleton, 
	// reply-and-recv
	// for client side: call
        CBEOperationFunction *pFunction = pCF->GetNewCallFunction();
        m_Functions.Add(pFunction);
        pFunction->SetComponentSide(false);
        pGroup->m_Functions.Add(pFunction);
	try
	{
	    pFunction->CreateBackEnd(pFEOperation);
	}
	catch (CBECreateException *e)
        {
	    e->Print();
	    delete e;
	    m_Functions.Remove(pFunction);
            delete pFunction;

	    exc += " failed, because call func couldn't be created for " +
		pFEOperation->GetName();
            throw new CBECreateException(exc);
        }

        // for server side: reply-and-wait, reply-and-recv, skeleton
        pFunction = pCF->GetNewComponentFunction();
        m_Functions.Add(pFunction);
        pFunction->SetComponentSide(true);
        pGroup->m_Functions.Add(pFunction);
	try
	{
	    pFunction->CreateBackEnd(pFEOperation);
	}
	catch (CBECreateException *e)
        {
	    e->Print();
	    delete e;
	    m_Functions.Remove(pFunction);
            delete pFunction;
	    
	    exc += " failed, because component func couldn't be created for " +
		pFEOperation->GetName();
            throw new CBECreateException(exc);
        }

        if (pFEOperation->m_Attributes.Find(ATTR_ALLOW_REPLY_ONLY))
        {
            pFunction = pCF->GetNewReplyFunction();
            m_Functions.Add(pFunction);
            pFunction->SetComponentSide(true);
            pGroup->m_Functions.Add(pFunction);
	    try
	    {
		pFunction->CreateBackEnd(pFEOperation);
	    }
	    catch (CBECreateException *e)
            {
		e->Print();
		delete e;
		m_Functions.Remove(pFunction);
                delete pFunction;

		exc += " failed, because reply function could not be " \
		    "created for " + pFEOperation->GetName();
                throw new CBECreateException(exc);
            }
        }
    }
    else
    {
        // the MP case
        // we need the functions send, recv, wait
        bool bComponent = (pFEOperation->m_Attributes.Find(ATTR_OUT));
        // sender: send
        CBEOperationFunction *pFunction = pCF->GetNewSndFunction();
        m_Functions.Add(pFunction);
        pFunction->SetComponentSide(bComponent);
        pGroup->m_Functions.Add(pFunction);
	try
	{
	    pFunction->CreateBackEnd(pFEOperation);
	}
	catch (CBECreateException *e)
        {
	    e->Print();
	    delete e;
	    m_Functions.Remove(pFunction);
            delete pFunction;

	    exc += " failed, because send function could not be created " \
		"for " + pFEOperation->GetName();
            throw new CBECreateException(exc);
        }

        // receiver: wait, recv, unmarshal
        pFunction = pCF->GetNewWaitFunction();
        m_Functions.Add(pFunction);
        pFunction->SetComponentSide(!bComponent);
        pGroup->m_Functions.Add(pFunction);
	try
	{
	    pFunction->CreateBackEnd(pFEOperation);
	}
	catch (CBECreateException *e)
        {
	    e->Print();
	    delete e;
	    m_Functions.Remove(pFunction);
            delete pFunction;

	    exc += " failed, because wait function could not be created for " +
		pFEOperation->GetName();
            throw new CBECreateException(exc);
        }

        pFunction = pCF->GetNewRcvFunction();
        m_Functions.Add(pFunction);
        pFunction->SetComponentSide(!bComponent);
        pGroup->m_Functions.Add(pFunction);
	try
	{
	    pFunction->CreateBackEnd(pFEOperation);
	}
	catch (CBECreateException *e)
        {
	    e->Print();
	    delete e;
	    m_Functions.Remove(pFunction);
            delete pFunction;

	    exc += " failed because receive function could not be created for "
		+ pFEOperation->GetName();
            throw new CBECreateException(exc);
        }

        // if we send oneway to the server we need a component function
        if (pFEOperation->m_Attributes.Find(ATTR_IN))
        {
            pFunction = pCF->GetNewComponentFunction();
            m_Functions.Add(pFunction);
            pFunction->SetComponentSide(true);
            pGroup->m_Functions.Add(pFunction);
	    try
	    {
		pFunction->CreateBackEnd(pFEOperation);
	    }
	    catch (CBECreateException *e)
            {
		e->Print();
		delete e;
		m_Functions.Remove(pFunction);
                delete pFunction;

		exc += " failed, because component function could not be " \
		    "created for " + pFEOperation->GetName();
                throw new CBECreateException(exc);
            }
        }
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "%s returns.\n", __func__);
}

/** \brief internal function to create the back-end functions
 *  \param pFEOperation the respective front-end function
 *  \return true if successful
 *
 * A function has to be generated depending on its attributes. If it is a call
 * function, we have to generate different back-end function than for a
 * message passing function.
 *
 * We depend on the fact, that either the [in] or the [out] attribute are
 * specified.  Never both may appear.
 *
 * Here we create functions, which depend on the class' message buffer.
 */
void 
CBEClass::CreateFunctionsClassDependency(CFEOperation *pFEOperation)
{
    string exc = string(__func__);
    // get function group of pFEOperation (should have been create above)
    vector<CFunctionGroup*>::iterator iterFG;
    for (iterFG = m_FunctionGroups.begin();
	 iterFG != m_FunctionGroups.end();
	 iterFG++)
    {
        if ((*iterFG)->GetOperation() == pFEOperation)
            break;
    }
    assert(iterFG != m_FunctionGroups.end());

    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    if (!(pFEOperation->m_Attributes.Find(ATTR_IN)) &&
        !(pFEOperation->m_Attributes.Find(ATTR_OUT)))
    {
        // the call case:
        // we need the functions unmarshal, marshal
        CBEOperationFunction *pFunction = pCF->GetNewUnmarshalFunction();
        m_Functions.Add(pFunction);
        pFunction->SetComponentSide(true);
        (*iterFG)->m_Functions.Add(pFunction);
	try
	{
	    pFunction->CreateBackEnd(pFEOperation);
	}
	catch (CBECreateException *e)
        {
	    m_Functions.Remove(pFunction);
            delete pFunction;
	    e->Print();
	    delete e;

	    exc += " failed, because unmarshal function could not be created" \
		" for " + pFEOperation->GetName();
            throw new CBECreateException(exc);
        }

        pFunction = pCF->GetNewMarshalFunction();
        m_Functions.Add(pFunction);
        pFunction->SetComponentSide(true);
        (*iterFG)->m_Functions.Add(pFunction);
	try
	{
	    pFunction->CreateBackEnd(pFEOperation);
	}
	catch (CBECreateException *e)
        {
	    m_Functions.Remove(pFunction);
            delete pFunction;
	    e->Print();
	    delete e;

	    exc += " failed, because marshal function coudl not be created" \
		" for " + pFEOperation->GetName();
            throw new CBECreateException(exc);
        }
    }
    else
    {
        // the MP case
        // we need the functions send, recv, wait, unmarshal
        bool bComponent = (pFEOperation->m_Attributes.Find(ATTR_OUT));

        CBEOperationFunction *pFunction = pCF->GetNewUnmarshalFunction();
        m_Functions.Add(pFunction);
        pFunction->SetComponentSide(!bComponent);
        (*iterFG)->m_Functions.Add(pFunction);
	try
	{
	    pFunction->CreateBackEnd(pFEOperation);
	}
	catch (CBECreateException *e)
        {
	    m_Functions.Remove(pFunction);
            delete pFunction;
	    e->Print();
	    delete e;

	    exc += " failed because unmarshal function could not be created" \
		" for " + pFEOperation->GetName();
            throw new CBECreateException(exc);
        }
    }
}

/** \brief interna function to create an attribute
 *  \param pFEAttribute the respective front-end attribute
 *  \return true if successful
 */
void
CBEClass::CreateBackEnd(CFEAttribute *pFEAttribute)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s(attr) called\n",
	__func__);

    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEAttribute *pAttribute = pCF->GetNewAttribute();
    m_Attributes.Add(pAttribute);
    try
    {
	pAttribute->CreateBackEnd(pFEAttribute);
    }
    catch (CBECreateException *e)
    {
	m_Attributes.Remove(pAttribute);
        delete pAttribute;
	e->Print();
	delete e;

	string exc = string(__func__);
	exc += " failed because attribute could not be created";
	throw new CBECreateException(exc);
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s finished\n", __func__);
}

/** \brief adds the Class to the header file
 *  \param pHeader file the header file to add to
 *  \return true if successful
 *
 * An Class adds its included types, constants and functions.
 */
bool CBEClass::AddToFile(CBEHeaderFile *pHeader)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL,
	"CBEClass::%s(header: %s) for class %s called\n", __func__,
        pHeader->GetFileName().c_str(), GetName().c_str());
    // add this class to the file
    assert(pHeader);
    if (IsTargetFile(pHeader))
        pHeader->m_Classes.Add(this);

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::%s(header: %s) for class %s returns true\n", __func__,
	pHeader->GetFileName().c_str(), GetName().c_str());
    return true;
}

/** \brief adds this class (or its members) to an implementation file
 *  \param pImpl the implementation file to add this class to
 *  \return true if successful
 *
 * if the options PROGRAM_FILE_FUNCTION is set, we have to add each function
 * seperately for the client implementation file. Otherwise we add the
 * whole class.
 */
bool CBEClass::AddToFile(CBEImplementationFile *pImpl)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::%s(impl: %s) for class %s called\n", __func__,
        pImpl->GetFileName().c_str(), GetName().c_str());
    // check compiler option
    assert(pImpl);
    if (CCompiler::IsFileOptionSet(PROGRAM_FILE_FUNCTION) &&
        dynamic_cast<CBEClient*>(pImpl->GetTarget()))
    {
        vector<CBEFunction*>::iterator iter;
	for (iter = m_Functions.begin();
	     iter != m_Functions.end();
	     iter++)
        {
            (*iter)->AddToFile(pImpl);
        }
    }
    // add this class to the file
    if (IsTargetFile(pImpl))
        pImpl->m_Classes.Add(this);

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::%s(impl: %s) for class %s returns true\n", __func__,
        pImpl->GetFileName().c_str(), GetName().c_str());
    return true;
}

/** \brief counts the parameters with a specific type
 *  \param nFEType the front-end type to test for
 *  \param bSameCount true if the count is the same for all functions (false \
 *         if functions have different count)
 *  \param nDirection the direction to count
 *  \return the number of parameters of this type
 */
int
CBEClass::GetParameterCount(int nFEType,
    bool& bSameCount,
    int nDirection)
{
    if (nDirection == 0)
    {
        // count both and return max
        int nCountIn = GetParameterCount(nFEType, bSameCount, DIRECTION_IN);
        int nCountOut = GetParameterCount(nFEType, bSameCount, DIRECTION_OUT);
        return (nCountIn > nCountOut) ? nCountIn : nCountOut;
    }

    int nCount = 0, nCurr;
    vector<CBEFunction*>::iterator iter;
    for (iter = m_Functions.begin();
	 iter != m_Functions.end();
	 iter++)
    {
        nCurr = (*iter)->GetParameterCount(nFEType, nDirection);
         if ((nCount > 0) && (nCurr != nCount) && (nCurr > 0))
            bSameCount = false;
        if (nCurr > nCount) nCount = nCurr;
    }

    return nCount;
}

/** \brief counts the number of string parameter needed for this interface
 *  \param nDirection the direction to count
 *  \param nMustAttrs the attributes which have to be set for the parameters
 *  \param nMustNotAttrs the attributes which must not be set for the parameters
 *  \return the number of strings needed
 */
int
CBEClass::GetStringParameterCount(int nDirection,
    ATTR_TYPE nMustAttrs,
    ATTR_TYPE nMustNotAttrs)
{
    if (nDirection == 0)
    {
        int nStringsIn = GetStringParameterCount(DIRECTION_IN, nMustAttrs,
	    nMustNotAttrs);
        int nStringsOut = GetStringParameterCount(DIRECTION_OUT, nMustAttrs,
	    nMustNotAttrs);
        return ((nStringsIn > nStringsOut) ? nStringsIn : nStringsOut);
    }

    int nCount = 0, nCurr;

    vector<CFunctionGroup*>::iterator iterG;
    for (iterG = m_FunctionGroups.begin();
	 iterG != m_FunctionGroups.end();
	 iterG++)
    {
        vector<CBEFunction*>::iterator iterF;
	for (iterF = (*iterG)->m_Functions.begin();
	     iterF != (*iterG)->m_Functions.end();
	     iterF++)
        {
	    if (!dynamic_cast<CBEOperationFunction*>(*iterF))
		continue;
            nCurr = (*iterF)->GetStringParameterCount(nDirection, nMustAttrs,
		nMustNotAttrs);
            nCount = (nCount > nCurr) ? nCount : nCurr;
        }
    }

    return nCount;
}

/** \brief calculates the size of the interface function
 *  \param nDirection the direction to count
 *  \return the number of bytes needed to transmit any of the functions
 */
int CBEClass::GetSize(int nDirection)
{
    if (nDirection == 0)
    {
        int nSizeIn = GetSize(DIRECTION_IN);
        int nSizeOut = GetSize(DIRECTION_OUT);
        return ((nSizeIn > nSizeOut) ? nSizeIn : nSizeOut);
    }

    int nSize = 0, nCurr;

    vector<CBEFunction*>::iterator iter;
    for (iter = m_Functions.begin();
	 iter != m_Functions.end();
	 iter++)
    {
	if (!dynamic_cast<CBEOperationFunction*>(*iter))
	    continue;
        nCurr = (*iter)->GetSize(nDirection);
        nSize = (nSize > nCurr) ? nSize : nCurr;
    }

    return nSize;
}

/** \brief tries to find a function
 *  \param sFunctionName the name of the function to search for
 *  \param nFunctionType the type of function to find
 *  \return a reference to the searched class or 0
 */
CBEFunction* CBEClass::FindFunction(string sFunctionName,
    FUNCTION_TYPE nFunctionType)
{
    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, "%s(%s) called\n", __func__,
	sFunctionName.c_str());
    
    if (sFunctionName.empty())
        return 0;

    // simply scan the function for a match
    vector<CBEFunction*>::iterator iter;
    for (iter = m_Functions.begin();
	 iter != m_Functions.end();
	 iter++)
    {
	CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, 
	    "%s checking function %s (compare to %s)\n", __func__,
	    (*iter)->GetName().c_str(), sFunctionName.c_str());

        if ((*iter)->GetName() == sFunctionName &&
	    (*iter)->IsFunctionType(nFunctionType))
            return *iter;
    }

    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, "%s function %s not found, return 0\n",
	__func__, sFunctionName.c_str());
    return 0;
}

/** \brief adds the opcodes of this class' functions to the header file
 *  \param pFile the file to write to
 *  \return true if successfule
 *
 * This implementation adds the base opcode for this class and all opcodes for
 * its functions.
 */
bool CBEClass::AddOpcodesToFile(CBEHeaderFile *pFile)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s(header: %s) called\n", 
	__func__, pFile->GetFileName().c_str());
    // check if the file is really our target file
    if (!IsTargetFile(pFile))
    {
	CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
	    "CBEClass::%s wrong target file\n", __func__);
        return true;
    }

    // first create classes in reverse order, so we can build correct parent
    // relationships
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    CBEConstant *pOpcode = pCF->GetNewConstant();
    CBEOpcodeType *pType = pCF->GetNewOpcodeType();
    pType->SetParent(pOpcode);
    CBEExpression *pBrace = pCF->GetNewExpression();
    pBrace->SetParent(pOpcode);
    CBEExpression *pInterfaceCode = pCF->GetNewExpression();
    pInterfaceCode->SetParent(pBrace);
    CBEExpression *pValue = pCF->GetNewExpression();
    pValue->SetParent(pInterfaceCode);
    CBEExpression *pBits = pCF->GetNewExpression();
    pBits->SetParent(pInterfaceCode);

    // now call Create functions, which require correct parent relationshsips

    // create opcode type
    try
    {
	pType->CreateBackEnd();
	// get opcode number
	int nInterfaceNumber = GetClassNumber();
	// create value
	pValue->CreateBackEnd(nInterfaceNumber);
	// create shift bits
	string sShift = pNF->GetInterfaceNumberShiftConstant();
	pBits->CreateBackEnd(sShift);
	// create value << bits
	pInterfaceCode->CreateBackEndBinary(pValue, EXPR_LSHIFT, pBits);
	// brace it
	pBrace->CreateBackEndPrimary(EXPR_PAREN, pInterfaceCode);
    }
    catch (CBECreateException *e)
    {
	e->Print();
	delete e;

        delete pOpcode;
        delete pValue;
        delete pType;
        delete pBrace;
        delete pInterfaceCode;
        delete pBits;
	CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
	    "CBEClass::%s opcode type creation failed\n", __func__);
        return false;
    }
    // create constant
    // create opcode name
    string sName = pNF->GetOpcodeConst(this);
    // add const to file
    pFile->m_Constants.Add(pOpcode);
    try
    {
	pOpcode->CreateBackEnd(pType, sName, pBrace, true/* always define*/);
    }
    catch (CBECreateException *e)
    {
	e->Print();
	delete e;

	pFile->m_Constants.Remove(pOpcode);
        delete pOpcode;
        delete pValue;
        delete pType;
	CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
	    "CBEClass::%s opcode var creation failed\n", __func__);
        return false;
    }

    // iterate over functions
    vector<CFunctionGroup*>::iterator iter;
    for (iter = m_FunctionGroups.begin();
	 iter != m_FunctionGroups.end();
	 iter++)
    {
        if (!AddOpcodesToFile((*iter)->GetOperation(), pFile))
	{
	    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
		"CBEClass::%s could not add opcodes of function group\n", __func__);
            return false;
	}
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s finished\n", __func__);
    return true;
}

/** \brief adds the opcode for a single function
 *  \param pFEOperation the function to add the opcode for
 *  \param pFile the file to add the opcode to
 */
bool
CBEClass::AddOpcodesToFile(CFEOperation *pFEOperation,
    CBEHeaderFile *pFile)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::AddOpcodesToFile(operation: %s) called\n",
        pFEOperation->GetName().c_str());

    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    // first create classes, so we can build parent relationship correctly
    CBEConstant *pOpcode = pCF->GetNewConstant();
    CBEOpcodeType *pType = pCF->GetNewOpcodeType();
    pType->SetParent(pOpcode);
    CBEExpression *pTopBrace = pCF->GetNewExpression();
    pTopBrace->SetParent(pOpcode);
    CBEExpression *pValue = pCF->GetNewExpression();
    pValue->SetParent(pTopBrace);
    CBEExpression *pBrace = pCF->GetNewExpression();
    pBrace->SetParent(pValue);
    CBEExpression *pFuncCode = pCF->GetNewExpression();
    pFuncCode->SetParent(pBrace);
    CBEExpression *pBase = pCF->GetNewExpression();
    pBase->SetParent(pValue);
    CBEExpression *pNumber = pCF->GetNewExpression();
    pNumber->SetParent(pFuncCode);
    CBEExpression *pBitMask = pCF->GetNewExpression();
    pBitMask->SetParent(pFuncCode);

    // now call the create functions, which require an correct parent
    // relationship 
    try
    {
	// get base opcode name
	string sBase = pNF->GetOpcodeConst(this);
	pBase->CreateBackEnd(sBase);
	// create number
	int nOperationNb = GetOperationNumber(pFEOperation);
	pNumber->CreateBackEnd(nOperationNb);
	// create bitmask
	string sBitMask =  pNF->GetFunctionBitMaskConstant();
	pBitMask->CreateBackEnd(sBitMask);
	// create function code
	pFuncCode->CreateBackEndBinary(pNumber, EXPR_BITAND, pBitMask);
	// create braces
	pBrace->CreateBackEndPrimary(EXPR_PAREN, pFuncCode);
	// create value
	pValue->CreateBackEndBinary(pBase, EXPR_PLUS, pBrace);
	// create top brace
	pTopBrace->CreateBackEndPrimary(EXPR_PAREN, pValue);
	// create opcode type
	pType->CreateBackEnd();
	// create opcode name
	string sName = pNF->GetOpcodeConst(pFEOperation);
	// create constant
	pFile->m_Constants.Add(pOpcode);
	pOpcode->CreateBackEnd(pType, sName, pTopBrace, true/*always defined*/);
    }
    catch (CBECreateException *e)
    {
	e->Print();
	delete e;
	
	pFile->m_Constants.Remove(pOpcode);
        delete pOpcode;
        delete pType;
        delete pTopBrace;
        delete pValue;
        delete pNumber;
        delete pBase;
        delete pBrace;
        delete pFuncCode;
        delete pBitMask;
	CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
	    "CBEClass::%s opcode const creation failed\n", __func__);
        return false;
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s finished\n", __func__);
    return true;
}

/** \brief calculates the number used as base opcode number
 *  \return the interface number
 *
 * AN interface number is first of all it's uuid. If it is not available
 * the base interfaces are counted and this interface's number is the
 * highest number of it's base interfaces + 1.
 */
int CBEClass::GetClassNumber()
{
    CBEAttribute *pUuidAttr = m_Attributes.Find(ATTR_UUID);
    if (pUuidAttr)
    {
        if (pUuidAttr->IsOfType(ATTR_CLASS_INT))
        {
            return pUuidAttr->GetIntValue();
        }
    }

    int nNumber = 1;
    vector<CBEClass*>::iterator iter;
    for (iter = m_vBaseClasses.begin();
	 iter != m_vBaseClasses.end();
	 iter++)
    {
        int nBaseNumber = (*iter)->GetClassNumber();
        if (nBaseNumber >= nNumber)
            nNumber = nBaseNumber+1;
    }
    return nNumber;
}

/** \brief writes the class to the target file
 *  \param pFile the file to write to
 *
 * With the C implementation the class simply calls the Write methods
 * of its constants, typedefs and functions. The message buffer is first
 * defined in a forward declaration and later really defined. This way it is
 * known when checking the functions, but can use types declared inside the
 * interface.
 * 
 * However, this does not work for inlined functions. They need to fully know
 * the message buffer type when using it.
 */
void CBEClass::Write(CBEHeaderFile *pFile)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s called\n", __func__);

    // since message buffer is local for this class, the class declaration
    // wraps the message buffer
    // per default we derive from CORBA_Object
    if (CCompiler::IsBackEndLanguageSet(PROGRAM_BE_CPP))
    {
	// CPP TODO: should derive from CORBA_Object, _but_:
	// then we need to have a declaration of CORBA_Object which is not a
	// typedef of a pointer to CORBA_Object_base, which in turn messes up
	// the compilation of C++ files including generated header files for C
	// backend... We have to find some other way, such as a dice local
	// define for C++.
	*pFile << "\tclass ";
	WriteClassName(pFile);
	*pFile << "\n";
	*pFile << "\t{\n";

	WriteMemberVariables(pFile);
	WriteConstructor(pFile);
	WriteDestructor(pFile);
    }
    
    // write message buffer type seperately
    CBEMsgBuffer* pMsgBuf = GetMessageBuffer();
    CBETarget *pTarget = pFile->GetTarget();
    // test for client and if type is needed
    bool bWriteMsgBuffer = (pMsgBuf != 0) &&
	 pTarget->HasFunctionWithUserType(
	     pMsgBuf->m_Declarators.First()->GetName());

    // write all constants, typedefs
    WriteElements(pFile);
    // now write message buffer (it might use types defined above)
    if (bWriteMsgBuffer)
    {
	if (CCompiler::IsBackEndLanguageSet(PROGRAM_BE_C))
	{
	    pMsgBuf->WriteDeclaration(pFile);
	    *pFile << "\n";
	}
	if (CCompiler::IsBackEndLanguageSet(PROGRAM_BE_CPP))
	{
	    *pFile << "\tprotected:\n";
	    pFile->IncIndent();
	    pMsgBuf->WriteDeclaration(pFile);
	    pFile->DecIndent();
	}
    }
    // now write functions (which need definition of msg buffer)
    WriteFunctions(pFile);
    WriteHelperFunctions(pFile);
    
    if (CCompiler::IsBackEndLanguageSet(PROGRAM_BE_CPP))
	*pFile << "\t};\n";

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s finished\n", __func__);
}

/** \brief writes the class name, which might be different for clnt & srv
 *  \param pFile the file to write to
 */
void CBEClass::WriteClassName(CBEFile *pFile)
{
    *pFile << GetName();
    if (pFile->IsOfFileType(FILETYPE_COMPONENT))
	*pFile << "Server";
}

/** \brief write member variable for C++ classes
 *  \param pFile the file to write to
 */
void CBEClass::WriteMemberVariables(CBEHeaderFile *pFile)
{
    if (pFile->IsOfFileType(FILETYPE_CLIENTHEADER))
    {
	*pFile << "\tprotected:\n";
	pFile->IncIndent();
	*pFile << "\t/* contains the address of the server */\n";
	*pFile << "\tCORBA_Object_base _dice_server;\n";
	*pFile << "\n";
	pFile->DecIndent();
    }
    if (pFile->IsOfFileType(FILETYPE_COMPONENTHEADER))
    {
	*pFile << "\tprotected:\n";
	pFile->IncIndent();
	assert(m_pCorbaObject);
	*pFile << "\t";
	m_pCorbaObject->WriteDeclaration(pFile);
	*pFile << ";\n";

	assert(m_pCorbaEnv);
	*pFile << "\t";
	m_pCorbaEnv->WriteDeclaration(pFile);
	*pFile << ";\n";
	pFile->DecIndent();
    }
}

/** \brief write constructor for C++ class
 *  \param pFile the file to write to
 *
 * At the client side do take the ID of the server as an argument. At the
 * server side to not write arguments.
 *
 * For both sides define a private copy constructor.
 */
void CBEClass::WriteConstructor(CBEHeaderFile *pFile)
{
    
    *pFile << "\tpublic:\n";
    pFile->IncIndent();
    *pFile << "\t/* construct the client object */\n";
    *pFile << "\t";
    WriteClassName(pFile);
    if (pFile->IsOfFileType(FILETYPE_CLIENT))
    {
     	*pFile << " (CORBA_Object_base _server)\n";
	*pFile << "\t : _dice_server(_server)\n";
    }
    else /* COMPONENT */
    {
	*pFile << " (void)\n";
	assert(m_pCorbaObject);
	string sObj = m_pCorbaObject->m_Declarators.First()->GetName();
	*pFile << "\t : " << sObj << "(),\n";
	
	assert(m_pCorbaEnv);
	string sEnv = m_pCorbaEnv->m_Declarators.First()->GetName();
	*pFile << "\t   " << sEnv << "()\n";
    }
    *pFile << "\t{ }\n";
    *pFile << "\n";

    pFile->DecIndent();
    *pFile << "\tprivate:\n";
    pFile->IncIndent();
    *pFile << "\t/* copy constructor to avoid implicit assignment of object */\n";
    *pFile << "\t";
    WriteClassName(pFile);
    *pFile << " (const ";
    WriteClassName(pFile);
    *pFile << "&)\n";
    *pFile << "\t{ }\n";
    *pFile << "\n";
    pFile->DecIndent();
}

/** \brief write the destructore for C++ class
 *  \param pFile the file to write to
 */
void CBEClass::WriteDestructor(CBEHeaderFile *pFile)
{
    *pFile << "\tpublic:\n";
    pFile->IncIndent();
    *pFile << "\t/* destruct client object */\n";
    *pFile << "\tvirtual ~";
    WriteClassName(pFile);
    *pFile << " ()\n";
    *pFile << "\t{ }\n";
    *pFile << "\n";
    pFile->DecIndent();
}

/** \brief writes the class to the target file
 *  \param pFile the file to write to
 *
 * With the C implementation the class simply calls the Write methods
 * of its constants, typedefs and functions
 */
void CBEClass::Write(CBEImplementationFile *pFile)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s called\n", __func__);
    // write implementation for functions
    WriteElements(pFile);
    // write helper functions if any
    WriteHelperFunctions(pFile);

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s finished\n", __func__);
}

/** \brief writes the class to the target file
 *  \param pFile the file to write to
 *
 * With the C implementation the class simply calls the Write methods
 * of its constants and typedefs. The functions are written separetly, because
 * they might be inline and need a full definition of the message buffer which
 * in turn needs all types to be declared.
 */
void CBEClass::WriteElements(CBEHeaderFile *pFile)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::%s for %s called\n", __func__, GetName().c_str());

    // sort our members/elements depending on source line number
    // into extra vector
    CreateOrderedElementList();

    // members are public
    if (CCompiler::IsBackEndLanguageSet(PROGRAM_BE_CPP))
    {
	*pFile << "\tpublic:\n";
	pFile->IncIndent();
    }

    // write target file
    vector<CObject*>::iterator iter = m_vOrderedElements.begin();
    for (; iter != m_vOrderedElements.end(); iter++)
    {
	if (dynamic_cast<CBEFunction*>(*iter))
	    continue;

	WriteLineDirective(pFile, *iter);
        if (dynamic_cast<CBEConstant*>(*iter))
            WriteConstant((CBEConstant*)(*iter), pFile);
        else if (dynamic_cast<CBETypedef*>(*iter))
            WriteTypedef((CBETypedef*)(*iter), pFile);
        else if (dynamic_cast<CBEType*>(*iter))
            WriteTaggedType((CBEType*)(*iter), pFile);
    }

    if (CCompiler::IsBackEndLanguageSet(PROGRAM_BE_CPP))
	pFile->DecIndent();

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::Write(head, %s) finished\n", GetName().c_str());
}

/** \brief writes the function declarations for the header file
 *  \param pFile the header file to write to
 */
void
CBEClass::WriteFunctions(CBEHeaderFile *pFile)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL,
	"CBEClass::%s(head, %s) called\n", __func__, GetName().c_str());

    int nFuncCount = GetFunctionWriteCount(pFile);
    
    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s %d functions\n",
	__func__, nFuncCount);

    if (nFuncCount > 0)
    {
	// members are public
	if (CCompiler::IsBackEndLanguageSet(PROGRAM_BE_CPP))
	{
	    *pFile << "\tpublic:\n";
	    pFile->IncIndent();
	}
	WriteExternCStart(pFile);
    }
    // write target functions in ordered appearance
    vector<CObject*>::iterator iter = m_vOrderedElements.begin();
    for (; iter != m_vOrderedElements.end(); iter++)
    {
	if (dynamic_cast<CBEFunction*>(*iter))
	{
	    WriteLineDirective(pFile, *iter);
	    WriteFunction((CBEFunction*)(*iter), pFile);
	}
    }
    // write helper functions if any
    if (nFuncCount > 0)
    {
	WriteExternCEnd(pFile);
	if (CCompiler::IsBackEndLanguageSet(PROGRAM_BE_CPP))
	    pFile->DecIndent();
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s(head, %s) finished\n",
	__func__, GetName().c_str());
}

/** \brief writes the class to the target file
 *  \param pFile the file to write to
 *
 * With the C implementation the class simply calls it's
 * function's Write method.
 */
void CBEClass::WriteElements(CBEImplementationFile *pFile)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL,
	"CBEClass::Write(impl, %s) called\n", GetName().c_str());

    // sort our members/elements depending on source line number
    // into extra vector
    CreateOrderedElementList();
    WriteFunctions(pFile);

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::Write(impl, %s) finished\n", GetName().c_str());
}

/** \brief writes the function declarations for the implementation file
 *  \param pFile the implementation file to write to
 */
void
CBEClass::WriteFunctions(CBEImplementationFile *pFile)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL,
	"CBEClass::%s(impl, %s) called\n", __func__, GetName().c_str());

    int nFuncCount = GetFunctionWriteCount(pFile);
    
    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s %d functions\n",
	__func__, nFuncCount);

    if (nFuncCount > 0)
    {
        WriteExternCStart(pFile);
    }
    // write target functions in ordered appearance
    vector<CObject*>::iterator iter = m_vOrderedElements.begin();
    for (; iter != m_vOrderedElements.end(); iter++)
    {
        if (dynamic_cast<CBEFunction*>(*iter))
        {
	    WriteLineDirective(pFile, *iter);
            WriteFunction((CBEFunction*)(*iter), pFile);
        }
    }
    // write helper functions if any
    if (nFuncCount > 0)
    {
        WriteExternCEnd(pFile);
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s(impl, %s) finished\n",
	__func__, GetName().c_str());
}

/** \brief writes the line directive
 *  \param pFile the file to write to
 *  \param pObj the object for which to write the line directive
 */
void
CBEClass::WriteLineDirective(CBEFile *pFile,
    CObject *pObj)
{
    if (!CCompiler::IsOptionSet(PROGRAM_GENERATE_LINE_DIRECTIVE))
	return;
    *pFile << "# " << pObj->GetSourceLine() << " \"" <<
	pObj->GetSourceFileName() << "\"\n";
}

/** \brief writes a constant
 *  \param pConstant the constant to write
 *  \param pFile the file to write to
 */
void CBEClass::WriteConstant(CBEConstant *pConstant,
    CBEHeaderFile *pFile)
{
    assert(pConstant);
    if (!pConstant->IsTargetFile(pFile))
	return;

    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s called\n", __func__);

    pConstant->Write(pFile);

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s finished\n", __func__);
}

/** \brief write a type definition
 *  \param pTypedef the type definition to write
 *  \param pFile the file to write to
 */
void CBEClass::WriteTypedef(CBETypedef *pTypedef,
    CBEHeaderFile *pFile)
{
    assert(pTypedef);
    if (!pTypedef->IsTargetFile(pFile))
	return;

    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s called\n", __func__);

    pTypedef->WriteDeclaration(pFile);

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s finished\n", __func__);
}

/** \brief write a function to the header file
 *  \param pFunction the function to write
 *  \param pFile the file to write to
 */
void CBEClass::WriteFunction(CBEFunction *pFunction,
    CBEHeaderFile *pFile)
{
    assert(pFunction);
    assert(pFile);

    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s(%s, %s) called\n",
	__func__, pFunction->GetName().c_str(), pFile->GetFileName().c_str());

    if (pFunction->DoWriteFunction(pFile))
    {
        pFunction->Write(pFile);
        *pFile << "\n";
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s finished\n", __func__);
}

/** \brief write a function to the implementation file
 *  \param pFunction the function to write
 *  \param pFile the file to write to
 */
void CBEClass::WriteFunction(CBEFunction *pFunction,
    CBEImplementationFile *pFile)
{
    assert(pFunction);
    assert(pFile);

    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s(%s, %s) called\n",
	__func__, pFunction->GetName().c_str(), pFile->GetFileName().c_str());

    if (pFunction->DoWriteFunction(pFile))
    {
        pFunction->Write(pFile);
        *pFile << "\n";
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s finished\n", __func__);
}

/** \brief calculates the function identifier
 *  \param pFEOperation the front-end operation to get the identifier for
 *  \return the function identifier
 *
 * The function identifier is unique within an interface. Usually an
 * interface's scope is determined by its definition. But since the server
 * loop using the function identifier only differentiates number, we have to
 * be unique within the same interface number.
 *
 * It is sufficient to check base interfaces for their interface ID, because
 * derived interfaces will use this algorithm as well and thus regard our
 * generated function ID's.
 *
 * For all interfaces with the same ID as the operation's interface, we build
 * a grid of predefined function ID's. In the next step we iterate over the
 * functions and count them, skipping the predefined numbers. When we finally
 * reached the given function, we return the calculated function ID.
 *
 * But first of all, the Uuid attribute of a function determines it's ID. To
 * print warnings about same Uuids anyways, we test for this situation after
 * the FindPredefinedNumbers call.
 */
int CBEClass::GetOperationNumber(CFEOperation *pFEOperation)
{
    assert(pFEOperation);
    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, 
	"CBEClass::GetOperationNumber for %s\n",
	pFEOperation->GetName().c_str());
    CFEInterface *pFEInterface = pFEOperation->GetSpecificParent<CFEInterface>();
    int nInterfaceNumber = GetInterfaceNumber(pFEInterface);
    // search for base interfaces with same interface number and collect them
    vector<CFEInterface*> vSameInterfaces;
    FindInterfaceWithNumber(pFEInterface, nInterfaceNumber, &vSameInterfaces);
    // now search for predefined numbers
    map<unsigned int, string> functionIDs;
    vSameInterfaces.push_back(pFEInterface); // add this interface too
    FindPredefinedNumbers(&vSameInterfaces, &functionIDs);
    // find pFEInterface in vector
    vector<CFEInterface*>::iterator iterI = std::find(vSameInterfaces.begin(),
	vSameInterfaces.end(), pFEInterface);
    if (iterI != vSameInterfaces.end())
	vSameInterfaces.erase(iterI);
    // check for Uuid attribute and return its value
    if (HasUuid(pFEOperation))
        return GetUuid(pFEOperation);
    // now there is no Uuid, thus we have to calculate the function id
    // get the maximum function ID of the interfaces with the same number
    int nFunctionID = 0;
    iterI = vSameInterfaces.begin();
    while (iterI != vSameInterfaces.end())
    {
        CFEInterface *pCurrI = *iterI++;
        if (!pCurrI)
            continue;
        // we sum the max opcodes, because thy have to be disjunct
	// if we would use max(current, functionID) the function ID would be
	// the same ...
        //
        // interface 1: 1, 2, 3, 4    \_ if max
        // interface 2: 1, 2, 3, 4, 5 /
        //
        // interface 1: 1, 2, 3, 4    \_ if sum
        // interface 2: 5, 6, 7, 8, 9 /
        //
        nFunctionID += GetMaxOpcodeNumber(pCurrI);
    }
    // now iterate over functions, incrementing a counter, check if new value
    // is predefined and if the checked function is this function, then return
    // the counter value
    vector<CFEOperation*>::iterator iterO;
    for (iterO = pFEInterface->m_Operations.begin();
	 iterO != pFEInterface->m_Operations.end();
	 iterO++)
    {
        nFunctionID++;
        while (IsPredefinedID(&functionIDs, nFunctionID)) nFunctionID++;
        if ((*iterO) == pFEOperation)
            return nFunctionID;
    }
    // something went wrong -> if we are here, the operations parent interface
    // does not have this operation as a child
    assert(false);
    return 0;
}
/** \brief checks if a function ID is predefined
 *  \param pFunctionIDs the predefined IDs
 *  \param nNumber the number to test
 *  \return true if number is predefined
 */
bool
CBEClass::IsPredefinedID(map<unsigned int, string> *pFunctionIDs,
    int nNumber)
{
    return pFunctionIDs->find(nNumber) != pFunctionIDs->end();
}

/** \brief calculates the maximum function ID
 *  \param pFEInterface the interface to test
 *  \return the maximum function ID for this interface
 *
 * Theoretically the max opcode count is the number of its operations. This
 * can be calculated calling CFEInterface::GetOperationCount(). This estimate
 * can be wrong if the interface contains functions with uuid's, which are
 * bigger than the operation count.
 */
int CBEClass::GetMaxOpcodeNumber(CFEInterface *pFEInterface)
{
    int nMax = pFEInterface->GetOperationCount(false);
    // now check operations
    vector<CFEOperation*>::iterator iter;
    for (iter = pFEInterface->m_Operations.begin();
	 iter != pFEInterface->m_Operations.end();
	 iter++)
    {
        // if this function has uuid, we check if uuid is bigger than count
        if (!HasUuid(*iter))
            continue;
        // now check
        int nUuid = GetUuid(*iter);
        if (nUuid > nMax)
            nMax = nUuid;
    }
    return nMax;
}

/** \brief returns the functions Uuid
 *  \param pFEOperation the operation to get the UUID from
 *  \return the Uuid or -1 if none exists
 */
int CBEClass::GetUuid(CFEOperation *pFEOperation)
{
    CFEAttribute *pUuidAttr = pFEOperation->m_Attributes.Find(ATTR_UUID);
    if (pUuidAttr)
    {
        if (dynamic_cast<CFEIntAttribute*>(pUuidAttr))
        {
            return ((CFEIntAttribute*)pUuidAttr)->GetIntValue();
        }
    }
    return -1;
}

/** \brief checks if this functio has a Uuid
 *  \param pFEOperation the operation to check
 *  \return true if this operation has a Uuid
 */
bool CBEClass::HasUuid(CFEOperation *pFEOperation)
{
    CFEAttribute *pUuidAttr = pFEOperation->m_Attributes.Find(ATTR_UUID);
    if (pUuidAttr)
    {
        if (dynamic_cast<CFEIntAttribute*>(pUuidAttr))
        {
            return true;
        }
    }
    return false;
}

/** \brief calculates the number used as base opcode number
 *  \param pFEInterface the interface to check.
 *  \return the interface number
 *
 * AN interface number is first of all it's uuid. If it is not available
 * the base interfaces are counted and this interface's number is the
 * highest number of it's base interfaces + 1.
 */
int CBEClass::GetInterfaceNumber(CFEInterface *pFEInterface)
{
    assert(pFEInterface);

    CFEAttribute *pUuidAttr = pFEInterface->m_Attributes.Find(ATTR_UUID);
    if (pUuidAttr)
    {
        if (dynamic_cast<CFEIntAttribute*>(pUuidAttr))
        {
            return ((CFEIntAttribute*)pUuidAttr)->GetIntValue();
        }
    }

    int nNumber = 1;
    vector<CFEInterface*>::iterator iterI;
    for (iterI = pFEInterface->m_BaseInterfaces.begin();
	 iterI != pFEInterface->m_BaseInterfaces.end();
	 iterI++)
    {
        int nBaseNumber = GetInterfaceNumber(*iterI);
        if (nBaseNumber >= nNumber)
            nNumber = nBaseNumber+1;
    }
    return nNumber;
}

/** \brief searches all interfaces with the same interface number
 *  \param pFEInterface its base interfaces are searched
 *  \param nNumber the number to compare with
 *  \param pCollection the vector to add the same interfaces to
 *
 * We only search the base interfaces, because if we would search derived
 * interfaces as well, we might produce different client bindings if a base
 * interface is suddenly derived from. Instead, we generate errors in the
 * derived interface, that function id's might overlap.
 */
int
CBEClass::FindInterfaceWithNumber(CFEInterface *pFEInterface,
    int nNumber,
    vector<CFEInterface*> *pCollection)
{
    assert(pCollection);
    int nCount = 0;
    // search base interfaces
    vector<CFEInterface*>::iterator iter;
    for (iter = pFEInterface->m_BaseInterfaces.begin();
	 iter != pFEInterface->m_BaseInterfaces.end();
	 iter++)
    {
        if (GetInterfaceNumber(*iter) == nNumber)
        {
            // check if we already got this interface (use pointer)
            vector<CFEInterface*>::const_iterator iterI;
            for (iterI = pCollection->begin();
		 iterI != pCollection->end();
		 iterI++)
            {
                if (*iterI == *iter)
                    break; // stops for(iterI) loop
            }
            if (iterI == pCollection->end()) // no match found
                pCollection->push_back(*iter);
            nCount++;
        }
        nCount += FindInterfaceWithNumber(*iter, nNumber, pCollection);
    }
    return nCount;
}

/** \brief find predefined function IDs
 *  \param pCollection the vector containing the interfaces to test
 *  \param pNumbers the array containing the numbers
 *  \return number of predefined function IDs
 *
 * To find predefined function id'swe have to iterate over the interface's
 * function, check if they have a UUID attribute and get it's number. If it
 * has a number we extend the pNumber array and add this number at the correct
 * position (the array is ordered). If the number exists already, we print a
 * warning containing both function's names and the function ID.
 */
int
CBEClass::FindPredefinedNumbers(vector<CFEInterface*> *pCollection,
    map<unsigned int, string> *pNumbers)
{
    assert (pCollection);
    assert (pNumbers);
    int nCount = 0;
    // iterate over interfaces with same interface number
    vector<CFEInterface*>::iterator iterI = pCollection->begin();
    CFEInterface *pFEInterface;
    while (iterI != pCollection->end())
    {
        pFEInterface = *iterI++;
        if (!pFEInterface)
            continue;
        // iterate over current interface's operations
        vector<CFEOperation*>::iterator iterO;
	for (iterO = pFEInterface->m_Operations.begin();
	     iterO != pFEInterface->m_Operations.end();
	     iterO++)
        {
            // check if operation has Uuid attribute
            if (HasUuid(*iterO))
            {
                int nOpNumber = GetUuid(*iterO);
                // check if this number is already defined somewhere
		if (pNumbers->find(nOpNumber) != pNumbers->end())
		{
		    if (CCompiler::IsWarningSet(PROGRAM_WARNING_IGNORE_DUPLICATE_FID))
		    {
			CCompiler::GccWarning(*iterO, 0,
			    "Function \"%s\" has same Uuid (%d) as function \"%s\"",
			    (*iterO)->GetName().c_str(), nOpNumber,
			    (*pNumbers)[nOpNumber].c_str());
			break;
		    }
		    else
		    {
			CCompiler::GccError(*iterO, 0,
			    "Function \"%s\" has same Uuid (%d) as function \"%s\"",
			    (*iterO)->GetName().c_str(), nOpNumber,
			    (*pNumbers)[nOpNumber].c_str());
			exit(1);
		    }
		}
		// no element with this number is present

		// check if this number might collide with base interface numbers
		CheckOpcodeCollision(pFEInterface, nOpNumber, pCollection,
		    *iterO);
		// add uuid attribute to list
		pNumbers->insert(std::make_pair(nOpNumber,
			(*iterO)->GetName()));
		// incremenet count
		nCount++;
            }
        }
    }
    return nCount;
}

/** \brief checks if the opcode could be used by base interfaces
 *  \param pFEInterface the currently investigated interface
 *  \param nOpNumber the opcode to check for
 *  \param pCollection the collection of predefined function ids
 *  \param pFEOperation a reference to the currently tested function
 */
int
CBEClass::CheckOpcodeCollision(CFEInterface *pFEInterface,
    int nOpNumber,
    vector<CFEInterface*> *pCollection,
    CFEOperation *pFEOperation)
{
    int nBaseNumber = 0;
    vector<CFEInterface*>::iterator iterI;
    for (iterI = pFEInterface->m_BaseInterfaces.begin();
	 iterI != pFEInterface->m_BaseInterfaces.end();
	 iterI++)
    {
        // test current base interface only if numbers are identical
        if (GetInterfaceNumber(pFEInterface) !=
	    GetInterfaceNumber(*iterI))
            continue;
        // first check the base interface's base interfaces
        nBaseNumber += CheckOpcodeCollision(*iterI, nOpNumber, pCollection,
	    pFEOperation);
        // now check interface's range
        nBaseNumber += GetMaxOpcodeNumber(*iterI);
        if ((nOpNumber > 0) && (nOpNumber <= nBaseNumber))
        {
            if (CCompiler::IsWarningSet(PROGRAM_WARNING_IGNORE_DUPLICATE_FID))
            {
                CCompiler::GccWarning(pFEOperation, 0,
		    "Function \"%s\" has Uuid (%d) which is used by compiler for base interface \"%s\"",
		    pFEOperation->GetName().c_str(), nOpNumber,
		    (*iterI)->GetName().c_str());
                break;
            }
            else
            {
                CCompiler::GccError(pFEOperation, 0, "Function \"%s\" has Uuid (%d) which is used by compiler for interface \"%s\"",
		    pFEOperation->GetName().c_str(), nOpNumber,
		    (*iterI)->GetName().c_str());
                exit(1);
            }
        }
    }
    // return checked range
    return nBaseNumber;
}

/** \brief tries to find the function group for a specific function
 *  \param pFunction the function to search for
 *  \return a reference to the function group or 0
 */
CFunctionGroup* CBEClass::FindFunctionGroup(CBEFunction *pFunction)
{
    // iterate over function groups
    vector<CFunctionGroup*>::iterator iter;
    for (iter = m_FunctionGroups.begin();
	 iter != m_FunctionGroups.end();
	 iter++)
    {
        // iterate over its functions
        vector<CBEFunction*>::iterator iterF;
	for (iterF = (*iter)->m_Functions.begin();
	     iterF != (*iter)->m_Functions.end();
	     iterF++)
        {
            if (*iterF == pFunction)
                return *iter;
        }
    }
    return 0;
}

/** \brief tries to find a type definition
 *  \param sTypeName the name of the searched type
 *  \return a reference to the type definition
 *
 * We also have to check the message buffer type (if existent).
 */
CBETypedef* CBEClass::FindTypedef(string sTypeName)
{
    vector<CBETypedef*>::iterator iter;
    for (iter = m_Typedefs.begin();
	 iter != m_Typedefs.end();
	 iter++)
    {
        if ((*iter)->m_Declarators.Find(sTypeName))
            return *iter;
        if ((*iter)->GetType() &&
	    (*iter)->GetType()->HasTag(sTypeName))
	    return *iter;
    }
    CBETypedef *pMsgBuf = GetMessageBuffer();
    if (pMsgBuf)
    {
        if (pMsgBuf->m_Declarators.Find(sTypeName))
            return pMsgBuf;
        if (pMsgBuf->GetType() &&
            pMsgBuf->GetType()->HasTag(sTypeName))
            return pMsgBuf;
    }
    return 0;
}

/** \brief test if this class belongs to the file
 *  \param pFile the file to test
 *  \return true if the given file is a target file for the class
 *
 * A file is a target file for the class if its the target file for at least
 * one function.
 */
bool CBEClass::IsTargetFile(CBEImplementationFile * pFile)
{
    vector<CBEFunction*>::iterator iter;
    for (iter = m_Functions.begin();
	 iter != m_Functions.end();
	 iter++)
    {
        if ((*iter)->IsTargetFile(pFile))
            return true;
    }
    return false;
}

/** \brief test if this class belongs to the file
 *  \param pFile the file to test
 *  \return true if the given file is a target file for the class
 *
 * A file is a target file for the class if its teh target class for at least
 * one function.
 */
bool CBEClass::IsTargetFile(CBEHeaderFile * pFile)
{
    vector<CBEFunction*>::iterator iter;
    for (iter = m_Functions.begin();
	 iter != m_Functions.end();
	 iter++)
    {
        if ((*iter)->IsTargetFile(pFile))
            return true;
    }
    return false;
}

/** \brief searches for a type using its tag
 *  \param nType the type of the searched type
 *  \param sTag the tag to search for
 *  \return a reference to the type
 */
CBEType* CBEClass::FindTaggedType(int nType, string sTag)
{
    vector<CBEType*>::iterator iter;
    for (iter = m_TypeDeclarations.begin();
	 iter != m_TypeDeclarations.end();
	 iter++)
    {
        int nFEType = (*iter)->GetFEType();
        if (nType != nFEType)
            continue;
        if (nFEType == TYPE_STRUCT ||
	    nFEType == TYPE_UNION ||
	    nFEType == TYPE_ENUM)
        {
            if ((*iter)->HasTag(sTag))
                return *iter;
        }
    }
    return 0;
}

/** \brief tries to create a new back-end representation of a tagged type declaration
 *  \param pFEType the respective front-end type
 *  \return true if successful
 */
void
CBEClass::CreateBackEnd(CFEConstructedType *pFEType)
{
    CCompiler::VerboseI(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s(constr type) called\n",
	__func__);

    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEType *pType = pCF->GetNewType(pFEType->GetType());
    m_TypeDeclarations.Add(pType);
    pType->SetParent(this);
    try
    {
	pType->CreateBackEnd(pFEType);
    }
    catch (CBECreateException *e)
    {
	m_TypeDeclarations.Remove(pType);
        delete pType;
	e->Print();
	delete e;

	string exc = string(__func__);
	exc += " failed because tagged type could not be created";
	throw new CBECreateException(exc);
    }
    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, "CBEClass::%s(constr type) returns\n",
	__func__);
}

/** \brief writes a tagged type declaration
 *  \param pType the type to write
 *  \param pFile the file to write to
 */
void CBEClass::WriteTaggedType(CBEType *pType,
    CBEHeaderFile *pFile)
{
    assert(pType);
    if (!pType->IsTargetFile(pFile))
	return;
    // get tag
    string sTag;
    if (dynamic_cast<CBEStructType*>(pType))
        sTag = ((CBEStructType*)pType)->GetTag();
    if (dynamic_cast<CBEUnionType*>(pType))
        sTag = ((CBEUnionType*)pType)->GetTag();
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    sTag = pNF->GetTypeDefine(sTag);
    *pFile << "#ifndef " << sTag << "\n";
    *pFile << "#define " << sTag << "\n";
    pType->Write(pFile);
    *pFile << ";\n";
    *pFile << "#endif /* !" << sTag << " */\n";
    *pFile << "\n";
}

/** \brief searches for a function with the given type
 *  \param sTypeName the name of the type to look for
 *  \param pFile the file to write to (its used to test if a function shall be written)
 *  \return true if a parameter of that type is found
 *
 * Search functions for a parameter with that type.
 */
bool CBEClass::HasFunctionWithUserType(string sTypeName, CBEFile *pFile)
{
    vector<CBEFunction*>::iterator iter;
    for (iter = m_Functions.begin();
	 iter != m_Functions.end();
	 iter++)
    {
        if (dynamic_cast<CBEHeaderFile*>(pFile) &&
            (*iter)->DoWriteFunction((CBEHeaderFile*)pFile) &&
            (*iter)->FindParameterType(sTypeName))
            return true;
        if (dynamic_cast<CBEImplementationFile*>(pFile) &&
            (*iter)->DoWriteFunction((CBEImplementationFile*)pFile) &&
            (*iter)->FindParameterType(sTypeName))
            return true;
    }
    return false;
}

/** \brief count parameters according to their set and not set attributes
 *  \param nMustAttrs the attribute that must be set to count a parameter
 *  \param nMustNotAttrs the attribute that must not be set to count a parameter
 *  \param nDirection the direction to count
 *  \return the number of parameter with or without the specified attributes
 */
int 
CBEClass::GetParameterCount(ATTR_TYPE nMustAttrs,
    ATTR_TYPE nMustNotAttrs,
    int nDirection)
{
    if (nDirection == 0)
    {
        int nCountIn = GetParameterCount(nMustAttrs, nMustNotAttrs,
	    DIRECTION_IN);
        int nCountOut = GetParameterCount(nMustAttrs, nMustNotAttrs,
	    DIRECTION_OUT);
        return std::max(nCountIn, nCountOut);
    }

    int nCount = 0, nCurr;
    vector<CFunctionGroup*>::iterator iterG;
    for (iterG = m_FunctionGroups.begin();
	 iterG != m_FunctionGroups.end();
	 iterG++)
    {
        vector<CBEFunction*>::iterator iterF;
	for (iterF = (*iterG)->m_Functions.begin();
	     iterF != (*iterG)->m_Functions.end();
	     iterF++)
        {
	    if (!dynamic_cast<CBEOperationFunction*>(*iterF))
		continue;
            nCurr = (*iterF)->GetParameterCount(nMustAttrs, nMustNotAttrs,
		nDirection);
            nCount = std::max(nCount, nCurr);
        }
    }

    return nCount;
}

/** \brief try to find functions with the given attribute
 *  \param nAttribute the attribute to find
 *  \return true if such a function could be found
 */
bool 
CBEClass::HasFunctionWithAttribute(ATTR_TYPE nAttribute)
{
    // check own functions
    vector<CBEFunction*>::iterator iter;
    for (iter = m_Functions.begin();
	 iter != m_Functions.end();
	 iter++)
    {
	if ((*iter)->m_Attributes.Find(nAttribute))
	    return true;
    }
    // check base classes
    vector<CBEClass*>::iterator iterC;
    for (iterC = m_vBaseClasses.begin();
	 iterC != m_vBaseClasses.end();
	 iterC++)
    {
        if ((*iterC)->HasFunctionWithAttribute(nAttribute))
            return true;
    }
    // nothing found
    return false;
}

/** \brief try to find functions with parameters with the given attributes
 *  \param nAttribute1 the first attribute
 *  \param nAttribute2 the second attribute
 *  \return true if such function exists
 */
bool
CBEClass::HasParametersWithAttribute(ATTR_TYPE nAttribute1,
    ATTR_TYPE nAttribute2)
{
    // check own functions
    vector<CBEFunction*>::iterator iter;
    for (iter = m_Functions.begin();
	 iter != m_Functions.end();
	 iter++)
    {
	CBETypedDeclarator *pParameter;
        if ((pParameter = (*iter)->FindParameterAttribute(nAttribute1)) != 0)
        {
            if (nAttribute2 == ATTR_NONE)
                return true;
            if (pParameter->m_Attributes.Find(nAttribute2))
		return true;
        }
    }
    // check base classes
    vector<CBEClass*>::iterator iterC;
    for (iterC = m_vBaseClasses.begin();
	 iterC != m_vBaseClasses.end();
	 iterC++)
    {
        if ((*iterC)->HasParametersWithAttribute(nAttribute1, nAttribute2))
            return true;
    }
    // nothing found
    return false;
}

/** \brief creates a list of ordered elements
 *
 * This method iterates each member vector and inserts their
 * elements into the ordered element list using bubble sort.
 * Sort criteria is the source line number.
 */
void CBEClass::CreateOrderedElementList(void)
{
    // clear vector
    m_vOrderedElements.clear();
    // typedef
    vector<CBEFunction*>::iterator iterF;
    for (iterF = m_Functions.begin();
	 iterF != m_Functions.end();
	 iterF++)
    {
        InsertOrderedElement(*iterF);
    }
    // typedef
    vector<CBETypedef*>::iterator iterT;
    for (iterT = m_Typedefs.begin();
	 iterT != m_Typedefs.end();
	 iterT++)
    {
        InsertOrderedElement(*iterT);
    }
    // tagged types
    vector<CBEType*>::iterator iterTa;
    for (iterTa = m_TypeDeclarations.begin();
	 iterTa != m_TypeDeclarations.end();
	 iterTa++)
    {
        InsertOrderedElement(*iterTa);
    }
    // consts
    vector<CBEConstant*>::iterator iterC;
    for (iterC = m_Constants.begin();
	 iterC != m_Constants.end();
	 iterC++)
    {
        InsertOrderedElement(*iterC);
    }
}

/** \brief insert one element into the ordered list
 *  \param pObj the new element
 *
 * This is the insert implementation
 */
void CBEClass::InsertOrderedElement(CObject *pObj)
{
    // get source line number
    int nLine = pObj->GetSourceLine();
    // search for element with larger number
    vector<CObject*>::iterator iter = m_vOrderedElements.begin();
    for (; iter != m_vOrderedElements.end(); iter++)
    {
        if ((*iter)->GetSourceLine() > nLine)
        {
            // insert before that element
            m_vOrderedElements.insert(iter, pObj);
            return;
        }
    }
    // new object is bigger that all existing
    m_vOrderedElements.push_back(pObj);
}

/** \brief write helper functions, if any
 *  \param pFile the file to write to
 *
 * Writes platform specific helper functions.
 *
 * In case a default function was defined for the server loop we write its
 * declaration into the header file. The implementation has to be user
 * provided.
 */
void CBEClass::WriteHelperFunctions(CBEHeaderFile* pFile)
{
    // check for function prototypes
    if (!pFile->IsOfFileType(FILETYPE_COMPONENT))
	return;

    CBEAttribute *pAttr = m_Attributes.Find(ATTR_DEFAULT_FUNCTION);
    if (!pAttr)
	return;
    string sDefaultFunction = pAttr->GetString();
    if (sDefaultFunction.empty())
	return;

    WriteExternCStart(pFile);
    CBEMsgBuffer *pMsgBuffer = GetMessageBuffer();
    string sMsgBuffer = pMsgBuffer->m_Declarators.First()->GetName();
    // int \<name\>(\<corba object\>, \<msg buffer type\>*,
    //              \<corba environment\>*)
    *pFile << "\tint " << sDefaultFunction << " (CORBA_Object, " <<
	sMsgBuffer << "*, CORBA_Server_Environment*);\n";
    WriteExternCEnd(pFile);
}

/** \brief write helper functions, if any
 *  \param pFile the file to write to
 *
 * Writes platform specific helper functions.
 */
void CBEClass::WriteHelperFunctions(CBEImplementationFile* /*pFile*/)
{
}

/** \brief retrieve reference to the class' server loop function
 *  \return reference to the class' server loop function
 */
CBESrvLoopFunction* CBEClass::GetSrvLoopFunction()
{
    vector<CBEFunction*>::iterator iter;
    for (iter = m_Functions.begin();
	 iter != m_Functions.end();
	 iter++)
    {
        if (dynamic_cast<CBESrvLoopFunction*>(*iter))
            return (CBESrvLoopFunction*)(*iter);
    }
    return 0;
}

/** \brief writes the start of extern "C" statement
 *  \param pFile the file to write to
 */
void 
CBEClass::WriteExternCStart(CBEFile *pFile)
{
    if (CCompiler::IsBackEndLanguageSet(PROGRAM_BE_CPP))
	return;

    *pFile << "#ifdef __cplusplus\n" <<
        "extern \"C\" {\n" <<
        "#endif\n\n";
}

/** \brief writes the end of extern "C" statement
 *  \param pFile the file to write to
 */
void 
CBEClass::WriteExternCEnd(CBEFile *pFile)
{
    if (CCompiler::IsBackEndLanguageSet(PROGRAM_BE_CPP))
	return;

    *pFile << "#ifdef __cplusplus\n" <<
        "}\n" <<
        "#endif\n\n";
}

/** \brief creates the CORBA_Object variable (and member)
 *  \return true if successful
 */
void
CBEClass::CreateObject()
{
    // trace remove old object
    if (m_pCorbaObject)
    {
        delete m_pCorbaObject;
        m_pCorbaObject = 0;
    }
    
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    string exc = string(__func__);
    
    string sTypeName("CORBA_Object");
    string sName = pNF->GetCorbaObjectVariable();
    m_pCorbaObject = pCF->GetNewTypedDeclarator();
    m_pCorbaObject->SetParent(this);
    try
    {
	m_pCorbaObject->CreateBackEnd(sTypeName, sName, 0);
    }
    catch (CBECreateException *e)
    {
        delete m_pCorbaObject;
        m_pCorbaObject = 0;
	e->Print();
	delete e;

	exc += " failed, because CORBA Object could not be created.";
	throw new CBECreateException(exc);
    }
    // CORBA_Object is always in
    CBEAttribute *pAttr = pCF->GetNewAttribute();
    try
    {
	pAttr->CreateBackEnd(ATTR_IN);
    }
    catch (CBECreateException *e)
    {
        delete pAttr;
	e->Print();
	delete e;

	exc += " failed, because IN attribute for CORBA Object could not be" \
	    " created.";
        throw new CBECreateException(exc);
    }
    m_pCorbaObject->m_Attributes.Add(pAttr);
}

/** \brief creates the CORBA_Environment variable (and parameter)
 *  \return true if successful
 */
void
CBEClass::CreateEnvironment()
{
    // clean up
    if (m_pCorbaEnv)
    {
        delete m_pCorbaEnv;
        m_pCorbaEnv = 0;
    }

    CBENameFactory *pNF = CCompiler::GetNameFactory();
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    // if function is at server side, this is a CORBA_Server_Environment
    string sTypeName = "CORBA_Server_Environment";
    string sName = pNF->GetCorbaEnvironmentVariable();
    m_pCorbaEnv = pCF->GetNewTypedDeclarator();
    m_pCorbaEnv->SetParent(this);
    try
    {
	m_pCorbaEnv->CreateBackEnd(sTypeName, sName, 1);
    }
    catch (CBECreateException *e)
    {
        delete m_pCorbaEnv;
        m_pCorbaEnv = 0;
	throw;
    }
}

