#ifndef HX_SLAB_ALLOC_H__
#define HX_SLAB_ALLOC_H__

#include <l4/cxx/std_alloc.h>
#include <l4/sys/consts.h>

namespace cxx {

template< int Obj_size, int Slab_size = L4_PAGESIZE,
  int Max_free = 2, template<typename A> class Alloc = New_allocator >
class Base_slab
{
private:
  struct Free_o
  {
    Free_o *next;
  };

public:
  struct Slab_i;

private:
  struct Slab_head
  {
    unsigned num_free;
    Free_o *free;
    Base_slab<Obj_size, Slab_size, Max_free, Alloc> *cache;
    Slab_i *free_next;
    Slab_i *free_prev;
    Slab_i *next;
    Slab_i *prev;

    inline Slab_head() : num_free(0), free(0),
			 cache(0), free_next(0), free_prev(0),
			 next(0), prev(0) 
    {}
  };

public:
  enum
  {
    object_size      = Obj_size,
    slab_size        = Slab_size,
    objects_per_slab = (Slab_size - sizeof(Slab_head)) / object_size,
    max_free_slabs   = Max_free,
  };
  
  struct Slab_i
  {
    Slab_head _h;
    char _o[slab_size - sizeof(Slab_head)];
  };
  
public:
  typedef Alloc<Slab_i> Slab_alloc;

private:
  Slab_alloc _alloc;
  unsigned _num_free;
  unsigned _num_slabs;
  Slab_i *_slabs;
  Slab_i *_free_slabs;

  void sort_free_list()
  {
    Slab_i *s = _free_slabs;
    if (!s)
      return;

    if (s->_h.free_prev && (s->_h.free_prev->_h.num_free > s->_h.num_free))
      {
	do
	  {
	    Slab_i *tmp = s->_h.free_prev;
	    s->_h.free_prev = tmp->_h.free_prev;
	    if (s->_h.free_prev)
	      s->_h.free_prev->_h.free_next = s;

	    tmp->_h.free_next = s->_h.free_next;
	    if (tmp->_h.free_next)
	      tmp->_h.free_next->_h.free_prev = tmp;

	    s->_h.free_next = tmp;
	    tmp->_h.free_prev = s;
	  }
	while (s->_h.free_prev && 
	       (s->_h.num_free > s->_h.free_prev->_h.num_free));
      }

    if (s->_h.free_next && (s->_h.num_free > s->_h.free_next->_h.num_free))
      {
	do
	  {
	    Slab_i *tmp = s->_h.free_next;
	    s->_h.free_next = tmp->_h.free_next;
	    if (s->_h.free_next)
	      s->_h.free_next->_h.free_prev = s;

	    tmp->_h.free_prev = s->_h.free_prev;
	    if (tmp->_h.free_prev)
	      tmp->_h.free_prev->_h.free_next = tmp;
	    else
	      _free_slabs = tmp;

	    s->_h.free_prev = tmp;
	    tmp->_h.free_next = s;
	  }
	while (s->_h.free_next &&
	       (s->_h.num_free > s->_h.free_next->_h.num_free));
      }
  }

  void add_slab(Slab_i *s)
  {
    s->_h.num_free = objects_per_slab;
    s->_h.cache = this;

    // initialize free list
    Free_o *f = s->_h.free = reinterpret_cast<Free_o*>(s->_o);
    for (unsigned i = 0; i < objects_per_slab; ++i)
      {
	f->next = reinterpret_cast<Free_o*>(s->_o + i*object_size);
	f = f->next;
      }
    f->next = 0;

    // insert slab into cache's list
    s->_h.next = _slabs;
    if (_slabs)
      _slabs->_h.prev = s;
    _slabs = s;
    ++_num_slabs;

    // insert slab into cache's list of free slabs
    s->_h.free_next = _free_slabs;
    if (_free_slabs)
      _free_slabs->_h.free_prev = s;
    _free_slabs = s;
    ++_num_free;

    sort_free_list();
  }

  bool grow()
  {
    Slab_i *s = _alloc.alloc();
    if (!s)
      return false;

    new (s) Slab_i();

    add_slab(s);
    return true;
  }

  void shrink()
  {
    if (!_alloc.can_free)
      return;

    Slab_i *s = _free_slabs;
    while (_num_free > max_free_slabs)
      {
	if (!s)
	  return;

	if (s->_h.num_free == objects_per_slab)
	  {
	    if (!s->_h.free_prev)
	      {
		// first in list
		_free_slabs = _free_slabs->_h.free_next;
		if (_free_slabs)
		  _free_slabs->_h.free_prev = 0;
	      }
	    else
	      {
		s->_h.free_prev->_h.free_next = s->_h.free_next;
		if (s->_h.free_next)
		  s->_h.free_next->_h.free_prev = s->_h.free_prev;
	      }
	    --_num_free;
	    
	    if (!s->_h.prev)
	      {
		// first in list
		_slabs = _slabs->_h.next;
		if (_slabs)
		  _slabs->_h.prev = 0;
	      }
	    else
	      {
		s->_h.prev->_h.next = s->_h.next;
		if (s->_h.next)
		  s->_h.next->_h.prev = s->_h.prev;
	      }
	    --_num_slabs;

	      {
		Slab_i *f = s;
		s = s->_h.free_next;
		f->~Slab_i();
		_alloc.free(f);
	      }
	  }
	else
	  s = s->_h.free_next;
      }
  }

public:
  Base_slab(Slab_alloc const &alloc = Slab_alloc()) 
    : _alloc(alloc), _num_free(0), _num_slabs(0), _slabs(0), _free_slabs(0)
  {}

  ~Base_slab()
  {
    Slab_i *s = _slabs;
    while (s)
      {
	Slab_i *o = s;
	s = s->_h.next;
	_alloc.free(o);
      }
  }

  void *alloc()
  {
    if (!_free_slabs && !grow())
      return 0;

    Free_o *o = _free_slabs->_h.free;
    _free_slabs->_h.free = o->next;

    if (_free_slabs->_h.num_free == objects_per_slab)
      --_num_free;

    --(_free_slabs->_h.num_free);

    if (!_free_slabs->_h.free)
      {
	_free_slabs = _free_slabs->_h.free_next;
	if (_free_slabs)
	  _free_slabs->_h.free_prev = 0;
      }

    return o;
  }

  void free(void *_o)
  {
    if (!_o)
      return;

    unsigned long addr = (unsigned long)_o;
    addr = (addr / slab_size) * slab_size;
    Slab_i *s = (Slab_i*)addr;

    if (s->_h.cache != this)
      return;

    Free_o *o = reinterpret_cast<Free_o*>(_o);

    o->next = s->_h.free;
    s->_h.free = o;
    if (!s->_h.num_free)
      {
	s->_h.free_next = _free_slabs;
	s->_h.free_prev = 0;
	if (_free_slabs)
	  _free_slabs->_h.free_prev = s;
	_free_slabs = s;
      }

    ++(s->_h.num_free);

    if (s->_h.num_free == objects_per_slab)
      {
	++_num_free;
	if (_num_free > max_free_slabs)
	  shrink();
	else
	  sort_free_list();
      }
    else
      sort_free_list();
  }
  
};

template<typename Type, int Slab_size = L4_PAGESIZE,
  int Max_free = 2, template<typename A> class Alloc = New_allocator > 
class Slab : public Base_slab<sizeof(Type), Slab_size, Max_free, Alloc>
{
private:
  typedef Base_slab<sizeof(Type), Slab_size, Max_free, Alloc> Base_type;
public:
  Slab(typename Base_type::Slab_alloc const &alloc 
      = typename Base_type::Slab_alloc())
    : Base_slab<sizeof(Type), Slab_size, Max_free, Alloc>(alloc) {}

  Type *alloc() 
  { 
    return (Type*)Base_slab<sizeof(Type), Slab_size,
      Max_free, Alloc>::alloc(); 
  }
  
  void free(Type *o) 
  { Base_slab<sizeof(Type), Slab_size, Max_free, Alloc>::free(o); }
};

template< int Obj_size, int Slab_size = L4_PAGESIZE,
  int Max_free = 2, template<typename A> class Alloc = New_allocator >
class Base_slab_static
{
private:
  typedef Base_slab<Obj_size, Slab_size, Max_free, Alloc> _A;
  static _A _a;
public:
  void *alloc() { return _a.alloc(); }
  void free(void *p) { _a.free(p); }
};

template< int _O, int _S, int _M, template<typename A> class Alloc >
typename Base_slab_static<_O,_S,_M,Alloc>::_A 
  Base_slab_static<_O,_S,_M,Alloc>::_a; 

template<typename Type, int Slab_size = L4_PAGESIZE,
  int Max_free = 2, template<typename A> class Alloc = New_allocator > 
class Slab_static 
: public Base_slab_static<sizeof(Type), Slab_size, Max_free, Alloc>
{
public:
  Type *alloc() 
  { 
    return (Type*)Base_slab_static<sizeof(Type), Slab_size,
      Max_free, Alloc>::alloc(); 
  }
};

};

#endif

