/**
 *  Implementing the real functionality of the VTPM interface using RSA.
 */

#include <l4/crtx/ctor.h>
#include <l4/sys/types.h>
#include <l4/env/errno.h>

#include <malloc.h>
#include <string.h>
#include <assert.h>

#include "rand.h"
#include "vtpmif.h"

#include "rsaglue.h"

//#define NDEBUG

#ifdef NDEBUG
# define debug(...) 
#else
# include <stdio.h>
# define debug(...) printf(__VA_ARGS__)
#endif


struct vtpm_entry_t
{
	vtpm_data_t data;
	l4_threadid_t owner;
	struct vtpm_entry_t *next;
};

typedef struct vtpm_entry_t *vtpm_pentry_t;

#define FIND_ENTRY(entry,id)			\
{						\
	entry=find_entry(id);			\
	if (entry==NULL)			\
		return -L4_ENOTASK;		\
	assert(*entry);				\
}

#define CHECK_COUNT(count,mincount)		\
{						\
	if (count<mincount)			\
		return -L4_EINVAL;		\
}


enum {
	KEYBITS=512
};

vtpm_pentry_t head = NULL;
rsa_key_t mykey;



/**
 * find an entry and return the pointer holding it
 */
static vtpm_pentry_t *
find_entry(l4_threadid_t id)
{
	vtpm_pentry_t *res;       

	res=&head;

	while ((*res)&&(!l4_tasknum_equal((*res)->data.id,id)))
		res=&((*res)->next);
	if (*res)
		return res;
	else 
	{
	  //debug("entry #%x not found\n",id.id.task);
		return NULL;
	}
}


/**
 * Return the number of items in the database.
 *
 * This is only a debugging function.
 */
int
vtpm_count(void)
{
	vtpm_pentry_t e;
	int count;
	
	count=0;
	e=head;
	while (e!=NULL)
	{
		count+=1;
		debug("owner #%x id #%x parent #%x ",e->owner.id.task,e->data.id.id.task,e->data.parent.id.task);
		debug("name '%s' hash '%s'\n",e->data.name,e->data.hash);
		e=e->next;
	}
	return count;
}


/**
 * Insert a hash of a new started task in the hash database.
 *
 * If the parent does not exist L4_ENOTASK is returned. Use NilThread as the first parent.
 */
int
vtpm_add    (l4_threadid_t owner_id,
	     l4_threadid_t new_id,
	     l4_threadid_t parent_id,
	     const char *name,
	     const char *hash)
{
	vtpm_pentry_t e;

	debug("\tadd #%x name %s\n",new_id.id.task,name);

	if (find_entry(new_id)!=NULL)
		return -L4_EEXISTS;

	if (l4_is_nil_id(parent_id) && (find_entry(parent_id)==NULL))
		return -L4_ENOTASK;
              
	e=malloc(sizeof(struct vtpm_entry_t));	
	if (e==NULL)
		return -L4_ENOMEM;

	debug("\tadd e: %p\n",e);

	e->owner=owner_id;
	e->data.id=new_id;
	e->data.parent=parent_id;
	strncpy(e->data.name,name,sizeof(e->data.name));
	strncpy(e->data.hash,hash,sizeof(e->data.hash));

	e->next=head;
	head=e;

	assert(find_entry(new_id)!=NULL);

	return 0;
}

/**
 * Delete a hash from the hash database.
 * The parent-value of all childs is reset to L4_NIL_ID.
 */
int 
vtpm_del    (l4_threadid_t owner_id,
	     l4_threadid_t new_id)
{
	vtpm_pentry_t *e;
	vtpm_pentry_t old;

	debug("\tdelete #%x\n",new_id.id.task);

	FIND_ENTRY(e,new_id);

	if (!l4_thread_equal((*e)->owner,owner_id))
		return -L4_ENOTOWNER;
	
	debug("\tdelete e: %p next: %p\n",*e,(*e)->next);
	old=*e;
	*e=(*e)->next;
	free(old);

	// reset parent value of the childs
	old=head;
	while (old)
	{
		if (l4_thread_equal(old->data.parent,new_id))
			old->data.parent=L4_NIL_ID;
		old=old->next;
	}

	return 0;
}


/**
 * Signs a key for a given id.
 */
int
vtpm_link   (l4_threadid_t id,
	     int keylen,
	     const char *key,
	     int dstlen,
	     char *dst)
{	
  return vtpm_quote(id,keylen,key,dstlen,dst);
}

/**
 * Sign a quote for a thread using a nonce.
 *
 * Returnes the number of bytes written to dst or an error.
 */
int
vtpm_quote  (l4_threadid_t id,
	     int nlen,
	     const char *nonce,
	     int dstlen,
	     char *dst)
{	
	vtpm_pentry_t *e;
	vtpm_quote_t *quote;
	int res;
	
	if (dst==NULL)
		return -L4_EINVAL;

	quote=(vtpm_quote_t *)dst;

	CHECK_COUNT(dstlen,sizeof(*quote))
	FIND_ENTRY(e,id);

	memcpy(quote,&((*e)->data),sizeof(quote->data));
	res=rsa_sign(&mykey, sizeof(quote->sig), quote->sig, nlen, nonce, sizeof(quote->data), (char *)&(quote->data), 0);
	if (res>=0)
	  res+=sizeof(quote->data);
	return res;
}

/**
 * Seal some memory.
 * If a hash is given, seal to that hash.
 *
 * Returns the number of bytes written to dst or an error.
 */
int
vtpm_seal(l4_threadid_t id, 
	  int hashlen,
	  const char *hash, 
	  int srclen,
	  const char *src, 
	  int dstlen,
	  char *dst)
{
	vtpm_pentry_t *e;
	
	if (src==NULL || dst==NULL)
		return -L4_EINVAL;

	printf("vtpm_seal(%s,%s)\n",src,dst);

	if (!hash){
		FIND_ENTRY(e,id);
		hash = (*e)->data.hash;
		hashlen= sizeof((*e)->data.hash);
	}
	printf("try to encrypt something\n");

	dstlen=rsa_encrypt(&mykey,dstlen,dst,hashlen,hash,srclen,src,0);
	return dstlen;
}



/**
 * Unseal some memory for a given thread.
 * dst must be large enough to hold a hash and the unsealed data.
 *
 * Returns the number of bytes written to dst or an error.
 */
int
vtpm_unseal(l4_threadid_t id,
	    int srclen,
	    const char *src,
	    int dstlen,
	    char *dst
	    )
{
	vtpm_pentry_t *e;
	int res;

	FIND_ENTRY(e,id);

	res=rsa_decrypt(&mykey, dstlen, dst, srclen, src, 0);
	if (res<0) 
		return res;

	if (!memcmp(dst,(*e)->data.hash,sizeof((*e)->data.hash)))
	{
		memset(dst,0,dstlen);
		return -L4_EPERM;
	}
	res-=sizeof((*e)->data.hash);

	if (res>0)
		memmove(dst,dst+sizeof((*e)->data.hash),res);	
	return res;
}

/**
 * Init the functions.
 */
static void
init_vtpm(void)
{
	static char buf[1024];
	int res;
	debug("%s()\n",__func__);

	res=1;
	while (res>0)
	{
		rand_buffer(buf,sizeof(buf));
		res=rsa_insertrandom(&mykey,sizeof(buf),buf);
	}
	assert (!res);
	res=rsa_create(&mykey,KEYBITS);
	debug("%s() generate key = %#x len: %d\n",__func__,res, mykey.pub.bits);
	assert (!res);	
	assert(mykey.pub.bits);
}


L4C_CTOR(init_vtpm,L4CTOR_AFTER_BACKEND);
