/**
 * Simple glue code for different rsa implementations.
 */

#include <stdarg.h>
#include <assert.h>
#include "rsaglue.h"


#define RSA_FUNCTION(NAME, VARS, INIT, BODY, END)		\
int rsa_##NAME(rsa_key_t *key, int dstlen, char *dst, ...)	\
{								\
	int res;						\
	va_list argp;						\
	int len;						\
	char *data;						\
	VARS;							\
								\
	va_start(argp, dst);					\
	assert(dst);						\
	INIT;							\
								\
	if (res)						\
		return -1;					\
	while (1) {						\
		len = (int) va_arg(argp, int);			\
		if (len<=0)					\
			break;					\
		data = (char *) va_arg(argp, char *);		\
		assert(data);					\
		BODY;						\
                assert(!res);					\
	}							\
	END;							\
	return res;						\
}

/**
 * Sign data with a key. The variable arguments are (char *value, int
 * count) pairs. The last variable argument must be 0!
 */

RSA_FUNCTION(sign,
	      R_SIGNATURE_CTX ctx;
	      ,
	      assert(dstlen>=MAX_SIGNATURE_LEN);
	      res=R_SignInit(&ctx, DA_MD5);
	      ,
	      res=R_SignUpdate(&ctx,data,len);
	      ,
	      res=R_SignFinal(&ctx, dst, &dstlen, &key->priv);
	      if (res)
	      		return -2;
	      res=dstlen;
	);

/**
 * Encrypt data with a pubkey. The varible arguments are (char *value,
 * int count) pairs, which are encrypted into dst.
 *
 * Returns the number of bytes written.
 *
 * The output is consists of [enckey_len,enckey,encdata]
 */

RSA_FUNCTION(encrypt,
	      R_ENVELOPE_CTX ctx;
	      unsigned char *encryptedKeys[1];
	      R_RSA_PUBLIC_KEY *pubkey[1];
	      unsigned char iv[8];
	      int outlen;
	      int l;
	      ,
	      // store the size and the encrypted key as first part of the dst
	      outlen=sizeof(unsigned int)+MAX_ENCRYPTED_KEY_LEN;
	      encryptedKeys[0]=dst+sizeof(unsigned int);

	      assert(dstlen>=outlen);
	      
	      pubkey[0]=&key->pub;
	      res=R_SealInit(&ctx, 
			     encryptedKeys, 
			     (unsigned int *)dst,
			     iv,
			     1, 
			     pubkey,
			     EA_DES_EDE3_CBC,
			     &key->random);

	      // correct outlen
	      l=sizeof(unsigned int)+(*(unsigned int *)dst);
	      assert(l<=outlen);
	      outlen=l;
	      ,
	      if ((dstlen-outlen)<((len+7)/8)*8)
		{
		  printf("=%s() dstlen %d outlen %d len %d\n",__func__,dstlen,outlen,len);
		  return -2;
		}
	      res=R_SealUpdate(&ctx,dst+outlen,&l,data,len);
	      assert(l>0);
	      outlen+=l;
	      ,
	      if (dstlen<(outlen+8))
	        return -3;
	      res=R_SealFinal(&ctx,dst,&l);
	      if (res)
	      return -4;
	      assert(l==8);
	      outlen+=l;
	      res=outlen;	      
	);



/**
 * Decrypt data with a privkey. The varible arguments are (char
 * *value, int count) pairs, which are decrypted into dst.
 *
 * Returns the number of bytes written.
 */
RSA_FUNCTION(decrypt,
	      R_ENVELOPE_CTX ctx;
	      unsigned char iv[8];
	      int outlen;
	      int l;
	      ,
	      outlen=sizeof(unsigned int) + *(unsigned int *)dst;
	      assert(dstlen>=outlen);
	      res=R_OpenInit(&ctx,
			     EA_DES_EDE3_CBC,
			     dst+sizeof(unsigned int),
			     *(unsigned int *)dst,
			     iv,
			     &key->priv);	      
	      ,	      
	      if ((dstlen-outlen)<((len+7)/8)*8)
			return -2;
	      res=R_OpenUpdate(&ctx,dst+outlen,&l,data,len);
	      assert(l>0);
	      outlen+=l;
	      ,
	      if (dstlen<(outlen+8))
			return -3;
	      res=R_OpenFinal(&ctx,dst,&l);
	      assert(l<8);
	      outlen+=l;
	      if (res)
	      		return -4;
	      res=outlen;
	);


/**
 * Create a new rsa key.
 */
int
rsa_create(rsa_key_t *key, int bits)
{
	int res;
	R_RSA_PROTO_KEY proto;

	proto.bits=bits;
	proto.useFermat4=1;

	res = R_GeneratePEMKeys(&key->pub,&key->priv,&proto,&key->random);
	return res;
}


/**
 * Insert random for key generation.
 *
 * Returns the bytes of random needed after this insert.
 */
int
rsa_insertrandom(rsa_key_t *key, int srclen, char *src)
{
	int res;
	res=R_RandomUpdate(&key->random,src,srclen);
	if (res)
		return res;
	R_GetRandomBytesNeeded(&res,&key->random);
	return res;
}
