/****************************************************************************/
/*                                                                          */
/*                           SEAL/UNSEAL routines                           */
/*                                                                          */
/* This file is copyright 2003 IBM. See "License" for details               */
/****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <tcg/buildbuff.h>
#include <tcg/oiaposap.h>
#include <tcg/hmac.h>
#include <tcg/pcrs.h>
#include <tcg/rand.h>
#include <tcg/ord.h>

/****************************************************************************/
/*                                                                          */
/* Seal a data object with caller Specified PCR infro                       */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* keyhandle is the TCG_KEY_HANDLE of the key used to seal the data        */
/*           0x40000000 for the SRK                                         */
/* pcrinfo   is a pointer to a TCG_PCR_INFO structure containing           */
/*           a bit map of the PCR's to seal the data to, and a              */
/*           pair of TCG_COMPOSITE_HASH values for the PCR's               */
/* pcrinfosize is the length of the pcrinfo structure                       */
/* keyauth   is the authorization data (password) for the key               */
/* dataauth  is the authorization data (password) for the data being sealed */
/*           both authorization values must be 20 bytes long ?              */
/* data      is a pointer to the data to be sealed                          */
/* datalen   is the length of the data to be sealed (max 256?)              */
/* blob      is a pointer to an area to received the sealed blob            */
/*           it should be long enough to receive the encrypted data         */
/*           which is 256 bytes, plus some overhead. 512 total recommended? */
/* bloblen   is a pointer to an integer which will receive the length       */
/*           of the sealed blob                                             */
/*                                                                          */
/****************************************************************************/

#if 0
TPM_TRANSMIT_OSAP_FUNC(Seal,
		       (unsigned long keyhandle,
			unsigned char *pcrinfo, unsigned long pcrinfosize,
			unsigned char *keyauth,
			unsigned char *dataauth,
			unsigned char *data, unsigned int datalen,
			unsigned char *blob, unsigned int *bloblen)
		       ,
		       keyauth,
		       0x0001,
		       keyhandle
		       ,
		       unsigned char encauth[TCG_HASH_SIZE];
		       unsigned char xorwork[TCG_HASH_SIZE * 2];
		       unsigned char xorhash[TCG_HASH_SIZE];
		       int i;

		       printf("%s() %p %ld %p %d\n",__func__,pcrinfo,pcrinfosize,data,datalen);
		       if (dataauth == NULL || blob == NULL)
		         return -1;
		       PRINT_HASH(keyauth);
		       ,
		       /* calculate encrypted authorization value */
		       memcpy(xorwork, sess.ssecret, TCG_HASH_SIZE);
		       memcpy(xorwork + TCG_HASH_SIZE, sess.enonce, TCG_HASH_SIZE);
		       sha1(xorwork, TCG_HASH_SIZE * 2, xorhash);
		       /* encrypt data authorization key */
		       for (i = 0; i < TCG_HASH_SIZE; ++i)
		         encauth[i] = xorhash[i] ^ dataauth[i];
		       PRINT_HASH(encauth);
		       ,
		       ,
		       "L I % @ @",
		       "L L % @ @",
		       TPM_ORD_Seal,
		       keyhandle,
		       TCG_HASH_SIZE, encauth,
		       pcrinfosize, pcrinfo,
		       datalen, data);
		       
#endif		       

#if 1
unsigned long TPM_Seal(unsigned long keyhandle,
		       unsigned char *pcrinfo, unsigned long pcrinfosize,
		       unsigned char *keyauth,
		       unsigned char *dataauth,
		       unsigned char *data, unsigned int datalen,
		       unsigned char *blob, unsigned int *bloblen)
{
    unsigned char seal_fmt[] = "00 C2 T l l % @ @ L % o %";
    unsigned char tcpadata[TCG_MAX_BUFF_SIZE];
    unsigned char encauth[TCG_HASH_SIZE];
    unsigned char pubauth[TCG_HASH_SIZE];
    unsigned char xorwork[TCG_HASH_SIZE * 2];
    unsigned char xorhash[TCG_HASH_SIZE];
    unsigned char nonceodd[TCG_NONCE_SIZE];
    osapsess sess;
    unsigned long ret;
    int i;
    unsigned char c;
    unsigned long ordinal;
    unsigned long pcrsize;
    unsigned long datsize;
    unsigned long keyhndl;
    unsigned short keytype;
    int sealinfosize;
    int encdatasize;
    int storedsize;

    /* check input arguments */
    if (keyauth == NULL || dataauth == NULL || data == NULL
        || blob == NULL)
        return -1;
    PRINT_HASH(keyauth);

    if (pcrinfosize != 0 && pcrinfo == NULL)
        return -2;    
    //if (keyhandle == 0x40000000)
    //        keytype = 0x0004;
    //else
    keytype = 0x0001;
    /* Open OSAP Session */
    ret = TPM_OSAP(&sess, keyauth, keytype, keyhandle);
    if (ret)
        return ret;
    printf("%s() TPM_OSAP %ld\n",__func__,ret);

    /* calculate encrypted authorization value */
    memcpy(xorwork, sess.ssecret, TCG_HASH_SIZE);
    memcpy(xorwork + TCG_HASH_SIZE, sess.enonce, TCG_HASH_SIZE);
    sha1(xorwork, TCG_HASH_SIZE * 2, xorhash);
    /* encrypt data authorization key */
    for (i = 0; i < TCG_HASH_SIZE; ++i)
        encauth[i] = xorhash[i] ^ dataauth[i];
    /* move Network byte order data to variables for hmac calculation */
    ordinal = htonl(0x17);
    datsize = htonl(datalen);
    keyhndl = htonl(keyhandle);
    pcrsize = htonl(pcrinfosize);

    c = 0;
    rand_buffer(nonceodd, TCG_NONCE_SIZE);
    PRINT_HASH(encauth);
    /* calculate authorization HMAC value */
    if (pcrinfosize == 0) {
        /* no pcr info specified */
        ret = authhmac(pubauth, sess.ssecret, TCG_HASH_SIZE,
                       sess.enonce, nonceodd, &c, 4,
                       &ordinal, TCG_HASH_SIZE, encauth, 4,
                       &pcrsize, 4, &datsize, datalen, data, 0, 0);
    } else {
        /* pcr info specified */
        ret = authhmac(pubauth, sess.ssecret, TCG_HASH_SIZE,
                       sess.enonce, nonceodd, &c, 4,
                       &ordinal, TCG_HASH_SIZE, encauth, 4,
                       &pcrsize, pcrinfosize, pcrinfo, 4,
                       &datsize, datalen, data, 0, 0);
    }
    printf("%s() authhmac %ld\n",__func__,ret);
    PRINT_HASH(pubauth);
    if (ret < 0) {
        TPM_Terminate_Handle(sess.handle);
        return -3;
    }
    /* build the request buffer */
    ret = buildbuff(seal_fmt, tcpadata,
                    ordinal,
                    keyhndl,
                    TCG_HASH_SIZE, encauth,
                    pcrinfosize, pcrinfo,
                    datalen, data,
                    sess.handle,
                    TCG_NONCE_SIZE, nonceodd, c, TCG_HASH_SIZE, pubauth);

    printf("%s() buildbuff %ld\n",__func__,ret);

    PRINT_HASH_SIZE(tcpadata,ret);
    if (ret <= 0) {
        TPM_Terminate_Handle(sess.handle);
        return -4;
    }

    /* transmit the request buffer to the TPM device and read the reply */

    ret = TPM_Transmit(tcpadata, "Seal");
    printf("%s() transmit %ld\n",__func__,ret);

    if (ret != 0) {
        TPM_Terminate_Handle(sess.handle);
        return ret;
    }
    /* calculate the size of the returned Blob */
    sealinfosize = ntohl(*(unsigned long *) (tcpadata + TCG_DATA_OFFSET + 4));
    encdatasize = ntohl(*(unsigned long *) (tcpadata +
                                       TCG_DATA_OFFSET + 4 + 4 +
                                       sealinfosize));
    storedsize = 4 + 4 + sealinfosize + 4 + encdatasize;
    /* check the HMAC in the response */
    ret = checkhmac1(tcpadata, ordinal, nonceodd, sess.ssecret,
                     TCG_HASH_SIZE, storedsize, TCG_DATA_OFFSET, 0, 0);
    printf("%s() checkhmac1 %ld\n",__func__,ret);
    if (ret != 0) {
        TPM_Terminate_Handle(sess.handle);
        return -5;
    }
    /* copy the returned blob to caller */
    memcpy(blob, tcpadata + TCG_DATA_OFFSET, storedsize);
    *bloblen = storedsize;
    TPM_Terminate_Handle(sess.handle);
    return 0;
}

#endif

/****************************************************************************/
/*                                                                          */
/* Seal a data object with current PCR information                          */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* keyhandle is the TCG_KEY_HANDLE of the key used to seal the data        */
/*           0x40000000 for the SRK                                         */
/* pcrmap    is a 32 bit integer containing a bit map of the PCR register   */
/*           numbers to be used when sealing. e.g 0x0000001 specifies       */
/*           PCR 0. 0x00000003 specifies PCR's 0 and 1, etc.                */
/* keyauth   is the authorization data (password) for the key               */
/* dataauth  is the authorization data (password) for the data being sealed */
/*           both authorization values must be 20 bytes long ?              */
/* data      is a pointer to the data to be sealed                          */
/* datalen   is the length of the data to be sealed (max 256?)              */
/* blob      is a pointer to an area to received the sealed blob            */
/*           it should be long enough to receive the encrypted data         */
/*           which is 256 bytes, plus some overhead. 512 total recommended? */
/* bloblen   is a pointer to an integer which will receive the length       */
/*           of the sealed blob                                             */
/*                                                                          */
/****************************************************************************/
unsigned long TPM_Seal_CurrPCR(unsigned long keyhandle, unsigned long pcrmap,
			       unsigned char *keyauth,
			       unsigned char *dataauth,
			       unsigned char *data, unsigned int datalen,
			       unsigned char *blob, unsigned int *bloblen)
{
    unsigned char pcrinfo[MAXPCRINFOLEN];
    unsigned int pcrlen;

    pcrlen=0;
    GenPCRInfo(pcrmap, pcrinfo, &pcrlen);
    return TPM_Seal(keyhandle, pcrinfo,pcrlen,
                    keyauth, dataauth, data, datalen, blob, bloblen);
}

/****************************************************************************/
/*                                                                          */
/* Unseal a data object                                                     */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* keyhandle is the TCG_KEY_HANDLE of the key used to seal the data        */
/*           0x40000000 for the SRK                                         */
/* keyauth   is the authorization data (password) for the key               */
/* dataauth  is the authorization data (password) for the data being sealed */
/*           both authorization values must be 20 bytes long ?              */
/* blob      is a pointer to an area to containing the sealed blob          */
/* bloblen   is the length of the sealed blob                               */
/* rawdata   is a pointer to an area to receive the unsealed data (max 256?)*/
/* datalen   is a pointer to a int to receive the length of the data        */
/*                                                                          */
/****************************************************************************/
unsigned long TPM_Unseal(unsigned long keyhandle,
			 unsigned char *keyauth,
			 unsigned char *dataauth,
			 unsigned char *blob, unsigned int bloblen,
			 unsigned char *rawdata, unsigned int *datalen)
{
    unsigned char unseal_fmt[] = "00 C3 T l l % L % o % L % o %";
    unsigned char tcpadata[TCG_MAX_BUFF_SIZE];
    unsigned char nonceodd[TCG_NONCE_SIZE];
    unsigned char enonce1[TCG_NONCE_SIZE];
    unsigned char enonce2[TCG_NONCE_SIZE];
    unsigned char authdata1[TCG_HASH_SIZE];
    unsigned char authdata2[TCG_HASH_SIZE];
    unsigned long ret;
    unsigned char c;
    unsigned long ordinal;
    unsigned long keyhndl;
    unsigned long authhandle1;
    unsigned long authhandle2;

    /* check input arguments */
    if (keyauth == NULL || dataauth == NULL || rawdata == NULL
        || blob == NULL)
        return 1;
    /* open TWO OIAP sessions, one for the Key and one for the Data */
    ret = TPM_OIAP(&authhandle1, enonce1);
    if (ret)
        return ret;
    ret = TPM_OIAP(&authhandle2, enonce2);
    if (ret)
        return ret;
    /* move data to Network byte order variables for HMAC calculation */
    ordinal = htonl(0x18);
    keyhndl = htonl(keyhandle);
    /* generate odd nonce */
    rand_buffer(nonceodd, TCG_NONCE_SIZE);
    c = 0;
    /* calculate KEY authorization HMAC value */
    ret = authhmac(authdata1, keyauth, TCG_HASH_SIZE, enonce1, nonceodd,
                   &c, 4, &ordinal, bloblen, blob, 0, 0);
    if (ret < 0) {
        TPM_Terminate_Handle(authhandle1);
        TPM_Terminate_Handle(authhandle2);
        return 1;
    }
    /* calculate DATA authorization HMAC value */
    ret = authhmac(authdata2, dataauth, TCG_HASH_SIZE, enonce2,
                   nonceodd, &c, 4, &ordinal, bloblen, blob, 0, 0);
    if (ret < 0) {
        TPM_Terminate_Handle(authhandle1);
        TPM_Terminate_Handle(authhandle2);
        return -1;
    }
    /* build the request buffer */
    ret = buildbuff(unseal_fmt, tcpadata,
                    ordinal,
                    keyhndl,
                    bloblen, blob,
                    authhandle1,
                    TCG_NONCE_SIZE, nonceodd,
                    c,
                    TCG_HASH_SIZE, authdata1,
                    authhandle2,
                    TCG_NONCE_SIZE, nonceodd,
                    c, TCG_HASH_SIZE, authdata2);

    if (ret <= 0) {
        TPM_Terminate_Handle(authhandle1);
        TPM_Terminate_Handle(authhandle2);
        return 1;
    }
    /* transmit the request buffer to the TPM device and read the reply */
    ret = TPM_Transmit(tcpadata, "Unseal");
    if (ret != 0) {
        TPM_Terminate_Handle(authhandle1);
        TPM_Terminate_Handle(authhandle2);
        return ret;
    }
    /* XXX
       check HMAC in response
     */
    /* copy decrypted data back to caller */
    *datalen = ntohl(*(unsigned long *) (tcpadata + TCG_DATA_OFFSET));
    memcpy(rawdata, tcpadata + TCG_DATA_OFFSET + 4, *datalen);
    TPM_Terminate_Handle(authhandle1);
    TPM_Terminate_Handle(authhandle2);
    return 0;
}
