#include <stdio.h>
#include <netinet/in.h>
#include <tcg/quote.h>
#include <tcg/buildbuff.h>
#include <tcg/rand.h>
#include <tcg/hmac.h>
#include <tcg/transmit.h>
#include <tcg/oiaposap.h>
#include <tcg/ord.h>

/****************************************************************************/
/*                                                                          */
/* Return all PCR values and a signature for it with a given key            */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* keyhandle is the TCG_KEY_HANDLE of the key used to sign the pcrs        */
/*           0x40000000 for the SRK                                         */
/* pcrselect  is a pointer to a TCG_PCR_SELECT structure containing        */
/*           a bit map of the PCR's to output                               */
/* keyauth   is the authorization data (password) for the key               */
/* pcrdata   is a pointer to an area to recive the pcr values               */
/* nounce    is a pointer to an area containing a user supplied nounce      */
/*           which is 20 Bytes long                                         */
/* blob      is a pointer to an area to received the signature              */
/*           it should be long enough to receive the signature              */
/* bloblen   is a pointer to an integer which will receive the length       */
/*           of the signature blob                                          */
/*                                                                          */
/****************************************************************************/
unsigned long
TPM_Quote(unsigned long keyhandle,
	  unsigned char *keyauth,
	  unsigned char *pcrselect,
	  unsigned char *nounce,
	  unsigned char *pcrcomposite,
	  unsigned char *blob, 
	  unsigned int *bloblen)
{
	unsigned char quote_fmt[] = "00 C2 T l l % % L % o %";
	osapsess sess;
	int      pcrselectsize;
	int      storedsize;
	unsigned short keytype;
	unsigned long ret;
	unsigned long ordinal;
	unsigned char c;
	unsigned char nonceodd[TCG_NONCE_SIZE];
	unsigned char tcpadata[TCG_MAX_BUFF_SIZE];
	unsigned char pubauth[TCG_HASH_SIZE];


	/* check input arguments */
	if (keyauth == NULL || pcrselect == NULL || blob == NULL)
		return -1;
	//	if (keyhandle == 0x40000000)
	//		keytype = 0x0004;
	//	else
	keytype = 0x0001;

	/* Open OSAP Session */
	ret = TPM_OSAP(&sess, keyauth, keytype, keyhandle);
	if (ret)
		return ret;


	// the command_code in network byte order
	ordinal = htonl(TPM_ORD_Quote);
	// the size the pcrselect occupies
	pcrselectsize = ntohs(((struct tpm_pcr_selection2 *)pcrselect)->size)+sizeof(unsigned short);

	rand_buffer(nonceodd, TCG_NONCE_SIZE);
    
	// continue the authorisation session
	c = 0;
    
	/* calculate authorization HMAC value */
	ret=authhmac(pubauth,
		     sess.ssecret, TCG_HASH_SIZE,	     
		     sess.enonce, nonceodd, &c,
		     4, &ordinal,
		     TCG_HASH_SIZE, nounce,
		     pcrselectsize, pcrselect, 
		     0, 0);

	if (ret < 0) {
		TPM_Terminate_Handle(sess.handle);
		return -2;
	}
	/* build the request buffer */
	ret = buildbuff(quote_fmt, tcpadata,
			// the params 3-6
			ordinal,
			htonl(keyhandle),
			TCG_HASH_SIZE, nounce,
			pcrselectsize, pcrselect,
			// the authenication informations follow
			sess.handle,   
			TCG_NONCE_SIZE, nonceodd, 
			c, 
			TCG_HASH_SIZE, pubauth);
	if (ret <= 0) {
		TPM_Terminate_Handle(sess.handle);
		return -3;
	}
	/* transmit the request buffer to the TPM device and read the reply */
	ret = TPM_Transmit(tcpadata, "Quote");
	if (ret != 0) {
	    
		TPM_Terminate_Handle(sess.handle);
		return ret;
	}
	/* calculate the size of the returned Blob */
	storedsize = ntohl(*(unsigned long *) (tcpadata + TCG_PARAMSIZE_OFFSET))-47;

	/* check the HMAC in the response */
	ret = checkhmac1(tcpadata, ordinal, nonceodd, sess.ssecret,
			 TCG_HASH_SIZE, storedsize, TCG_DATA_OFFSET, 0, 0);
	if (ret != 0) {
		TPM_Terminate_Handle(sess.handle);
		return -4;
	}
	{
		int pcrcompositesize = pcrselectsize + 4 + ntohl(((struct tpm_pcr_composite*)(tcpadata + TCG_DATA_OFFSET))->valuesize);
		int sigsize = ntohl(*((unsigned long *)(tcpadata + TCG_DATA_OFFSET+pcrcompositesize)));
      
		memcpy(pcrcomposite, tcpadata + TCG_DATA_OFFSET, pcrcompositesize);

		/* copy the returned sig to caller */
		memcpy(blob, tcpadata + TCG_DATA_OFFSET+pcrcompositesize+4, sigsize);
		*bloblen = sigsize;
	}
	TPM_Terminate_Handle(sess.handle);
	return 0;
}

#include <tcg/basic.h>
#include <stdlib.h>

/**
 * XXX This is a test function for quote...
 */
int quote_stdout(int argc, char *argv[])
{
	int major, minor, version, rev, i, j;

	if (argc<4){
		printf("Usage: %s keyhandle keypassword nounce\n",argv[0]?argv[0]:__func__);
		return(-1);
	}

	if (TPM_Reset()){
		printf("TPM reset failed\n");	
		return(-2);
	}
	else
		printf("TPM successfully reset\n");

	if (TPM_GetCapability_Version(&major,&minor,&version,&rev)){
		printf("TPM version failed\n");	
		return(-3);
	} else
		printf("TPM version: %d.%d.%d.%d\n",major,minor,version,rev);

	{
		struct tpm_pcr_selection2 pcrselect;
		unsigned char output[1024];
		unsigned char passhash[20];
		unsigned char nouncehash[20];
		struct tpm_pcr_composite pcrcomposite;
		unsigned int outputlen;
		unsigned int keyhandle;
		int res;

		pcrselect.size=0x0200;
		pcrselect.select=0xFFFF;
      
		keyhandle=strtol(argv[1],NULL,16);
		// fill the variables to find bugs
		for (i=0; i<20; i++)
			passhash[i]=i;
		for (i=0; i<20; i++)
			nouncehash[i]=i;

		sha1(argv[2], strlen(argv[2]), passhash);
		sha1(argv[3], strlen(argv[3]), nouncehash);

		printf("keyhandle: %#x\n",keyhandle);
		res=TPM_Quote(keyhandle, passhash, 
			      (unsigned char *) &pcrselect, 
			      nouncehash, 
			      (unsigned char *) &pcrcomposite,
			      (unsigned char *) &output, &outputlen);
		printf("TPM_Quote() = %d\n",res);

		if (res)
			return(-4);

		printf("pcrcomposite (%d pcrs):\n",ntohl(pcrcomposite.valuesize)/20);
      
		for(i=0;i<ntohl(pcrcomposite.valuesize)/20;i++){
			printf("%8s%02d: ","PCR-",i);
			for(j=0;j<20;j++)
				printf("%02x",pcrcomposite.value[i].value[j]);
			printf("\n");
		}

		printf("signature (%d Bytes):\n",outputlen);
		for(i=0;i<(outputlen>>4);i++){
			printf("    ");
			for(j=0;j<16;j++)
				printf("%02x",output[(i<<4)+j]);      
			printf("\n");
		}
	}
	printf("signature end\n");
	return (0);
}
