/****************************************************************************/
/*                                                                          */
/*                         TCG Key Handling Routines                       */
/*                                                                          */
/* This file is copyright 2003 IBM. See "License" for details               */
/****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <tcg/buildbuff.h>
#include <tcg/keys.h>
#include <tcg/oiaposap.h>
#include <tcg/hmac.h>
#include <tcg/rand.h>
#include <tcg/tpm.h>
#include <tcg/ord.h>

/**
 * Read the public part of the EndorsementKey
 */
TPM_TRANSMIT_FUNC(ReadPubek,
		  (pubkeydata * k),
		  unsigned char nonce[TCG_HASH_SIZE];
		  if (k == NULL)
		  return -2;
		  if (rand_buffer(nonce, TCG_NONCE_SIZE)<1)
		  return -3;
		  ,
		  PubKeyExtract(buffer + TCG_DATA_OFFSET, k, 0);,
		  "L %",		      
		  TPM_ORD_ReadPubek,
		  TCG_HASH_SIZE, 
		  nonce);



/****************************************************************************/
/*                                                                          */
/* Create and Wrap a Key                                                    */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* keyhandle is the TCG_KEY_HANDLE of the parent key of the new key        */
/*           0x40000000 for the SRK                                         */
/* keyauth   is the authorization data (password) for the parent key        */
/* newauth   is the authorization data (password) for the new key           */
/* migauth   is the authorization data (password) for migration of the new  */
/*           key, or NULL if the new key is not migratable                  */
/*           all authorization values must be 20 bytes long                 */
/* keyparms  is a pointer to a keydata structure with parms set for the new */
/*           key                                                            */
/* key       is a pointer to a keydata structure returned filled in         */
/*           with the public key data for the new key                       */
/*                                                                          */
/****************************************************************************/
unsigned long TPM_CreateWrapKey(unsigned long keyhandle,
                           unsigned char *keyauth,
                           unsigned char *newauth,
                           unsigned char *migauth,
                           keydata *keyparms, keydata *key)
{
    unsigned char create_key_fmt[] = "00 c2 T l l % % % L % o %";
    unsigned char tcpadata[TCG_MAX_BUFF_SIZE];
    unsigned char kparmbuf[TCG_MAX_BUFF_SIZE];
    unsigned char encauth1[TCG_HASH_SIZE];
    unsigned char encauth2[TCG_HASH_SIZE];
    unsigned char xorwork[TCG_HASH_SIZE * 2];
    unsigned char xorhash[TCG_HASH_SIZE];
    unsigned char nonceodd[TCG_NONCE_SIZE];
    unsigned char pubauth[TCG_HASH_SIZE];
    osapsess sess;
    unsigned long ret;
    int i;
    unsigned char c;
    unsigned long ordinal;
    unsigned long keyhndl;
    unsigned short keytype;
    int kparmbufsize;

    /* check input arguments */
    if (keyauth == NULL || newauth == NULL || keyparms == NULL
        || key == NULL)
        return -1;
    //    if (keyhandle == 0x40000000)        /* SRK */
    //        keytype = 0x0004;
    //    else
    keytype = 0x0001;
    /* generate odd nonce */
    rand_buffer(nonceodd, TCG_NONCE_SIZE);
    /* Open OSAP Session */
    ret = TPM_OSAP(&sess, keyauth, keytype, keyhandle);
    if (ret)
        return ret;
    /* calculate encrypted authorization value for new key */
    memcpy(xorwork, sess.ssecret, TCG_HASH_SIZE);
    memcpy(xorwork + TCG_HASH_SIZE, sess.enonce, TCG_HASH_SIZE);
    sha1(xorwork, TCG_HASH_SIZE * 2, xorhash);
    for (i = 0; i < TCG_HASH_SIZE; ++i)
        encauth1[i] = xorhash[i] ^ newauth[i];
    /* calculate encrypted authorization value for migration of new key */
    if (migauth != NULL) {
        memcpy(xorwork, sess.ssecret, TCG_HASH_SIZE);
        memcpy(xorwork + TCG_HASH_SIZE, nonceodd, TCG_HASH_SIZE);
        sha1(xorwork, TCG_HASH_SIZE * 2, xorhash);
        for (i = 0; i < TCG_HASH_SIZE; ++i)
            encauth2[i] = xorhash[i] ^ migauth[i];
    } else
        memset(encauth2, 0, TCG_HASH_SIZE);
    /* move Network byte order data to variables for hmac calculation */
    ordinal = htonl(0x1F);
    keyhndl = htonl(keyhandle);
    c = 0;
    /* convert keyparm structure to buffer */
    kparmbufsize = BuildKey(kparmbuf, keyparms);
    if (kparmbufsize < 0) {
        TPM_Terminate_Handle(sess.handle);
        return 1;
    }
    ret = authhmac(pubauth, sess.ssecret, TCG_HASH_SIZE, sess.enonce,
                   nonceodd, &c, 4, &ordinal, TCG_HASH_SIZE,
                   encauth1, TCG_HASH_SIZE, encauth2, kparmbufsize,
                   kparmbuf, 0, 0);

    if (ret < 0) {
        TPM_Terminate_Handle(sess.handle);
        return 1;
    }
    /* build the request buffer */
    ret = buildbuff(create_key_fmt, tcpadata,
                    ordinal,
                    keyhndl,
                    TCG_HASH_SIZE, encauth1,
                    TCG_HASH_SIZE, encauth2,
                    kparmbufsize, kparmbuf,
                    sess.handle,
                    TCG_NONCE_SIZE, nonceodd, c, TCG_HASH_SIZE, pubauth);
    if (ret <= 0) {
        TPM_Terminate_Handle(sess.handle);
        return -1;
    }
    /* transmit the request buffer to the TPM device and read the reply */
    ret = TPM_Transmit(tcpadata, "CreateWrapKey");
    if (ret != 0) {
        TPM_Terminate_Handle(sess.handle);
        return ret;
    }
    kparmbufsize = KeySize(tcpadata + TCG_DATA_OFFSET);
    ret = checkhmac1(tcpadata, ordinal, nonceodd, sess.ssecret,
                     TCG_HASH_SIZE, kparmbufsize, TCG_DATA_OFFSET, 0, 0);
    if (ret != 0) {
        TPM_Terminate_Handle(sess.handle);
        return -1;
    }
    TPM_Terminate_Handle(sess.handle);
    /* convert the returned key to a structure */
    KeyExtract(tcpadata + TCG_DATA_OFFSET, key);
    return 0;
}

/****************************************************************************/
/*                                                                          */
/* Load a new Key into the TPM                                              */
/*                                                                          */
/* The arguments are...                                                     */
/*                                                                          */
/* keyhandle is the TCG_KEY_HANDLE of parent key for the new key           */
/*           0x40000000 for the SRK                                         */
/* keyauth   is the authorization data (password) for the parent key        */
/* keyparms  is a pointer to a keydata structure with all data  for the new */
/*           key                                                            */
/* newhandle is a pointer to a 32bit word which will receive the handle     */
/*           of the new key                                                 */
/*                                                                          */
/****************************************************************************/
unsigned long TPM_LoadKey(unsigned long keyhandle, unsigned char *keyauth,
                     keydata *keyparms, unsigned long *newhandle)
{
    unsigned char load_key_fmt[] = "00 c2 T l l % l % o %";
    unsigned char tcpadata[TCG_MAX_BUFF_SIZE];
    unsigned char kparmbuf[TCG_MAX_BUFF_SIZE];
    unsigned char nonceodd[TCG_NONCE_SIZE];
    unsigned char evennonce[TCG_NONCE_SIZE];
    unsigned char pubauth[TCG_HASH_SIZE];
    unsigned long ret;
    unsigned char c;
    unsigned long ordinal;
    unsigned long keyhndl;
    unsigned long authhandle;
    int kparmbufsize;

    /* check input arguments */
    if (keyauth == NULL || keyparms == NULL || newhandle == NULL)
        return 1;
    /* generate odd nonce */
    rand_buffer(nonceodd, TCG_NONCE_SIZE);
    /* Open OIAP Session */
    ret = TPM_OIAP(&authhandle, evennonce);
    if (ret != 0)
        return ret;
    /* move Network byte order data to variables for hmac calculation */
    ordinal = htonl(0x20);
    keyhndl = htonl(keyhandle);
    c = 0;
    /* convert keyparm structure to buffer */
    kparmbufsize = BuildKey(kparmbuf, keyparms);
    if (kparmbufsize < 0) {
        TPM_Terminate_Handle(authhandle);
        return 1;
    }
    /* calculate authorization HMAC value */
    ret = authhmac(pubauth, keyauth, TCG_HASH_SIZE, evennonce, nonceodd,
                   &c, 4, &ordinal, kparmbufsize, kparmbuf, 0, 0);
    if (ret < 0) {
        TPM_Terminate_Handle(authhandle);
        return 1;
    }
    /* build the request buffer */
    ret = buildbuff(load_key_fmt, tcpadata,
                    ordinal,
                    keyhndl,
                    kparmbufsize, kparmbuf,
                    htonl(authhandle),
                    TCG_NONCE_SIZE, nonceodd, c, TCG_HASH_SIZE, pubauth);
    if (ret <= 0) {
        TPM_Terminate_Handle(authhandle);
        return 1;
    }
    /* transmit the request buffer to the TPM device and read the reply */
    ret = TPM_Transmit(tcpadata, "LoadKey");
    if (ret != 0) {
        TPM_Terminate_Handle(authhandle);
        return ret;
    }
    TPM_Terminate_Handle(authhandle);
    ret = checkhmac1(tcpadata, ordinal, nonceodd, keyauth,
                     TCG_HASH_SIZE, 4, TCG_DATA_OFFSET, 0, 0);
    if (ret != 0)
        return 1;
    *newhandle = ntohl(*(unsigned long *) (tcpadata + TCG_DATA_OFFSET));
    return 0;
}

/**
 * Delete a key from the tpm.
 */
TPM_TRANSMIT_FUNC(EvictKey,
		      (unsigned long keyhandle),
		      ,
		      ,
		      "L L",
		      TPM_ORD_EvictKey,
		      keyhandle);

/****************************************************************************/
/*                                                                          */
/* Create a TCG_KEY buffer from a keydata structure                        */
/*                                                                          */
/****************************************************************************/
int BuildKey(unsigned char *buffer, keydata * k)
{
    unsigned char build_key_fmt[] = "% S L o L S S L L L L @ @ @";
    int ret;

    ret = buildbuff(build_key_fmt, buffer,
                    4, k->version,
                    k->keyusage,
                    k->keyflags,
                    k->authdatausage,
                    k->pub.algorithm,
                    k->pub.encscheme,
                    k->pub.sigscheme,
                    12,
                    k->pub.keybitlen,
                    k->pub.numprimes,
                    0,
                    k->pub.pcrinfolen, k->pub.pcrinfo,
                    k->pub.keylength, k->pub.modulus,
                    k->privkeylen, k->encprivkey);
    return ret;
}


/****************************************************************************/
/*                                                                          */
/* Walk down the TCG_Key Structure extracting information                  */
/*                                                                          */
/****************************************************************************/
int KeyExtract(unsigned char *keybuff, keydata * k)
{
    int offset;
    int pubkeylen;

    /* fill in  keydata structure */
    offset = 0;
    memcpy(k->version, keybuff + offset, sizeof(k->version));
    offset += 4;
    k->keyusage = ntohs(*(unsigned short *) (keybuff + offset));
    offset += 2;
    k->keyflags = ntohl(*(unsigned long *) (keybuff + offset));
    offset += 4;
    k->authdatausage = keybuff[offset];
    offset += 1;
    pubkeylen = PubKeyExtract(keybuff + offset, &(k->pub), 1);
    offset += pubkeylen;
    k->privkeylen = ntohl(*(unsigned long *) (keybuff + offset));
    offset += 4;
    if (k->privkeylen > 0 && k->privkeylen <= 1024)
        memcpy(k->encprivkey, keybuff + offset, k->privkeylen);
    offset += k->privkeylen;
    return offset;
}

/****************************************************************************/
/*                                                                          */
/* Walk down the TCG_PUBKey Structure extracting information               */
/*                                                                          */
/****************************************************************************/
int PubKeyExtract(unsigned char *keybuff, pubkeydata * k, int pcrpresent)
{
    unsigned long parmsize;
    unsigned long pcrisize;
    int offset;

    offset = 0;
    k->algorithm = ntohl(*(unsigned long *) (keybuff + offset));
    offset += 4;
    k->encscheme = ntohs(*(unsigned short *) (keybuff + offset));
    offset += 2;
    k->sigscheme = ntohs(*(unsigned short *) (keybuff + offset));
    offset += 2;
    parmsize = ntohl(*(unsigned long *) (keybuff + offset));
    offset += 4;
    if (k->algorithm == 0x00000001 && parmsize > 0) {   /* RSA */
        k->keybitlen = ntohl(*(unsigned long *) (keybuff + offset));
        offset += 4;
        k->numprimes = ntohl(*(unsigned long *) (keybuff + offset));
        offset += 4;
        k->expsize = ntohl(*(unsigned long *) (keybuff + offset));
        offset += 4;
    } else {
        offset += parmsize;
    }
    if (k->expsize != 0)
        offset += k->expsize;
    else {
        k->exponent[0] = 0x01;
        k->exponent[1] = 0x00;
        k->exponent[2] = 0x01;
        k->expsize = 3;
    }
    if (pcrpresent) {
        pcrisize = ntohl(*(unsigned long *) (keybuff + offset));
        offset += 4;
        if (pcrisize > 0 && pcrisize <= 256)
            memcpy(k->pcrinfo, keybuff + offset, pcrisize);
        offset += pcrisize;
        k->pcrinfolen = pcrisize;
    }
    else{
      k->pcrinfolen = 0;
    }
    k->keylength = ntohl(*(unsigned long *) (keybuff + offset));
    offset += 4;
    if (k->keylength > 0 && k->keylength <= 256)
        memcpy(k->modulus, keybuff + offset, k->keylength);
    offset += k->keylength;
    return offset;
}

/****************************************************************************/
/*                                                                          */
/* Get the size of a TCG_KEY                                               */
/*                                                                          */
/****************************************************************************/
int KeySize(unsigned char *keybuff)
{
    int offset;
    int privkeylen;

    offset = 0 + 4 + 2 + 4 + 1;
    offset += PubKeySize(keybuff + offset, 1);
    privkeylen = ntohl(*(unsigned long *) (keybuff + offset));
    offset += 4 + privkeylen;
    return offset;
}

/****************************************************************************/
/*                                                                          */
/* Get the size of a TCG_PUBKEY                                            */
/*                                                                          */
/****************************************************************************/
int PubKeySize(unsigned char *keybuff, int pcrpresent)
{
    unsigned long parmsize;
    unsigned long pcrisize;
    unsigned long keylength;

    int offset;

    offset = 0 + 4 + 2 + 2;
    parmsize = ntohl(*(unsigned long *) (keybuff + offset));
    offset += 4;
    offset += parmsize;
    if (pcrpresent) {
        pcrisize = ntohl(*(unsigned long *) (keybuff + offset));
        offset += 4;
        offset += pcrisize;
    }
    keylength = ntohl(*(unsigned long *) (keybuff + offset));
    offset += 4;
    offset += keylength;
    return offset;
}
