#include<stdio.h>
#include<vector>

#include<l4/names/libnames.h>
#include<l4/log/l4log.h>
#include<l4/env/errno.h>
#include<l4/util/util.h>

#include<l4/cbbmi/bmi_types.h>
#include<l4/cbbmi/bmi_classes.h>
#include<l4/cbbmi/bmi.h>

#include "bmi-client.h"

PDController::PDController()
{
    pdid = -2;
}

PDController::PDController(int id)
{
    pdid = id;
}

char* PDController::getConfigParam(const char* name)
{

    /* Find the bmi server */
    DICE_DECLARE_ENV(env);
    l4_threadid_t id_cbbmi = get_bmi_id();

    /* Create a buffer where the parameter value will be stored */
    char* buf = new char[16];
    sprintf(buf,"%d",-1);

    /* Retrieve the parameter value */
    
    /* 
     * /system/memory/min 
     * /system/memory/max
     */
    if( (!(strcmp(name,"/system/memory/min"))) || (!(strcmp(name,"/system/memory/max"))) )
    {
        /* Ask the bmi server for the value of the memory extrema */
        int minmem = -1;
        int maxmem = -1; 
     
        int res = bmi_getMemoryLimits_call(&id_cbbmi, this->pdid, &minmem, &maxmem, &env);

        if(res)
            LOG("(pdcon) ERROR PD(%d): MemLimits could not be received.\n",
                this->pdid);
        else
        {
            /* Transform the parameter value into a char* and write it in the buffer */
            if(!(strcmp(name,"/system/memory/min")))
                sprintf(buf,"%d",minmem);
            else if(!(strcmp(name,"/system/memory/max")))
                sprintf(buf,"%d",maxmem);
        }
    }
 
    /* 
     * /system/memory/current 
     */
    else if( (!(strcmp(name,"/system/memory/current"))) )
    {
        /* Ask the bmi server for the value of the parameter */
        int curmem = -1;

        int res = bmi_getMemoryCurrent_call(&id_cbbmi, this->pdid, &curmem, &env);

        if(res)
            LOG("(pdcon) ERROR PD(%d): MemCurrent could not be received.\n",
                   this->pdid);
        else
        {
            /* Transform the parameter value into a char* and write it in the buffer */
            sprintf(buf,"%d",curmem);
        }
    }

    /* 
     * /system/cpu/min 
     * /system/cpu/max
     * /system/cpu/current
     */
    else if( (!(strcmp(name,"/system/cpu/min"))) || (!(strcmp(name,"/system/cpu/max"))) || (!(strcmp(name,"/system/cpu/current"))) )
    {

            /* In L4, only cpu==1 is supported */
            int cpunb = 1;
            
            /* Convert the parameter value into a char* and write it in the buffer */
            sprintf(buf,"%d",cpunb);
            
    }

    /* 
     * /capability/pause 
     * /capability/restart
     * /capability/suspend
     * /capability/resume
     */
    else if( (!(strcmp(name,"/capability/pause"))) || (!(strcmp(name,"/capability/restart"))) || (!(strcmp(name,"/capability/suspend"))) || (!(strcmp(name,"/capability/resume"))) )
    {

        /* Give the corresponding PDStatus */     
        int status = -1;

        if(!(strcmp(name,"/capability/pause")))
            status = SHUTDOWN;

        else if(!(strcmp(name,"/capability/restart")))
            status = RUNNING;

        else if(!(strcmp(name,"/capability/suspend")))
            status = INVALID;

        else if(!(strcmp(name,"/capability/resume")))
            status = INVALID;
        
        /* Convert PDStatus into char* */
        sprintf(buf,"%d",status);
           
    }

    else
    {
        LOG("(pdcon) ERROR PD(%d): (%s) not a valid ConfigParam!\n",this->pdid,name);
        return NULL;
    }

    return buf;
}

void PDController::setConfigParam(const char* name, const char* value)
{

    /* Find the bmi server */
    DICE_DECLARE_ENV(env);
    l4_threadid_t id_cbbmi = get_bmi_id();
    
    /* Only the configuration parameters in /system/memory/ are allowed to be set. */
    
    /* 
     * /system/memory/min 
     * /system/memory/max
     */
    if((!(strcmp(name,"/system/memory/min"))) || (!(strcmp(name,"/system/memory/max"))))
    {
        int minmem = -1;
        int maxmem = -1;

        /* Convert char* to int */
        if(!(strcmp(name,"/system/memory/min")))
            minmem = atoi(value);
        else
            maxmem = atoi(value);

        /* Ask the bmi server to update the mem attributes of the PDEntry */
        int res = bmi_setMemoryLimits_call(&id_cbbmi, this->pdid, minmem, maxmem, &env);

        if(res)
            LOG("(pdcon) ERROR for PD(%d): MemLimits(%d - %d) could not be set.\n",
                this->pdid,
                minmem,
                maxmem);
    }

    /* 
     * /system/memory/current 
     */
    else if( (!(strcmp(name,"/system/memory/current"))) ) 
    {
        int curmem = -1;
     
        /* Convert char* to int */
        curmem = atoi(value);

        /* Ask the bmi server to update the value of the parameter */
        int res = bmi_setMemoryCurrent_call(&id_cbbmi, this->pdid, curmem, &env);

        if(res)
            LOG("(pdcon) ERROR for PD(%d): MemCurrent(%d) could not be set.\n",
                this->pdid,
                curmem);
    }

    else 
    {
        LOG("(pdcon) Config Parameter(%s) is read-only !\n",name);
    }
}

char* PDController::getRuntimeParam(const char* name)
{
    return getConfigParam(name);
}

void PDController::setRuntimeParam(const char* name, const char* value)
{
    LOG("(pdcon) Runtime Parameter(%s) is read-only !\n",name);
}


void PDController::setupPD(PDImage* img)
{
    /* Find the bmi server */
    DICE_DECLARE_ENV(env);
    l4_threadid_t id_cbbmi = get_bmi_id();
   
    printf("(bmilib) get_data(%s)\n",img->get_data());

    /* Ask the BMI to setup the controlled PD.  */ 
    int res = bmi_setImage_call(&id_cbbmi,
                               this->getPDID(), 
                               img->get_size(),
                               img->get_data(),
                               &env);

}

int PDController::getPDID()
{
    return this->pdid;
}

int PDController::getPriority()
{
    /* Find the bmi server */
    DICE_DECLARE_ENV(env);
    l4_threadid_t id_cbbmi = get_bmi_id();
    
    /* Get the Priority */
    int priority;
    int res = bmi_getPrio_call(&id_cbbmi, this->pdid, &priority, &env);

    return priority;
}

void PDController::setPriority(int newprio)
{
    /* Find the bmi server */
    DICE_DECLARE_ENV(env);
    l4_threadid_t id_cbbmi = get_bmi_id();
    
    /* Set the Priority */
    int res = bmi_setPrio_call(&id_cbbmi, this->pdid, newprio, &env);
}

PDStatus PDController::getCurrentStatus()
{
    
    /* Find the bmi server */
    DICE_DECLARE_ENV(env);
    l4_threadid_t id_cbbmi = get_bmi_id();

    /* Get the Current Status */
    int status = -1;
    
    int res_status = bmi_getStatus_call(&id_cbbmi, this->pdid, &status, &env);
    
    PDStatus* pstatus = (PDStatus*)(malloc(sizeof(PDStatus)));
    
    *pstatus = (PDStatus)status;

    return *pstatus;
}

void PDController::requestStatusChange(PDStatus newstatus)
{
    /* Find the bmi server */
    DICE_DECLARE_ENV(env);
    l4_threadid_t id_cbbmi = get_bmi_id();
   
    /* If SHUTDOWN is required, the bmilx must have been previously registered */
    if(newstatus==SHUTDOWN)
    {

        int res = -2;
        int i = 0;
        while(i<200)
        {
            /* 0 is returned when the bmilx was found */
            res = bmi_setStatus_call(&id_cbbmi, this->pdid, (int)newstatus, &env);
        
            if(res)
                LOGd(dbgpdcon,"(pdcon) ------- [%d] bmilx not registered yet. I try later!\n",i);
            else
            {
                LOGd(dbgpdcon,"(pdcon) ------- [%d] bmilx registered!\n",i);
                int stat3 = this->getCurrentStatus();
                LOGd(dbgpdcon,"(pdcon) -------- [%d] Current status(%d)\n",i,stat3);
                return;
            }

            l4_sleep(100);
            i++;
        }

        if((i==200) && res!=0)
        {
            LOG("(pdcon) [%d] bmilx was not registered. PD Shutting down process aborted!\n",i);
        }

        else
            LOGd(dbgpdcon,"(pdcon) [%d] bmilx finally registered. \n",i);
    }

    else
    {
        /* Request the Status Change */
        int res = bmi_setStatus_call(&id_cbbmi, this->pdid, (int)newstatus, &env);
    }
}

void PDController::destroy()
{
    /* Find the bmi server */
    DICE_DECLARE_ENV(env);
    l4_threadid_t id_cbbmi = get_bmi_id();
   
#if 0 
    /* Destroy the PD in PDcon_map */
    LOGd(dbgpdcon,"(pdcon): Deregistering PD(%d).\n",this->getPDID());
    int res0 = PDderegister_con(this);
    LOGd(dbgpdcon,"(pdcon): PD(%d) deregistered.\n",this->getPDID());
#endif

    LOG("(pdcon) START CHECK 0\n");
    /* Wait until the PD was registered at the BMI. */
    int res = -2;
    int i = 0;
    l4_threadid_t testid = L4_INVALID_ID;
    LOG("(pdcon) START CHECK 01\n");
    //*testid = L4_INVALID_ID;
    LOG("(pdcon) START CHECK 1\n");
    while(i<200)
    {
        /* 0 is returned when the bmilx was found */
        LOG("(pdcon) START CHECK 2\n");

        res = bmi_lookupThreadid_call(&id_cbbmi, this->pdid, &testid, &env);
        LOG("(pdcon) START CHECK 3\n");

        if(l4_is_invalid_id(testid))
            LOG("(pdcon) [%d] INVALID L4ID for (%d)\n",i,this->pdid);

        else
        {
            LOG("(pdcon) [%d] OK VALID L4ID for (%d)\n",i,this->pdid);
            break;
        }

        l4_sleep(100);

        i++;
    }


    if((i==200) && l4_is_invalid_id(testid))
    {
        LOG("(pdcon) [%d] NO VALID L4ID for (%d). ABORTED!\n",i,this->pdid);
        return ;
    }

    /* Destroy the PD at the bmi_server */

    LOGd(dbgpdcon,"(pdcon): Destroying PD(%d).\n",this->getPDID());
    int res2 = bmi_killPD_call(&id_cbbmi, this->pdid, &env);
    LOGd(dbgpdcon,"(pdcon): PD(%d) destroyed.\n",this->getPDID());
}

void PDController::setAllowedConnections(int tid,
                                         ConnectionPolicy allowed_connections[],
                                         bool disable_existing)
{
    /* Find the bmi server */
    DICE_DECLARE_ENV(env);
    l4_threadid_t id_cbbmi = get_bmi_id();
    
    /* Set the Allowed Connections :
     * 0. If (!disable_existing), ask the BMI server for the existing bitmap
     * 1. Transform the list of ConnectionPolicy instances into the resulting bitmap 
     * 2. Ask the bmi server to set the Connections */

    int con_code = 0;

    ConnectionPolicy* tmp = allowed_connections;
    
    int count = 0;
    while(tmp->con_type!=INVALID)
    {
        /* Compute the con_code */
        if(tmp->con_type==IPC)
        {
            if(tmp->mode==SEND)
                con_code |= IPC_S;

            else if(tmp->mode==RECV)
               con_code |= IPC_R;
        }

        else if(tmp->con_type==VLAN)
        {
            if(tmp->mode==SEND)
                con_code |= VLAN_S;

            else if(tmp->mode==RECV)
                con_code |= VLAN_R;
        }

        count += 1;
        tmp+=1;
    }

    int disable = (int)disable_existing;
    
    int res = bmi_setAllowedCon_call(&id_cbbmi, this->pdid, tid, con_code, disable, &env);
}

ConnectionPolicy* PDController::getAllowedConnections(int tid)
{
    /* Find the bmi server */
    DICE_DECLARE_ENV(env);
    l4_threadid_t id_cbbmi = get_bmi_id();
    
    /* Ask the bmi server for the conmap of this tid */
    int conmap = 0;
    int res = bmi_getAllowedCon_call(&id_cbbmi, this->pdid, tid, &conmap, &env);

    /* From the conmap, compute the list of ConnectionPolicy instances */
    int length = 0;
    std::vector<ConnectionPolicy*> vec;

    if(conmap & IPC_S)
    {
        vec.push_back(new ConnectionPolicy(IPC,SEND));
        length += 1;
    }

    if(conmap & IPC_R)
    {
        vec.push_back(new ConnectionPolicy(IPC,RECV));
        length += 1;
    }

    if(conmap & VLAN_S)
    {
        vec.push_back(new ConnectionPolicy(VLAN,SEND));
        length += 1;
    }

    if(conmap & VLAN_R)
    {
        vec.push_back(new ConnectionPolicy(VLAN,RECV));
        length += 1;
    }

    ConnectionPolicy *con_list = new ConnectionPolicy[length+1];

    int i=0;
    for(i=0; i<length; i++)
    {
        con_list[i] = *(vec[i]);
    }

    /* At the end, append a delimitor to show the list is over */
    con_list[length] = ConnectionPolicy(INVALID,SEND);

    return con_list;
}

