#ifndef __BMI_CLASSES_H_
#define __BMI_CLASSES_H_

#include <l4/cbbmi/bmi_types.h>
#include <l4/cbbmi/bmi_types_cpp.h>


/* Class of Connection Policies, defines the allowed connections between PDs. */
class ConnectionPolicy {
        public:
                ConnectionType con_type;
                int mode;

        public:
                ConnectionPolicy(){con_type=INVALID; mode=0;}
                ConnectionPolicy(ConnectionType c,int m){con_type=c; mode=m;}
};

/* Class of PD Images, abstracts the data image corresponding to a PD. */
class PDImage {

        private:
                int size;
                char* data;

        public:
                PDImage();
                PDImage(int mysize, char* mydata);

        public:
                /* Default set and get methods */
                int get_size(){return size;}
                void set_size(int mysize){size=mysize;}

                char* get_data(){return data;}
                void set_data(char* mydata){data=mydata;}

                /* BMI methods */

                /*!\brief get the size of the binary image of the PD 
                 */
                int getImageSize();
                
                /*!\brief get the data describing the PD 
                 */
                char* getData();

};

/* Class of PD Controllers, allows to set up and configure a PD. */ 
class PDController 
{
        private:
                int pdid;
                PDDescription desc;
               
        public:
                PDController(int pdid);
                PDController();

        public:
                /* Default set and get methods */
                int get_pdid(){return pdid;}
                void set_pdid(int myid){pdid=myid;}
             
                PDDescription get_desc(){return desc;}
                void set_desc(PDDescription mydesc){desc=mydesc;}

                /* BMI methods */
                
                /*!\brief get the BMI pdid of the controlled PD 
                */
                int getPDID();
                
                /*!\brief change the priority of the controlled PD
                 * \param priority new priority (in [-10,10], default=0)
                 */
                void setPriority(int priority);

                /*!\brief get the current priority of the controlled PD 
                 * \param name name of the parameter to set
                 * \param value value to give to the parameter 
                 */
                int getPriority();

                /*!\brief get the current status of the controlled PD 
                 */
                PDStatus getCurrentStatus();
                
                /*!\brief request a change of status of the controlled PD
                 * \param status new required status of the controlled PD 
                 */
                void requestStatusChange(PDStatus status);
               
                /*!\brief destroy the controlled PD and free the resources it was using
                 */
                void destroy();
                
                /*!\brief ask the BMI to setup the controlled PD  
                 * \param desc description of the data of the controlled PD
                 */
                void setupPD(PDImage* desc);
                
                /*!\brief define a connection between the controlled PD and another one
                 * \param tid pdid of the partner PD 
                 * \param allowed_connections list of communication types allowed for this connection
                 * \param disable_existing set if existing connections should be disabled 
                 */
                void setAllowedConnections(int tid,
                                           ConnectionPolicy* allowed_connections,
                                           bool disable_existing);
                
                /*!\brief get the list of connections between the controlled PD and the PD with pdid tid
                 * \param tid pdid of the partner PD 
                 */ 
                ConnectionPolicy* getAllowedConnections(int tid);

                /*!\brief set a configuration parameter for the PD being created
                 * \param name name of the parameter to set
                 * \param value value to give to the parameter 
                 */
                void setConfigParam(const char* name, const char* value);
    
                /*!\brief retrieve the value of a configuration parameter
                 * \param name name of the parameter to be read 
                 */
                char* getConfigParam(const char* name);

                /*!\brief set a runtime parameter for the PD being created
                 * \param name name of the parameter to set
                 * \param value value to give to the parameter 
                 */
                void setRuntimeParam(const char* name, const char* value);
                
                /*!\brief retrieve the value of a runtime parameter
                 * \param name name of the parameter to be read 
                 */
                char* getRuntimeParam(const char* name);

                /*!\brief list the currently defined runtime parameters of the family root
                 * \param root the family of parameters to be listed  
                 */
                RuntimeCapability* listRuntimeParams(const char* root);
};


/* Class of Protection Domain Managers, manipulate PDs through PD Controllers. */
class PDManagement {
        private:
                /* A list of pointers of the locally registered PDControllers. 
                 * Their index in the list corresponds to the pdid of the 
                 * related PD (at the BMI). 
                 */
                PDController* pdcons_map[MAX_PDS];

        private:
                /* Internal methods */
                int addCon(PDController*);
                PDController* getCon(int);
                
                /*!\brief list the l4lx flags for all PDs currently registered at the BMI 
                 * */

        public:
                void printCons();
                int hasCon(int pdid);
                int* listflags();

        public:
                /* BMI methods */

                /*!\brief return the current version of the BMI
                 */
                int getVersion();

                /*!\brief allocate resources to create a new PD
                 * \param desc resource requirement for the new PD  
                 */
                PDController* allocatePD(PDDescription desc);

                /*!\brief get the PDController for this PD (if necessary create a new one) 
                 * \param pdid BMI pdid of the PD to get the PDController for   
                 */
                PDController* getController(int pdid);

                /*!\brief get the PDController for the current PD
                 */
                PDController* getController();
                
                /*!\brief list the PDs currently registered at the BMI 
                 */
                int* listPD();

};



#endif 
