/**
 *  \file    dice/src/be/BEHeaderFile.cpp
 *  \brief   contains the implementation of the class CBEHeaderFile
 *
 *  \date    01/11/2002
 *  \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "BEHeaderFile.h"
#include "BEContext.h"
#include "BEConstant.h"
#include "BETypedef.h"
#include "BEType.h"
#include "BEFunction.h"
#include "BEDeclarator.h"
#include "BENameSpace.h"
#include "BEClass.h"
#include "BEStructType.h"
#include "BEUnionType.h"
#include "IncludeStatement.h"
#include "BETrace.h"
#include "Compiler.h"
#include "fe/FEFile.h"
#include "fe/FELibrary.h"
#include "fe/FEInterface.h"
#include "fe/FEOperation.h"
#include <iostream>
#include <cassert>

CBEHeaderFile::CBEHeaderFile()
: m_Constants(0, (CObject*)0),
  m_Typedefs(0, (CObject*)0),
  m_TaggedTypes(0, (CObject*)0)
{ }

CBEHeaderFile::CBEHeaderFile(CBEHeaderFile & src)
: CBEFile(src),
  m_Constants(0, (CObject*)0),
  m_Typedefs(0, (CObject*)0),
  m_TaggedTypes(0, (CObject*)0)
{
    m_sIncludeName = src.m_sIncludeName;
    // only copy references
    vector<CBEConstant*>::iterator iC;
    for (iC = src.m_Constants.begin(); 
	 iC != src.m_Constants.end(); 
	 iC++)
	m_Constants.Add(*iC);
    vector<CBETypedef*>::iterator iT;
    for (iT = src.m_Typedefs.begin(); 
	 iT != src.m_Typedefs.end(); 
	 iT++)
	m_Typedefs.Add(*iT);
    vector<CBEType*>::iterator iTa;
    for (iTa = src.m_TaggedTypes.begin(); 
	 iTa != src.m_TaggedTypes.end(); 
	 iTa++)
	m_TaggedTypes.Add(*iTa);
}

/** \brief destructor
 */
CBEHeaderFile::~CBEHeaderFile()
{ }

/** \brief prepares the header file for the back-end
 *  \param pFEFile the corresponding front-end file
 *  \param nFileType the type of the file
 *  \return true if the creation was successful
 *
 * This function should only add the file name and the names of the included
 * files to this instance.
 *
 * If the name of the front-end file included a relative path, this path
 * should be stripped of for the file name of this file. But it should be used
 * when writing include statements. E.g. a file included with \#include
 * "l4/test.idl" should get the header file name "test-client.h" or
 * "test-server.h", but should be included using "l4/test-client.h" or
 * "l4/test-server.h".
 */
void
CBEHeaderFile::CreateBackEnd(CFEFile * pFEFile, FILE_TYPE nFileType)
{
    assert(pFEFile);

    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, "CBEHeaderFile::%s(file: %s) called\n",
	__func__, pFEFile->GetFileName().c_str());

    m_nFileType = nFileType;
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    m_sFilename = pNF->GetFileName(pFEFile, m_nFileType);
    m_sIncludeName = pNF->GetIncludeFileName(pFEFile, m_nFileType);

    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL,
	"CBEHeaderFile::%s m_sFilename=%s, m_sIncludeName=%s\n", __func__,
	m_sFilename.c_str(), m_sIncludeName.c_str());

    CFEFile *pFERoot = dynamic_cast<CFEFile*>(pFEFile->GetRoot());
    assert(pFERoot);
    vector<CIncludeStatement*>::iterator iterI;
    for (iterI = pFEFile->m_Includes.begin();
	 iterI != pFEFile->m_Includes.end();
	 iterI++)
    {
        // check if we shall add an include statement
        if ((*iterI)->m_bPrivate)
            continue;
        // find the corresponding file
        CFEFile *pIncFile = pFERoot->FindFile((*iterI)->m_sFilename);
        // get name for include (with prefix, etc.)
        string sIncName;
        if (pIncFile)
            sIncName = pNF->GetIncludeFileName(pIncFile, m_nFileType);
        else
            sIncName = pNF->GetIncludeFileName((*iterI)->m_sFilename);
        // if the compiler option is FILE_ALL, then we only add non-IDL files
        if (CCompiler::IsFileOptionSet(PROGRAM_FILE_ALL) &&
	    (*iterI)->m_bIDLFile)
            continue;

        AddIncludedFileName(sIncName, (*iterI)->m_bIDLFile,
	    (*iterI)->m_bStandard, *iterI);
    }

    CCompiler::VerboseD(PROGRAM_VERBOSE_NORMAL, 
	"CBEHeaderFile::%s(file: %s) finished\n", __func__,
	pFEFile->GetFileName().c_str());
}

/** \brief prepares the header file for the back-end
 *  \param pFELibrary the corresponding front-end library
 *  \param nFileType the type of the file
 *  \return true if creation was successful
 */
void
CBEHeaderFile::CreateBackEnd(CFELibrary * pFELibrary, FILE_TYPE nFileType)
{
    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, 
	"CBEHeaderFile::%s(library: %s) called\n", __func__,
        pFELibrary->GetName().c_str());

    m_nFileType = nFileType;
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    m_sFilename = pNF->GetFileName(pFELibrary, m_nFileType);
    m_sIncludeName = pNF->GetIncludeFileName(pFELibrary, m_nFileType);
}

/** \brief prepares the back-end file for usage as per interface file
 *  \param pFEInterface the respective interface to prepare for
 *  \param nFileType the type of the file
 *  \return true if code generation was successful
 */
void
CBEHeaderFile::CreateBackEnd(CFEInterface * pFEInterface, 
    FILE_TYPE nFileType)
{
    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, 
	"CBEHeaderFile::%s(interface: %s) called\n", __func__,
        pFEInterface->GetName().c_str());
    
    m_nFileType = nFileType;
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    m_sFilename = pNF->GetFileName(pFEInterface, m_nFileType);
    m_sIncludeName = pNF->GetIncludeFileName(pFEInterface, m_nFileType);
}

/** \brief prepares the back-end file for usage as per operation file
 *  \param pFEOperation the respective front-end operation to prepare for
 *  \param nFileType the type of the file
 *  \return true if back-end was created correctly
 */
void
CBEHeaderFile::CreateBackEnd(CFEOperation * pFEOperation, 
    FILE_TYPE nFileType)
{
    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, 
	"CBEHeaderFile::%s(operation: %s) called\n", __func__,
        pFEOperation->GetName().c_str());

    m_nFileType = nFileType;
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    m_sFilename = pNF->GetFileName(pFEOperation, m_nFileType);
    m_sIncludeName= pNF->GetIncludeFileName(pFEOperation, m_nFileType);
}

/** \brief writes the content of the header file
 *
 * The content of the header file includes the functions, constants and type
 * definitions.
 *
 * Before we can actually write anything we have to create the file.
 *
 * The content of a header file is always braced by a symbol, so a multiple
 * include of this file will not result in multiple constant, type or function
 * declarations.
 *
 * In C we start with the constants, then the type definitions and after that
 * we write the functions using the base class' Write operation.  For the
 * server's side we print the typedefs before the includes. This way we can
 * use the message buffer type of the derived server-loop for the functions of
 * the base interface.
 */
void CBEHeaderFile::Write(void)
{
    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, "%s called\n", __func__);
    string sOutputDir = CCompiler::GetOutputDir();
    string sFilename;
    if (!sOutputDir.empty())
        sFilename = sOutputDir;
    sFilename += GetFileName();
    if (!Open(sFilename))
    {
	std::cerr << "Could not open header file " << sFilename << "\n";
        return;
    }
    // sort our members/elements depending on source line number
    // into extra vector
    CreateOrderedElementList();

    // write intro
    WriteIntro();
    // write include define
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sDefine = pNF->GetHeaderDefine(GetFileName());
    if (!sDefine.empty())
    {
	m_file << "#if !defined(" << sDefine << ")\n";
	m_file << "#define " << sDefine << "\n";
    }
    m_file << "\n";

    // default includes always come first, because they define standard headers
    // needed by other includes
    WriteDefaultIncludes();

    // write target file
    vector<CObject*>::iterator iter = m_vOrderedElements.begin();
    int nLastType = 0, nCurrType = 0;
    for (; iter != m_vOrderedElements.end(); iter++)
    {
        if (dynamic_cast<CIncludeStatement*>(*iter))
            nCurrType = 1;
        else if (dynamic_cast<CBEClass*>(*iter))
            nCurrType = 2;
        else if (dynamic_cast<CBENameSpace*>(*iter))
            nCurrType = 3;
        else if (dynamic_cast<CBEConstant*>(*iter))
            nCurrType = 4;
        else if (dynamic_cast<CBETypedef*>(*iter))
            nCurrType = 5;
        else if (dynamic_cast<CBEType*>(*iter))
            nCurrType = 6;
        else if (dynamic_cast<CBEFunction*>(*iter))
        {
	    /* only write functions if this is client header or component
	     * header */
	    if (IsOfFileType(FILETYPE_CLIENTHEADER) ||
                IsOfFileType(FILETYPE_COMPONENTHEADER))
                nCurrType = 7;
        }
        else
            nCurrType = 0;
        if (nCurrType != nLastType)
        {
            // brace functions with extern C
            if (nLastType == 6)
            {
		m_file << "#ifdef __cplusplus\n" <<
		    "}\n" <<
		    "#endif\n\n";
            }
	    m_file << "\n";
            nLastType = nCurrType;
            // brace functions with extern C
            if (nCurrType == 6)
            {
		m_file << "#ifdef __cplusplus\n" <<
		    "extern \"C\" {\n" <<
		    "#endif\n\n";
            }
        }
        // add pre-processor directive to denote source line
        if (CCompiler::IsOptionSet(PROGRAM_GENERATE_LINE_DIRECTIVE))
        {
	    m_file << "# " << (*iter)->GetSourceLine() << " \"" <<
		(*iter)->GetSourceFileName() << "\"\n";
        }
        switch (nCurrType)
        {
        case 1:
            WriteInclude((CIncludeStatement*)(*iter));
            break;
        case 2:
            WriteClass((CBEClass*)(*iter));
            break;
        case 3:
            WriteNameSpace((CBENameSpace*)(*iter));
            break;
        case 4:
            WriteConstant((CBEConstant*)(*iter));
            break;
        case 5:
            WriteTypedef((CBETypedef*)(*iter));
            break;
        case 6:
            WriteTaggedType((CBEType*)(*iter));
            break;
        case 7:
            WriteFunction((CBEFunction*)(*iter));
            break;
        default:
            break;
        }
    }
    // if last element was function, close braces
    if (nLastType == 6)
    {
	m_file << "#ifdef __cplusplus\n" <<
	    "}\n" <<
	    "#endif\n\n";
    }

    // write helper functions, if any
    /* only write functions if this is client header or component header */
    if (IsOfFileType(FILETYPE_CLIENTHEADER) ||
        IsOfFileType(FILETYPE_COMPONENTHEADER))
    {
        WriteHelperFunctions();
    }

    // write include define closing statement
    if (!sDefine.empty())
    {
	m_file << "#endif /* " << sDefine << " */\n";
    }
    m_file << "\n";

    // close file
    Close();
}

/** \brief writes includes, which have to appear before any type definition
 */
void CBEHeaderFile::WriteDefaultIncludes(void)
{
    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, "%s called\n", __func__);

    char *major = strdup(VERSION);
    char *minor = strchr(major, '.');
    char *submin = NULL;
    if (minor)
    {
	*minor = 0;
	minor++;
	submin = strchr(minor, '.');
	if (submin)
	{
	    *submin = 0;
	    submin++;
	}
    }
    m_file << 
	"#ifndef DICE_MAJOR_VERSION\n" <<
	"#define DICE_MAJOR_VERSION " << major << "\n" <<
	"#define DICE_MINOR_VERSION " << minor << "\n" <<
	"#define DICE_SUBMINOR_VERSION " << submin << "\n" <<
	"#endif\n\n";
    free(major);
    
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBETrace *pTrace = pCF->GetNewTrace();
    pTrace->DefaultIncludes(this);
    delete pTrace;

    m_file << 
	"/* needed for CORBA types */\n" <<
	"#include \"dice/dice.h\"\n" <<
	"\n";
}

/** \brief creates a list of ordered elements
 *
 * This method iterates each member vector and inserts their
 * elements into the ordered element list using bubble sort.
 * Sort criteria is the source line number.
 */
void CBEHeaderFile::CreateOrderedElementList(void)
{
    // first call base class
    CBEFile::CreateOrderedElementList();

    // add own vectors
    // typedef
    vector<CBETypedef*>::iterator iterT;
    for (iterT = m_Typedefs.begin();
	 iterT != m_Typedefs.end();
	 iterT++)
    {
        InsertOrderedElement(*iterT);
    }
    // tagged types
    vector<CBEType*>::iterator iterTa;
    for (iterTa = m_TaggedTypes.begin();
	 iterTa != m_TaggedTypes.end();
	 iterTa++)
    {
        InsertOrderedElement(*iterTa);
    }
    // consts
    vector<CBEConstant*>::iterator iterC;
    for (iterC = m_Constants.begin();
	 iterC != m_Constants.end();
	 iterC++)
    {
        InsertOrderedElement(*iterC);
    }
}

/** \brief writes a class
 *  \param pClass the class to write
 */
void CBEHeaderFile::WriteClass(CBEClass *pClass)
{
    assert(pClass);
    pClass->Write(this);
}

/** \brief writes the namespace
 *  \param pNameSpace the namespace to write
 */
void CBEHeaderFile::WriteNameSpace(CBENameSpace *pNameSpace)
{
    assert(pNameSpace);
    pNameSpace->Write(this);
}

/** \brief writes the function
 *  \param pFunction the function to write
 */
void CBEHeaderFile::WriteFunction(CBEFunction *pFunction)
{
    assert(pFunction);
    if (pFunction->DoWriteFunction(this))
        pFunction->Write(this);
}

/** \brief  writes a constant
 *  \param pConstant the constant to write
 */
void CBEHeaderFile::WriteConstant(CBEConstant *pConstant)
{
    assert(pConstant);
    pConstant->Write(this);
}

/** \brief write a typedef
 *  \param pTypedef the typedef to write
 */
void CBEHeaderFile::WriteTypedef(CBETypedef *pTypedef)
{
    assert(pTypedef);
    CCompiler::Verbose(PROGRAM_VERBOSE_NORMAL, "%s called for %s.\n", __func__,
        pTypedef->m_Declarators.First()->GetName().c_str());
    pTypedef->WriteDeclaration(this);
}

/** \brief writes a tagged type
 *  \param pType the type to write
 */
 void CBEHeaderFile::WriteTaggedType(CBEType *pType)
 {
    assert(pType);
    // get tag
    string sTag;
    if (dynamic_cast<CBEStructType*>(pType))
        sTag = ((CBEStructType*)pType)->GetTag();
    if (dynamic_cast<CBEUnionType*>(pType))
        sTag = ((CBEUnionType*)pType)->GetTag();
    sTag = CCompiler::GetNameFactory()->GetTypeDefine(sTag);
    m_file << 
	"#ifndef " << sTag << "\n" <<
	"#define " << sTag << "\n";
    pType->Write(this);
    m_file << ";\n" <<
	"#endif /* !" << sTag << " */\n" <<
	"\n";
}

/** \brief retrieves the maximum line number in the file
 *  \return the maximum line number in this file
 *
 * If line number is not that, i.e., is zero, then we iterate the elements and
 * check their end line number. The maximum is out desired maximum line
 * number.
 */
int
CBEHeaderFile::GetSourceLineEnd()
{
    if (m_nSourceLineNbEnd != 0)
       return m_nSourceLineNbEnd;

    // get maximum of members in base class;
    CBEFile::GetSourceLineEnd();
    
    // constants
    vector<CBEConstant*>::iterator iterC;
    for (iterC = m_Constants.begin();
	 iterC != m_Constants.end();
	 iterC++)
    {
       int sLine = (*iterC)->GetSourceLine();
       int eLine = (*iterC)->GetSourceLineEnd();
       m_nSourceLineNbEnd = std::max(sLine, std::max(eLine, m_nSourceLineNbEnd));
    }
    // typedef
    vector<CBETypedef*>::iterator iterT;
    for (iterT = m_Typedefs.begin();
	 iterT != m_Typedefs.end();
	 iterT++)
    {
       int sLine = (*iterT)->GetSourceLine();
       int eLine = (*iterT)->GetSourceLineEnd();
       m_nSourceLineNbEnd = std::max(sLine, std::max(eLine, m_nSourceLineNbEnd));
    }
    // tagged types
    vector<CBEType*>::iterator iterTa;
    for (iterTa = m_TaggedTypes.begin();
	 iterTa != m_TaggedTypes.end();
	 iterTa++)
    {
       int sLine = (*iterTa)->GetSourceLine();
       int eLine = (*iterTa)->GetSourceLineEnd();
       m_nSourceLineNbEnd = std::max(sLine, std::max(eLine, m_nSourceLineNbEnd));
    }

    return m_nSourceLineNbEnd;
}

