/* $Id: sched.c,v 1.2 2004/08/09 12:23:52 adam Exp $ */
/*****************************************************************************/
/**
 * \file	dde_linux26/lib/src/sched.c
 *
 * \brief	Scheduling
 *
 * \author	Marek Menzer <mm19@os.inf.tu-dresden.de>
 *
 * Based on dde_linux version by Christian Helmuth <ch12@os.inf.tu-dresden.de>
 */
/*****************************************************************************/

/* L4 */
#include <l4/sys/syscalls.h>
#include <l4/thread/thread.h>
#include <l4/semaphore/semaphore.h>

#include <l4/dde_linux/dde.h>

/* Linux */
#include <linux/sched.h>
#include <linux/spinlock.h>

/* local */
#include "internal.h"
#include "__config.h"
#include "__macros.h"

/* some old bullshit */
#define WQ_BUG()
#define CHECK_MAGIC(x)
#define CHECK_MAGIC_WQHEAD(x)
#define WQ_CHECK_LIST_HEAD(list)
#define WQ_NOTE_WAKER(tsk)

/*****************************************************************************/
/**
 * \name Scheduling Primitives
 *
 * To circumvent \e real user-level threads and scheduling we support only sane
 * user context states generated by \e bug-free drivers.
 *
 * The task structure has a new member - dde_sem binary semaphore - and each
 * process could sleep on this until awakened by any event.
 *
 * <em>This is from kernel/%sched.c</em>
 * @{ */
/*****************************************************************************/

/** Generic wake up for user contexts */
static inline int try_to_wake_up(struct task_struct *p, int synchronous)
{
  p->state = TASK_RUNNING;

  /* unlock process' self_lock */
  l4semaphore_up(&p->dde_sem);

  return 1;
}

int default_wake_function(wait_queue_t *curr, unsigned mode, int sync)
{
   task_t *p = curr->task;
   return try_to_wake_up(p, 0);
}

/*****************************************************************************/
/** Wake up dedicated user context
 * \ingroup mod_proc */
/*****************************************************************************/
inline int fastcall wake_up_process(struct task_struct *p)
{
  return try_to_wake_up(p, 0);
}

/** Handle Timeout for schedule_timeout() */
static void process_timeout(unsigned long __data)
{
  struct task_struct *p = (struct task_struct *) __data;

  wake_up_process(p);
}

/*****************************************************************************/
/** Schedule process but wake me at least after timeout
 * \ingroup mod_proc */
/*****************************************************************************/
signed long fastcall schedule_timeout(signed long timeout)
{
  struct timer_list timer;
  unsigned long expire;

  switch (timeout)
    {
    case MAX_SCHEDULE_TIMEOUT:
      /*
       * These two special cases are useful to be comfortable
       * in the caller. Nothing more. We could take
       * MAX_SCHEDULE_TIMEOUT from one of the negative value
       * but I' d like to return a valid offset (>=0) to allow
       * the caller to do everything it want with the retval.
       */
      schedule();
      goto out;
    default:
      /*
       * Another bit of PARANOID. Note that the retval will be
       * 0 since no piece of kernel is supposed to do a check
       * for a negative retval of schedule_timeout() (since it
       * should never happens anyway). You just have the printk()
       * that will tell you if something is gone wrong and where.
       */
      if (timeout < 0)
	{
	  LOG_Error("schedule_timeout: wrong timeout "
		 "value %lx", timeout);
	  current->state = TASK_RUNNING;
	  goto out;
	}
    }

  expire = timeout + jiffies;

  init_timer(&timer);
  timer.expires = expire;
  timer.data = (unsigned long) current;
  timer.function = process_timeout;

  add_timer(&timer);
  schedule();
  del_timer_sync(&timer);

  timeout = expire - jiffies;

out:
  return timeout < 0 ? 0 : timeout;
}

/*****************************************************************************/
/** Schedule process
 * \ingroup mod_proc */
/*****************************************************************************/
void schedule(void)
{

  switch (current->state)
    {
    case TASK_RUNNING:
      /* release CPU on any way */
#if SCHED_YIELD_OPT
      l4thread_usleep(SCHED_YIELD_TO);
#else
      l4_yield();
#endif
      break;

    case TASK_UNINTERRUPTIBLE:
    case TASK_INTERRUPTIBLE:
      /* lock yourself on current semaphore */
      l4semaphore_down(&current->dde_sem);
      break;

    default:
      Panic("current->state unknown (%ld)\n", current->state);
    }
}

/** Generic wake up for user contexts in wait queues */
static inline void __wake_up_common(wait_queue_head_t * q, unsigned int mode,
				    int nr_exclusive, const int sync)
{
  struct list_head *tmp, *head;
  struct task_struct *p;

  CHECK_MAGIC_WQHEAD(q);
  head = &q->task_list;
  WQ_CHECK_LIST_HEAD(head);
  tmp = head->next;
  while (tmp != head)
    {
      unsigned int state;
      wait_queue_t *curr = list_entry(tmp, wait_queue_t, task_list);

      tmp = tmp->next;
      CHECK_MAGIC(curr->__magic);
      p = curr->task;
      state = p->state;
      if (state & mode)
	{
	  WQ_NOTE_WAKER(curr);
	  if (try_to_wake_up(p, sync) && curr->flags && !--nr_exclusive)
	    break;
	}
    }
}

/*****************************************************************************/
/** Wake up wait queue entries
 * \ingroup mod_proc */
/*****************************************************************************/
void fastcall __wake_up(wait_queue_head_t * q, unsigned int mode, int nr)
{
  if (q)
    {
      unsigned long flags;
      spin_lock_irqsave(&q->lock, flags);
      __wake_up_common(q, mode, nr, 0);
      spin_unlock_irqrestore(&q->lock, flags);
    }
}

/*****************************************************************************/
/** Wake up wait queue entries (sync)
 * \ingroup mod_proc */
/*****************************************************************************/
void fastcall __wake_up_sync(wait_queue_head_t * q, unsigned int mode, int nr)
{
  if (q)
    {
      unsigned long flags;
      spin_lock_irqsave(&q->lock, flags);
      __wake_up_common(q, mode, nr, 1);
      spin_unlock_irqrestore(&q->lock, flags);
    }
}

void fastcall complete(struct completion *x)
{
	unsigned long flags;

	spin_lock_irqsave(&x->wait.lock, flags);
	x->done++;
	__wake_up_common(&x->wait, TASK_UNINTERRUPTIBLE | TASK_INTERRUPTIBLE, 1, 0);
	spin_unlock_irqrestore(&x->wait.lock, flags);
}

void fastcall wait_for_completion(struct completion *x)
{
	might_sleep();
	spin_lock_irq(&x->wait.lock);
	if (!x->done) {
		DECLARE_WAITQUEUE(wait, current);

		wait.flags |= WQ_FLAG_EXCLUSIVE;
		__add_wait_queue_tail(&x->wait, &wait);
		do {
			__set_current_state(TASK_UNINTERRUPTIBLE);
			spin_unlock_irq(&x->wait.lock);
			schedule();
			spin_lock_irq(&x->wait.lock);
		} while (!x->done);
		__remove_wait_queue(&x->wait, &wait);
	}
	x->done--;
	spin_unlock_irq(&x->wait.lock);
}

#define	SLEEP_ON_VAR				\
	unsigned long flags;			\
	wait_queue_t wait;			\
	init_waitqueue_entry(&wait, current);

#define	SLEEP_ON_HEAD					\
	spin_lock_irqsave(&q->lock,flags);		\
	__add_wait_queue(q, &wait);			\
	spin_unlock(&q->lock);

#define	SLEEP_ON_TAIL						\
	spin_lock_irq(&q->lock);				\
	__remove_wait_queue(q, &wait);				\
	spin_unlock_irqrestore(&q->lock,flags);

/*****************************************************************************/
/** Sleep on wait queue (interruptible by signals)
 * \ingroup mod_proc */
/*****************************************************************************/
void fastcall interruptible_sleep_on(wait_queue_head_t * q)
{
  SLEEP_ON_VAR current->state = TASK_INTERRUPTIBLE;

  SLEEP_ON_HEAD schedule();
  SLEEP_ON_TAIL
}

/*****************************************************************************/
/** Sleep on wait queue (interruptible by signals and timeout)
 * \ingroup mod_proc */
/*****************************************************************************/
long fastcall interruptible_sleep_on_timeout(wait_queue_head_t * q, long timeout)
{
  SLEEP_ON_VAR current->state = TASK_INTERRUPTIBLE;

  SLEEP_ON_HEAD timeout = schedule_timeout(timeout);
  SLEEP_ON_TAIL return timeout;
}

/*****************************************************************************/
/** Sleep on wait queue
 * \ingroup mod_proc */
/*****************************************************************************/
void fastcall sleep_on(wait_queue_head_t * q)
{
  SLEEP_ON_VAR current->state = TASK_UNINTERRUPTIBLE;

  SLEEP_ON_HEAD schedule();
  SLEEP_ON_TAIL
}

/*****************************************************************************/
/** Sleep on wait queue (interruptible by timeout)
 * \ingroup mod_proc */
/*****************************************************************************/
long fastcall sleep_on_timeout(wait_queue_head_t * q, long timeout)
{
  SLEEP_ON_VAR current->state = TASK_UNINTERRUPTIBLE;

  SLEEP_ON_HEAD timeout = schedule_timeout(timeout);
  SLEEP_ON_TAIL return timeout;
}

/*****************************************************************************/
/** Put all the gunge required to become a kernel thread without attached user
 * resources in one place where it belongs. (dummy)
 * \ingroup mod_proc */
/*****************************************************************************/
void daemonize(const char *text,...) {
  LOG("dde: dummy daemonize() call");
}
/** @} */
