#include <stdio.h>
#include <unistd.h>

#include <l4/sys/types.h>
#include <l4/sys/syscalls.h>
#include <l4/sys/ipc.h>
#include <l4/sys/ktrace.h>
#include <l4/util/l4_macros.h>
#include <l4/env_support/panic.h>
#include <l4/sigma0/sigma0.h>

#include "globals.h"
#include "config.h"

#include "memmap.h"

owner_t		 *__memmap;
__superpage_t	 *__memmap4mb;
owner_t		 *__iomap;
l4_addr_t	 mem_high;
l4_kernel_info_t *l4_info;
l4_addr_t        tbuf_status;

static void find_free(l4_umword_t *d1, l4_umword_t *d2,
		      owner_t owner, int super);

static inline void
warn_old(l4_umword_t d1, l4_umword_t d2, l4_threadid_t t)
{
#ifdef SIGMA0_REQ_MAGIC
  printf("SIGMA0: \033[33mClient "l4util_idfmt" sent old request "
         "d1=%08lx d2=%08lx\033[m\n", l4util_idstr(t), d1, d2);
#endif
}

#ifdef SIGMA0_REQ_MAGIC
static inline void*
handle_extended_sigma0(l4_umword_t *d1, l4_umword_t *d2, l4_threadid_t t)
{
  l4_fpage_t fp = { .raw = *d2 };
  int fn = *d1 & SIGMA0_REQ_ID_MASK;

  switch (fn)
    {
    case SIGMA0_REQ_ID_FPAGE_RAM:	   // dedicated fpage cached RAM
    case SIGMA0_REQ_ID_FPAGE_IOMEM:	   // dedicated fpage uncached I/O mem
    case SIGMA0_REQ_ID_FPAGE_IOMEM_CACHED: // dedicated fpage uncached I/O mem
	{
          int       super = fp.fp.size >= L4_LOG2_SUPERPAGESIZE;
          int       shift = super ? L4_SUPERPAGESHIFT : L4_PAGESHIFT;
          l4_addr_t addr  = fp.fp.page << L4_PAGESHIFT, a;
          l4_addr_t add   = super ? L4_SUPERPAGESIZE : L4_PAGESIZE;
          l4_addr_t page  = addr >> shift, p;
          int       num   = (1U << fp.fp.size) >> shift;

          /* don't allow cached access above RAM */
          if (fn == SIGMA0_REQ_ID_FPAGE_RAM &&
              page + num > (mem_high >> shift))
            return L4_IPC_SHORT_MSG;

	  for (p = page, a = addr; p < page + num; p++, a += add)
	    {
	      owner_t o = super ? memmap_owner_superpage(a)
				: memmap_owner_page(a);
	      if (o != O_FREE && o != t.id.task)
		return L4_IPC_SHORT_MSG;
	    }
	  for (p = page, a = addr; p < page + num; p++, a += add)
	    {
	      if (( super && !memmap_alloc_superpage(a, t.id.task)) ||
		  (!super && !memmap_alloc_page     (a, t.id.task)))
		{
		  /* a race -- this cannot happen! */
		  panic("Something evil happened!\n");
		}
	    }

	  /* the Fiasco kernel makes the page non-cachable if the frame
	   * address is greater than mem_high */
	  *d1 = addr;
	  *d2 = l4_fpage(addr, fp.fp.size, L4_FPAGE_RW, L4_FPAGE_MAP).fpage;
	  if (fn == SIGMA0_REQ_ID_FPAGE_IOMEM_CACHED
	      || fn == SIGMA0_REQ_ID_FPAGE_RAM)
	    *d2 |= L4_FPAGE_CACHE_ENABLE;
	  else
	    *d2 |= L4_FPAGE_CACHE_DISABLE;

	  return L4_IPC_SHORT_FPAGE;
	}

     case SIGMA0_REQ_ID_FPAGE_ANY:
	if (fp.fp.size == L4_LOG2_PAGESIZE)
	  {
	    find_free(d1, d2, t.id.task, 0);
	    if (*d2 != 0)
	      {
		memmap_alloc_page(*d1, t.id.task);
		if (t.id.task < ROOT_TASKNO) /* sender == kernel? */
		  {
		    *d2 |= 1; /* kernel wants page granted */
		    if (jochen)
		      *d2 &= ~2; /* work around bug in Jochen's L4 */
		  }

		*d2 |= L4_FPAGE_CACHE_ENABLE;
		return L4_IPC_SHORT_FPAGE;
	      }
	  }
	else if (fp.fp.size == L4_LOG2_SUPERPAGESIZE)
	  {
	    find_free(d1, d2, t.id.task, 1);
	    if (*d2 != 0)
	      {
		memmap_alloc_superpage(*d1, t.id.task);
		*d2 |= L4_FPAGE_CACHE_ENABLE;
		return L4_IPC_SHORT_FPAGE;
	      }
	  }
	break;

    case SIGMA0_REQ_ID_KIP: // kernel info page
      *d1 = 0;
      *d2 = l4_fpage((l4_umword_t) l4_info, L4_LOG2_PAGESIZE,
		      L4_FPAGE_RO, L4_FPAGE_MAP).fpage;
      *d2 |= L4_FPAGE_CACHE_ENABLE;
      return L4_IPC_SHORT_FPAGE;

    case SIGMA0_REQ_ID_TBUF:
      if (tbuf_status != 0x00000000 && tbuf_status != ~0UL)
	{
	  *d1 = 0;
	  *d2 = l4_fpage(tbuf_status, L4_LOG2_PAGESIZE,
			 L4_FPAGE_RW, L4_FPAGE_MAP).fpage;
	  *d2 |= L4_FPAGE_CACHE_ENABLE;
	  return L4_IPC_SHORT_FPAGE;
	}
      break;
    }

  return L4_IPC_SHORT_MSG;
}
#endif

void
pager(void)
{
  l4_umword_t d1, d2, pfa;
  void *desc;
  int err;
  l4_threadid_t t;
  l4_msgdope_t result;

  /* now start serving the subtasks */
  for (;;)
    {
      err = l4_ipc_wait(&t, 0, &d1, &d2, L4_IPC_NEVER, &result);

      while (!err)
	{
	  /* we received a paging request here */
	  /* handle the sigma0 protocol */

	  if (debug)
	    printf("SIGMA0: received d1=0x%lx, d2=0x%lx "
		   "from thread="l4util_idfmt"\n", d1, d2, l4util_idstr(t));

	  desc = L4_IPC_SHORT_MSG;

	  if (t.id.task > O_MAX)
	    {
	      if (debug)
		printf("SIGMA0: task number %x is too big\n", t.id.task);
	      /* OOPS.. can't map to this sender. */
	      d1 = d2 = 0;
	    }
	  else			/* valid requester */
	    {
	      pfa = d1 & ~3UL;
	      d1 &=      ~2UL;	/* the L4 kernel seems to get this bit wrong */

#ifdef SIGMA0_REQ_MAGIC
	      if (SIGMA0_IS_MAGIC_REQ(d1))
		{
		  desc = handle_extended_sigma0(&d1, &d2, t);
		}
	      else
#endif
	      if (d1 == ~3UL)
		{
		  /* map any free page back to sender */
		  warn_old(d1, d2, t);

		  find_free(&d1, &d2, t.id.task, 0);

		  if (d2 != 0)	/* found a free page? */
		    {
		      memmap_alloc_page(d1, t.id.task);
		      desc = L4_IPC_SHORT_FPAGE;

		      if (t.id.task < ROOT_TASKNO) /* sender == kernel? */
			{
			  d2 |= 1; /* kernel wants page granted */
			  if (jochen)
			    d2 &= ~2; /* work around bug in Jochen's L4 */
			}
		      d2 |= L4_FPAGE_CACHE_ENABLE;
		    }
		}
	      else if (d1 == 1 && (d2 & 0xff) == 1)
		{
		  /* kernel info page requested */
		  warn_old(d1, d2, t);

		  d1 = 0;
		  d2 = l4_fpage((l4_umword_t) l4_info, L4_LOG2_PAGESIZE,
				 L4_FPAGE_RO, L4_FPAGE_MAP).fpage;
		  d2 |= L4_FPAGE_CACHE_ENABLE;
		  desc = L4_IPC_SHORT_FPAGE;
		}
	      else if (d1 == 1 && (d2 & 0xff) == 0 && t.id.task < ROOT_TASKNO)
		{
		  /* kernel requests recommended kernel-internal RAM
                     size (in number of pages) */
		  warn_old(d1, d2, t);

		  d1 = mem_high / 8 / L4_PAGESIZE;
		}
	      else if (d1 == 1 && (d2 & 0xff) == 0xff)
		{
		  /* tbuf descriptor page requested */
		  warn_old(d1, d2, t);

		  if (tbuf_status != 0x00000000 && tbuf_status != ~0UL)
		    {
		      d1 = 0;
		      d2 = l4_fpage(tbuf_status, L4_LOG2_PAGESIZE,
				    L4_FPAGE_RW, L4_FPAGE_MAP).fpage;
		      d2 |= L4_FPAGE_CACHE_ENABLE;
		      desc = L4_IPC_SHORT_FPAGE;
		    }
		}
	      else if (pfa < 0x40000000)
		{
		  /* map a specific page */
		  if ((d1 & 1) && (d2 == (L4_LOG2_SUPERPAGESIZE << 2)))
		    {
		      /* superpage request */
		      if (memmap_alloc_superpage(pfa, t.id.task))
			{
			  d1 &= L4_SUPERPAGEMASK;
			  d2 = l4_fpage(d1, L4_LOG2_SUPERPAGESIZE,
					L4_FPAGE_RW, L4_FPAGE_MAP).fpage;
			  d2 |= L4_FPAGE_CACHE_ENABLE;
			  desc = L4_IPC_SHORT_FPAGE;

			  /* flush the superpage first so that
                             contained 4K pages which already have
                             been mapped to the task don't disturb the
                             4MB mapping */
			  l4_fpage_unmap(l4_fpage(d1, L4_LOG2_SUPERPAGESIZE,
						  0, 0),
					 L4_FP_FLUSH_PAGE|L4_FP_OTHER_SPACES);

			  goto reply;
			}
		    }
		  /* failing a superpage allocation, try a single page
                     allocation */
		  if (memmap_alloc_page(d1, t.id.task))
		    {
		      d1 &= L4_PAGEMASK;
		      d2 = l4_fpage(d1, L4_LOG2_PAGESIZE,
				    L4_FPAGE_RW, L4_FPAGE_MAP).fpage;
		      d2 |= L4_FPAGE_CACHE_ENABLE;
		      desc = L4_IPC_SHORT_FPAGE;
		    }
		  else if (pfa >= (l4_info->semi_reserved.start & L4_PAGEMASK)
			   && pfa < l4_info->semi_reserved.end)
		    {
		      /* adapter area, page faults are OK */
		      d1 &= L4_PAGEMASK;
		      d2 = l4_fpage(d1, L4_LOG2_PAGESIZE,
				    L4_FPAGE_RW, L4_FPAGE_MAP).fpage;
		      d2 |= L4_FPAGE_CACHE_ENABLE;
		      desc = L4_IPC_SHORT_FPAGE;
		    }
		  else
		    {
		      if (debug)
			printf("SIGMA0: no appropriate page found\n");
		    }
		}
	      else if (pfa >= 0x40000000 && pfa < 0xC0000000 && !(d1 & 1))
		{
		  /* request physical 4 MB area at pfa + 0x40000000 */
		  warn_old(d1, d2, t);

		  pfa &= L4_SUPERPAGEMASK;

		  if (memmap_alloc_superpage(pfa + 0x40000000, t.id.task))
		    {
		      /* map the superpage to the subtask */

		      d1 = 0;

		      /* Not only does the Sigma0 interface specify
                         that requests between 0x40000000 and
                         0xC0000000 are satisfied with memory from
                         0x80000000 to 0xffffffff -- the kernel
                         interface is also so kludged.  This is to
                         allow I/O mappings to start with addresses
                         like 0xf0... . */
		      d2 = l4_fpage(pfa, /* maps pfa + 0x40000000 */
				    L4_LOG2_SUPERPAGESIZE,
				    L4_FPAGE_RW, L4_FPAGE_MAP).fpage;
		      d2 |= L4_FPAGE_CACHE_DISABLE;
		      desc = L4_IPC_SHORT_FPAGE;
		    }
		}

	      else if (l4_is_io_page_fault(d1))
		{
		  unsigned i, port, size, end, got_all;
		  port = ((l4_fpage_t)(d1)).iofp.iopage;
		  size = ((l4_fpage_t)(d1)).iofp.iosize;
		  end = port + (1L << size);
		  if (end > IO_MAX)
		    end = IO_MAX;
		  got_all = 1;
		  for(i = port; i < end; i++)
		    {
		      if (iomap_alloc_port(i, t.id.task))
			continue;
		      got_all = 0;
		      break;
		    }
		  if (got_all)
		    {
		      d2 = l4_iofpage(port, size, 0).fpage;
		      d1 = 0;
		      desc = L4_IPC_SHORT_FPAGE;
		    }
		}
	      else
		{
		  printf("SIGMA0: can't handle d1=0x%lx, d2=0x%lx "
			 "from thread="l4util_idfmt"\n",
			 d1, d2, l4util_idstr(t));
		}

	      if (desc == L4_IPC_SHORT_MSG)
		d1 = d2 = 0;
	    }

	reply:

	  if (debug)
	    printf("SIGMA0: sending d1=0x%lx, d2=0x%lx "
		   "to thread="l4util_idfmt"\n", d1, d2, l4util_idstr(t));

	  /* send reply and wait for next message */
	  err = l4_ipc_reply_and_wait(t, desc, d1, d2,
				      &t, 0, &d1, &d2,
				      L4_IPC_SEND_TIMEOUT_0, &result);

	}
    }
}

static void
find_free(l4_umword_t *d1, l4_umword_t *d2, owner_t owner, int super)
{
  /* for kernel tasks, start looking at the back */
  l4_addr_t address = owner < ROOT_TASKNO
			   ? (mem_high - 1) & L4_SUPERPAGEMASK
			   : 0;

  for (;;)
    {
      unsigned f = memmap_nrfree_superpage(address);
      if (super)
	{
	  if (f == L4_SUPERPAGESIZE/L4_PAGESIZE)
	    {
	      *d1 = address;
	      *d2 = l4_fpage(address, L4_LOG2_SUPERPAGESIZE,
			     L4_FPAGE_RW, L4_FPAGE_MAP).fpage;
	      return;
	    }
	}
      else
	{
	  if (f != 0)
	    {
	      if (owner < ROOT_TASKNO)
		address += L4_SUPERPAGESIZE - L4_PAGESIZE;

	      for (;;)
		{
		  if (memmap_owner_page(address) != O_FREE)
		    {
		      address += owner < ROOT_TASKNO ? -L4_PAGESIZE
						     : L4_PAGESIZE;
		      continue;
		    }

		  /* found page */
		  *d1 = address;
		  *d2 = l4_fpage(address, L4_LOG2_PAGESIZE,
				 L4_FPAGE_RW, L4_FPAGE_MAP).fpage;
		  return;
		}
	    }
	}

      if (owner < ROOT_TASKNO)
	{
	  if (address == 0)
	    break;
	  address -= L4_SUPERPAGESIZE;
	}
      else
	{
	  address += L4_SUPERPAGESIZE;
	  if (address >= mem_high)
	    break;
	}
    }

  /* nothing found! */
  *d1 = *d2 = 0;
}

void
memmap_dump_area(l4_addr_t start, l4_addr_t end, owner_t owner)
{
  printf("  ["l4_addr_fmt"-"l4_addr_fmt") [%7ldKB] ",
      start, end, (end - start) >> 10);
  if (owner == O_FREE)
    printf("free");
  else if (owner == O_RESERVED)
    printf("reserved");
  else
    printf("#%02x", owner);
  putchar('\n');
}

void
memmap_dump(void)
{
  l4_addr_t start, end;
  owner_t state;

  printf("Sigma0 memmap:\n");
  state = memmap_owner_page(0);
  for (start = end = 0;
       end < (l4_addr_t)mem_high;
       end += L4_PAGESIZE)
    {
      if (memmap_owner_superpage(end) == O_RESERVED)
	{
	  if (state != memmap_owner_page(end))
	    {
	      memmap_dump_area(start, end, state);
	      state = memmap_owner_page(end);
	      start = end;
	    }
	  end += L4_PAGESIZE;
	}
      else
	{
	  if (state != memmap_owner_superpage(end))
	    {
	      memmap_dump_area(start, end, state);
	      state = memmap_owner_superpage(end);
	      start = end;
	    }
	}
    }
  memmap_dump_area(start, end, state);
}
