/**
 *    \file    dice/src/fe/FEUnionType.cpp
 *  \brief   contains the implementation of the class CFEUnionType
 *
 *    \date    01/31/2001
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "FEUnionType.h"
#include "FEUnionCase.h"
#include "FEFile.h"
#include "Compiler.h"
#include "File.h"
#include <iostream>
using namespace std;

CFEUnionType::CFEUnionType(string sTag,
    vector<CFEUnionCase*> * pUnionBody)
: CFEConstructedType(TYPE_UNION)
{
    m_sTag = sTag;
    if (pUnionBody)
        m_vUnionBody.swap(*pUnionBody);
    else
        m_bForwardDeclaration = true;
    vector<CFEUnionCase*>::iterator iter;
    for (iter = m_vUnionBody.begin(); iter != m_vUnionBody.end(); iter++)
    {
        (*iter)->SetParent(this);
    }
}

CFEUnionType::CFEUnionType(CFEUnionType & src)
: CFEConstructedType(src)
{
    m_sTag = src.m_sTag;
    COPY_VECTOR(CFEUnionCase, m_vUnionBody, iter);
}

/** cleans up a union type object */
CFEUnionType::~CFEUnionType()
{
    DEL_VECTOR(m_vUnionBody);
}

/** retrives a pointer to the first union case
 *  \return an iterator which points to the first union case object
 */
vector<CFEUnionCase*>::iterator CFEUnionType::GetFirstUnionCase()
{
    return m_vUnionBody.begin();
}

/** \brief retrieves the next union case object
 *  \param iter the iterator, which points to the next union case object
 *  \return the next union case object
 */
CFEUnionCase*
CFEUnionType::GetNextUnionCase(vector<CFEUnionCase*>::iterator &iter)
{
    if (iter == m_vUnionBody.end())
        return 0;
    return *iter++;
}

/** creates a copy of this object
 *  \return a reference to a new union type object
 */
CObject *CFEUnionType::Clone()
{
    return new CFEUnionType(*this);
}

/** \brief checks the integrity of an union
 *  \return true if everything is fine
 *
 * A union is consistent if it has a union body and the elements of this body
 * are consistent.
 */
bool CFEUnionType::CheckConsistency()
{
    CCompiler::Verbose(0, "%s called\n", __FUNCTION__);

    if (m_bForwardDeclaration)
    {
        // check if this is an alias
        CFEFile *pRoot = dynamic_cast<CFEFile*>(GetRoot());
        if (pRoot->FindTaggedDecl(GetTag()))
            return true;
    }
    if (m_vUnionBody.empty())
    {
        CCompiler::GccError(this, 0, "A union without members is not allowed.");
        return false;
    }
    vector<CFEUnionCase*>::iterator iter = GetFirstUnionCase();
    CFEUnionCase *pUnionCase;
    while ((pUnionCase = GetNextUnionCase(iter)) != 0)
    {
        if (!(pUnionCase->CheckConsistency()))
            return false;
    }
    return true;
}

