/**
 *    \file    dice/src/be/sock/BESocket.cpp
 *  \brief   contains the declaration of the class CBESocket
 *
 *    \date    08/18/2003
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */
#include "be/sock/BESocket.h"
#include "be/BEFile.h"
#include "be/BEContext.h"
#include "be/BEFunction.h"
#include "be/BEType.h"
#include "be/BEMsgBuffer.h"
#include "be/BEDeclarator.h"
#include "be/BENameFactory.h"
#include "Compiler.h"
#include "TypeSpec-Type.h"

CBESocket::CBESocket()
 : CBECommunication()
{
}

/** destroys the socket object */
CBESocket::~CBESocket()
{
}

/** \brief writes the sending of the message
 *  \param pFile the file to write to
 *  \param pFunction the function to write to
 *  \param bUseEnv true if socket is stroed in environment (otherwise 'sd' is socket descriptor)
 *  \param nDirection the direction to use for calculating send sizes
 *  \param sFunc the name of the calling function

 */
void
CBESocket::WriteSendTo(CBEFile* pFile,
    CBEFunction* pFunction,
    bool bUseEnv,
    int nDirection,
    const char* sFunc)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sCorbaObj = pNF->GetCorbaObjectVariable();

    CBEMsgBuffer *pMsgBuffer = pFunction->GetMessageBuffer();
    string sPtrName, sSizeName;
    if (pMsgBuffer->GetDeclarator()->GetStars() == 0)
	sPtrName = "&";
    else
	sSizeName = "*";
    sPtrName += pMsgBuffer->GetDeclarator()->GetName();
    sSizeName += pMsgBuffer->GetDeclarator()->GetName();

    *pFile << "\tdice_ret_size = sendto (";
    WriteSocketDescriptor(pFile, pFunction, bUseEnv);
    *pFile << ", " << sPtrName << ", sizeof(" << sSizeName << "), 0, " <<
	"(struct sockaddr*)" << sCorbaObj << ", dice_fromlen);\n";

    *pFile << "\tif (dice_ret_size < sizeof(" << sSizeName << "))\n";
    *pFile << "\t{\n";
    pFile->IncIndent();
    *pFile << "\tperror(\"" << sFunc << "\");\n";
    pFunction->WriteReturn(pFile);
    pFile->DecIndent();
    *pFile << "\t}\n";
}

/** \brief writes the receiving of a message
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 *  \param bUseEnv true if environment contains socket descriptor
 *  \param bUseMaxSize true if the variable sized message buffer should be ignored
 *  \param sFunc the name of the calling function

 */
void
CBESocket::WriteReceiveFrom(CBEFile* pFile,
    CBEFunction* pFunction,
    bool bUseEnv,
    bool bUseMaxSize,
    const char* sFunc)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sCorbaObj = pNF->GetCorbaObjectVariable();
    string sOffset = pNF->GetOffsetVariable();
    CBEMsgBuffer *pMsgBuffer = pFunction->GetMessageBuffer();
    string sPtrName, sSizeName;
    if (pMsgBuffer->GetDeclarator()->GetStars() == 0)
	sPtrName = "&";
    else
	sSizeName = "*";
    sPtrName += pMsgBuffer->GetDeclarator()->GetName();
    sSizeName += pMsgBuffer->GetDeclarator()->GetName();

    *pFile << "\tdice_ret_size = recvfrom(";
    WriteSocketDescriptor(pFile, pFunction, bUseEnv);
    *pFile << ", " << sPtrName << ", sizeof(" << sSizeName << "), 0, " <<
	"(struct sockaddr*)" << sCorbaObj << ", &dice_fromlen);\n";

    *pFile << "\tif (dice_ret_size < 0)\n";
    *pFile << "\t{\n";
    pFile->IncIndent();
    *pFile << "\tperror (\"" << sFunc << "\");\n";
    pFunction->WriteReturn(pFile);
    pFile->DecIndent();
    *pFile << "\t}\n";
}

/** \brief prints the socket descriptor to the target file
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 *  \param bUseEnv true if environment variable should be used
 */
void CBESocket::WriteSocketDescriptor(CBEFile* pFile,
    CBEFunction* pFunction,
    bool bUseEnv)
{
    CBEDeclarator *pEnvDecl = 0;
    if (bUseEnv)
    {
        CBETypedDeclarator *pEnv = pFunction->GetEnvironment();
        if (pEnv)
        {
            vector<CBEDeclarator*>::iterator iterE = pEnv->GetFirstDeclarator();
            pEnvDecl = *iterE;
        }
    }
    if (pEnvDecl)
    {
        if (pEnvDecl->GetStars())
	    *pFile << pEnvDecl->GetName() << "->cur_socket";
        else
	    *pFile << pEnvDecl->GetName() << ".cur_socket";
    }
    else
	*pFile << "sd";
}

/** \brief writes the call implementation of the socket layer
 *  \param pFile the file to write to
 *  \param pFunction the function to write for

 */
void CBESocket::WriteCall(CBEFile* pFile, CBEFunction* pFunction)
{
    // send message
    WriteSendTo(pFile, pFunction, false, 0/* both directions*/, "call");
    // offset might have been overwritten, so it has to be reinitialized
    CBEMsgBuffer *pMsgBuffer = pFunction->GetMessageBuffer();
    if (pMsgBuffer->IsVariableSized(0))
        pMsgBuffer->WriteInitialization(pFile);
    // zero msgbuffer
    WriteZeroMsgBuffer(pFile, pFunction);
    // receive response
    WriteReceiveFrom(pFile, pFunction, false, false, "call");
}

/** \brief writes the reply-and-wait implementation of the socket layer
 *  \param pFile the file to write to
 *  \param pFunction the function to write for

 */
void CBESocket::WriteReplyAndWait(CBEFile* pFile, CBEFunction* pFunction)
{
    // send message
    WriteSendTo(pFile, pFunction, true, pFunction->GetSendDirection(),
	"reply-wait");
    // reset msg buffer to zeros
    WriteZeroMsgBuffer(pFile, pFunction);
    // want to receive from any client again
    string sCorbaObj = CCompiler::GetNameFactory()->GetCorbaObjectVariable();
    *pFile << "\t" << sCorbaObj << "->sin_addr.s_addr = INADDR_ANY;\n";
    // wait for new request
    WriteReceiveFrom(pFile, pFunction, true, true, "reply-wait");
}

/** \brief zeros the message buffer
 *  \param pFile the file to write to
 *  \param pFunction the function to write for

 */
void CBESocket::WriteZeroMsgBuffer(CBEFile* pFile,
    CBEFunction* pFunction)
{
    string sOffset = CCompiler::GetNameFactory()->GetOffsetVariable();
    CBEMsgBuffer *pMsgBuffer = pFunction->GetMessageBuffer();
    // msgbuffer is always a pointer: either variable sized or char[]
    string sPtrName, sSizeName;
    if (pMsgBuffer->GetDeclarator()->GetStars() == 0)
	sPtrName = "&";
    else
	sSizeName = "*";
    sPtrName += pMsgBuffer->GetDeclarator()->GetName();
    sSizeName += pMsgBuffer->GetDeclarator()->GetName();

    *pFile << "\tbzero (" << sPtrName << ", sizeof(" << sSizeName << ") );\n";
}

/** \brief writes the wait implementation of the socket layer
 *  \param pFile the file to write to
 *  \param pFunction the function to write for

 */
void CBESocket::WriteWait(CBEFile* pFile, CBEFunction* pFunction)
{
    WriteZeroMsgBuffer(pFile, pFunction);
    // wait for new request
    WriteReceiveFrom(pFile, pFunction, true, true, "wait");
}

/** \brief writes the initialization
 *  \param pFile the file to write to
 *  \param pFunction the funtion to write for
 */
void CBESocket::WriteInitialization(CBEFile *pFile,
    CBEFunction *pFunction)
{
    bool bUseEnv = pFunction->IsComponentSide();

    *pFile << "\t";
    WriteSocketDescriptor(pFile, pFunction, bUseEnv);
    *pFile << " = socket(PF_INET, SOCK_DGRAM, 0);\n";

    *pFile << "\tif (";
    WriteSocketDescriptor(pFile, pFunction, bUseEnv);
    *pFile << " < 0)\n";
    *pFile << "\t{\n";
    pFile->IncIndent();
    *pFile << "\tperror(\"socket creation\");\n";
    pFunction->WriteReturn(pFile);
    pFile->DecIndent();
    *pFile << "\t}\n";
}

/** \brief writes the assigning of a local name to a communication port
 *  \param pFile the file to write to
 *  \param pFunction the funtion to write for
 */
void CBESocket::WriteBind(CBEFile *pFile,
    CBEFunction *pFunction)
{
    bool bUseEnv = pFunction->IsComponentSide();
    string sCorbaObj = CCompiler::GetNameFactory()->GetCorbaObjectVariable();

    *pFile << "\tif (bind(";
    WriteSocketDescriptor(pFile, pFunction, bUseEnv);
    *pFile << ", (struct sockaddr*)" << sCorbaObj << 
	", sizeof(struct sockaddr)) < 0)\n";
    *pFile << "\t{\n";
    pFile->IncIndent();
    *pFile << "\tperror(\"bind\");\n";
    pFunction->WriteReturn(pFile);
    pFile->DecIndent();
    *pFile << "\t}\n";
}

/** \brief writes the clean up code
 *  \param pFile the file to write to
 *  \param pFunction the funtion to write for
 */
void CBESocket::WriteCleanup(CBEFile *pFile,
    CBEFunction *pFunction)
{
    *pFile << "\tclose (";
    WriteSocketDescriptor(pFile, pFunction, pFunction->IsComponentSide());
    *pFile << ");\n";
}

