/**
 *    \file    dice/src/be/l4/L4BEMsgBuffer.cpp
 *  \brief   contains the implementation of the class CL4BEMsgBuffer
 *
 *    \date    02/02/2005
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2005
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "L4BEMsgBuffer.h"
#include "L4BEMsgBufferType.h"
#include "L4BESizes.h"
#include "L4BENameFactory.h"
#include "be/BEMsgBuffer.h"
#include "be/BEFunction.h" // required for DIRECTION_IN/_OUT
#include "be/BEContext.h"
#include "be/BEMsgBufferType.h"
#include "be/BEInterfaceFunction.h"
#include "be/BESrvLoopFunction.h"
#include "be/BEStructType.h"
#include "be/BEAttribute.h"
#include "be/BEExpression.h"
#include "be/BEDeclarator.h"
#include "be/BEUnionCase.h"
#include "be/BEClassFactory.h"
#include "Compiler.h"
#include "fe/FEInterface.h"
#include "fe/FEOperation.h"
#include "TypeSpec-Type.h"
#include "TypeSpec-L4Types.h"
#include <cassert>
using namespace std;

CL4BEMsgBuffer::CL4BEMsgBuffer()
 : CBEMsgBuffer()
{
}

CL4BEMsgBuffer::CL4BEMsgBuffer(CL4BEMsgBuffer & src)
 : CBEMsgBuffer(src)
{
}

/** destroys the object */
CL4BEMsgBuffer::~CL4BEMsgBuffer()
{
}

/** \brief add platform specific members to specific struct
 *  \param pFunction the function of the message buffer
 *  \param pStruct the struct to add the members to
 *  \param nDirection the direction of the struct
 *  \return true if successful
 *
 *  In this implementation we should add the L4 specific receive
 *  window for flexpages, the size and the send dope.
 */
bool
CL4BEMsgBuffer::AddPlatformSpecificMembers(CBEFunction *pFunction,
    CBEStructType *pStruct,
    int nDirection)
{
    if (!CBEMsgBuffer::AddPlatformSpecificMembers(pFunction, pStruct, 
	    nDirection))
	return false;

    // create receive flexpage
    CBETypedDeclarator *pFlexpage = GetFlexpageVariable();
    if (!pFlexpage)
	return false;
    // check if struct already has flexpage
    if (pStruct->FindMember(pFlexpage->GetDeclarator()->GetName()))
	delete pFlexpage;
    else
	pStruct->AddMember(pFlexpage);
    
    // create size dope
    CBETypedDeclarator *pSizeDope = GetSizeDopeVariable();
    if (!pSizeDope)
	return false;
    if (pStruct->FindMember(pSizeDope->GetDeclarator()->GetName()))
	delete pSizeDope;
    else
	pStruct->AddMember(pSizeDope);
    
    // create send dope
    CBETypedDeclarator *pSendDope = GetSendDopeVariable();
    if (!pSendDope)
	return false;
    if (pStruct->FindMember(pSendDope->GetDeclarator()->GetName()))
	delete pSendDope;
    else
	pStruct->AddMember(pSendDope);

    return true;
}

/** \brief creates a member variable for the receive flexpage
 *  \return a reference to the variable or NULL if none could be created
 */
CBETypedDeclarator* 
CL4BEMsgBuffer::GetFlexpageVariable()
    throw (CBECreateException*)
{
    string exc = string(__FUNCTION__);
    
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEType *pType = pCF->GetNewType(TYPE_RCV_FLEXPAGE);
    string sName = CCompiler::GetNameFactory()->
	GetMessageBufferMember(TYPE_RCV_FLEXPAGE);
    CBETypedDeclarator *pFpage = pCF->GetNewTypedDeclarator();
    try
    {
	pType->CreateBackEnd(false, 0, TYPE_RCV_FLEXPAGE);
	pFpage->CreateBackEnd(pType, sName);
    }
    catch (CBECreateException *e)
    {
	delete pFpage;
	delete pType;
	e->Print();
	delete e;
	exc += " failed, because flexpage variable could not be created";
	throw new CBECreateException(exc);
    }
    delete pType; // cloned in CBETypedDeclarator::CreateBackEnd
    
    CBEAttribute *pAttr = pCF->GetNewAttribute();
    try
    {
	// add directional attribute so later checks when marshaling work
	pAttr->SetParent(pFpage);
	pAttr->CreateBackEnd(ATTR_IN);
	pFpage->AddAttribute(pAttr);
	// add directional attribute so later checks when marshaling work
	pAttr = pCF->GetNewAttribute();
	pAttr->SetParent(pFpage);
	pAttr->CreateBackEnd(ATTR_OUT);
	pFpage->AddAttribute(pAttr);
    }
    catch (CBECreateException *e)
    {
	delete pAttr;
	delete pFpage;
	e->Print();
	delete e;
	exc += " failed, because attribute could not be created";
	throw new CBECreateException(exc);
    }

    return pFpage;
}

/** \brief creates a member variable for the size dope

 *  \return a reference to the created variable or NULL if failed
 */
CBETypedDeclarator* 
CL4BEMsgBuffer::GetSizeDopeVariable()
    throw (CBECreateException*)
{
    string exc = string(__FUNCTION__);
    
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEType *pType = pCF->GetNewType(TYPE_MSGDOPE_SIZE);
    string sName = CCompiler::GetNameFactory()->
	GetMessageBufferMember(TYPE_MSGDOPE_SIZE);
    CBETypedDeclarator *pSize = pCF->GetNewTypedDeclarator();
    try
    {
	pType->CreateBackEnd(false, 0, TYPE_MSGDOPE_SIZE);
	pSize->CreateBackEnd(pType, sName);
    }
    catch (CBECreateException *e)
    {
	delete pSize;
	delete pType;
	e->Print();
	delete e;
	exc += " failed, because size dope could not be created.";
	throw new CBECreateException(exc);
    }
    delete pType; // cloned in CBETypedDeclarator::CreateBackEnd
    
    // add directional attribute so later checks when marshaling work
    CBEAttribute *pAttr = pCF->GetNewAttribute();
    pAttr->SetParent(pSize);
    try
    {
	pAttr->CreateBackEnd(ATTR_IN);
	pSize->AddAttribute(pAttr);
	// add directional attribute so later checks when marshaling work
	pAttr = pCF->GetNewAttribute();
	pAttr->SetParent(pSize);
	pAttr->CreateBackEnd(ATTR_OUT);
	pSize->AddAttribute(pAttr);
    }
    catch (CBECreateException *e)
    {
	delete pAttr;
	delete pSize;
	e->Print();
	delete e;
	exc += " failed, because attribute could not be created.";
	throw new CBECreateException(exc);
    }

    return pSize;
}

/** \brief creates a member variable for the send dope
 *  \return a reference to the send dope variable
 */
CBETypedDeclarator* 
CL4BEMsgBuffer::GetSendDopeVariable()
    throw (CBECreateException*)
{
    string exc = string(__FUNCTION__);
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBEType *pType = pCF->GetNewType(TYPE_MSGDOPE_SEND);
    string sName = CCompiler::GetNameFactory()->
	GetMessageBufferMember(TYPE_MSGDOPE_SEND);
    CBETypedDeclarator *pSend = pCF->GetNewTypedDeclarator();
    try
    {
	pType->CreateBackEnd(false, 0, TYPE_MSGDOPE_SEND);
	pSend->CreateBackEnd(pType, sName);
    }
    catch (CBECreateException *e)
    {
	delete pSend;
	delete pType;
	e->Print();
	delete e;
	exc += " failed, because send dope could not be created.";
	throw new CBECreateException(exc);
    }
    delete pType; // cloned in CBETypedDeclarator::CreateBackEnd
    
    // add directional attribute so later checks when marshaling work
    CBEAttribute *pAttr = pCF->GetNewAttribute();
    pAttr->SetParent(pSend);
    try
    {
	pAttr->CreateBackEnd(ATTR_IN);
	pSend->AddAttribute(pAttr);
	// add directional attribute so later checks when marshaling work
	pAttr = pCF->GetNewAttribute();
	pAttr->SetParent(pSend);
	pAttr->CreateBackEnd(ATTR_OUT);
	pSend->AddAttribute(pAttr);
    }
    catch (CBECreateException *e)
    {
	delete pAttr;
	delete pSend;
	e->Print();
	delete e;
	exc += " failed, because attribute could not be created.";
	throw new CBECreateException(exc);
    }

    return pSend;
}

/** \brief sorts one struct in the message buffer
 *  \param pStruct the struct to sort
g
 *  \return true if sorting was successful
 *
 * This implementation first moves the special message buffer members to the
 * front and then calls the base class' Sort method.  This order is important,
 * because the Sort method calls GetStartOfPayload, which in turn tries to
 * skip the special message buffer members.  Thus they have to be at the begin
 * of the message buffer before calling base class' Sort.
 *
 * The same procedure has to be done after calling the base class' Sort,
 * because after the payload sort opcode and exception are moved to the start
 * of the message buffer.  Thus we have to move the special members past them
 * again.
 *
 * This implementation takes also care of flexpages and indirect strings.
 */
bool
CL4BEMsgBuffer::Sort(CBEStructType *pStruct)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    // doing this in reverse order, so we can insert them at
    // position 0
    // get send dope
    string sName = pNF->GetMessageBufferMember(TYPE_MSGDOPE_SEND);
    pStruct->MoveMember(sName, 0);
    // get size dope
    sName = pNF->GetMessageBufferMember(TYPE_MSGDOPE_SIZE);
    pStruct->MoveMember(sName, 0);
    // get flexpage
    sName = pNF->GetMessageBufferMember(TYPE_RCV_FLEXPAGE);
    pStruct->MoveMember(sName, 0);

    // the following call, calls SortPayload which uses DoExchangeMembers to
    // sort the members.
    if (!CBEMsgBuffer::Sort(pStruct))
	return false;

    // move all flexpages to begin (to override opcode and exception
    // placement) -- therefore we do not start with the payload, but with
    // iterate all members of the struct
    vector<CBETypedDeclarator*>::iterator iter = pStruct->GetFirstMember(); 
    CBETypedDeclarator *pMember;
    while ((pMember = pStruct->GetNextMember(iter)) != 0)
    {
	vector<CBETypedDeclarator*>::iterator iterS = iter;
	CBETypedDeclarator *pSuccessor = pStruct->GetNextMember(iterS);
	if (!pSuccessor)
	    break; // finished
	
	int nFirstType = pMember->GetType()->GetFEType();
	int nSecondType = pSuccessor->GetType()->GetFEType();

	if ((nFirstType == TYPE_FLEXPAGE) &&
	    (nSecondType != TYPE_FLEXPAGE))
	    continue;
	
	if ((nFirstType != TYPE_FLEXPAGE) &&
	    (nSecondType == TYPE_FLEXPAGE))
	{
	    pStruct->MoveMember(pSuccessor->GetDeclarator()->GetName(),
		    pMember->GetDeclarator()->GetName());
	    iter = pStruct->GetFirstMember();
	}
    }

    // and now move the L4 message header back up front
    // get send dope
    sName = pNF->GetMessageBufferMember(TYPE_MSGDOPE_SEND);
    pStruct->MoveMember(sName, 0);
    // get size dope
    sName = pNF->GetMessageBufferMember(TYPE_MSGDOPE_SIZE);
    pStruct->MoveMember(sName, 0);
    // get flexpage
    sName = pNF->GetMessageBufferMember(TYPE_RCV_FLEXPAGE);
    pStruct->MoveMember(sName, 0);
    
    return true;
}

/** \brief checks if two members of a struct should be exchanged
 *  \param pFirst the first member
 *  \param pSecond the second member - first's successor
 *  \return true if exchange
 *
 * This method check for refstrings in the message buffer. They should always
 * stay at the end.
 */
bool
CL4BEMsgBuffer::DoExchangeMembers(CBETypedDeclarator *pFirst,
	CBETypedDeclarator *pSecond)
{
    assert(pFirst);
    assert(pSecond);
    CCompiler::Verbose(0, "%s called with (%s, %s)\n", __FUNCTION__,
	pFirst->GetDeclarator()->GetName().c_str(),
	pSecond->GetDeclarator()->GetName().c_str());

    int nFirstType = pFirst->GetType()->GetFEType();
    int nSecondType = pSecond->GetType()->GetFEType();

    // test refstrings
    if ((nFirstType == TYPE_REFSTRING) &&
	(nSecondType != TYPE_REFSTRING))
	return true;
    if ((nFirstType != TYPE_REFSTRING) &&
	(nSecondType == TYPE_REFSTRING))
	return false;

    return CBEMsgBuffer::DoExchangeMembers(pFirst, pSecond);
}

/** \brief return the offset where the payload starts
 *  \return the offset in bytes
 */
int
CL4BEMsgBuffer::GetPayloadOffset()
{
    CBESizes *pSizes = CCompiler::GetSizes();
    int nPage = pSizes->GetSizeOfType(TYPE_RCV_FLEXPAGE);
    int nDope = pSizes->GetSizeOfType(TYPE_MSGDOPE_SEND);
    return nPage + 2*nDope;
}

/** \brief adds a generic structure to the message buffer
 *  \param pFunction the function which owns the message buffer
 *  \param pFEOperation the front-end operation to use as reference
 *  \return true if successful
 *
 * For non interface functions we test if the members for both directions are
 * sufficient for short IPC (2/3 word sized members at the begin). If not, we
 * add an extra struct to the union containing base elements and 2/3 word
 * sized members.
 */
bool
CL4BEMsgBuffer::AddGenericStruct(CBEFunction *pFunction,
    CFEOperation *pFEOperation)
{
    // test if this an interface function
    if (dynamic_cast<CBEInterfaceFunction*>(pFunction))
	return true;
   
    bool bWordMembers = HasWordMembers(pFunction, 
	pFunction->GetSendDirection());
    if (bWordMembers)
    {
	bWordMembers = HasWordMembers(pFunction, 
	    pFunction->GetReceiveDirection());
    }
   
    if (bWordMembers)
	return true;

    return CBEMsgBuffer::AddGenericStruct(pFunction, pFEOperation);
}

/** \brief adds the members of to the generic struct
 *  \param pClass the class to count for
 *  \param pStruct the struct to add the members to
 *  \return true if successful
 */
bool
CL4BEMsgBuffer::AddGenericStructMembers(CBEClass *pClass,
    CBEStructType *pStruct)
{
    // count word members
    DTRACE("%s for class %s called\n", __FUNCTION__,
	pClass->GetName().c_str());
    int nWords = GetWordMemberCount(pClass);
    DTRACE("%s for class %s. GetWordMemberCount returned %d\n", __FUNCTION__,
	pClass->GetName().c_str(), nWords);
    // count refstring members
    int nStrings = CBEMsgBuffer::GetMemberSize(TYPE_REFSTRING);
    // create word member and add to struct
    CBETypedDeclarator *pMember = GetWordMemberVariable(nWords);
    if (!pMember)
	return false;
    pStruct->AddMember(pMember);
    // create restring member and add to struct
    if (nStrings > 0)
    {
	pMember = GetRefstringMemberVariable(nStrings);
	if (!pMember)
	    return false;
	pStruct->AddMember(pMember);
    }

    // return status
    return true;
}

/** \brief creates an array member of word sized elements for the message buffer
 *  \param nNumber the number of elements
 *  \return a reference to the newly created opcode member
 */
CBETypedDeclarator*
CL4BEMsgBuffer::GetRefstringMemberVariable(int nNumber)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sName = pNF->GetMessageBufferMember(TYPE_REFSTRING);
    return GetMemberVariable(TYPE_REFSTRING, false, sName, nNumber);
}

/** \brief writes the initialization of specific members
 *  \param pFile the file to write to
 *  \param nType the type of the members to initialize
 *  \param nDirection the direction of the struct to initialize
 */
void
CL4BEMsgBuffer::WriteInitialization(CBEFile *pFile,
	int nType,
	int nDirection)
{
    if ((nType != TYPE_MSGDOPE_SIZE) &&
	(nType != TYPE_MSGDOPE_SEND) &&
	(nType != TYPE_REFSTRING) &&
	(nType != TYPE_RCV_FLEXPAGE))
    {
	CBEMsgBuffer::WriteInitialization(pFile, nType, nDirection);
	return;
    }

    if (nType == TYPE_REFSTRING)
    {
	WriteRefstringInitialization(pFile, nDirection);
	return;
    }
    if (nType == TYPE_RCV_FLEXPAGE)
    {
	WriteRcvFlexpageInitialization(pFile, nDirection);
	return;
    }
   
    CBEFunction *pFunction = GetSpecificParent<CBEFunction>();
    DTRACE("%s for func %s and dir %d\n", __FUNCTION__,
	pFunction ? pFunction->GetName().c_str() : "(no func)",
	nDirection);

    // if direction is 0 we have to get maximum of IN and OUT
    int nWords = 0;
    int nStrings = 0;
    CBESizes *pSizes = CCompiler::GetSizes();
    int nRefSize = pSizes->GetSizeOfType(TYPE_REFSTRING) /
	pSizes->GetSizeOfType(TYPE_MWORD);
    if (nDirection == 0)
    {
	int nWordsIn = CBEMsgBuffer::GetMemberSize(TYPE_MWORD, pFunction,
	    DIRECTION_IN, false);
	DTRACE("%s words in: %d\n", __FUNCTION__, nWordsIn);
	if (nWordsIn < 0)
	    nWordsIn = CBEMsgBuffer::GetMemberSize(TYPE_MWORD, pFunction,
		DIRECTION_IN, true);
	DTRACE("%s words in: %d\n", __FUNCTION__, nWordsIn);
	int nWordsOut = CBEMsgBuffer::GetMemberSize(TYPE_MWORD, pFunction,
	    DIRECTION_OUT, false);
	DTRACE("%s words out: %d\n", __FUNCTION__, nWordsOut);
	if (nWordsOut < 0)
	    nWordsOut = CBEMsgBuffer::GetMemberSize(TYPE_MWORD, pFunction,
		DIRECTION_OUT, true);
	DTRACE("%s words out: %d\n", __FUNCTION__, nWordsOut);

	// for generic struct we also have to subtract the strings from the
	// word count, because the strings have been counted when counting
	// TYPE_MWORD as well.
	int nStringsIn = CBEMsgBuffer::GetMemberSize(TYPE_REFSTRING, pFunction,
	    DIRECTION_IN, false);
	DTRACE("%s strings in: %d\n", __FUNCTION__, nStringsIn);
	int nStringsOut = CBEMsgBuffer::GetMemberSize(TYPE_REFSTRING,
	    pFunction, DIRECTION_OUT, false);
	DTRACE("%s strings out: %d\n", __FUNCTION__, nStringsOut);
	nStrings = max(nStringsIn, nStringsOut);
	DTRACE("%s strings: %d\n", __FUNCTION__, nStrings);
	
	DTRACE("%s words in: %d words out: %d\n", __FUNCTION__, nWordsIn, 
	    nWordsOut);
	if (nWordsIn > 0 && nWordsOut > 0)
	    nWords = max(nWordsIn, nWordsOut);
	else
	    nWords = (nWordsIn < nWordsOut) ? nWordsIn : nWordsOut;
	DTRACE("%s words: %d\n", __FUNCTION__, nWords);
    }
    else
    {
	nWords = CBEMsgBuffer::GetMemberSize(TYPE_MWORD, pFunction,
	    nDirection, false);
	DTRACE("%s words: %d\n", __FUNCTION__, nWords);
	nStrings = CBEMsgBuffer::GetMemberSize(TYPE_REFSTRING, pFunction,
	    nDirection, false);
	DTRACE("%s strings: %d\n", __FUNCTION__, nStrings);
    }
    // check minimum number of words
    int nMinWords = GetWordMemberCount(pFunction);
    if (nWords >= 0)
	nWords = max(nWords, nMinWords);
    DTRACE("%s words (min): %d\n", __FUNCTION__, nWords);

    // get name of member
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sName = pNF->GetMessageBufferMember(nType);
    // get member
    CBETypedDeclarator *pMember = FindMember(sName, nDirection);
    // check if we have member of that type
    if (!pMember && (nDirection == 0))
    {
	nDirection = DIRECTION_IN;
	// maybe we do not have a generic struct
	pMember = FindMember(sName, nDirection);
    }
    if (!pMember)
	return;

    *pFile << "\t";

    WriteAccess(pFile, pFunction, nDirection, pMember);
    *pFile << " = L4_IPC_DOPE(";
    if (nWords > 0)
	*pFile << nWords;
    else // variable sized elements
    {
	// sizeof(<msgbufvar>.<structname>)/sizeof(long)-3
	*pFile << "sizeof(";
	WriteAccessToStruct(pFile, pFunction, nDirection);
	*pFile << ")/sizeof(long)-";
	*pFile << 3 + nStrings*nRefSize;
    }
    *pFile << ", " << nStrings << ");\n";
}

/** \brief writes the initialization of the refstring members
 *  \param pFile the file to write to
 *  \param nDirection the direction of the struct to initialize
 *
 * initialize refstring members: check if they are OUT parameters for this
 * function (have to find them as parameter and check the OUT attribute). If
 * so, set the rcv_size and rcv_str members. This depends on attributes of the
 * parameter (prealloc -> pointer is parameter, size is size attribute). If no
 * attribute given, allocate using CBEContext::WriteMalloc().
 */
void
CL4BEMsgBuffer::WriteRefstringInitialization(CBEFile *pFile,
	int nDirection)
{
    CBEFunction *pFunction = GetSpecificParent<CBEFunction>();
    if (!pFunction)
	return;
    CBEClass *pClass = GetSpecificParent<CBEClass>();
    CBEStructType *pStruct = GetStruct(nDirection);
    assert(pStruct);

    int nIndex = -1;
    vector<CBETypedDeclarator*>::iterator iter = pStruct->GetFirstMember();
    CBETypedDeclarator *pMember;
    while ((pMember = pStruct->GetNextMember(iter)) != 0)
    {
	if (!pMember->GetType()->IsOfType(TYPE_REFSTRING))
	    continue;

	// check if declarator is array (as can be for generic part of message
	// buffer...
	CBEDeclarator *pMemberDecl = pMember->GetDeclarator();
	vector<CBEExpression*>::iterator arrIter =
	    pMemberDecl->GetFirstArrayBound();
	CBEExpression *pArrayBound = pMemberDecl->GetNextArrayBound(arrIter);
	do // using 'do { } while' so we pass here even without array bounds
	{
	    // for each element in the array
	    int nArrayCount = (pArrayBound) ? pArrayBound->GetIntValue() : 1;
	    while (nArrayCount--)
	    {

		// increment here, so we catch 'continue' in the middle of the
		// loop.  Because we increment at begin of loop, we have to
		// init index with -1, so first iteration makes it zero.
		nIndex++;

		if (WriteRefstringInitFunction(pFile, pFunction, pClass, 
			nIndex, nDirection))
		    continue;

		// member is refstring, initialize:
		WriteRefstringInitParameter(pFile, pFunction, pMember, nIndex,
		    nDirection);

	    } // array iteration
	// array bound iteration
	} while ((pArrayBound = pMemberDecl->GetNextArrayBound(arrIter)) != 0);
    }
}

/** \brief writes the initialisation of a refstring member with the init func
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 *  \param pClass the class to write for
 *  \param nIndex the index of the refstring in an array
 *  \param nDirection the direction of the struct
 *  \return true if we wrote something, false if not
 */
bool
CL4BEMsgBuffer::WriteRefstringInitFunction(CBEFile *pFile,
    CBEFunction *pFunction,
    CBEClass *pClass,
    int nIndex,
    int nDirection)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    // check if this is server side, and if so, check for 
    // init-rcvstring attribute of class
    if (pFunction->IsComponentSide() &&
	(CCompiler::IsOptionSet(PROGRAM_INIT_RCVSTRING) ||
	 (pClass &&
	  (pClass->FindAttribute(ATTR_INIT_RCVSTRING) ||
	   pClass->FindAttribute(ATTR_INIT_RCVSTRING_CLIENT) ||
	   pClass->FindAttribute(ATTR_INIT_RCVSTRING_SERVER)))))
    {
	string sFunction;
	if (!CCompiler::IsOptionSet(PROGRAM_INIT_RCVSTRING))
	{
	    CBEAttribute *pAttr = 0;
	    if ((pAttr = pClass->FindAttribute(ATTR_INIT_RCVSTRING)) 
		!= 0)
		sFunction = pAttr->GetString();
	    if (((pAttr = pClass->FindAttribute(
			    ATTR_INIT_RCVSTRING_CLIENT)) != 0) && 
		pFile->IsOfFileType(FILETYPE_CLIENT))
		sFunction = pAttr->GetString();
	    if (((pAttr = pClass->FindAttribute(
			    ATTR_INIT_RCVSTRING_SERVER)) != 0) &&
		pFile->IsOfFileType(FILETYPE_COMPONENT))
		sFunction = pAttr->GetString();
	}
	if (sFunction.empty())
	    sFunction = pNF->GetString(STR_INIT_RCVSTRING_FUNC);
	else
	    sFunction = pNF->GetString(STR_INIT_RCVSTRING_FUNC, 
		(void*)&sFunction);

	CBETypedDeclarator *pEnvVar = pFunction->GetEnvironment();
	CBEDeclarator *pEnv = pEnvVar->GetDeclarator();
	// call the init function for the indirect string
	*pFile << "\t" << sFunction << " ( " << nIndex << ", &(";
	WriteMemberAccess(pFile, pFunction, nDirection, TYPE_REFSTRING,
	    nIndex);
	*pFile << ".rcv_str), &(";
	WriteMemberAccess(pFile, pFunction, nDirection, TYPE_REFSTRING,
	    nIndex);
	*pFile << ".rcv_size), " << pEnv->GetName() << ");\n";

	// OK, next!
	return true;
    }
    return false;
}

/** \brief writes the refstring member init for given parameter
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 *  \param pMember the member to write
 *  \param nIndex the index of the refstring to initialize
 *  \param nDirection the direction of the struct
 */
void
CL4BEMsgBuffer::WriteRefstringInitParameter(CBEFile *pFile,
    CBEFunction *pFunction,
    CBETypedDeclarator *pMember,
    int nIndex,
    int nDirection)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sWord = pNF->GetTypeName(TYPE_MWORD, true);
    // get parameter
    CBETypedDeclarator *pParameter = pFunction->FindParameter(
	pMember->GetDeclarator()->GetName());

    if (pParameter && pParameter->FindAttribute(ATTR_PREALLOC))
    {
	*pFile << "\t";
	WriteAccess(pFile, pFunction, nDirection, pMember);
	*pFile << ".rcv_str = (" << sWord << ")(";
	CBEDeclarator *pDecl = pParameter->GetDeclarator();
	int nStars = pDecl->GetStars();
	CBEType *pType = pParameter->GetType();
	if (!pType->IsPointerType())
	    nStars--;
	for (; nStars > 0; nStars--)
	    *pFile << "*";
	*pFile << pDecl->GetName() << ");\n";

	*pFile << "\t";
	WriteAccess(pFile, pFunction, nDirection, pMember);
	*pFile << ".rcv_size = ";
	if ((pParameter->FindAttribute(ATTR_SIZE_IS)) ||
	    (pParameter->FindAttribute(ATTR_LENGTH_IS)))
	{
	    if (pType->GetSize() > 1)
		*pFile << "(";
	    pParameter->WriteGetSize(pFile, NULL, pFunction);
	    if (pType->GetSize() > 1)
	    {
		*pFile << ")*sizeof";
		pType->WriteCast(pFile, false);
	    }
	}
	else
	    // write max-is
	    pParameter->WriteGetSize(pFile, NULL, pFunction);
	*pFile << ";\n";

    }
    else
    {
	bool bUseArray = pMember->GetDeclarator()->IsArray() &&
	    pMember->GetDeclarator()->GetArrayDimensionCount() > 0;
	*pFile << "\t";
	WriteAccess(pFile, pFunction, nDirection, pMember);
	if (bUseArray)
	    *pFile << "[" << nIndex << "]";
	*pFile << ".rcv_str = (" << sWord << ")";
	CBEContext::WriteMalloc(pFile, pFunction);
	*pFile << "(";
	WriteMaxRefstringSize(pFile, pFunction, pMember, pParameter, nIndex, 
	    nDirection);
	*pFile << ");\n";

	*pFile << "\t";
	WriteAccess(pFile, pFunction, nDirection, pMember);
	if (bUseArray)
	    *pFile << "[" << nIndex << "]";
	*pFile << ".rcv_size = ";
	WriteMaxRefstringSize(pFile, pFunction, pMember, pParameter, nIndex,
	    nDirection);
	*pFile << ";\n";
    }
}

/** \brief writes the maximu size of a specific refstring
 *  \param pFile the file to write to
 *  \param pFunction the function to write for
 *  \param pMember the member to write
 *  \param nIndex the index of the refstring to initialize
 *  \param nDirection the direction of the struct
 *
 * The maxmimum size can either be a max-is attribute or the maximum size of
 * the type of the member.  The max-is attribute can only be determined if we
 * have a parameter or the member has one, but it might be difficult to find
 * if there is no parameter, which means we have to initialize for a generic
 * refstring in the server's message buffer.
 *
 * For the latter case we have to iterate the max values of all refstrings for
 * this specific position and determine their maximum.
 */
void
CL4BEMsgBuffer::WriteMaxRefstringSize(CBEFile *pFile,
    CBEFunction *pFunction,
    CBETypedDeclarator *pMember,
    CBETypedDeclarator *pParameter,
    int nIndex,
    int nDirection)
{
    // if parameter has a max_is attribute use that
    CBEAttribute *pMaxAttr = 0;
    if (pParameter)
	pMaxAttr = pParameter->FindAttribute(ATTR_MAX_IS);
    if (!pMaxAttr)
	pMaxAttr = pMember->FindAttribute(ATTR_MAX_IS);
    if (pMaxAttr)
    {
	pMaxAttr->Write(pFile);
	return;
    }

    // ok, seems to be a generic struct, get the maximum of the max-is
    // attributes of the refstrings of the other message buffer structs
    if (!pParameter)
    {
	string sMaxStr = string();

	bool bInterfaceFunc = (dynamic_cast<CBEInterfaceFunction*>(pFunction));
	// get type
	CBEMsgBufferType *pMsgType;
	if (bInterfaceFunc)
	{
	    CBEClass *pClass = GetSpecificParent<CBEClass>();
	    assert(pClass);
	    pMsgType = dynamic_cast<CBEMsgBufferType*>
		(pClass->GetMessageBuffer()->GetType());
	}
	else
	    pMsgType = dynamic_cast<CBEMsgBufferType*>(GetType());
	assert(pMsgType);
	// iterate structs
	CBEStructType *pStruct = 0;
	CBENameFactory *pNF = CCompiler::GetNameFactory();
	string sNameIn = pNF->GetMessageBufferStructName(DIRECTION_IN,
	    string(), string());
	string sNameOut = pNF->GetMessageBufferStructName(DIRECTION_OUT,
	    string(), string());
	int nLenIn = sNameIn.length();
	int nLenOut = sNameOut.length();
	vector<CBEUnionCase*>::iterator iter = pMsgType->GetFirstUnionCase();
	CBEUnionCase *pUnionCase;
	while ((pUnionCase = pMsgType->GetNextUnionCase(iter)) != 0)
	{
	    // check direction
	    string sUnion = pUnionCase->GetDeclarator()->GetName();
	    if (sUnion.substr(sUnion.length() - nLenIn) == sNameIn ||
		sUnion.substr(sUnion.length() - nLenOut) == sNameOut)
	    {		
		pStruct = static_cast<CBEStructType*>(pUnionCase->GetType());

		int nCurr = nIndex;
		
		vector<CBETypedDeclarator*>::iterator iM = 
		    pMsgType->GetStartOfPayload(pStruct);
		CBETypedDeclarator *pM;
		while ((pM = pStruct->GetNextMember(iM)) != 0)
		{
		    // get refstring member at index
		    if (!pM->GetType()->IsOfType(TYPE_REFSTRING))
			continue;
		    if (nCurr-- > 0)
			continue;
		    
		    // check max-is attribute of member
		    CBEAttribute *pA = pM->FindAttribute(ATTR_MAX_IS);
		    if (!pA)
			break;

		    if (sMaxStr.empty())
			pA->WriteToStr(sMaxStr);
		    else
		    {
			string s = "_dice_max(" + sMaxStr + ", ";
			pA->WriteToStr(s);
			sMaxStr = s + ")";
		    }
		}
	    }
	}

	// if max-is string is not empty print and return
	if (!sMaxStr.empty())
	{
	    *pFile << sMaxStr;
	    return;
	}
    }

    // the last resort: use the type's max size
    CBESizes *pSizes = CCompiler::GetSizes();
    int nMaxStrSize = pSizes->GetMaxSizeOfType(TYPE_CHAR);
    *pFile << nMaxStrSize;
}

/** \brief initializes the dopes to a value pair for short IPC
 *  \param pFile the file to write to
 *  \param nType the member to initialize
 *  \param nDirection the direction of the struct to set the member in
 *
 * \todo This is not X.2 conform (have map,grant items as well).
 */
void
CL4BEMsgBuffer::WriteDopeShortInitialization(CBEFile *pFile,
    int nType,
    int nDirection)
{
    if ((nType != TYPE_MSGDOPE_SIZE) &&
	(nType != TYPE_MSGDOPE_SEND))
    {
	return;
    }
   
    // get name of member
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sName = pNF->GetMessageBufferMember(nType);
    // get member
    CBETypedDeclarator *pMember = FindMember(sName, nDirection);
    // check if we have member of that type
    if (!pMember)
	return;
    CBEFunction *pFunction = GetSpecificParent<CBEFunction>();

    *pFile << "\t";
    WriteAccess(pFile, pFunction, nDirection, pMember);
    // get short IPC values
    CL4BESizes *pSizes = static_cast<CL4BESizes*>(CCompiler::GetSizes());
    int nMinSize = pSizes->GetMaxShortIPCSize(DIRECTION_IN);
    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);
    // set short IPC dope
    *pFile << " = L4_IPC_DOPE(" << nMinSize / nWordSize << ", 0);\n";
}

/** \brief writes the initialization of the receive flexpage
 *  \param pFile the file to write to
 *  \param nDirection the direction of the struct to initialize
 */
void
CL4BEMsgBuffer::WriteRcvFlexpageInitialization(CBEFile *pFile,
	int nDirection)
{
    // get receive flexpage member
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    string sFlexName = pNF->GetMessageBufferMember(TYPE_RCV_FLEXPAGE);

    CBETypedDeclarator *pFlexpage = FindMember(sFlexName, nDirection);
    assert (pFlexpage);

    // get environment
    CBEFunction *pFunction = GetSpecificParent<CBEFunction>();
    CBETypedDeclarator *pEnv = pFunction->GetEnvironment();
    assert (pEnv);
    string sEnv = pEnv->GetDeclarator()->GetName();
    if (pEnv->GetDeclarator()->GetStars() > 0)
      	sEnv += "->";
    else
	sEnv += ".";
    
    // assign environment's flexpage to message buffer's flexpage
    *pFile << "\t";
    WriteAccess(pFile, pFunction, nDirection, pFlexpage);
    *pFile << " = " << sEnv << "rcv_fpage;\n";
}

/** \brief tests if the struct for a direction has enough word sized members \
 *         for short IPC
 *  \param pFunction the function of the message buffer
 *  \param nDirection the direction to test
 *  \return true if enough word sized members exist
 */
bool
CL4BEMsgBuffer::HasWordMembers(CBEFunction *pFunction, 
    int nDirection)
{
    CCompiler::Verbose(0, "%s called for func %s and direction %d\n", 
	__FUNCTION__, pFunction->GetName().c_str(), nDirection);
    DTRACE("%s(%s, %d) called\n", __FUNCTION__, pFunction->GetName().c_str(),
	nDirection);

    int nWordSize = CCompiler::GetSizes()->GetSizeOfType(TYPE_MWORD);
    int nShortMembers = GetWordMemberCount(pFunction);
    DTRACE("%s: word size %d, short size %d\n", __FUNCTION__, nWordSize,
	nShortMembers);
    // get structure
    CBEStructType *pStruct = GetStruct(pFunction, nDirection);
    assert(pStruct);
    // test word sized members
    CBEMsgBufferType *pType = pStruct->GetSpecificParent<CBEMsgBufferType>();
    assert(pType);
    vector<CBETypedDeclarator*>::iterator iter = 
	pType->GetStartOfPayload(pStruct);
    CBETypedDeclarator *pMember;
    while (((pMember = pStruct->GetNextMember(iter)) != 0) && 
	(nShortMembers-- > 0))
    {
	DTRACE("%s: testing member %s %d for word size (short members %d)\n",
	    __FUNCTION__, pMember->GetDeclarator()->GetName().c_str(),
	    pMember->GetSize(), nShortMembers);
	if (pMember->GetSize() != nWordSize)
	{
	    DTRACE("%s: not word size, return false\n", __FUNCTION__);
	    return false;
	}
    }
    DTRACE("%s: short members left: %d\n", __FUNCTION__, nShortMembers);
    if (nShortMembers > 0)
	return false;

    CCompiler::Verbose(0, "%s returns true\n", __FUNCTION__);
    return true;
}

/** \brief calculate the number of word sized members required for short IPC
 *  \param pFunction the function to get the number of word sized members for
 *  \return the number of word sized members
 *
 * This implementation has to return the number of required word sized members
 * in the generic message buffer. Since this is for functions, not class, the
 * maximum we need is the number of words for the IPC bindings.
 */
int
CL4BEMsgBuffer::GetWordMemberCount(CBEFunction *pFunction)
{
    CL4BESizes *pSizes = static_cast<CL4BESizes*>(CCompiler::GetSizes());
    int nShort = pSizes->GetMaxShortIPCSize(DIRECTION_IN);
    DTRACE("%s: sizes says, max short size is %d -> make words from that\n",
	__FUNCTION__, nShort);
    return pSizes->WordsFromBytes(nShort);
}

/** \brief calculate the number of word sized members required for short IPC
 *  \param pClass the class to get the number of word sized members for
 *  \return the number of word sized members
 */
int
CL4BEMsgBuffer::GetWordMemberCount(CBEClass *pClass)
{
    DTRACE("%s(class %s) called\n", __FUNCTION__, pClass->GetName().c_str());
    // get value from base class
    int nSize = CBEMsgBuffer::GetWordMemberCount(pClass);
    CL4BESizes *pSizes = static_cast<CL4BESizes*>(CCompiler::GetSizes());
    int nShort = pSizes->GetMaxShortIPCSize(DIRECTION_IN);
    nShort = pSizes->WordsFromBytes(nShort);
    // check minimum
    DTRACE("%s(class %s) return max(%d, %d)\n", __FUNCTION__,
	pClass->GetName().c_str(), nSize, nShort);
    return max(nSize, nShort);
}

/** \brief determine the size of a specific member
 *  \param nType the type requested to determine the size
 *  \param pMember the member to get the size for
 *  \param bMax true if the maximum size should be determined
 *  \return the size of the member
 *
 * This version of the member size determination filters out refstrings,
 * because sometimes (especially with transmit_as attribute _and_ ref
 * attribute) even though a parameter is transmitted as refstring and has
 * therefore the member type refstring, its size determination uses the
 * transmit-as type.
 *
 * (Maybe we should remove automatically added transmit-as types when creating
 * a refstring member).
 */
int
CL4BEMsgBuffer::GetMemberSize(int nType,
    CBETypedDeclarator *pMember,
    bool bMax)
{
    DTRACE("%s (%d, %s, %s called)\n", __FUNCTION__, nType,
	pMember->GetDeclarator()->GetName().c_str(),
	bMax ? "true" : "false");
    // filter out refstring members
    CBESizes *pSizes = CCompiler::GetSizes();
    bool bMemberIsRef = pMember->FindAttribute(ATTR_REF) ||
	pMember->GetType()->IsOfType(TYPE_REFSTRING);
    if (bMemberIsRef &&	nType == TYPE_REFSTRING)
	return pSizes->GetSizeOfType(TYPE_REFSTRING);
    if (bMemberIsRef && nType != TYPE_REFSTRING)
	return 0;

    return CBEMsgBuffer::GetMemberSize(nType, pMember, bMax);
}

/** \brief try to get the position of a member counting word sizes
 *  \param sName the name of the member
 *  \param nDirection the direction of the struct
 *  \return the position (index) of the member, -1 if not found
 *
 * The position obtained here will be used as index into the word size member
 * array (_word). Because flexpage and send/size dope are not part of this
 * array we return -1 to indicate an error.
 */
int
CL4BEMsgBuffer::GetMemberPosition(string sName,
    int nDirection)
{
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    if (sName == pNF->GetMessageBufferMember(TYPE_RCV_FLEXPAGE))
	return -1;
    // test send dope
    if (sName == pNF->GetMessageBufferMember(TYPE_MSGDOPE_SEND))
	return -1;
    // test size dope
    if (sName == pNF->GetMessageBufferMember(TYPE_MSGDOPE_SIZE))
	return -1;

    // not found 
    return CBEMsgBuffer::GetMemberPosition(sName, nDirection);
}

/** \brief pads elements of the message buffer
 *  \return false if an error occured
 *
 * This implementation has to pad indirect strings to NOT reside in the first
 * two dwords and aligns all refstrings to the same position.
 */
bool
CL4BEMsgBuffer::Pad()
{
    CCompiler::Verbose(0, "%s called\n", __FUNCTION__);
    // get minimal size for short IPC
    CL4BESizes *pSizes = static_cast<CL4BESizes*>(CCompiler::GetSizes());
    int nMinSize = pSizes->GetMaxShortIPCSize(DIRECTION_IN);
    // get maximum position of any refstring in message buffer
    int nMax = GetMaxPosOfRefstringInMsgBuffer();
    DTRACE("%s: min size %d, max pos of refstring %d\n", __FUNCTION__,
	nMinSize, nMax);
    if (nMinSize > nMax)
	nMax = nMinSize;
    // align refstrings in all message buffers to the max of the two
    return PadRefstringToPosition (nMax);
}

/** \brief pad refstrings to specified position
 *  \param nPosition the position to align refstring to
 *  \return false if something went wrong
 *
 * Refstrings should start after the short IPC words.
 */
bool
CL4BEMsgBuffer::PadRefstringToPosition(int nPosition)
{
    CCompiler::Verbose(0, "%s(%d) called\n", __FUNCTION__, nPosition);

    CBEMsgBufferType *pType = dynamic_cast<CBEMsgBufferType*>(GetType());
    assert(pType);
    vector<CBEUnionCase*>::iterator iter = pType->GetFirstStruct();
    CBEStructType* pStruct;
    while ((pStruct = pType->GetNextStruct(iter)) != 0)
    {
	if (!PadRefstringToPosition(pStruct, nPosition))
	    return false;
    }
    return true;
}

/** \brief pad refstrings to specified position
 *  \param pStruct the struct to pad the refstrings in
 *  \param nPosition the position to align refstring to
 *  \return false if something went wrong
 *
 * Refstrings should start after the specified position.
 */
bool
CL4BEMsgBuffer::PadRefstringToPosition(CBEStructType *pStruct, 
    int nPosition)
{
    // get start of payload
    CBEMsgBufferType *pType = dynamic_cast<CBEMsgBufferType*>(GetType());
    assert(pType);
    vector<CBETypedDeclarator*>::iterator iter = 
	pType->GetStartOfPayload(pStruct);
    // iterate
    CBETypedDeclarator *pMember;
    int nSize = 0;
    while ((pMember = pStruct->GetNextMember(iter)) != 0)
    {
	if (pMember->GetType()->IsOfType(TYPE_REFSTRING))
	    break;
	int nMemberSize = pMember->GetSize();
	if (nMemberSize < 0)
	    pMember->GetMaxSize(true, nMemberSize);
	nSize += nMemberSize;
	if (nSize > nPosition)
	    return true;
    }
    
    // if no member now, than all members are smaller than position
    if (!pMember)
	return true;

    // otherwise, the member is a refstring and the previous members are less
    // than the minimal position. Pad them with words until minimal size
    // is reached.
    if (nSize < nPosition)
    {
	int nPadding = nPosition - nSize;
    	int nWordSize = CCompiler::GetSizes()->GetSizeOfType(TYPE_MWORD);
	if (nPadding % nWordSize)
	    InsertPadMember(TYPE_BYTE, nPadding % nWordSize,
		pMember, pStruct);
	if (nPadding / nWordSize)
	    InsertPadMember(TYPE_MWORD, nPadding / nWordSize,
		pMember, pStruct);
    }
    
    return true;
}

/** \brief insert a padding member
 *  \param nFEType the type of the member to insert (usually byte or word)
 *  \param nSize the size to insert
 *  \param pMember the member to insert the padding before
 *  \param pStruct the struct where to insert the padding

 *  \return true if successful
 *
 * The implementation first checks if there exists a member with the same
 * name (of the padding variable), which indicates that there has been some
 * padding before (at client side, for instance). If there is such member,
 * then this member's size is increased by the given size (nSize). If no such
 * member existed before, then a new member is created and inserted before
 * pMember.
 */
bool
CL4BEMsgBuffer::InsertPadMember(int nFEType,
    int nSize,
    CBETypedDeclarator *pMember,
    CBEStructType *pStruct)
{
    try
    {
	CBETypedDeclarator *pPadMember = 0; 
	CBEClassFactory *pCF = CCompiler::GetClassFactory();
	CBENameFactory *pNF = CCompiler::GetNameFactory();
	
	// get name
	string sName = pNF->GetPaddingMember(nFEType, 
	    TYPE_REFSTRING);
	// try to find padding member
	if ((pPadMember = pStruct->FindMember(sName)) == 0)
	{
	    // create type
	    CBEType *pType = pCF->GetNewType(nFEType);
	    pType->CreateBackEnd(true, 1, nFEType);
	    // create padding member
	    pPadMember = pCF->GetNewTypedDeclarator();
	    pPadMember->CreateBackEnd(pType, sName);
	    // add array with size difference
	    if (nSize > 1)
	    {
		CBEExpression *pBound = pCF->GetNewExpression();
		pBound->CreateBackEnd(nSize);
		pPadMember->GetDeclarator()->AddArrayBound(pBound);
	    }
	    // add member to struct
	    pStruct->AddMember(pPadMember);
	    // move member just before refstring member
	    pStruct->MoveMember(sName, pMember->GetDeclarator()->GetName());
	}
	else
	{
	    CBEDeclarator *pDecl = pPadMember->GetDeclarator();
	    CBEExpression *pBound = 0;
	    if (pDecl->IsArray())
	    {
		vector<CBEExpression*>::iterator iter = 
		    pDecl->GetFirstArrayBound();
		pBound = pDecl->GetNextArrayBound(iter);
		nSize += pBound->GetIntValue();
	    }
	    else
	    {
		pBound = pCF->GetNewExpression();
		pDecl->AddArrayBound(pBound);
		nSize++;
	    }
	    pBound->CreateBackEnd(nSize);
	}
    }
    catch (CBECreateException *e)
    {
	e->Print();
	delete e;
	return false;
    }
    return true;
}

/** \brief determine maximum position of a refstring in the message buffer
 *  \return the maximum position or 0 if no refstring member
 *
 *  Iterates all message buffer structs and determines the position of the
 *  first refstring in the payload. This requires that the structs have been
 *  sorted before!
 */
int
CL4BEMsgBuffer::GetMaxPosOfRefstringInMsgBuffer()
{
    int nMax = 0;
    // determine maximum distance to refstring parameters
    CBEMsgBufferType *pType = dynamic_cast<CBEMsgBufferType*>(GetType());
    assert(pType);
    vector<CBEUnionCase*>::iterator iterS = pType->GetFirstStruct();
    CBEStructType* pStruct;
    while ((pStruct = pType->GetNextStruct(iterS)) != 0)
    {
	DTRACE("%s: Checking next struct\n", __FUNCTION__);
	// iterate over struct members
	vector<CBETypedDeclarator*>::iterator iter = 
	    pType->GetStartOfPayload(pStruct);
	CBETypedDeclarator *pMember;
	int nSize = 0;
	while ((pMember = pStruct->GetNextMember(iter)) != 0)
	{
	    if (pMember->GetType()->IsOfType(TYPE_REFSTRING))
		break;
	    DTRACE("%s: size of member %s is %d\n", __FUNCTION__,
		pMember->GetDeclarator()->GetName().c_str(),
		pMember->GetSize());
	    int nMemberSize = pMember->GetSize();
	    if (nMemberSize < 0)
		pMember->GetMaxSize(true, nMemberSize);
	    DTRACE("%s: adding size (max?) %d\n", __FUNCTION__, nMemberSize);
	    nSize += nMemberSize;
	}
	DTRACE("%s: Size of this struct is %d\n", __FUNCTION__, nSize);
	// check if new max
	if (nSize > nMax)
	    nMax = nSize;
    }

    DTRACE("%s: returns %d\n", __FUNCTION__, nMax);
    return nMax;
}

/** \brief test if the message buffer has a certain property
 *  \param nProperty the property to test for
 *  \param nDirection the direction to test
 *  \return true if property is available for direction
 *
 * This implementation checks the short IPC property.
 */
bool
CL4BEMsgBuffer::HasProperty(int nProperty,
	int nDirection)
{
    if (nProperty == MSGBUF_PROP_SHORT_IPC)
    {
	CBEFunction *pFunction = GetSpecificParent<CBEFunction>();

    	// if direction is 0 we have to get maximum of IN and OUT
	int nWords = 0;
	int nStrings = 0;
	CL4BESizes *pSizes = static_cast<CL4BESizes*>(CCompiler::GetSizes());
	int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);
	if (nDirection == 0)
	{
	    int nWordsIn = CBEMsgBuffer::GetMemberSize(TYPE_MWORD, pFunction,
		DIRECTION_IN, false);
	    int nWordsOut = CBEMsgBuffer::GetMemberSize(TYPE_MWORD, pFunction,
		DIRECTION_OUT, false);
	    
	    int nStringsIn = CBEMsgBuffer::GetMemberSize(TYPE_REFSTRING,
		pFunction, DIRECTION_IN, false);
	    int nStringsOut = CBEMsgBuffer::GetMemberSize(TYPE_REFSTRING,
		pFunction, DIRECTION_OUT, false);
	    nStrings = max(nStringsIn, nStringsOut);
	    if (nWordsIn > 0 && nWordsOut > 0)
		nWords = max(nWordsIn, nWordsOut);
	    else
		nWords = (nWordsIn < nWordsOut) ? nWordsIn : nWordsOut;
	}
	else
	{
	    nWords = CBEMsgBuffer::GetMemberSize(TYPE_MWORD, pFunction,
		nDirection, false);
	    nStrings = CBEMsgBuffer::GetMemberSize(TYPE_REFSTRING, pFunction,
		nDirection, false);
	}
	// check minimum number of words
	int nMinWords = GetWordMemberCount(pFunction);
	if (nWords >= 0)
	    nWords = max(nWords, nMinWords);
    
       	// get short IPC values
	int nShortSize = pSizes->GetMaxShortIPCSize(DIRECTION_IN) / nWordSize;
	return (nStrings == 0) && (nWords <= nShortSize) && (nWords > 0);
    }

    // send everything else to base class
    return CBEMsgBuffer::HasProperty(nProperty, nDirection);
}
