/**
 *    \file    dice/src/be/BEUnionType.cpp
 * \brief   contains the implementation of the class CBEUnionType
 *
 *    \date    01/15/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "BEUnionType.h"
#include "BEContext.h"
#include "BEUnionCase.h"
#include "BEFile.h"
#include "BEDeclarator.h"
#include "BETypedef.h"
#include "BEDeclarator.h"
#include "BEExpression.h"
#include "BERoot.h"
#include "BESizes.h"
#include "BEStructType.h"
#include "BEUserDefinedType.h"
#include "Compiler.h"
#include "fe/FEFile.h"
#include "fe/FEUnionType.h"
#include "Attribute-Type.h"
#include "Compiler.h"
#include <cassert>
using namespace std;

CBEUnionType::CBEUnionType()
: m_sTag()
{
    m_vUnionCases.clear();
}

CBEUnionType::CBEUnionType(CBEUnionType & src)
: CBEType(src)
{
    m_sTag = src.m_sTag;
    COPY_VECTOR(CBEUnionCase, m_vUnionCases, iter);
}

/** \brief destructor of this instance */
CBEUnionType::~CBEUnionType()
{
    DEL_VECTOR(m_vUnionCases);
}

/** \brief creates this class' part of the back-end
 *  \param pFEType the respective type to crete from
 *
 * This implementation calls the base class to perform basic initialization
 * first.
 */
void
CBEUnionType::CreateBackEnd(CFETypeSpec * pFEType)
    throw (CBECreateException*)
{
    CCompiler::VerboseI(0, "CBEUnionType::%s(fe) called\n", __FUNCTION__);
    
    // sets m_sName to "union"
    CBEType::CreateBackEnd(pFEType);

    string exc = string(__FUNCTION__);

    CFEUnionType *pFEUnion = dynamic_cast<CFEUnionType*>(pFEType);
    assert (pFEUnion);
    
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    vector<CFEUnionCase*>::iterator iterUC = pFEUnion->GetFirstUnionCase();
    CFEUnionCase *pFEUnionCase;
    while ((pFEUnionCase = pFEUnion->GetNextUnionCase(iterUC)) != 0)
    {
        CBEUnionCase *pUnionCase = pCF->GetNewUnionCase();
        AddUnionCase(pUnionCase);
	try
	{
	    pUnionCase->CreateBackEnd(pFEUnionCase);
	}
	catch (CBECreateException *e)
        {
            RemoveUnionCase(pUnionCase);
            delete pUnionCase;
	    e->Print();
	    delete e;

	    exc += " failed because union case could not be added";
	    throw new CBECreateException(exc);
        }
    }
    // set tag
    string sTag = pFEUnion->GetTag();
    if (!sTag.empty())
    {
	CBENameFactory *pNF = CCompiler::GetNameFactory();
        // see if we can find the original struct
        CFEFile *pFERoot = dynamic_cast<CFEFile*>(pFEType->GetRoot());
        assert(pFERoot);
        CFEConstructedType *pFETaggedDecl = pFERoot->FindTaggedDecl(sTag);
        if (pFETaggedDecl)
            m_sTag = pNF->GetTypeName(pFETaggedDecl, sTag);
        else
            m_sTag = sTag;
    }

    CCompiler::VerboseD(0, "CBEUnionType::%s(fe) returns\n", __FUNCTION__);
}

/** \brief creates a union
 *  \param sTag the tag of the union
 */
void
CBEUnionType::CreateBackEnd(string sTag)
    throw (CBECreateException*)
{
    CCompiler::VerboseI(0, "CBEUnionType::%s(%s) called\n", __FUNCTION__,
	sTag.c_str());
    
    string exc = string(__FUNCTION__);

    CBENameFactory *pNF = CCompiler::GetNameFactory();
    m_sName = pNF->GetTypeName(TYPE_UNION, false);
    if (m_sName.empty())
    {
        // user defined type overloads this function -> m_sName.c_str()
        // should always be set
	exc += " failed, because no type name could be assigned";
	throw new CBECreateException(exc);
    }
    m_nFEType = TYPE_UNION;
    m_sTag = sTag;

    CCompiler::VerboseD(0, "CBEUnionType::%s(%s) returns\n", __FUNCTION__,
	sTag.c_str());
}

/** \brief adds a new union case to the vector
 *  \param pUnionCase the new union case
 */
void CBEUnionType::AddUnionCase(CBEUnionCase * pUnionCase)
{
    if (!pUnionCase)
        return;
    m_vUnionCases.push_back(pUnionCase);
    pUnionCase->SetParent(this);
}

/** \brief removes a union case from the union case vector
 *  \param pCase the union case to remove
 */
void CBEUnionType::RemoveUnionCase(CBEUnionCase * pCase)
{
    if (!pCase)
        return;
    vector<CBEUnionCase*>::iterator iter;
    for (iter = m_vUnionCases.begin(); iter != m_vUnionCases.end(); iter++)
    {
        if (*iter == pCase)
        {
            m_vUnionCases.erase(iter);
            return;
        }
    }
}

/** \brief retrieves a pointer to the first union case element
 *  \return a pointer to the first union case
 */
vector<CBEUnionCase*>::iterator CBEUnionType::GetFirstUnionCase()
{
    return m_vUnionCases.begin();
}

/** \brief retrieves a reference to the next union case
 *  \param iter the pointer to the next union case
 *  \return a reference to the next union case
 */
CBEUnionCase *CBEUnionType::GetNextUnionCase(vector<CBEUnionCase*>::iterator &iter)
{
    if (iter == m_vUnionCases.end())
        return 0;
    return *iter++;
}

/** \brief writes the union to the target file
 *  \param pFile the file to write to
 *
 * A union is rather simple:
 * union &lt;tag&gt;
 * {
 *  &lt;member list&gt;
 * }
 */
void CBEUnionType::Write(CBEFile * pFile)
{
    // write union
    *pFile << m_sName;
    if (!m_sTag.empty())
	*pFile << " " << m_sTag;
    if (!m_vUnionCases.empty())
    {
	*pFile << "\n";
	*pFile << "\t{\n";
        pFile->IncIndent();

        // write members
        vector<CBEUnionCase*>::iterator iterU = GetFirstUnionCase();
        CBEUnionCase *pCase;
        while ((pCase = GetNextUnionCase(iterU)) != 0)
        {
	    *pFile << "\t";
            pCase->WriteDeclaration(pFile);
	    *pFile << ";\n";
        }

        // close union
        pFile->DecIndent();
	*pFile << "\t}";
    }
}

/** \brief write the declaration of this type
 *  \param pFile the file to write to
 *
 * Only write a 'struct &lt;tag&gt;'.
 */
void CBEUnionType::WriteDeclaration(CBEFile * pFile)
{
    *pFile << m_sName;
    if (!m_sTag.empty())
	*pFile << " " << m_sTag;
}

/** \brief calculates the size of this union type
 *  \return the size in bytes
 *
 * The union type's size is the size of the biggest member and the switch type.
 */
int CBEUnionType::GetSize()
{
    CCompiler::VerboseI(0, "CBEUnionType::%s (m_nSize = %d)\n",
	__FUNCTION__, m_nSize);
    
    if (m_nSize != 0)
	return m_nSize;
    
    vector<CBEUnionCase*>::iterator iter = GetFirstUnionCase();
    CBEUnionCase *pUnionCase;
    while ((pUnionCase = GetNextUnionCase(iter)) != 0)
    {
	CCompiler::Verbose(0, 
	    "CBEUnionType::%s determining size of union case %s\n", 
	    __FUNCTION__, pUnionCase->GetDeclarator()->GetName().c_str());
	
        int nUnionSize = pUnionCase->GetSize();
	
	CCompiler::Verbose(0, "CBEUnionType::%s size of union case %s is %d\n",
	    __FUNCTION__, pUnionCase->GetDeclarator()->GetName().c_str(),
	    nUnionSize);

        // if we have one variable sized union member, the
        // whole union is variable sized, because we cannot pinpoint
        // its exact size to determine a message buffer size
        if (nUnionSize < 0)
	{
	    m_nSize = nUnionSize;
	    CCompiler::VerboseD(0, "CBEUnionType::%s var sized, return -1\n",
		__FUNCTION__);
            return nUnionSize;
	}
        if (m_nSize < nUnionSize)
            m_nSize = nUnionSize;
    }

    if ((m_vUnionCases.size() == 0) && (m_nSize == 0))
    {
	// forward declared union -> find definition of union
	if (m_sTag.empty())
	{
	    CCompiler::VerboseD(0, "CBEUnionType::%s returns %d (empty)\n",
		__FUNCTION__, m_nSize);
	    return m_nSize;
	}

	CBERoot *pRoot = GetSpecificParent<CBERoot>();
	assert(pRoot);
	CBEType *pType = pRoot->FindTaggedType(TYPE_UNION, m_sTag);
	if (!pType)
	{
	    CCompiler::VerboseD(0, 
		"CBEUnionType::%s returns %d (no union %s found)\n",
		__FUNCTION__, m_nSize, m_sTag.c_str());
	    return m_nSize;
	}
	m_nSize = pType->GetSize();
    }

    CCompiler::VerboseD(0, "CBEUnionType::%s return %d\n", __FUNCTION__,
	m_nSize);
    return m_nSize;
}

/** \brief calculates the maximum size of this union type
 *  \return the maxmimum size in bytes
 *
 * This implemetation tries to determine the usual size. If that is negative,
 * i.e., the union is of variable size, we have rerun the algorithm and
 * retrieve the maximum sizes for the members.
 */
int CBEUnionType::GetMaxSize()
{
    int m_nMaxSize = GetSize();
    if (m_nMaxSize > 0)
	return m_nMaxSize;
    
    // reset m_nMaxSize, because it could have been negative before
    m_nMaxSize = 0;
    vector<CBEUnionCase*>::iterator iter = GetFirstUnionCase();
    CBEUnionCase *pUnionCase;
    while ((pUnionCase = GetNextUnionCase(iter)) != 0)
    {
        int nUnionSize = 0;
	pUnionCase->GetMaxSize(true, nUnionSize);

	// here we try to determine the maximum size. If there is a variable
	// sized member, it has some fixed size or anything else replica
	// somewhere that will fit its needs. Therefore we simply skip these
	// members here.
        if (nUnionSize < 0)
	{
	    continue;
	}
        if (m_nMaxSize < nUnionSize)
            m_nMaxSize = nUnionSize;
    }
    
    if ((m_vUnionCases.size() == 0) && (m_nMaxSize == 0))
    {
	// forward declared union -> find definition of union
	if (m_sTag.empty())
	    return m_nMaxSize;

	CBERoot *pRoot = GetSpecificParent<CBERoot>();
	assert(pRoot);
	CBEType *pType = pRoot->FindTaggedType(TYPE_UNION, m_sTag);
	if (!pType)
	    return m_nMaxSize;
	m_nMaxSize = pType->GetMaxSize();
    }

    return m_nMaxSize;
}

/** \brief counts the union cases
 *  \return the number of union cases
 */
int CBEUnionType::GetUnionCaseCount()
{
    return m_vUnionCases.size();
}

/** \brief writes the cast for this type
 *  \param pFile the file to write to
 *  \param bPointer true if the cast should produce a pointer
 *
 * The cast of a union type is '(union tag)'.
 */
void CBEUnionType::WriteCast(CBEFile * pFile, bool bPointer)
{
    *pFile << "(";
    if (m_sTag.empty())
    {
        // no tag -> we need a typedef to save us
        // the alias can be used for the cast
        CBETypedef *pTypedef = GetTypedef();
        assert(pTypedef);
        // get first declarator (without stars)
        vector<CBEDeclarator*>::iterator iterD = pTypedef->GetFirstDeclarator();
        CBEDeclarator *pDecl;
        while ((pDecl = pTypedef->GetNextDeclarator(iterD)) != 0)
        {
            if (pDecl->GetStars() <= (bPointer?1:0))
                break;
        }
        assert(pDecl);
	*pFile << pDecl->GetName();
        if (bPointer && (pDecl->GetStars() == 0))
	    *pFile << "*";
    }
    else
    {
	*pFile << m_sName << " " << m_sTag;
        if (bPointer)
	    *pFile << "*";
    }
    *pFile << ")";
}

/** \brief write the zero init code for a union
 *  \param pFile the file to write to
 *
 * init union similar to struct, but use only first member.
 */
void CBEUnionType::WriteZeroInit(CBEFile * pFile)
{
    *pFile << "{ ";
    pFile->IncIndent();
    vector<CBEUnionCase*>::iterator iter = GetFirstUnionCase();
    CBEUnionCase *pMember = GetNextUnionCase(iter);
    if (pMember)
    {
	// get type
	CBEType *pType = pMember->GetType();
	// get declarator
	vector<CBEDeclarator*>::iterator iterD = pMember->GetFirstDeclarator();
	CBEDeclarator *pDecl;
	while ((pDecl = pMember->GetNextDeclarator(iterD)) != 0)
	{
	    // be C99 compliant:
	    *pFile << pDecl->GetName() << " : ";
	    if (pDecl->IsArray())
		WriteZeroInitArray(pFile, pType, pDecl, 
		    pDecl->GetFirstArrayBound());
	    else if (pType)
		pType->WriteZeroInit(pFile);
	}
    }
    pFile->DecIndent();
    *pFile << " }";
}

/** \brief used to determine if this type writes a zero init string
 *  \return false
 */
bool CBEUnionType::DoWriteZeroInit()
{
    return false;
}

/** \brief if struct is variable size, it has to write the size
 *  \param pFile the file to write to
 *  \param pStack the declarator stack for constructed types with variable
 *         sized members 
 *
 * This is not the maximum of all cases, because this would mean that all
 * cases are of equal type (to determine their size), but they are not.
 * Therefore we have to write code which test the switch variable.  We can
 * write it as recursive function.
 *
 * \todo what if default case is variable sized?
 */
void CBEUnionType::WriteGetSize(CBEFile * pFile,
    vector<CDeclaratorStackLocation*> *pStack,
    CBEFunction *pUsingFunc)
{
    int nFixedSize = GetFixedSize();
    // build vector with var sized members
    vector<CBEUnionCase*> vVarSizedUnionCase;
    vector<CBEUnionCase*>::iterator iter = GetFirstUnionCase();
    CBEUnionCase *pCase;
    while ((pCase = GetNextUnionCase(iter)) != 0)
    {
        if (pCase->IsVariableSized())
            vVarSizedUnionCase.push_back(pCase);
    }
    // call recursive write
    iter = vVarSizedUnionCase.begin();
    if (iter != vVarSizedUnionCase.end())
    {
	*pFile << "_dice_max(";
        WriteGetMaxSize(pFile, &vVarSizedUnionCase, iter, pStack, pUsingFunc);
	*pFile << ", " << nFixedSize << ")";
    }
    else
	// should never get here, becaus WriteGetSize is only called if
	// Var-Sized
	*pFile << nFixedSize;
}

/** \brief calculates the maximum size of the fixed sized mebers
 *  \return the maximum fixed size in bytes
 */
int CBEUnionType::GetFixedSize()
{
    int nSize = 0;
    vector<CBEUnionCase*>::iterator iter = GetFirstUnionCase();
    CBEUnionCase *pUnionCase;
    while ((pUnionCase = GetNextUnionCase(iter)) != 0)
    {
        int nUnionSize = pUnionCase->GetSize();
        // if this is negative size, then its variable sized
        // and we skip it here (the max operation will simply ignore it)
        if (nSize < nUnionSize)
            nSize = nUnionSize;
    }

    if ((m_vUnionCases.size() == 0) && (nSize == 0))
    {
       // forward declared union -> find definition of union
       if (m_sTag.empty())
           return nSize;

        CBERoot *pRoot = GetSpecificParent<CBERoot>();
        assert(pRoot);
        CBEType *pType = pRoot->FindTaggedType(TYPE_UNION, m_sTag);
       if (!pType)
           return nSize;
       nSize = pType->GetSize();
    }

    return nSize;
}

/** \brief writes the maximum of the variable sized members
 *  \param pFile the file to write to
 *  \param pMembers the vector containing the variable sized members
 *  \param iter the current position in the vector
 *  \param pStack contains the declarator stack for variables sized parameters
 */
void CBEUnionType::WriteGetMaxSize(CBEFile *pFile,
    const vector<CBEUnionCase*> *pMembers,
    vector<CBEUnionCase*>::iterator iter,
    vector<CDeclaratorStackLocation*> *pStack,
    CBEFunction *pUsingFunc)
{
    assert(pMembers);
    assert(iter != pMembers->end());
    CBEUnionCase *pMember = *iter++;
    bool bNext = (iter != pMembers->end());
    if (bNext)
	*pFile << "_dice_max(";
    WriteGetMemberSize(pFile, pMember, pStack, pUsingFunc);
    // if we have successor, add ':(<max(next)>)'
    // otherwise nothing
    if (bNext)
    {
	*pFile << ", ";
        WriteGetMaxSize(pFile, pMembers, iter, pStack, pUsingFunc);
	*pFile << ")";
    }
}

/** \brief writes the whole size string for a member
 *  \param pFile the file to write to
 *  \param pMember the member to write for
 *  \param pStack contains the declarator stack for variable sized parameters
 *
 * This code is taken from
 * CBEMsgBufferType::WriteInitializationVarSizedParameters if something is not
 * working, check if something changed there as well.
 */
void CBEUnionType::WriteGetMemberSize(CBEFile *pFile,
    CBEUnionCase *pMember,
    vector<CDeclaratorStackLocation*> *pStack,
    CBEFunction *pUsingFunc)
{
    pMember->WriteGetSize(pFile, pStack, pUsingFunc);
    if ((pMember->GetType()->GetSize() > 1) && !(pMember->IsString()))
    {
	*pFile << "*sizeof";
        pMember->GetType()->WriteCast(pFile, false);
    }
    else if (pMember->IsString())
    {
        // add terminating zero
	*pFile << "+1";
        bool bHasSizeAttr = pMember->HasSizeAttr(ATTR_SIZE_IS) ||
                pMember->HasSizeAttr(ATTR_LENGTH_IS) ||
                pMember->HasSizeAttr(ATTR_MAX_IS);
        if (!bHasSizeAttr)
	    *pFile << "+" << 
		CCompiler::GetSizes()->GetSizeOfType(TYPE_INTEGER);
    }
}

/** \brief find a union case with a specific name
 *  \param sName the name of the element to find
 *  \return a reference to the union case with the specified name (switch)
 */
CBETypedDeclarator*
CBEUnionType::FindMember(string sName)
{
    vector<CBEUnionCase*>::iterator iter = GetFirstUnionCase();
    CBEUnionCase *pUnionCase;
    while ((pUnionCase = GetNextUnionCase(iter)) != 0)
    {
	if (pUnionCase->FindDeclarator(sName))
	    return pUnionCase;
    }
    // nothing found, return NULL
    return 0;
}

/** \brief tries to find a member with a declarator stack
 *  \param pStack contains the list of members to search for
 *  \param iCurr the iterator pointing to the currently searched element
 *  \return the member found or 0 if not found
 *
 * Gets the first element on the stack and tries to find
 */
CBETypedDeclarator* 
CBEUnionType::FindMember(vector<CDeclaratorStackLocation*> *pStack,
    vector<CDeclaratorStackLocation*>::iterator iCurr)
{
    // if at end, return
    if (iCurr == pStack->end())
	return 0;
    // try to find member for current declarator
    string sName = (*iCurr)->pDeclarator->GetName();
    CBETypedDeclarator *pMember = FindMember(sName);
    if (!pMember)
	return pMember;

    // no more elements in stack, we are finished
    if (++iCurr == pStack->end())
	return pMember;

    // check member types
    CBEType *pType = pMember->GetType();
    while (dynamic_cast<CBEUserDefinedType*>(pType))
	pType = dynamic_cast<CBEUserDefinedType*>(pType);

    CBEStructType *pStruct = dynamic_cast<CBEStructType*>(pType);
    if (pStruct)
	return pStruct->FindMember(pStack, iCurr);

    CBEUnionType *pUnion = dynamic_cast<CBEUnionType*>(pType);
    if (pUnion)
	return pUnion->FindMember(pStack, iCurr);

    return 0;
}
