/**
 *    \file    dice/src/be/BEUnionCase.cpp
 *  \brief   contains the implementation of the class CBEUnionCase
 *
 *    \date    01/15/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "be/BEUnionCase.h"
#include "be/BEContext.h"
#include "be/BEExpression.h"
#include "be/BETypedDeclarator.h"
#include "be/BEDeclarator.h"
#include "BEClassFactory.h"
#include "fe/FEUnionCase.h"
#include "Compiler.h"
#include <cassert>
using namespace std;

CBEUnionCase::CBEUnionCase()
{
    m_bDefault = false;
    m_vCaseLabels.clear();
}

CBEUnionCase::CBEUnionCase(CBEUnionCase & src)
: CBETypedDeclarator(src)
{
    m_bDefault = src.m_bDefault;
    COPY_VECTOR(CBEExpression, m_vCaseLabels, iter);
}

/** \brief destructor of this instance */
CBEUnionCase::~CBEUnionCase()
{
    DEL_VECTOR(m_vCaseLabels);
}

/** \brief creates the back-end structure for a union case
 *  \param pFEUnionCase the corresponding front-end union case
 *  \return true if code generation was successful
 */
void
CBEUnionCase::CreateBackEnd(CFEUnionCase * pFEUnionCase)
    throw (CBECreateException*)
{
    assert(pFEUnionCase);
    // the union arm is the typed declarator we initialize the base class with:
    CBETypedDeclarator::CreateBackEnd(pFEUnionCase->GetUnionArm());
    // now init union case specific stuff
    m_bDefault = pFEUnionCase->IsDefault();
    if (!m_bDefault)
    {
	CBEClassFactory *pCF = CCompiler::GetClassFactory();
	
        vector<CFEExpression*>::iterator iter = 
	    pFEUnionCase->GetFirstUnionCaseLabel();
        CFEExpression *pFELabel;
        while ((pFELabel = pFEUnionCase->GetNextUnionCaseLabel(iter)) != 0)
        {
            CBEExpression *pLabel = pCF->GetNewExpression();
            AddLabel(pLabel);
	    try
	    {
		pLabel->CreateBackEnd(pFELabel);
	    }
	    catch (CBECreateException *e)
            {
                RemoveLabel(pLabel);
                delete pLabel;
		throw;
            }
        }
    }
}

/** \brief creates the union case
 *  \param pType the type of the union arm
 *  \param sName the name of the union arm
 *  \param pCaseLabel the case label
 *  \param bDefault true if this is the default arm
 *  \return true if successful
 *
 * If neither pCaseLabel nor bDefault is set, then this is the member of a C
 * style union.
 */
void CBEUnionCase::CreateBackEnd(CBEType *pType,
    string sName,
    CBEExpression *pCaseLabel,
    bool bDefault)
    throw (CBECreateException*)
{
    CCompiler::VerboseI(0, "CBEUnionCase::%s called for %s\n",
	__FUNCTION__, sName.c_str());

    CBETypedDeclarator::CreateBackEnd(pType, sName);
    m_bDefault = bDefault;
    if (pCaseLabel)
    {
        AddLabel(pCaseLabel);
    }

    CCompiler::VerboseD(0, "CBEUnionCase::%s returns true\n", __FUNCTION__);
}

/** \brief adds another label to the vector
 *  \param pLabel the new label
 */
void CBEUnionCase::AddLabel(CBEExpression * pLabel)
{
    if (!pLabel)
        return;
    m_vCaseLabels.push_back(pLabel);
    pLabel->SetParent(this);
}

/** \brief removes a label from the labels list
 *  \param pLabel the label to remove
 */
void CBEUnionCase::RemoveLabel(CBEExpression * pLabel)
{
    if (!pLabel)
        return;
    vector<CBEExpression*>::iterator iter;
    for (iter = m_vCaseLabels.begin(); iter != m_vCaseLabels.end(); iter++)
    {
        if (*iter == pLabel)
        {
            m_vCaseLabels.erase(iter);
            return;
        }
    }
}

/** \brief retrieves pointer to first label
 *  \return the pointer to the first label
 */
vector<CBEExpression*>::iterator CBEUnionCase::GetFirstLabel()
{
    return m_vCaseLabels.begin();
}

/** \brief retrieves a reference to the next label
 *  \param iter the pointer to this label
 *  \return a reference to the expression
 */
CBEExpression *CBEUnionCase::GetNextLabel(vector<CBEExpression*>::iterator &iter)
{
    if (iter == m_vCaseLabels.end())
        return 0;
    return *iter++;
}
