/**
 *    \file    dice/src/be/BETarget.h
 *  \brief   contains the declaration of the class CBETarget
 *
 *    \date    01/11/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

/** preprocessing symbol to check header file */
#ifndef __DICE_BETARGET_H__
#define __DICE_BETARGET_H__

#include "be/BEObject.h"
#include <vector>
using namespace std;

class CFEFile;
class CFEOperation;
class CFEInterface;
class CFELibrary;
class CFEFile;
class CFETypedDeclarator;
class CFEConstDeclarator;

class CBEFile;
class CBEHeaderFile;
class CBEImplementationFile;
class CBEContext;
class CBETypedef;
class CBEFunction;

/** \class CBETarget
 *  \ingroup backend
 *  \brief the base class for the target components (client, component,
 *          testsuite, root)
 *
 * This class is introduced as base class to client, component, testsuite and
 * root, because these target classes all have the posession of files in
 * common. The client and component obviously posess several header and
 * implementation files. The root may also contain some header files, such as
 * opcode files or similar. The testsuite posseses the implementation files
 * containing the tesuite application.
 */
class CBETarget : public CBEObject
{
// Constructor
public:
    /** \brief constructor
     */
    CBETarget();
    virtual ~CBETarget();

protected:
    /** \brief copy constructor */
    CBETarget(CBETarget &src);

public:
    virtual void CreateBackEnd(CFEFile *pFEFile)
	throw (CBECreateException*);
    
    virtual CBEFunction* FindFunction(string sFunctionName, int nFunctionType);
    virtual CBETypedef* FindTypedef(string sTypeName);
    
    virtual CBEImplementationFile* GetNextImplementationFile(
	vector<CBEImplementationFile*>::iterator &iter);
    virtual vector<CBEImplementationFile*>::iterator 
	GetFirstImplementationFile();
    virtual void AddFile(CBEImplementationFile *pImplementationFile);
    virtual void RemoveFile(CBEImplementationFile *pImplementation);
    
    virtual CBEHeaderFile* GetNextHeaderFile(
	vector<CBEHeaderFile*>::iterator &iter);
    virtual vector<CBEHeaderFile*>::iterator GetFirstHeaderFile();
    virtual void AddFile(CBEHeaderFile *pHeaderFile);
    virtual void RemoveFile(CBEHeaderFile *pHeader);
    
    /** \brief generates the output files and code */
    virtual void Write() = 0;
    virtual void PrintTargetFiles(FILE *output, int &nCurCol, int nMaxCol);
    virtual bool HasFunctionWithUserType(string sTypeName);

protected:
    virtual void WriteImplementationFiles();
    virtual void WriteHeaderFiles();
    virtual bool DoAddIncludedFiles();
    // constants
    virtual bool AddConstantToFile(CBEFile *pFile, 
	CFEConstDeclarator *pFEConstant);
    virtual bool AddConstantToFile(CBEFile *pFile, CFEInterface *pFEInterface);
    virtual bool AddConstantToFile(CBEFile *pFile, CFELibrary *pFELibrary);
    virtual bool AddConstantToFile(CBEFile *pFile, CFEFile *pFEFile);
    // type definitions
    virtual bool AddTypedefToFile(CBEFile *pFile, 
	CFETypedDeclarator *pFETypedDeclarator);
    virtual bool AddTypedefToFile(CBEFile *pFile, CFEInterface *pFEInterface);
    virtual bool AddTypedefToFile(CBEFile *pFile, CFELibrary *pFELibrary);
    virtual bool AddTypedefToFile(CBEFile *pFile, CFEFile *pFEFile);
    // header file search
    virtual CBEHeaderFile* FindHeaderFile(string sFileName);
    virtual CBEHeaderFile* FindHeaderFile(CFEOperation *pFEOperation,
	int nFileType);
    virtual CBEHeaderFile* FindHeaderFile(CFEInterface *pFEInterface,
	int nFileType);
    virtual CBEHeaderFile* FindHeaderFile(CFELibrary *pFELibrary,
	int nFileType);
    virtual CBEHeaderFile* FindHeaderFile(CFEFile *pFEFile, int nFileType); 

    /** \brief create target for header file 
     *  \param pFEFile the front-end file to use as reference
     */
    virtual void CreateBackEndHeader(CFEFile *pFEFile)
	throw (CBECreateException*) = 0;
    /** \brief create target for implementation file 
     *  \param pFEFile the front-end file to use as reference
     */
    virtual void CreateBackEndImplementation(CFEFile *pFEFile)
	throw (CBECreateException*) = 0;
    
    virtual void PrintTargetFileName(FILE *output, string sFilename, 
	int &nCurCol, int nMaxCol);

protected:
    /** \var vector<CBEHeaderFile*> m_vHeaderFiles
     *  \brief contains the header files of the respective target part
     *
     * Because the handling for header and implementation files is different
     * at some points, we keep them in different vectors.
     */
    vector<CBEHeaderFile*> m_vHeaderFiles;
    /** \var vector<CBEImplementationFile*> m_vImplementationFiles
     *  \brief contains the implementation files for the respective target part
     */
    vector<CBEImplementationFile*> m_vImplementationFiles;
};

#endif // !__DICE_BETARGET_H__
