/**
 *    \file    dice/src/be/BEStructType.cpp
 * \brief   contains the implementation of the class CBEStructType
 *
 *    \date    01/15/2002
 *    \author  Ronald Aigner <ra3@os.inf.tu-dresden.de>
 */
/*
 * Copyright (C) 2001-2004
 * Dresden University of Technology, Operating Systems Research Group
 *
 * This file contains free software, you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, Version 2 as
 * published by the Free Software Foundation (see the file COPYING).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * For different licensing schemes please contact
 * <contact@os.inf.tu-dresden.de>.
 */

#include "BEStructType.h"
#include "BEUserDefinedType.h"
#include "BEContext.h"
#include "BETypedef.h"
#include "BEDeclarator.h"
#include "BERoot.h"
#include "BEAttribute.h"
#include "BEStructType.h"
#include "BEClassFactory.h"
#include "BESizes.h"
#include "BEUnionType.h"
#include "Compiler.h"
#include "fe/FEStructType.h"
#include "fe/FEInterface.h"
#include "fe/FELibrary.h"
#include "fe/FEFile.h"
#include "fe/FEArrayType.h"
#include "fe/FESimpleType.h"
#include "fe/FEIsAttribute.h"
#include "fe/FEDeclarator.h"
#include <stdexcept>
#include <cassert>
using namespace std;

CBEStructType::CBEStructType()
: m_sTag()
{
    m_bForwardDeclaration = false;
    m_vMembers.clear();
}

CBEStructType::CBEStructType(CBEStructType & src)
 : CBEType(src)
{
    m_sTag = src.m_sTag;
    m_bForwardDeclaration = src.m_bForwardDeclaration;
    COPY_VECTOR(CBETypedDeclarator, m_vMembers, iter);
}

/** \brief destructor of this instance */
CBEStructType::~CBEStructType()
{
    DEL_VECTOR(m_vMembers);
}

/** \brief prepares this instance for the code generation
 *  \param pFEType the corresponding front-end attribute
 *  \return true if the code generation was successful
 *
 * This implementation calls the base class' implementatio first to set
 * default values and then adds the members of the struct to this class.
 */
void
CBEStructType::CreateBackEnd(CFETypeSpec * pFEType)
    throw (CBECreateException*)
{
    CCompiler::VerboseI(0, "CBEStructType::%s(fe) called\n", __FUNCTION__);
    
    // sets m_sName to "struct"
    CBEType::CreateBackEnd(pFEType);
    // if sequence create own members
    if (pFEType->GetType() == TYPE_ARRAY)
    {
        CreateBackEndSequence((CFEArrayType*)pFEType);
	CCompiler::VerboseD(0, "CBEStructType::%s(fe) returns\n", __FUNCTION__);
	return;
    }
    // get forward declaration
    m_bForwardDeclaration = 
	((CFEConstructedType*) pFEType)->IsForwardDeclaration();
    CBEClassFactory *pCF = CCompiler::GetClassFactory();
    CBENameFactory *pNF = CCompiler::GetNameFactory();
    // iterate over members
    CFEStructType *pFEStruct = (CFEStructType *) pFEType;
    vector<CFETypedDeclarator*>::iterator iterM = pFEStruct->GetFirstMember();
    CFETypedDeclarator *pFEMember;
    while ((pFEMember = pFEStruct->GetNextMember(iterM)) != 0)
    {
        CBETypedDeclarator *pMember = pCF->GetNewTypedDeclarator();
        AddMember(pMember);
	try
	{
	    pMember->CreateBackEnd(pFEMember);
	}
	catch (CBECreateException *e)
        {
            RemoveMember(pMember);
            delete pMember;
            throw;
        }
    }
    // set tag
    string sTag = pFEStruct->GetTag();
    if (!sTag.empty())
    {
	// we start with the parent interface and walk all the way up to the
	// root
	CFEConstructedType *pFETaggedDecl = 0;
        CFEInterface *pFEInterface = pFEType->GetSpecificParent<CFEInterface>();
        if (pFEInterface)
        {
            pFETaggedDecl = pFEInterface->FindTaggedDecl(sTag);
            if (!pFETaggedDecl)
            {
                CFELibrary *pParentLib = 
		    pFEInterface->GetSpecificParent<CFELibrary>();
                while (pParentLib && !pFETaggedDecl)
                {
                    pFETaggedDecl = pParentLib->FindTaggedDecl(sTag);
                    pParentLib = pParentLib->GetSpecificParent<CFELibrary>();
                }
            }
        }
        if (!pFETaggedDecl)
        {
            CFEFile *pFERoot = dynamic_cast<CFEFile*>(pFEType->GetRoot());
            // we definetly have a root
            assert(pFERoot);
            // we definetly have this decl in there
            pFETaggedDecl = pFERoot->FindTaggedDecl(sTag);
        }
        // now we can assign a global tag name
        if (pFETaggedDecl)
            m_sTag = pNF->GetTypeName(pFETaggedDecl, sTag);
        else
        {
            // if this is a complete type, than this should
            // be made a full name as well, since it is defined in
            // an idl file
            if (!m_bForwardDeclaration)
                m_sTag = pNF->GetTypeName(pFEType, sTag);
            else
                m_sTag = sTag;
            // still no original struct found, than this might be a user
            // defined struct
            // get the size from there
	    CBERoot *pRoot = GetSpecificParent<CBERoot>();
	    assert(pRoot);
	    CBETypedef *pTypedef = pRoot->FindTypedef(m_sTag);
	    if (!pTypedef)
	    {
		CCompiler::VerboseD(0, "CBEStructType::%s(fe) returns\n",
		    __FUNCTION__);
		return;
	    }
	    /* since the typedef is a CBETypedDeclarator, it would evaluate
	     * the size of it's base type and sum it for all it's declarators.
	     * We only want it for the declarator we are using. That's why we
	     * use a specific GetSize function instead of the generic one. */
	    m_nSize = pTypedef->GetSize(m_sTag);
	    // determine max size later
	    m_nMaxSize = 0;
        }
    }

    CCompiler::VerboseD(0, "CBEStructType::%s(fe) returns\n", __FUNCTION__);
}

/** \brief initialize instance of object
 *  \param sTag the tag of the struct
 *  \param pRefObj a reference object
 *  \return true if successful
 *
 * The members are added later using AddMember.
 */
void
CBEStructType::CreateBackEnd(string sTag,
    CFEBase *pRefObj)
    throw (CBECreateException*)
{
    CCompiler::VerboseI(0, "CBEStructType::%s(%s, %p) called\n", __FUNCTION__,
	sTag.c_str(), pRefObj);
    // skip CBEType::CreateBackEnd to avoid asserts
    CBEObject::CreateBackEnd(pRefObj);
    m_nFEType = TYPE_STRUCT;

    CBENameFactory *pNF = CCompiler::GetNameFactory();
    m_sName = pNF->GetTypeName(TYPE_STRUCT, false);
    m_sTag = sTag;
    // determine size later
    m_nSize = 0;
    m_nMaxSize = 0;

    CCompiler::VerboseD(0, "CBEStructType::%s(%s,) called\n", __FUNCTION__,
	sTag.c_str());
}

/** \brief prepares this instance for the code generation
 *  \param pFEType the corresponding front-end type
 *  \return true if the code generation was successful
 */
void 
CBEStructType::CreateBackEndSequence(CFEArrayType * pFEType)
    throw (CBECreateException*)
{
    CCompiler::VerboseI(0, "CBEStructType::%s(fe-array) called\n", 
	__FUNCTION__);
    // if sequence create own members
    if (pFEType->GetType() != TYPE_ARRAY)
    {
	string exc = string(__FUNCTION__);
	exc += " failed, because array type is no array type";
	throw new CBECreateException(exc);
    }
    // CLM states that (1.11)
    // that 'sequence <type, size>' will be mapped to
    // struct {
    // unsigned long _maximum;
    // unsigned long _length;
    // type* _buffer;
    // }
    // we extend this to add the attributes max_is and length_is to buffer, so
    // the marshaller can perform range checks.

    // the member vector
    vector<CFETypedDeclarator*> *pMembers = new vector<CFETypedDeclarator*>();

    // create the _maximum member
    CBESizes *pSizes = CCompiler::GetSizes();
    int nLongSize = pSizes->GetSizeOfType(TYPE_LONG, 4);
    CFETypeSpec* pFEMType = new CFESimpleType(TYPE_INTEGER, true,
	true, nLongSize, false);
    CFEDeclarator *pFEDeclarator = new CFEDeclarator(DECL_IDENTIFIER, 
	string("_maximum"));
    vector<CFEDeclarator*> *pFEDeclarators = new vector<CFEDeclarator*>();
    pFEDeclarators->push_back(pFEDeclarator);
    CFETypedDeclarator *pFEMember = new CFETypedDeclarator(TYPEDECL_FIELD,
	pFEMType,
	pFEDeclarators);
    pFEMType->SetParent(pFEMember);
    pFEDeclarator->SetParent(pFEMember);
    pMembers->push_back(pFEMember);
    pFEDeclarators->clear();

    // create _length member
    pFEMType = new CFESimpleType(TYPE_INTEGER, true,
	true, nLongSize, false);
    pFEDeclarator = new CFEDeclarator(DECL_IDENTIFIER, string("_length"));
    pFEDeclarators->push_back(pFEDeclarator);
    pFEMember = new CFETypedDeclarator(TYPEDECL_FIELD,
	pFEMType,
	pFEDeclarators);
    pFEMType->SetParent(pFEMember);
    pFEDeclarator->SetParent(pFEMember);
    pMembers->push_back(pFEMember);
    pFEDeclarators->clear();

    // add attributes
    // attribute [max_is(_maximum)]
    vector<CFEAttribute*> *pAttributes = new vector<CFEAttribute*>();
    pFEDeclarator = new CFEDeclarator(DECL_IDENTIFIER, string("_maximum"));
    vector<CFEDeclarator*> *pAttrParams = new vector<CFEDeclarator*>();
    pAttrParams->push_back(pFEDeclarator);
    CFEAttribute *pFEAttribute = new CFEIsAttribute(ATTR_MAX_IS,
                                        pAttrParams);
    delete pAttrParams;
    pFEDeclarator->SetParent(pFEAttribute);
    pAttributes->push_back(pFEAttribute);
    // attribute [length_is(_length)]
    pFEDeclarator = new CFEDeclarator(DECL_IDENTIFIER, string("_length"));
    pAttrParams = new vector<CFEDeclarator*>();
    pAttrParams->push_back(pFEDeclarator);
    pFEAttribute = new CFEIsAttribute(ATTR_LENGTH_IS,
                                        pAttrParams);
    delete pAttrParams;
    pFEDeclarator->SetParent(pFEAttribute);
    pAttributes->push_back(pFEAttribute);
    // create the *_buffer member
    pFEMType = pFEType->GetBaseType();
    pFEDeclarator = new CFEDeclarator(DECL_IDENTIFIER, string("_buffer"), 1);
    pFEDeclarators->push_back(pFEDeclarator);
    pFEMember = new CFETypedDeclarator(TYPEDECL_FIELD,
                                        pFEMType, pFEDeclarators, pAttributes);
    pFEMType->SetParent(pFEMember);
    pFEDeclarator->SetParent(pFEMember);
    pMembers->push_back(pFEMember);
    pFEDeclarators->clear();

    // create struct
    CFEStructType *pFEStruct = new CFEStructType(string(), pMembers);
    pFEStruct->SetParent(pFEType->GetParent());
    delete pMembers;
    delete pAttributes;

    // recusively call CreateBackEnd to initialize struct
    CreateBackEnd(pFEStruct);

    CCompiler::VerboseD(0, "CBEStructType::%s(fe-array) returns\n", 
	__FUNCTION__);
}

/** \brief adds a new member
 *  \param pMember the new member to add
 */
void CBEStructType::AddMember(CBETypedDeclarator * pMember)
{
    if (!pMember)
        return;

    if (pMember->GetDeclarator() &&
	FindMember(pMember->GetDeclarator()->GetName()))
	return; // member already exists
    m_vMembers.push_back(pMember);
    pMember->SetParent(this);
}

/** \brief removes a member from the members vector
 *  \param pMember the member to remove
 */
void CBEStructType::RemoveMember(CBETypedDeclarator * pMember)
{
    if (!pMember)
        return;
    vector<CBETypedDeclarator*>::iterator iter;
    for (iter = m_vMembers.begin(); iter != m_vMembers.end(); iter++)
    {
        if (*iter == pMember)
        {
            m_vMembers.erase(iter);
            return;
        }
    }
}

/** \brief retrieves a pointer to the first member
 *  \return a pointer to the first member
 */
vector<CBETypedDeclarator*>::iterator CBEStructType::GetFirstMember()
{
    return m_vMembers.begin();
}

/** \brief retrieves reference to next member
 *  \param iter the pointer to the next member
 *  \return a reference to the member pIter points to
 */
CBETypedDeclarator *CBEStructType::GetNextMember(vector<CBETypedDeclarator*>::iterator &iter)
{
    if (iter == m_vMembers.end())
        return 0;
    return *iter++;
}

/** \brief writes the structure into the target file
 *  \param pFile the target file
 *
 * A struct looks like this:
 * <code>
 * struct &lt;tag&gt;
 * {
 *   &lt;member list&gt;
 * }
 * </code>
 */
void CBEStructType::Write(CBEFile * pFile)
{
    if (!pFile->IsOpen())
        return;

    CCompiler::Verbose(0, "%s called.\n", __FUNCTION__);
    // open struct
    *pFile << m_sName;
    if (!m_sTag.empty())
        *pFile << " " << m_sTag;
    if (GetMemberCount() > 0)
    {
        *pFile << "\n";
        *pFile << "\t{\n";
        pFile->IncIndent();
        // print members
        vector<CBETypedDeclarator*>::iterator iter = GetFirstMember();
        CBETypedDeclarator *pMember;
        while ((pMember = GetNextMember(iter)) != 0)
        {
            // this might happend (if we add return types to a struct)
            if (pMember->GetType()->IsVoid() &&
                (pMember->GetSize() == 0))
                continue;
            *pFile << "\t";
            pMember->WriteDeclaration(pFile);
            *pFile << ";\n";
        }
        // close struct
        pFile->DecIndent();
        *pFile << "\t}";
    }
    else if (!m_bForwardDeclaration)
    {
        *pFile << " { }";
    }
    CCompiler::Verbose(0, "%s returned.\n", __FUNCTION__);
}

/** \brief calculates the size of the written string
 *  \return the length of the written string
 *
 * This function is used to see how long the type of a parameter (or return type) is.
 * Thus no members are necessary, but we have to consider the tag if it exists.
 */
int CBEStructType::GetStringLength()
{
    int nSize = m_sName.length();
    if (!m_sTag.empty())
        nSize += m_sTag.length();
    return nSize;
}

/** \brief generates an exact copy of this class
 *  \return a reference to the new object
 */
CObject *CBEStructType::Clone()
{
    return new CBEStructType(*this);
}

/** \brief calculate the size of a struct type
 *  \return the size in bytes of the struct
 *
 * The struct's size is the sum of the member's size.  We test m_nSize on
 * zero. If it is greater than zero, the size has already been set. If it
 * isn't we calculate it.
 *
 * If the declarator's GetSize returns 0, the member is a bitfield. We have to
 * add the bits of the declarators. If a bitfield declarator is followed by a
 * non-bitfield declarator, the size has to be aligned to the next byte.
 *
 * \todo If member is indirect, we should add size of pointer instead of \
 *       size of type
 */
int CBEStructType::GetSize()
{
    CCompiler::VerboseI(0, "CBEStructType::%s called (m_nSize = %d)\n",
	__FUNCTION__, m_nSize);
    DTRACE("%s called (m_nSize = %d)\n", __FUNCTION__, m_nSize);
    
    if (m_nSize != 0)
    {
	CCompiler::VerboseD(0, "CBEStructType::%s returns %d\n", __FUNCTION__,
	    m_nSize);
        return m_nSize;
    }

    // if this is a tagged struct without members, we have to find the
    // original struct
    if (m_bForwardDeclaration)
    {
	CCompiler::Verbose(0, "CBEStructType::%s forward decl of %s\n",
	    __FUNCTION__, GetTag().c_str());

        // search for tag
        CBERoot *pRoot = GetSpecificParent<CBERoot>();
        assert(pRoot);
        CBEStructType *pTaggedType = 
	    (CBEStructType*)pRoot->FindTaggedType(TYPE_STRUCT, GetTag());
        // if found, marshal this instead
        if ((pTaggedType) && (pTaggedType != this))
        {
            m_nSize = pTaggedType->GetSize();
	    CCompiler::VerboseD(0, "CBEStructType::%s returns %d\n",
		__FUNCTION__, m_nSize);
            return m_nSize;
        }
        // if no tagged struct found, this is a user defined type
        // we asked the sizes class in CreateBackEnd, maybe it knows my size
        // -> m_nSize should be > 0
    }

    vector<CBETypedDeclarator*>::iterator iter = GetFirstMember();
    CBETypedDeclarator *pMember;
    int nBitSize = 0;
    while ((pMember = GetNextMember(iter)) != 0)
    {
	CCompiler::Verbose(0, "CBEStructType::%s testing member %s\n",
	    __FUNCTION__, pMember->GetDeclarator()->GetName().c_str());

        // special case handling:
        // if the member is a pointer to ourself, then we
        // handle this as a variable sized entry.
        // example:
        // struct list {
        //     struct list *prev, *next;
        //     ...
        // };
        // To catch this we test if the type of the
        // member is a tagged struct type with the same
        // tag as we have.
	// 
        // another special case:
        // typedef struct A A_t;
        // struct A {
        //   A_t *next;
        // };
        //
        // To catch this, we hav to check if the type is
        // a user defined type. If so, we get the original
        // type, which should be the struct, and then check
        // if it has the same tag.
        CBEType *pMemberType = pMember->GetType();

	CCompiler::Verbose(0, "CBEStructType::%s member %s is of type %d\n",
	    __FUNCTION__, pMember->GetDeclarator()->GetName().c_str(), 
	    pMemberType->GetFEType());
	
	while (dynamic_cast<CBEUserDefinedType*>(pMemberType))
    	    pMemberType = ((CBEUserDefinedType*)pMemberType)->GetRealType();
	assert(pMemberType);

	CCompiler::Verbose(0, "CBEStructType::%s member %s has real type %d\n",
	    __FUNCTION__, pMember->GetDeclarator()->GetName().c_str(), 
	    pMemberType->GetFEType());
	
        if ((dynamic_cast<CBEStructType*>(pMemberType)) &&
            ((CBEStructType*)pMemberType)->HasTag(m_sTag) &&
	    !m_sTag.empty())
        {
	    // FIXME: get size from
	    // CCompiler::GetSizes()->GetSizeOfType(TYPE_MWORD);
	    m_nSize = -1;

	    CCompiler::VerboseD(0, 
		"CBEStructType::%s self reference (%s): return -1\n",
		__FUNCTION__, m_sTag.c_str());
            return m_nSize;
        }

        int nSize = pMember->GetSize();
        if (pMember->IsString())
        {
            // a string is also variable sized member
            m_nSize = -1;

	    CCompiler::VerboseD(0, 
		"CBEStructType::%s string member: return -1\n", __FUNCTION__);
            return m_nSize;
        }
        else if (pMember->IsVariableSized())
        {
            // if one of the members is variable sized,
            // the whole struct is variable sized
            m_nSize = -1;

	    CCompiler::VerboseD(0, 
		"CBEStructType::%s var-sized member: return -1\n",
		__FUNCTION__);
            return -1;
        }
        else if (nSize == 0)
        {
            nBitSize += pMember->GetBitfieldSize();
        }
        else
        {
	    CBESizes *pSizes = CCompiler::GetSizes();
            // check for alignment:
	    // if current size (nSize) is 4 bytes or above then sum is aligned
	    // to dword size 
	    //
	    // if current size (nSize) is 2 bytes then sum is aligned to word
	    // size
            if (nSize >= 4)
                m_nSize = pSizes->WordRoundUp(m_nSize);
            if ((nSize == 2) && ((m_nSize % 2) > 0))
                m_nSize++; // word align
            m_nSize += nSize;
            // if bitfields before, align them and add them
            if (nBitSize > 0)
            {
                m_nSize += nBitSize / 8;
                if ((nBitSize % 8) > 0)
                    m_nSize++;
                nBitSize = 0;
            }
        }

	CCompiler::Verbose(0, 
	    "CBEStructType::%s size of member is %d, new total: %d\n",
	    __FUNCTION__, nSize, m_nSize);
    }
    // some bitfields left? -> align them and add them
    if (nBitSize > 0)
    {
        m_nSize += nBitSize / 8;
        if ((nBitSize % 8) > 0)
            m_nSize++;
    }

    CCompiler::VerboseD(0, "CBEStructType::%s returns %d\n", __FUNCTION__,
	m_nSize);
    return m_nSize;
}

/** \brief returns the maximum size of the struct
 *  \return the size in bytes
 *
 * First it calls the GetSize method. If that returns a negative value, it is
 * variable sized and a maximum size calculation is feasable. The algorithm is
 * basically the same as for GetSize, but when a member returns -1 (variable
 * sized) then this implementation tries to determine it's maximum size.
 */
int CBEStructType::GetMaxSize()
{
    CCompiler::VerboseI(0, "CBEStructType::%s called\n", __FUNCTION__);
    
    m_nMaxSize = GetSize();
    if (m_nMaxSize > 0)
    {
	CCompiler::VerboseD(0, "CBEStructType::%s normal size is %d, return\n",
	    __FUNCTION__, m_nMaxSize);
	return m_nMaxSize;
    }
    
    // if this is a tagged struct without members, we have to find the
    // original struct
    if (m_bForwardDeclaration)
    {
        // search for tag
        CBERoot *pRoot = GetSpecificParent<CBERoot>();
        assert(pRoot);
        CBEStructType *pTaggedType = 
	    (CBEStructType*)pRoot->FindTaggedType(TYPE_STRUCT, GetTag());
        // if found, marshal this instead
        if ((pTaggedType) && (pTaggedType != this))
        {
            m_nMaxSize = pTaggedType->GetMaxSize();
	    CCompiler::VerboseD(0, 
		"CBEStructType::%s size of forward decl struct is %d\n",
		__FUNCTION__, m_nMaxSize);
            return m_nMaxSize;
        }
        // if no tagged struct found, this is a user defined type
        // we asked the sizes class in CreateBackEnd, maybe it knows my size
        // -> m_nMaxSize should be > 0
    }

    CBESizes *pSizes = CCompiler::GetSizes();

    vector<CBETypedDeclarator*>::iterator iter = GetFirstMember();
    CBETypedDeclarator *pMember;
    int nBitSize = 0;
    // reset m_nMaxSize, which could have been negative before
    m_nMaxSize = 0;
    while ((pMember = GetNextMember(iter)) != 0)
    {
	CCompiler::Verbose(0, "CBEStructType::%s testing member %s\n",
	    __FUNCTION__, pMember->GetDeclarator()->GetName().c_str());
        // special case handling:
        // if the member is a pointer to ourself, then we
        // handle this as a variable sized entry.
        // example:
        // struct list {
        //     struct list *prev, *next;
        //     ...
        // };
        // To catch this we test if the type of the
        // member is a tagged struct type with the same
        // tag as we have.
	// 
        // another special case:
        // typedef struct A A_t;
        // struct A {
        //   A_t *next;
        // };
        //
        // To catch this, we hav to check if the type is
        // a user defined type. If so, we get the original
        // type, which should be the struct, and then check
        // if it has the same tag.
        CBEType *pMemberType = pMember->GetType();

	CCompiler::Verbose(0, "CBEStructType::%s member %s is of type %d\n",
	    __FUNCTION__, pMember->GetDeclarator()->GetName().c_str(), 
	    pMemberType->GetFEType());
	
	while (dynamic_cast<CBEUserDefinedType*>(pMemberType))
    	    pMemberType = ((CBEUserDefinedType*)pMemberType)->GetRealType();
	assert(pMemberType);

	CCompiler::Verbose(0, "CBEStructType::%s member %s has real type %d\n",
	    __FUNCTION__, pMember->GetDeclarator()->GetName().c_str(), 
	    pMemberType->GetFEType());
	
        if ((dynamic_cast<CBEStructType*>(pMemberType)) &&
            ((CBEStructType*)pMemberType)->HasTag(m_sTag))
        {
            m_nMaxSize = pSizes->GetSizeOfType(TYPE_VOID_ASTERISK);
	    CCompiler::VerboseD(0, 
		"CBEStructType::%s pointer to self, return %d\n",
		__FUNCTION__, m_nMaxSize);
            return m_nMaxSize;
        }

        int nSize = 0;
	if (!pMember->GetMaxSize(true, nSize))
	{
	    m_nMaxSize = -1;
	    CCompiler::VerboseD(0, 
		"CBEStructType::%s can't determine max of member, return -1\n",
		__FUNCTION__);
	    return -1;
	}
	else if (nSize == 0)
        {
            nBitSize += pMember->GetBitfieldSize();
        }
        else
        {
            // check for alignment:
	    // if current size (nSize) is 4 bytes or above then sum is aligned
	    // to dword size 
	    //
	    // if current size (nSize) is 2 bytes then sum is aligned to word
	    // size
	    int nWordSize = pSizes->GetSizeOfType(TYPE_MWORD);
            if (nSize >= nWordSize)
		m_nMaxSize = pSizes->WordRoundUp(m_nMaxSize);
            m_nMaxSize += nSize;
            // if bitfields before, align them and add them
            if (nBitSize > 0)
            {
                m_nMaxSize += nBitSize / 8;
                if ((nBitSize % 8) > 0)
                    m_nMaxSize++;
                nBitSize = 0;
            }
        }

	CCompiler::Verbose(0, 
	    "CBEStructType::%s: member %s has size %d (bits %d) - total: %d\n", 
	    __FUNCTION__, pMember->GetDeclarator()->GetName().c_str(),
	    nSize, nBitSize, m_nMaxSize);
    }
    // some bitfields left? -> align them and add them
    if (nBitSize > 0)
    {
        m_nMaxSize += nBitSize / 8;
        if ((nBitSize % 8) > 0)
            m_nMaxSize++;
    }

    CCompiler::VerboseD(0, "CBEStructType::%s return max size %d\n",
	__FUNCTION__, m_nMaxSize);
    return m_nMaxSize;
}

/** \brief writes code to initialize a variable of this type with a zero value
 *  \param pFile the file to write to
 *
 * A struct is usually initialized by writing a init value for all its members
 * in a comma seperated list, embraced by braces. E.g. { (CORBA_long)0,
 * (CORBA_float)0 }
 */
void CBEStructType::WriteZeroInit(CBEFile * pFile)
{
    *pFile << "{ ";
    pFile->IncIndent();
    vector<CBETypedDeclarator*>::iterator iter = GetFirstMember();
    CBETypedDeclarator *pMember;
    bool bComma = false;
    while ((pMember = GetNextMember(iter)) != 0)
    {
        // get type
        CBEType *pType = pMember->GetType();
        // get declarator
        vector<CBEDeclarator*>::iterator iterD = pMember->GetFirstDeclarator();
        CBEDeclarator *pDecl;
        while ((pDecl = pMember->GetNextDeclarator(iterD)) != 0)
        {
            if (bComma)
                *pFile << ", \n\t";
            // be C99 compliant:
	    *pFile << pDecl->GetName() << " : ";
            if (pDecl->IsArray())
                WriteZeroInitArray(pFile, pType, pDecl,
		    pDecl->GetFirstArrayBound());
            else if (pType)
                pType->WriteZeroInit(pFile);

            bComma = true;
        }
    }
    pFile->DecIndent();
    *pFile << " }";
}

/** \brief checks if this is a constructed type
 *  \return true, because this is a constructed type
 */
bool CBEStructType::IsConstructedType()
{
    return true;
}

/** \brief counts the members of the struct
 *  \return the number of members
 */
int CBEStructType::GetMemberCount()
{
    return m_vMembers.size();
}

/** \brief tests if this type has the given tag
 *  \param sTag the tag to check
 *  \return true if the same
 */
bool CBEStructType::HasTag(string sTag)
{
    return (m_sTag == sTag);
}

/** \brief writes a cast of this type
 *  \param pFile the file to write to
 *  \param bPointer true if the cast should produce a pointer
 *
 * A struct cast is '(struct tag)'.
 */
void CBEStructType::WriteCast(CBEFile * pFile, bool bPointer)
{
    *pFile << "(";
    if (m_sTag.empty())
    {
        // no tag -> we need a typedef to save us
        // the alias can be used for the cast
        CBETypedef *pTypedef = GetTypedef();
        assert(pTypedef);
        // get first declarator (without stars)
        vector<CBEDeclarator*>::iterator iterD = pTypedef->GetFirstDeclarator();
        CBEDeclarator *pDecl;
        while ((pDecl = pTypedef->GetNextDeclarator(iterD)) != 0)
        {
            if (pDecl->GetStars() <= (bPointer?1:0))
                break;
        }
        assert(pDecl);
	*pFile << pDecl->GetName();
        if (bPointer && (pDecl->GetStars() == 0))
	    *pFile << "*";
    }
    else
    {
	*pFile << m_sName << " " << m_sTag;
        if (bPointer)
	    *pFile << "*";
    }
    *pFile << ")";
}

/** \brief allows to access tag member
 *  \return a copy of the tag
 */
string CBEStructType::GetTag()
{
    return m_sTag;
}

/** \brief write the declaration of this type
 *  \param pFile the file to write to
 *
 * Only write a 'struct &lt;tag&gt;'.
 */
void CBEStructType::WriteDeclaration(CBEFile * pFile)
{
    *pFile << m_sName;
    if (!m_sTag.empty())
	*pFile << " " << m_sTag;
}

/** \brief if struct is variable size, it has to write the size
 *  \param pFile the file to write to
 *  \param pStack contains the declarator stack for constructed typed variables
 *
 * This is usually the sum of the member's sizes. Because this is only called
 * when the struct is variable sized, we have to first add all the fixed sized
 * members, use this number plus the variable sized members.
 *
 * This might also get called for a [ref, prealloc] parameter, so first check
 * if we really are variable sized. If not and we have a tag:
 * sizeof(struct tag).
 */
void CBEStructType::WriteGetSize(CBEFile * pFile,
    vector<CDeclaratorStackLocation*> *pStack,
    CBEFunction *pUsingFunc)
{
    /* check for variable sized members */
    bool bVarSized = false;
    vector<CBETypedDeclarator*>::iterator iter = GetFirstMember();
    CBETypedDeclarator *pMember;
    while (!bVarSized && ((pMember = GetNextMember(iter)) != 0))
    {
        if (pMember->IsVariableSized() ||
            pMember->IsString())
            bVarSized = true;
    }
    if (!bVarSized && !m_sTag.empty())
    {
        *pFile << "sizeof(struct " << m_sTag << ")";
        return;
    }

    int nFixedSize = GetFixedSize();
    bool bFirst = true;
    if (nFixedSize > 0)
    {
	*pFile << nFixedSize;
        bFirst = false;
    }
    iter = GetFirstMember();
    while ((pMember = GetNextMember(iter)) != 0)
    {
        if (!pMember->IsVariableSized() &&
            !pMember->IsString())
            continue;
        if (!bFirst)
	    *pFile << "+";
        bFirst = false;
        WriteGetMemberSize(pFile, pMember, pStack, pUsingFunc);
    }
}

/** \brief calculates the size of all fixed sized members
 *  \return the sum of the fixed-sized member's sizes in bytes
 */
int CBEStructType::GetFixedSize()
{
    int nSize = 0;

    // if this is a tagged struct without members,
    // we have to find the original struct
    if (m_bForwardDeclaration)
    {
        // search for tag
        CBERoot *pRoot = GetSpecificParent<CBERoot>();
        assert(pRoot);
        CBEStructType *pTaggedType = (CBEStructType*)pRoot->FindTaggedType(TYPE_STRUCT, GetTag());
        // if found, marshal this instead
        if ((pTaggedType) && (pTaggedType != this))
        {
            nSize = pTaggedType->GetSize();
            return nSize;
        }
    }

    vector<CBETypedDeclarator*>::iterator iter = GetFirstMember();
    CBETypedDeclarator *pMember;
    int nBitSize = 0;
    while ((pMember = GetNextMember(iter)) != 0)
    {
        int nMemberSize = pMember->GetSize();
        if (pMember->IsString() ||
            pMember->IsVariableSized())
        {
            // its of variable size
            // if bitfields before, align them and add them
            if (nBitSize > 0)
            {
                nSize += nBitSize / 8;
                if ((nBitSize % 8) > 0)
                    nSize++;
                nBitSize = 0;
            }
        }
        else if (nMemberSize == 0)
        {
            nBitSize += pMember->GetBitfieldSize();
        }
        else
        {
	    CBESizes *pSizes = CCompiler::GetSizes();
            // check for alignment:
	    // if current size (nSize) is 4 bytes or above then sum is aligned
	    // to dword size
	    //
	    // if current size (nSize) is 2 bytes then sum is aligned to word
	    // size
	    if (nMemberSize >= 4)
                nSize = pSizes->WordRoundUp(nSize); // dword align
            if ((nMemberSize == 2) && ((nSize % 2) > 0))
                nSize++; // word align
            nSize += nMemberSize;
            // if bitfields before, align them and add them
            if (nBitSize > 0)
            {
                nSize += nBitSize / 8;
                if ((nBitSize % 8) > 0)
                    nSize++;
                nBitSize = 0;
            }
        }
    }
    // some bitfields left? -> align them and add them
    if (nBitSize > 0)
    {
        nSize += nBitSize / 8;
        if ((nBitSize % 8) > 0)
            nSize++;
    }
    return nSize;
}

/** \brief writes the whole size string for a member
 *  \param pFile the file to write to
 *  \param pMember the member to write for
 *  \param pStack contains the declarator stack
 *
 * This code is taken from
 * CBEMsgBufferType::WriteInitializationVarSizedParameters if something is not
 * working, check if something changed there as well.
 */
void
CBEStructType::WriteGetMemberSize(CBEFile *pFile,
    CBETypedDeclarator *pMember,
    vector<CDeclaratorStackLocation*> *pStack,
    CBEFunction *pUsingFunc)
{
    bool bFirst = true;
    vector<CBEDeclarator*>::iterator iterD = pMember->GetFirstDeclarator();
    CBEDeclarator *pDecl;
    while ((pDecl = pMember->GetNextDeclarator(iterD)) != 0)
    {
        if (!bFirst)
        {
	    *pFile << "+";
            bFirst = false;
        }
        // add the current decl to the stack
        CDeclaratorStackLocation *pLoc = new CDeclaratorStackLocation(pDecl);
        pStack->push_back(pLoc);
        // add the member of the struct to the stack
        pMember->WriteGetSize(pFile, pStack, pUsingFunc);
        if ((pMember->GetType()->GetSize() > 1) && !(pMember->IsString()))
        {
	    *pFile << "*sizeof";
            pMember->GetType()->WriteCast(pFile, false);
        }
        else if (pMember->IsString())
        {
            // add terminating zero
	    *pFile << "+1";
            bool bHasSizeAttr = pMember->HasSizeAttr(ATTR_SIZE_IS) ||
                    pMember->HasSizeAttr(ATTR_LENGTH_IS) ||
                    pMember->HasSizeAttr(ATTR_MAX_IS);
	    CBESizes *pSizes = CCompiler::GetSizes();
            if (!bHasSizeAttr)
		*pFile << "+" << pSizes->GetSizeOfType(TYPE_INTEGER);
        }
        // remove the decl from the stack
        pStack->pop_back();
        delete pLoc;
    }
}

/** \brief test if this is a simple type
 *  \return false
 */
bool CBEStructType::IsSimpleType()
{
    return false;
}

/** \brief tries to find a member with the name sName
 *  \param sName the name of the member to search
 *  \return the member found or 0 if not found
 */
CBETypedDeclarator* CBEStructType::FindMember(string sName)
{
    CCompiler::Verbose(0, "CBEStructType::%s(%s) called\n", __FUNCTION__,
	sName.c_str());

    vector<CBETypedDeclarator*>::iterator iter = GetFirstMember();
    CBETypedDeclarator *pMember;
    while ((pMember = GetNextMember(iter)) != 0)
    {
	CCompiler::Verbose(0, "CBEStructType::%s compare to member %s\n",
	    __FUNCTION__, pMember->GetDeclarator()->GetName().c_str());

        if (pMember->FindDeclarator(sName))
            return pMember;
    }

    CCompiler::Verbose(0, "CBEStructType::%s returns NULL\n", __FUNCTION__);
    return 0;
}

/** \brief tries to find a member with a declarator stack
 *  \param pStack contains the list of members to search for
 *  \param iCurr the iterator pointing to the currently searched element
 *  \return the member found or 0 if not found
 *
 * Gets the first element on the stack and tries to find
 */
CBETypedDeclarator* 
CBEStructType::FindMember(vector<CDeclaratorStackLocation*> *pStack,
    vector<CDeclaratorStackLocation*>::iterator iCurr)
{
    // if at end, return
    if (iCurr == pStack->end())
	return 0;

    CCompiler::Verbose(0, "CBEStructType::%s(stack) called\n", __FUNCTION__);

    DUMP_STACK(i, pStack, "Find");
    // try to find member for current declarator
    string sName = (*iCurr)->pDeclarator->GetName();
    CBETypedDeclarator *pMember = FindMember(sName);
    if (!pMember)
	return pMember;

    // no more elements in stack, we are finished
    if (++iCurr == pStack->end())
	return pMember;

    // check member types
    CBEType *pType = pMember->GetType();
    while (dynamic_cast<CBEUserDefinedType*>(pType))
	pType = dynamic_cast<CBEUserDefinedType*>(pType);

    CBEStructType *pStruct = dynamic_cast<CBEStructType*>(pType);
    if (pStruct)
	return pStruct->FindMember(pStack, iCurr);

    CBEUnionType *pUnion = dynamic_cast<CBEUnionType*>(pType);
    if (pUnion)
	return pUnion->FindMember(pStack, iCurr);

    CCompiler::Verbose(0, 
	"CBEStructType::%s no member in struct or union found. return 0.\n",
	__FUNCTION__);
    return 0;
}

/** \brief tries to find a member with a specific attribute
 *  \param nAttributeType the attribute type to look for
 *  \return the first member with the given attribute
 */
CBETypedDeclarator* CBEStructType::FindMemberAttribute(int nAttributeType)
{
    vector<CBETypedDeclarator*>::iterator iter = GetFirstMember();
    CBETypedDeclarator *pMember;
    while ((pMember = GetNextMember(iter)) != 0)
    {
        if (pMember->FindAttribute(nAttributeType))
            return pMember;
    }
    return 0;
}

/** \brief tries to find a member with a specific IS attribute
 *  \param nAttributeType the attribute type to look for
 *  \param sAttributeParameter the name of the attributes parameter to look for
 *  \return the first member with the given attribute
 */
CBETypedDeclarator* 
CBEStructType::FindMemberIsAttribute(int nAttributeType, 
	string sAttributeParameter)
{
    vector<CBETypedDeclarator*>::iterator iter = GetFirstMember();
    CBETypedDeclarator *pMember;
    while ((pMember = GetNextMember(iter)) != 0)
    {
        CBEAttribute *pAttr = pMember->FindAttribute(nAttributeType);
        if (pAttr && pAttr->FindIsParameter(sAttributeParameter))
            return pMember;
    }
    return 0;
}

/** \brief moves a member in the struct
 *  \param sName the name of the member
 *  \param nPos the new position in the struct
 *  \return true if successful
 *
 * If nPos is -1 this means the end of the struct.
 * If nPos is 0 this means the begin of the struct.
 * nPos is regarded to be zero-based.
 * Otherwise the member is extracted from the struct
 * and inserted before the old member with the number nPos.
 * If the number nPos is larger than the struct size,
 * an error is returned.
 */
bool
CBEStructType::MoveMember(string sName, int nPos)
{
    CBETypedDeclarator *pMember = FindMember(sName);
    if (!pMember)
	return false;

    RemoveMember(pMember);
    if (nPos == -1)
	m_vMembers.push_back(pMember);

    // get position to insert at
    vector<CBETypedDeclarator*>::iterator iter = m_vMembers.begin();
    while ((nPos-- > 0) && (iter != m_vMembers.end())) iter++;
    // check if nPos was too big
    if ((nPos > 0) && (iter == m_vMembers.end()))
    {
	AddMember(pMember);
	return false;
    }
    // insert member
    m_vMembers.insert(iter, pMember);

    return true;
}

/** \brief moves a member in the struct
 *  \param sName the name of the member to move
 *  \param sBeforeHere the name of the member to move before
 *  \return true if successful
 */
bool
CBEStructType::MoveMember(string sName, string sBeforeHere)
{
    CBETypedDeclarator *pMember = FindMember(sName);
    if (!pMember)
	return false;
    if (sName == sBeforeHere)
	return false;

    RemoveMember(pMember);
    // get position of BeforeHere
    vector<CBETypedDeclarator*>::iterator iter = m_vMembers.begin();
    while ((iter != m_vMembers.end()) && 
	    (!(*iter)->FindDeclarator(sBeforeHere))) iter++;
    // check if BeforeHere was member of this struct
    if (iter == m_vMembers.end())
    {
	AddMember(pMember);
	return false;
    }
    // move member
    m_vMembers.insert(iter, pMember);
    
    return true;
}

